import os, datetime, time
from arcnagios import arcutils, vomsutils
from arcnagios.nagutils import NagiosPlugin, OK, CRITICAL, UNKNOWN, \
			       ServiceCRITICAL
from arcnagios.confargparse import UsageError
try:
    from subprocess import CalledProcessError
except ImportError:
    CalledProcessError = OSError

# Backward compatibility.
if not hasattr(__builtins__, 'any'):
    from arcnagios.compat import any

class GridStorageProbe(NagiosPlugin, vomsutils.NagiosPluginVomsMixin):

    main_config_section = 'gridstorage'

    def __init__(self):
	super(GridStorageProbe, self).__init__()
	ap = self.argparser.add_argument_group('Probe-Specific Options')
	ap.add_argument('--url', dest = 'url',
		help = 'The remote URL on which to perform the tests.')
	ap.add_argument('--write-url', dest = 'write_url',
		help = 'The URL for the initial write operation if different '
		       'from the other URLs.  This is primarily used for LFC.')
	ap.add_argument('--dir-url', dest = 'dir_url',
		help = 'The URL of the directory holding the test file. '
		       'A file name including the host name and a time stamp '
		       'will be appended. This option is only useful if you '
		       'enable write operations, and it will not work '
		       'correctly for LFC.')
	ap.add_argument('--disable-read', dest = 'enable_read',
		default = True, action = 'store_false',
		help = 'Disable the read check on the the URL.')
	ap.add_argument('--disable-list', dest = 'enable_list',
		default = True, action = 'store_false',
		help = 'Disable the list check on the the URL.')
	ap.add_argument('--list-dir', dest = 'list_dir',
		default = False, action = 'store_true',
		help = 'List the URL of the directory containing the file '
		       'rather than the file itself.  This will use '
		       '--dir-url if provided, otherwise it will use --url '
		       'after stripping the last component.')
	ap.add_argument('--enable-write', dest = 'enable_write',
		default = False, action = 'store_true',
		help = 'Enable write and delete operations on the url. '
		       'If enabled a file with a fairly unique content will '
		       'be written before any list and read operations, '
		       'and the file will be deleted after.')
	ap.add_argument('-t', '--timeout', dest = 'timeout',
		type = int, default = 120,
		help = 'Timeout.  This is divided among the sub-tasks, so '
		       'individual operations will get shorter times to '
		       'complete.  The minimum value is 5 seconds.')

    def parse_args(self, args):
	NagiosPlugin.parse_args(self, args)
	if not self.opts.url and not self.opts.dir_url:
	    raise UsageError('You must provide either a --dir-url or a --url.')

    def next_timeout(self, which):
	assert self.time_slots > 0
	t = (self.time_limit - time.time()) / self.time_slots - self.grace_time
	self.time_slots -= 1
	if t < 1:
	    if self.clean_url and self.time_slots > 0:
		timeout = self.time_limit - time.time()
		arcutils.arcrm(self.clean_url, log = self.log,
			       timeout = timeout)
	    raise ServiceCRITICAL('Insufficient time for %s.'%which)
	return int(t)

    def check(self):
	# Allocate up to 4 time slots for the main subtasks, and reserve a
	# fraction of the total time for the script itself and fork/exec
	# overhead.
	self.time_limit = time.time() + self.opts.timeout
	self.time_slots = \
		(self.opts.enable_read and 2 or 0) + \
		(self.opts.enable_write and 1 or 0) + \
		(self.opts.enable_list and 1 or 0)
	self.grace_time = self.opts.timeout * 0.02
	self.clean_url = None

	host = self.opts.host or 'localhost'
	timestamp = datetime.datetime.now().strftime('%Y%m%dT%H%M%S')

	self.require_voms_proxy()

	if self.opts.url:
	    url = self.opts.url
	    filename = os.path.basename(url)
	else:
	    filename = '%s_%s.data'%(host, timestamp)
	    url = os.path.join(self.opts.dir_url, filename)

	self.log.info('Performing checks on %s'%url)
	failed_ops = []
	if self.opts.enable_write:
	    timeout = self.next_timeout('write')
	    uploaded_contents = 'Created by check_gridstorage %s for %s.\n' \
		    % (timestamp, self.opts.host or 'localhost')
	    write_url = self.opts.write_url or url
	    try:
		fd, outbound = self.mkstemp(prefix = 'outbound')
		fh = os.fdopen(fd, 'w')
		fh.write(uploaded_contents)
		fh.close()
	    except OSError, xc:
		self.log.error('%s'%xc)
		self.nagios_exit(UNKNOWN, 'Could not create test file.')
	    try:
		arcutils.arccp(outbound, write_url, log = self.log,
			       timeout = timeout)
		self.clean_url = write_url
		self.log.info('Uploaded file.')
	    except CalledProcessError, xc:
		self.log.error('Upload failed: %s'%xc)
		failed_ops.append('upload')
	else:
	    uploaded_contents = None

	if self.opts.enable_list:
	    timeout = self.next_timeout('list')
	    if self.opts.list_dir:
		list_url = self.opts.dir_url or os.path.dirname(url)
	    else:
		list_url = url
	    try:
		listing = arcutils.arcls(list_url, log = self.log,
					 timeout = timeout)
		self.log.info('Listing contains %d entries.'%len(listing))
		if not any(os.path.basename(ent.filename) == filename
			   for ent in listing):
		    self.log.error('Did not find %s in listing.'%filename)
		    failed_ops.append('list')
	    except CalledProcessError, xc:
		self.log.error('Listing failed: %s'%xc)
		failed_ops.append('list')

	fetched_contents = None
	if self.opts.enable_read:
	    timeout = self.next_timeout('read')
	    inbound = self.mktemp(prefix = 'inbound')
	    try:
		arcutils.arccp(url, inbound, log = self.log, timeout = timeout)
		self.remove_at_exit(inbound)
		try:
		    fh = open(inbound)
		    fetched_contents = fh.read()
		    fh.close()
		    self.log.info('Fetched file.')
		except OSError, xc:
		    self.log.error('Could not open fetched file %s.'%xc)
		    failed_ops.append('open-read')
	    except CalledProcessError, xc:
		self.log.error('Fetch failed: %s'%xc)
		failed_ops.append('fetch')

	if uploaded_contents and fetched_contents:
	    if fetched_contents != uploaded_contents:
		self.log.error('Mismatched content in fetched file.')
		failed_ops.append('verify')
	    else:
		self.log.info('Verified content of fetched file.')

	if self.opts.enable_write:
	    timeout = self.next_timeout('remove')
	    try:
		arcutils.arcrm(url, log = self.log, timeout = timeout)
		self.log.info('Removed file.')
	    except CalledProcessError, xc:
		self.log.error('Removal failed: %s'%xc)
		failed_ops.append('remove')
	if failed_ops:
	    self.nagios_exit(CRITICAL, 'Failed to %s.'%', '.join(failed_ops))
	else:
	    self.nagios_exit(OK, 'Storage tests succeeded.')
