/* 
 * Copyright 1999,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.log4j.appender;

import java.util.Locale;
import java.util.zip.Deflater;

import org.apache.log4j.helpers.SynchronizedCountingQuietWriter;

/**
 * Configuration properties used by the {@link TimeAndSizeRollingAppender}.
 * 
 * @author <a href="mailto:simon_park_mail AT yahoo DOT co DOT uk">Simon Park</a>
 * @version 1.12
 */
final class AppenderRollingProperties {

  /** The default maximum file size is 10MB. */
  private long maxFileSize = 10 * 1024 * 1024;

  /**
   * Maximum number of files that will be accumulated before the appender begins
   * removing the oldest files.
   */
  private int maxRollFileCount = 10;

  /**
   * The date pattern. By default, the pattern is set to "'.'yyyy-MM-dd" meaning
   * daily rollover.
   */
  private String datePattern = "'.'yyyy-MM-dd";

  private Locale datePatternLocale = null;

  private boolean dateRollEnforced = false;

  /**
   * Default scavenge interval is 30 seconds.
   */
  private long scavengeInterval = 30 * 1000;

  /**
   * Default disk space that must not be used for log files.
   */
  private long minFreeDiscSpace = -1;

  /**
   * The name of the compression algorithm to use. Defaults to none (empty
   * string).
   */
  private String compressionAlgorithm = "";

  /**
   * The compression level used by the ZIP algorithm. Defaults to
   * {@link Deflater#BEST_SPEED}.
   */
  private int compressionLevel = Deflater.BEST_SPEED;

  /**
   * Default interval between checks for backup files due for compression is 1
   * second.
   */
  private long compressionMaxWait = 1 * 1000;

  /**
   * Default number of backup files that must be in the queue before any
   * compression takes place.
   */
  private int compressionMinQueueSize = 0;

  private boolean compressionBlocking = false;

  /**
   * The writer used by the appender that allows counting of bytes written.
   */
  private SynchronizedCountingQuietWriter countingQuietWriter = null;

  /**
   * The next time (UTC) a rollover is due, in milliseconds, or 0 if none has
   * yet been recorded. <em>Not</em> a configuration property.
   */
  private long nextRollOverTime = 0L;

  /**
   * Indicates whether the appender should roll upon receiving the first log
   * event after being activated, regardless of time or file size.
   */
  private boolean rollOnStartup = false;

  AppenderRollingProperties() {
    super();
  }

  final String getDatePattern() {
    return (this.datePattern != null) ? this.datePattern : "";
  }

  final void setDatePattern(final String datePattern) {
    this.datePattern = datePattern;
  }

  final Locale getDatePatternLocale() {
    return (this.datePatternLocale != null) ? this.datePatternLocale
        : Locale.ENGLISH;
  }

  final void setDatePatternLocale(Locale datePatternLocale) {
    this.datePatternLocale = datePatternLocale;
  }

  final boolean shouldRollOnStartup() {
    return this.rollOnStartup;
  }

  final void setRollOnStartup(final boolean rollOnStartup) {
    this.rollOnStartup = rollOnStartup;
  }

  final SynchronizedCountingQuietWriter getCountingQuietWriter() {
    return this.countingQuietWriter;
  }

  final void setCountingQuietWriter(
      final SynchronizedCountingQuietWriter countingQuietWriter) {
    this.countingQuietWriter = countingQuietWriter;
  }

  // --- synchronized methods follow
  
  synchronized final long getMaxFileSize() {
    return this.maxFileSize;
  }

  synchronized final void setMaxFileSize(final long maxFileSize) {
    this.maxFileSize = maxFileSize;
  }

  synchronized final int getMaxRollFileCount() {
    return this.maxRollFileCount;
  }

  synchronized final void setMaxRollFileCount(final int maxRollFileCount) {
    this.maxRollFileCount = maxRollFileCount;
  }

  synchronized final void setMinFreeDiscSpace(long minFreeDiskSpace) {
    this.minFreeDiscSpace = minFreeDiskSpace;
  }

  synchronized final long getMinFreeDiscSpace() {
    return this.minFreeDiscSpace;
  }

  /**
   * @return The number of bytes accounted for by the internal writer.
   */
  synchronized final long getBytesWrittenCount() {
    return (this.countingQuietWriter != null) ? this.countingQuietWriter
        .getCount() : 0L;
  }

  /**
   * @return the scavengeInterval
   */
  synchronized final long getScavengeInterval() {
    return this.scavengeInterval;
  }

  /**
   * @param byteCount
   *                The number of bytes already written to a file.
   */
  synchronized final void setBytesWrittenCount(final long byteCount) {
    if (this.countingQuietWriter != null) {
      this.countingQuietWriter.setCount(byteCount);
    }
  }

  /**
   * @param scavengeInterval
   *                the scavengeInterval to set
   */
  synchronized final void setScavengeInterval(final long scavengeInterval) {
    if (scavengeInterval >= -1) {
      this.scavengeInterval = scavengeInterval;
    }
  }

  synchronized final String getCompressionAlgorithm() {
    return compressionAlgorithm;
  }

  synchronized final void setCompressionAlgorithm(
      final String compressionAlgorithm) {
    this.compressionAlgorithm = (compressionAlgorithm != null) ? compressionAlgorithm
        .trim()
        : "";
  }

  synchronized final long getCompressionMaxWait() {
    return compressionMaxWait;
  }

  synchronized final void setCompressionMaxWait(final long compressionMaxWait) {
    if (compressionMaxWait >= 0) {
      this.compressionMaxWait = compressionMaxWait;
    }
  }

  synchronized final int getCompressionLevel() {
    return compressionLevel;
  }

  synchronized final void setCompressionLevel(final int compressionLevel) {
    if ((compressionLevel >= Deflater.DEFAULT_COMPRESSION)
        && (compressionLevel <= Deflater.BEST_COMPRESSION)) {
      this.compressionLevel = compressionLevel;
    }
  }

  synchronized final boolean isCompressionBlocking() {
    return compressionBlocking;
  }

  synchronized final void setCompressionBlocking(final boolean compressionBlocks) {
    this.compressionBlocking = compressionBlocks;
  }

  synchronized final int getCompressionMinQueueSize() {
    return compressionMinQueueSize;
  }

  synchronized final void setCompressionMinQueueSize(
      final int compressionMinQueueSize) {
    if (compressionMinQueueSize >= 0) {
      this.compressionMinQueueSize = compressionMinQueueSize;
    }
  }

  synchronized final long getNextRollOverTime() {
    return nextRollOverTime;
  }

  synchronized final void setNextRollOverTime(long nextRollOverTime) {
    this.nextRollOverTime = nextRollOverTime;
    this.notifyAll();
  }

  synchronized final boolean isDateRollEnforced() {
    return dateRollEnforced;
  }

  synchronized final void setDateRollEnforced(boolean dateRollEnforced) {
    this.dateRollEnforced = dateRollEnforced;
  }
}
