
/*****************************************************************************
 *****************************************************************************
 Copyright (c) 1999-2001, Intel Corporation 

 All rights reserved.

 Redistribution and use in source and binary forms, with or without 
 modification, are permitted provided that the following conditions are met:

 1. Redistributions of source code must retain the above copyright notice, 
 this list of conditions and the following disclaimer.

 2. Redistributions in binary form must reproduce the above copyright notice,
 this list of conditions and the following disclaimer in the documentation 
 and/or other materials provided with the distribution.

 3. Neither the name of Intel Corporation nor the names of its contributors 
 may be used to endorse or promote products derived from this software 
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 

 *****************************************************************************
****************************************************************************/

/**********************************************************************
*                                                                       *
* INTEL CORPORATION                                                     *
*                                                                       *
* This software is supplied under the terms of the license included     *
* above.  All use of this driver must be in accordance with the terms   *
* of that license.                                                      *
*                                                                       *
* Module Name:  e100_proc.c                                             *
*                                                                       *
* Abstract:     Functions to handle the proc file system.               *
*               Create the proc directories and files and run read and  *
*               write requests from the user                            *
*                                                                       *
* Environment:  This file is intended to be specific to the Linux       *
*               operating system.                                       *
*                                                                       *
**********************************************************************/


#include "e100_proc.h"

/***************************************************************************/
/*       /proc File System Interaface Support Functions                    */
/***************************************************************************/


#ifdef CONFIG_PROC_FS
static struct proc_dir_entry *adapters_proc_dir = 0;

/* externs from e100_main.c */
extern const char *e100_short_driver_name;
extern const char *e100_version;
extern struct net_device_stats *e100_get_stats(device_t * dev);
extern char *e100_GetBrandingMesg(bd_config_t * bdp);
extern void e100_MdiWrite (bd_config_t *, uint32_t, uint32_t, uint16_t);

static void e100_blink_timer(device_t * dev);
static void remove_e100_proc_subdir(device_t * dev);
static void e100_proc_cleanup();
static boolean_t e100_init_proc_dir(void);

#define WRITE_BUF_MAX_LEN 20  

/* flags */
#define PROC_INFO     0x0001
#define PROC_NEW_LINE 0x0002

typedef struct _e100_proc_entry
{
    char          name[256];
    read_proc_t  *read_proc;
    write_proc_t *write_proc;
    uint16_t      flags;
} e100_proc_entry;


static int generic_read(char *page, char **start, off_t off, int count, int *eof)
{
    int len;

    len = strlen(page);
    page[len++] = '\n';

    if (len <= off + count)
        *eof = 1;
    *start = page + off;
    len -= off;
    if (len > count)
        len = count;
    if (len < 0)
        len = 0;
    return len;
}

static int read_ulong(char *page, char **start, off_t off, 
    int count, int *eof, unsigned long l)
{
    sprintf(page, "%lu", l);

    return generic_read(page, start, off, count, eof);
}

static int read_ulong_hex(char *page, char **start, off_t off, 
    int count, int *eof, unsigned long l)
{
    sprintf(page, "0x%04lx", l);

    return generic_read(page, start, off, count, eof);
}

static int read_generic_0(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    return read_ulong(page, start, off, count, eof, 
        (unsigned long)0); 
}

static int read_hwaddr(char *page, char **start, off_t off,
    int count, int *eof, unsigned char *hwaddr)
{
    sprintf(page, "%02X:%02X:%02X:%02X:%02X:%02X",
        hwaddr[0], hwaddr[1], hwaddr[2], 
        hwaddr[3], hwaddr[4], hwaddr[5]);

    return generic_read(page, start, off, count, eof);
}


static int read_descr(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;
    char *msg;
        
    msg=e100_GetBrandingMesg(bdp);
    sprintf(page, "%s", msg);

    return generic_read(page, start, off, count, eof);
}

static int read_drvr_name(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    sprintf(page, "%s", e100_short_driver_name);

    return generic_read(page, start, off, count, eof);
}

static int read_drvr_ver(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    sprintf(page, "%s", e100_version);

    return generic_read(page, start, off, count, eof);
}

static int read_pci_vendor(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong_hex(page, start, off, count, eof, 
        (unsigned long) bdp->ppci_dev->vendor);
}
        
static int read_pci_device(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong_hex(page, start, off, count, eof, 
        (unsigned long) bdp->ppci_dev->device);
}
        
static int read_pci_sub_vendor(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong_hex(page, start, off, count, eof, 
        (unsigned long) bdp->sub_ven_id);
}
        
static int read_pci_sub_device(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong_hex(page, start, off, count, eof, 
        (unsigned long) bdp->sub_dev_id);
}
        
static int read_pci_revision(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong_hex(page, start, off, count, eof, 
        (unsigned long) bdp->rev_id);
}

static int read_pci_bus(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof,
        (unsigned long) (bdp->ppci_dev->bus->number));
}

static int read_pci_slot(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof,
        (unsigned long)(PCI_SLOT((bdp->ppci_dev->devfn))));
}

static int read_irq(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)(bdp->device->irq));
}

static int read_dev_name(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    sprintf(page, "%s", bdp->device->name);

    return generic_read(page, start, off, count, eof);
}

static int read_current_hwaddr(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;
    unsigned char *hwaddr = bdp->device->dev_addr;

    return read_hwaddr(page, start, off, count, eof, hwaddr);
}

static int read_permanent_hwaddr(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;
    unsigned char *hwaddr = bdp->bddp->perm_node_address;

    return read_hwaddr(page, start, off, count, eof, hwaddr);
}


static int read_part_number(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;
    sprintf(page, "%06lx-%03x",
            (unsigned long)(bdp->bddp->pwa_no >>8),
            (unsigned int)(bdp->bddp->pwa_no & 0xFF));
    return generic_read(page, start, off, count, eof);
}

static int read_link_status(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    if (bdp->flags & DF_LINK_UP)
        sprintf(page, "up");
    else
        sprintf(page, "down");

    return generic_read(page, start, off, count, eof);
}

static int read_speed(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    if (bdp->bddp->cur_line_speed)
        return read_ulong(page, start, off, count, eof,
            (unsigned long) (bdp->bddp->cur_line_speed));

    sprintf(page, "N/A");
    return generic_read(page, start, off, count, eof);
}

static int read_dplx_mode(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;
    char *dplx_mode;

    dplx_mode = bdp->bddp->cur_dplx_mode == FULL_DUPLEX ? "full" :
        ((bdp->bddp->cur_dplx_mode == NO_DUPLEX) ? "N/A" : "half");

    sprintf(page, "%s", dplx_mode);

    return generic_read(page, start, off, count, eof);
}

static int read_state(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    if (bdp->device->flags & IFF_UP)
        sprintf(page, "up");
    else
        sprintf(page, "down");

    return generic_read(page, start, off, count, eof);
}

static int read_rx_packets(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->perr_stats->gd_recvs);
}

static int read_tx_packets(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->perr_stats->gd_xmits);
}

static int read_rx_bytes(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->net_stats.rx_bytes);
}

static int read_tx_bytes(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->net_stats.tx_bytes);
}

static int read_rx_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;
        
    e100_get_stats(bdp->device);
    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->net_stats.rx_errors);
}

static int read_tx_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    e100_get_stats(bdp->device);
    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->net_stats.tx_errors);
}

static int read_rx_dropped(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->perr_stats->rcv_rsrc_err);
}

static int read_tx_dropped(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->net_stats.tx_dropped);
}

static int read_rx_multicast_packets(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->net_stats.multicast);
}

static int read_collisions(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->tx_tot_retries);
}

static int read_rx_length_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->perr_stats->rcv_runts);
}

static int read_rx_over_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->perr_stats->rcv_rsrc_err);
}

static int read_rx_crc_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->perr_stats->rcv_crc_err);
}

static int read_rx_frame_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->rcv_align_err);
}

static int read_rx_fifo_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->rcv_dma_orun);
}

static int read_rx_missed_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->net_stats.rx_missed_errors);
}

static int read_tx_aborted_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->tx_abrt_xs_col);
}

static int read_tx_carrier_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->tx_lost_csrs);
}

static int read_tx_fifo_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->tx_dma_urun);
}

static int read_tx_abort_late_coll(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->tx_late_col);
}

static int read_tx_deferred_ok(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->tx_ok_defrd);
}

static int read_tx_single_coll_ok(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->tx_one_retry);
}

static int read_tx_multi_coll_ok(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->tx_mt_one_retry);
}

static int read_rx_align_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->rcv_align_err);
}


static int read_tx_fc_packets (char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->xmt_fc_pkts);
}

static int read_rx_fc_packets (char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->rcv_fc_pkts);
}

static int read_rx_fc_unsupported(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->rcv_fc_unsupported);
}

static int read_tx_tco_packets (char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->xmt_tco_pkts);
}

static int read_rx_tco_packets (char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->rcv_tco_pkts);
}



static int write_blink_led_timer(struct file *file, const char *buffer,
                                 unsigned long count, void *data)
{
    bd_config_t  *bdp = (bd_config_t *) data;
    char          s_seconds[WRITE_BUF_MAX_LEN + 1];
    char         *res;
    unsigned long i_seconds;
    int           len = WRITE_BUF_MAX_LEN;
    
    if (!buffer)
    	return -EINVAL;
    
    if (count < len) {
        len = count;
    }
    
    copy_from_user(s_seconds, buffer, len);
    s_seconds[len] = '\0';
    i_seconds = simple_strtoul(s_seconds, &res, 0);
    if (res == s_seconds) {
        printk(KERN_WARNING "Invalid /proc argument used\n");
        return -EINVAL;
    }
    
    spin_lock(& (bdp->blink_led.blink_led_lock));
    
    if (bdp->blink_led.blink_is_running == B_TRUE) {
        spin_unlock(& (bdp->blink_led.blink_led_lock));
        return -EBUSY;
        
    } else {
        bdp->blink_led.blink_is_running = B_TRUE;
    }
    
    spin_unlock( &(bdp->blink_led.blink_led_lock) );
    
    e100_MdiWrite(bdp, PHY_82555_LED_SWITCH_CONTROL,
                  bdp->phy_addr,PHY_82555_LED_DRIVER_CONTROL);
    bdp->blink_led.blink_count  = i_seconds * BLINK_PER_SECOND * 2;
    bdp->blink_led.blink_state  = LED_OFF;
    
    init_timer( &(bdp->blink_led.blink_led_timer) );
    bdp->blink_led.blink_led_timer.data = (ulong_t) bdp->device;
    bdp->blink_led.blink_led_timer.function = (void *) &e100_blink_timer;
    
    e100_blink_timer(bdp->device);
    
    return count;
}

static e100_proc_entry e100_proc_list[] = {
    {"Description", read_descr, 0, PROC_INFO},
    {"Driver_Name", read_drvr_name, 0, PROC_INFO},
    {"Driver_Version", read_drvr_ver, 0, PROC_INFO},
    {"PCI_Vendor", read_pci_vendor, 0, PROC_INFO},
    {"PCI_Device_ID", read_pci_device, 0, PROC_INFO},
    {"PCI_Subsystem_Vendor", read_pci_sub_vendor, 0, PROC_INFO},
    {"PCI_Subsystem_ID", read_pci_sub_device, 0, PROC_INFO},
    {"PCI_Revision_ID", read_pci_revision, 0, PROC_INFO},
    {"PCI_Bus", read_pci_bus, 0, PROC_INFO},
    {"PCI_Slot", read_pci_slot, 0, PROC_INFO},
    {"IRQ", read_irq, 0, PROC_INFO},
    {"System_Device_Name", read_dev_name, 0, PROC_INFO},
    {"Current_HWaddr", read_current_hwaddr, 0, PROC_INFO},
    {"Permanent_HWaddr", read_permanent_hwaddr, 0, PROC_INFO},
    {"Part_Number", read_part_number, 0, PROC_INFO | PROC_NEW_LINE},
    {"Link", read_link_status, 0, PROC_INFO},
    {"Speed", read_speed, 0, PROC_INFO},
    {"Duplex", read_dplx_mode, 0, PROC_INFO},
    {"State", read_state, 0, PROC_INFO | PROC_NEW_LINE},
    {"Rx_Packets", read_rx_packets, 0, PROC_INFO},
    {"Tx_Packets", read_tx_packets, 0, PROC_INFO},
    {"Rx_Bytes", read_rx_bytes, 0, PROC_INFO},
    {"Tx_Bytes", read_tx_bytes, 0, PROC_INFO},
    {"Rx_Errors", read_rx_errors, 0, PROC_INFO},
    {"Tx_Errors", read_tx_errors, 0, PROC_INFO},
    {"Rx_Dropped", read_rx_dropped, 0, PROC_INFO},
    {"Tx_Dropped", read_tx_dropped, 0, PROC_INFO},
    {"Multicast", read_rx_multicast_packets, 0, PROC_INFO},
    {"Collisions", read_collisions, 0, PROC_INFO},
    {"Rx_Length_Errors", read_rx_length_errors, 0, PROC_INFO},
    {"Rx_Over_Errors", read_rx_over_errors, 0, PROC_INFO},
    {"Rx_CRC_Errors", read_rx_crc_errors, 0, PROC_INFO},
    {"Rx_Frame_Errors", read_rx_frame_errors, 0, PROC_INFO},
    {"Rx_FIFO_Errors", read_rx_fifo_errors, 0, PROC_INFO},
    {"Rx_Missed_Errors", read_rx_missed_errors, 0, PROC_INFO},
    {"Tx_Aborted_Errors", read_tx_aborted_errors, 0, PROC_INFO},
    {"Tx_Carrier_Errors", read_tx_carrier_errors, 0, PROC_INFO},
    {"Tx_FIFO_Errors", read_tx_fifo_errors, 0, PROC_INFO},
    {"Tx_Heartbeat_Errors", read_generic_0, 0, PROC_INFO},
    {"Tx_Window_Errors", read_generic_0, 0, PROC_INFO | PROC_NEW_LINE},
    {"Rx_TCP_Checksum_Good", read_generic_0, 0, PROC_INFO},
    {"Rx_TCP_Checksum_Bad", read_generic_0, 0, PROC_INFO},
    {"Tx_TCP_Checksum_Good", read_generic_0, 0, PROC_INFO},
    {"Tx_TCP_Checksum_Bad", read_generic_0, 0, PROC_INFO | PROC_NEW_LINE},
    {"Tx_Abort_Late_Coll", read_tx_abort_late_coll, 0, PROC_INFO},
    {"Tx_Deferred_Ok", read_tx_deferred_ok, 0, PROC_INFO},
    {"Tx_Single_Coll_Ok", read_tx_single_coll_ok, 0, PROC_INFO},
    {"Tx_Multi_Coll_Ok", read_tx_multi_coll_ok, 0, PROC_INFO},
    {"Rx_Long_Length_Errors", read_generic_0, 0, PROC_INFO},
    {"Rx_Align_Errors", read_rx_align_errors, 0, PROC_INFO | PROC_NEW_LINE},
    {"Tx_Flow_Control_Pause", read_tx_fc_packets, 0, PROC_INFO},
    {"Rx_Flow_Control_Pause", read_rx_fc_packets, 0, PROC_INFO},
    {"Rx_Flow_Control_Unsup", read_rx_fc_unsupported, 0, PROC_INFO | PROC_NEW_LINE},
    {"Tx_TCO_Packets", read_tx_tco_packets, 0, PROC_INFO},
    {"Rx_TCO_Packets", read_rx_tco_packets, 0, PROC_INFO},
    {"Identify_Adapter", 0, write_blink_led_timer, PROC_INFO},

#ifdef EEPROM_STB    
    {"EEPROM_STB", read_eeprom_stb, write_eeprom_stb, 0},
#endif

    {"", 0, 0, 0}
};



/* need to check page boundaries !!! */
static int read_info(char *page, char **start, off_t off,
                     int count, int *eof, void *data)
{
    bd_config_t     *bdp = (bd_config_t *) data;
    char            *pagep = page;
    e100_proc_entry *pe;
    int tmp;
    
    /* check that we aren't asked for data beyond a page boundary */
    if (count + off > PAGE_SIZE) {
        return -EINVAL;
    }
    
    for (pe = e100_proc_list; pe->name[0]; pe++) {
        if ((pe->flags & PROC_INFO) && (pe->read_proc)) {
            page += sprintf(page, "%-25s ", pe->name);
            page += pe->read_proc(page, start, 0, PAGE_SIZE, &tmp, bdp);
            
            if (pe->flags & PROC_NEW_LINE)
                page += sprintf(page, "\n");
        }
    }
    
    page += sprintf(page, "scbp = 0x%p\t bddp = 0x%p\n",bdp->bddp->scbp,
                    bdp->bddp);
   
    *page = 0; 
    return generic_read(pagep, start, off, count, eof);
}


static struct proc_dir_entry *create_proc_rw(char *name, bd_config_t *bdp,
                                             struct proc_dir_entry *parent,
                                             read_proc_t *read_proc,
                                             write_proc_t *write_proc)
{
    struct proc_dir_entry *pdep;
    mode_t mode = S_IFREG;

    if (write_proc) {
    	mode |= S_IWUSR;
        if (read_proc) {
            mode |= S_IRUSR;
        }
    
    } else if (read_proc) {
    	mode |= S_IRUGO;
    }
    
    if (!(pdep = create_proc_entry(name, mode , parent)))
        return NULL;
    
    pdep->read_proc = read_proc;
    pdep->write_proc = write_proc;
    pdep->data = bdp;
    return pdep;
}


int create_e100_proc_subdir(bd_config_t * bdp)
{
    struct proc_dir_entry *dev_dir;
    e100_proc_entry       *pe;
    char                   info[256];
    int                    len;
    
    /* create the main /proc dir if needed */
    if (!adapters_proc_dir) {
        if (e100_init_proc_dir() != B_TRUE)
            return -1;
    }
    
    strncpy(info, bdp->device->name, sizeof(info));
    len = strlen(info);
    strncat(info + len, ".info", sizeof(info) - len);
    
    /* info */
    if (!(create_proc_rw(info, bdp, adapters_proc_dir, read_info, 0))) {
        e100_proc_cleanup();
        return -1;
    }

    dev_dir = create_proc_entry(bdp->device->name, S_IFDIR, adapters_proc_dir);
    bdp->proc_parent = dev_dir;
    
    if (!dev_dir) {
        remove_e100_proc_subdir(bdp->device);
        return -1;
    }
    
    for (pe = e100_proc_list; pe->name[0]; pe++) {
        if (!(create_proc_rw(pe->name, bdp, dev_dir, pe->read_proc, pe->write_proc))) {
            remove_e100_proc_subdir(bdp->device);
            return -1;
        }
    }
    
    return 0;
}

static void remove_e100_proc_subdir(device_t * dev)
{
    bd_config_t     *bdp = dev->priv;
    e100_proc_entry *pe;
    char             info[256];
    int              len;

    len = strlen(dev->name);
    strncpy(info, dev->name, sizeof(info));
    strncat(info + len, ".info", sizeof(info) - len);

    if (bdp->proc_parent) {
        for (pe = e100_proc_list; pe->name[0]; pe++) {
            remove_proc_entry(pe->name, bdp->proc_parent);
        }
        
        remove_proc_entry(dev->name, adapters_proc_dir);
    }

    remove_proc_entry(info, adapters_proc_dir);
    
    /* try to remove the main /proc dir, if it's empty */
    e100_proc_cleanup();
}


/*****************************************************************************
 * Name:        e100_blink_timer
 *
 * Description: This routine blink the NIC led.
 *            The routine toggle the led state(ON/OFF) and fire a timer to call the routine again.
 *                  value.
 *
 * Arguments:
 *            dev - device structure
 *
 * Returns:
 *      (none)
 *
 *****************************************************************************/
static void
e100_blink_timer(device_t * dev)
{
    bd_config_t    *bdp = dev->priv;
    blink_led_t	   *blink_led = &bdp->blink_led;
    
    if(blink_led->blink_count){
        
        blink_led->blink_count--;
        
        switch(blink_led->blink_state){
            
        case LED_OFF:
            e100_MdiWrite(bdp,PHY_82555_LED_SWITCH_CONTROL,
                          bdp->phy_addr, PHY_82555_LED_OFF);
            blink_led->blink_state = LED_ON;
            break;
            
        case LED_ON: 
            if(bdp->rev_id >= D101MA_REV_ID)
                e100_MdiWrite(bdp,PHY_82555_LED_SWITCH_CONTROL, 
                              bdp->phy_addr, PHY_82555_LED_ON_559);
            else
                e100_MdiWrite(bdp,PHY_82555_LED_SWITCH_CONTROL,
                              bdp->phy_addr, PHY_82555_LED_ON_PRE_559);
            
            blink_led->blink_state = LED_OFF;
            break;
            
        case LED_NORMAL:
        default:
            break;
        }
        
    }
    
    if(blink_led->blink_count){
        
    	spin_lock( &(blink_led->blink_led_lock) );
        
        if( ! blink_led->at_cleanup){
            blink_led->blink_led_timer.expires = jiffies + (HZ/(2*BLINK_PER_SECOND));
            add_timer( &(blink_led->blink_led_timer) );
        }
        spin_unlock( &(blink_led->blink_led_lock) );
    }

    else{
        e100_MdiWrite(bdp,PHY_82555_LED_SWITCH_CONTROL, 
                      bdp->phy_addr, PHY_82555_LED_NORMAL_CONTROL);
        blink_led->blink_state = LED_NORMAL;
        spin_lock( &(blink_led->blink_led_lock) );
        
        blink_led->blink_is_running = B_FALSE;
        
        spin_unlock( &(blink_led->blink_led_lock) );
    }
}


/****************************************************************************
 * Name:          e100_init_proc_dir
 *
 * Description:   This routine creates the top-level /proc directory for the
 *                driver in /proc/net
 *
 * Arguments:     none
 *
 * Returns:       B_TRUE on success, B_FALSE on fail
 *
 ***************************************************************************/
static boolean_t e100_init_proc_dir(void)
{
    int len;
    
    /* first check if adapters_proc_dir already exists */
    len = strlen(ADAPTERS_PROC_DIR);
    for (adapters_proc_dir = proc_net->subdir;
         adapters_proc_dir;
         adapters_proc_dir = adapters_proc_dir->next) {
        
        if ((adapters_proc_dir->namelen == len) && 
            (!memcmp(adapters_proc_dir->name, ADAPTERS_PROC_DIR, len)))
            break;
    }
    
    if (!adapters_proc_dir)     
        adapters_proc_dir = create_proc_entry(ADAPTERS_PROC_DIR, S_IFDIR, proc_net);

    if (!adapters_proc_dir)
        return B_FALSE;

    return B_TRUE;
}


/****************************************************************************
 * Name:          e100_proc_remove_one
 *
 * Description:   This routine cleans the adapter's /proc dir
 *
 * Arguments:     device_t *dev - the device struct of the adapter
 *
 * Returns:       none
 *
 ***************************************************************************/
void e100_proc_remove_one(device_t *dev)
{
    bd_config_t *bdp = dev->priv;
    
    spin_lock(& (bdp->blink_led.blink_led_lock));
    
    bdp->blink_led.at_cleanup = B_TRUE ;
    del_timer(& (bdp->blink_led.blink_led_timer) );
    
    spin_unlock( &(bdp->blink_led.blink_led_lock) );
    
    e100_MdiWrite(bdp, PHY_82555_LED_SWITCH_CONTROL,
                  bdp->phy_addr,PHY_82555_LED_NORMAL_CONTROL);
    remove_e100_proc_subdir(dev);
}

/****************************************************************************
 * Name:          e100_proc_cleanup
 *
 * Description:   This routine clears the top-level /proc directory, if empty.
 *
 * Arguments:     none
 *
 * Returns:       none
 *
 ***************************************************************************/
static void e100_proc_cleanup()
{
    struct proc_dir_entry   *de;
    
    /* check if the subdir list is empty before removing adapters_proc_dir */
    for (de = adapters_proc_dir->subdir; de; de = de->next) {
        /* ignore . and .. */
        if (*(de->name) != '.')
            break;
    }
    
    if (de)
        return;

    remove_proc_entry(ADAPTERS_PROC_DIR, proc_net);
    adapters_proc_dir = NULL;
}

#endif /* CONFIG_PROC_FS */
