
/*****************************************************************************
 *****************************************************************************
 Copyright (c) 1999-2001, Intel Corporation 

 All rights reserved.

 Redistribution and use in source and binary forms, with or without 
 modification, are permitted provided that the following conditions are met:

 1. Redistributions of source code must retain the above copyright notice, 
 this list of conditions and the following disclaimer.

 2. Redistributions in binary form must reproduce the above copyright notice,
 this list of conditions and the following disclaimer in the documentation 
 and/or other materials provided with the distribution.

 3. Neither the name of Intel Corporation nor the names of its contributors 
 may be used to endorse or promote products derived from this software 
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 

 *****************************************************************************
****************************************************************************/

/**********************************************************************
*                                                                       *
* INTEL CORPORATION                                                    *
*                                                                       *
* This software is supplied under the terms of the license included  	*
* above.  All use of this driver must be in accordance with the terms  	*
* of that license.                                                    	*
*                                                                     	*
* Module Name:  e100_proc.c                                              *
*                                                                     	*
* Abstract:     Functions to handle the proc file system. 				*
*			  Create the proc directories and files and run read and	*
*             write requests from the user 								*
*                                                                     	*
* Environment:  This file is intended to be specific to the Linux     	*
*               operating system.                                     	*
*                                                                     	*
**********************************************************************/


#include "e100.h"

/***************************************************************************/
/*       /proc File System Interaface Support Functions                    */
/***************************************************************************/


#ifdef CONFIG_PROC_FS
struct proc_dir_entry *adapters_proc_dir;

/* externs from e100_main.c */
extern const char *e100_short_driver_name;
extern const char *e100_version;
extern struct net_device_stats *e100_get_stats(device_t * dev);
extern char *e100_GetBrandingMesg(bd_config_t * bdp);


static int generic_read(char *page, char **start, off_t off, int count, int *eof)
{
    int len;

    len = strlen(page);
    page[len++] = '\n';

    if (len <= off + count)
        *eof = 1;
    *start = page + off;
    len -= off;
    if (len > count)
        len = count;
    if (len < 0)
        len = 0;
    return len;
}

static int read_ulong(char *page, char **start, off_t off, 
    int count, int *eof, unsigned long l)
{
    sprintf(page, "%lu", l);

    return generic_read(page, start, off, count, eof);
}

static int read_ulong_hex(char *page, char **start, off_t off, 
    int count, int *eof, unsigned long l)
{
    sprintf(page, "0x%04lx", l);

    return generic_read(page, start, off, count, eof);
}

static int read_hwaddr(char *page, char **start, off_t off,
    int count, int *eof, unsigned char *hwaddr)
{
    sprintf(page, "%02X:%02X:%02X:%02X:%02X:%02X",
        hwaddr[0], hwaddr[1], hwaddr[2], 
        hwaddr[3], hwaddr[4], hwaddr[5]);

    return generic_read(page, start, off, count, eof);
}

/* need to check page boundaries !!! */
static int read_info(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t         *bdp = (bd_config_t *) data;
    struct net_device_stats     *stats;
    unsigned char               *hwaddr;
    char        *pagep = page;
    char        *msg;
        
    msg=e100_GetBrandingMesg(bdp);
    page += sprintf(page, "%-25s %s\n", DESCRIPTION_TAG, msg);

    page += sprintf(page, "%-25s %s\n", DRVR_NAME_TAG, e100_short_driver_name);

    page += sprintf(page, "%-25s %s\n", DRVR_VERSION_TAG, e100_version);
        
    page += sprintf(page, "%-25s 0x%04lx\n", 
        PCI_VENDOR_TAG, (unsigned long) bdp->bddp->ven_id);
    page += sprintf(page, "%-25s 0x%04lx\n", 
        PCI_DEVICE_ID_TAG, (unsigned long) bdp->bddp->dev_id);
    page += sprintf(page, "%-25s 0x%04lx\n", 
        PCI_SUBSYSTEM_VENDOR_TAG, 
        (unsigned long) bdp->bddp->sub_ven_id);
    page += sprintf(page, "%-25s 0x%04lx\n", 
        PCI_SUBSYSTEM_ID_TAG, 
        (unsigned long) bdp->bddp->sub_dev_id);
    page += sprintf(page, "%-25s 0x%02lx\n", 
        PCI_REVISION_ID_TAG, 
        (unsigned long) bdp->bddp->rev_id);
    page += sprintf(page, "%-25s %lu\n", 
        PCI_BUS_TAG, 
        (unsigned long) (bdp->bddp->pci_dev->bus->number));
    page += sprintf(page, "%-25s %lu\n", 
        PCI_SLOT_TAG, 
        (unsigned long)(PCI_SLOT((bdp->bddp->pci_dev->devfn))));
    page += sprintf(page, "%-25s %lu\n", 
        IRQ_TAG, 
        (unsigned long)(bdp->irq_level));
    page += sprintf(page, "%-25s %s\n", 
        SYSTEM_DEVICE_NAME_TAG, bdp->device->name);

    hwaddr = bdp->device->dev_addr;
    page += sprintf(page, "%-25s %02X:%02X:%02X:%02X:%02X:%02X\n",
        CURRENT_HWADDR_TAG,
        hwaddr[0], hwaddr[1], hwaddr[2], 
        hwaddr[3], hwaddr[4], hwaddr[5]);

    hwaddr = bdp->bddp->perm_node_address;
    page += sprintf(page, "%-25s %02X:%02X:%02X:%02X:%02X:%02X\n",
        PERMANENT_HWADDR_TAG,
        hwaddr[0], hwaddr[1], hwaddr[2], 
        hwaddr[3], hwaddr[4], hwaddr[5]);
    page += sprintf(page, "%-25s %06lx-%03x\n",
        PART_NUMBER_TAG,
        bdp->bddp->pwa_no >>8,
        bdp->bddp->pwa_no & 0xFF);

    page += sprintf(page, "\n");

    e100_phy_check(bdp);
    if (bdp->flags & DF_LINK_UP) msg = "up";
    else msg = "down";
    page += sprintf(page, "%-25s %s\n", LINK_TAG, msg);
        
    e100_FindPhySpeedAndDpx(bdp, bdp->bddp->PhyId);
    if (bdp->bddp->cur_line_speed)
        page += sprintf(page, "%-25s %lu\n", 
            SPEED_TAG, 
            (unsigned long)(bdp->bddp->cur_line_speed));
    else
        page += sprintf(page, "%-25s %s\n", SPEED_TAG, "N/A");

    msg = bdp->bddp->cur_dplx_mode == FULL_DUPLEX ? "full" :
        ((bdp->bddp->cur_dplx_mode == NO_DUPLEX) ? "N/A" : "half");
    page += sprintf(page, "%-25s %s\n", DUPLEX_TAG, msg);

    if (bdp->device->flags & IFF_UP) msg = "up";
    else msg = "down";
    page += sprintf(page, "%-25s %s\n", STATE_TAG, msg);

    page += sprintf(page, "\n");

    stats = e100_get_stats(bdp->device);
    page += sprintf(page, "%-25s %lu\n",
        RX_PACKETS_TAG, 
        (unsigned long) stats->rx_packets);
    page += sprintf(page, "%-25s %lu\n",
        TX_PACKETS_TAG, 
        (unsigned long) stats->tx_packets);
    page += sprintf(page, "%-25s %lu\n",
        RX_BYTES_TAG, 
        (unsigned long) stats->rx_bytes);
    page += sprintf(page, "%-25s %lu\n",
        TX_BYTES_TAG, 
        (unsigned long) stats->tx_bytes);
    page += sprintf(page, "%-25s %lu\n",
        RX_ERRORS_TAG, 
        (unsigned long) stats->rx_errors);
    page += sprintf(page, "%-25s %lu\n",
        TX_ERRORS_TAG, 
        (unsigned long) stats->tx_errors);
    page += sprintf(page, "%-25s %lu\n",
        RX_DROPPED_TAG, 
        (unsigned long) stats->rx_dropped);
    page += sprintf(page, "%-25s %lu\n",
        TX_DROPPED_TAG, 
        (unsigned long) stats->tx_dropped);
    page += sprintf(page, "%-25s %lu\n",
        MULTICAST_TAG, 
        (unsigned long) stats->multicast);
    page += sprintf(page, "%-25s %lu\n",
        COLLISIONS_TAG, 
        (unsigned long) stats->collisions);
    page += sprintf(page, "%-25s %lu\n",
        RX_LENGTH_ERRORS_TAG, 
        (unsigned long) stats->rx_length_errors);
    page += sprintf(page, "%-25s %lu\n",
        RX_OVER_ERRORS_TAG, 
        (unsigned long) stats->rx_over_errors);
    page += sprintf(page, "%-25s %lu\n",
        RX_CRC_ERRORS_TAG, 
        (unsigned long) stats->rx_crc_errors);
    page += sprintf(page, "%-25s %lu\n",
        RX_FRAME_ERRORS_TAG, 
        (unsigned long) stats->rx_frame_errors);
    page += sprintf(page, "%-25s %lu\n",
        RX_FIFO_ERRORS_TAG, 
        (unsigned long) stats->rx_fifo_errors);
    page += sprintf(page, "%-25s %lu\n",
        RX_MISSED_ERRORS_TAG, 
        (unsigned long) stats->rx_missed_errors);
    page += sprintf(page, "%-25s %lu\n",
        TX_ABORTED_ERRORS_TAG, 
        (unsigned long) stats->tx_aborted_errors);
    page += sprintf(page, "%-25s %lu\n",
        TX_CARRIER_ERRORS_TAG, 
        (unsigned long) stats->tx_carrier_errors);
    page += sprintf(page, "%-25s %lu\n",
        TX_FIFO_ERRORS_TAG, 
        (unsigned long) stats->tx_fifo_errors);
    page += sprintf(page, "%-25s %lu\n",
        TX_HEARTBEAT_ERRORS_TAG, 
        (unsigned long)0); /* !!! */
    page += sprintf(page, "%-25s %lu\n",
        TX_WINDOW_ERRORS_TAG, 
        (unsigned long)0); /* !!! */

    page += sprintf(page, "\n");

    page += sprintf(page, "%-25s %lu\n",
        RX_TCP_CHECKSUM_GOOD_TAG, 
        (unsigned long)0); /* !!! */
    page += sprintf(page, "%-25s %lu\n",
        RX_TCP_CHECKSUM_BAD_TAG, 
        (unsigned long)0); /* !!! */
    page += sprintf(page, "%-25s %lu\n",
        TX_TCP_CHECKSUM_GOOD_TAG, 
        (unsigned long)0); /* !!! */
    page += sprintf(page, "%-25s %lu\n",
        TX_TCP_CHECKSUM_BAD_TAG, 
        (unsigned long)0); /* !!! */

    page += sprintf(page, "\n");
    page += sprintf(page, "%-25s %lu\n",
        TX_ABORT_LATE_COLL_TAG, 
        (unsigned long)bdp->bddp->perr_stats->tx_late_col);
    page += sprintf(page, "%-25s %lu\n",
        TX_DEFERRED_OK_TAG, 
        (unsigned long)bdp->bddp->perr_stats->tx_ok_defrd);
    page += sprintf(page, "%-25s %lu\n",
        TX_SINGLE_COLL_OK_TAG, 
        (unsigned long)bdp->bddp->perr_stats->tx_one_retry);
    page += sprintf(page, "%-25s %lu\n",
        TX_MULTI_COLL_OK_TAG, 
        (unsigned long)bdp->bddp->perr_stats->tx_mt_one_retry);
    page += sprintf(page, "%-25s %lu\n",
        RX_LONG_LENGTH_ERRORS_TAG, 
        (unsigned long)0);
    page += sprintf(page, "%-25s %lu\n",
        RX_ALIGN_ERRORS_TAG,
        (unsigned long)bdp->bddp->perr_stats->rcv_align_err);

    page += sprintf(page, "\n");

    page += sprintf(page, "%-25s %lu\n",
        TX_FCTL_PACKETS_TAG,
        (unsigned long)bdp->bddp->perr_stats->xmt_fc_pkts);
    page += sprintf(page, "%-25s %lu\n",
        RX_FCTL_PACKETS_TAG,
        (unsigned long)bdp->bddp->perr_stats->rcv_fc_pkts);
    page += sprintf(page, "%-25s %lu\n",
        RX_FCTL_UNSUPPOORTED_TAG,
        (unsigned long)bdp->bddp->perr_stats->rcv_fc_unsupported);
    
    page += sprintf(page, "\n");

    page += sprintf(page, "%-25s %lu\n",
        TX_TCO_PACKETS_TAG,
        (unsigned long)bdp->bddp->perr_stats->xmt_tco_pkts);
    page += sprintf(page, "%-25s %lu\n",
        RX_TCO_PACKETS_TAG,
        (unsigned long)bdp->bddp->perr_stats->rcv_tco_pkts);
    page += sprintf(page, "scbp = 0x%X\t bddp = 0x%X\n",bdp->bddp->scbp,
                    bdp->bddp);
   
    *page = 0; 
    return generic_read(pagep, start, off, count, eof);
}

static int read_descr(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;
    char *msg;
        
    msg=e100_GetBrandingMesg(bdp);
    strncpy(page, msg, PAGE_SIZE);

    return generic_read(page, start, off, count, eof);
}

static int read_drvr_name(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    strncpy(page, e100_short_driver_name, PAGE_SIZE);

    return generic_read(page, start, off, count, eof);
}

static int read_drvr_ver(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    strncpy(page, e100_version, PAGE_SIZE);

    return generic_read(page, start, off, count, eof);
}

static int read_pci_vendor(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong_hex(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->ven_id);
}
        
static int read_pci_device(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong_hex(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->dev_id);
}
        
static int read_pci_sub_vendor(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong_hex(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->sub_ven_id);
}
        
static int read_pci_sub_device(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong_hex(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->sub_dev_id);
}
        
static int read_pci_revision(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong_hex(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->rev_id);
}
        
static int read_dev_name(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    strncpy(page, bdp->device->name, PAGE_SIZE);

    return generic_read(page, start, off, count, eof);
}

static int read_pci_bus(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof,
        (unsigned long) (bdp->bddp->pci_dev->bus->number));
}

static int read_pci_slot(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof,
        (unsigned long)(PCI_SLOT((bdp->bddp->pci_dev->devfn))));
}

static int read_irq(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)(bdp->irq_level));
}

static int read_current_hwaddr(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;
    unsigned char *hwaddr = bdp->device->dev_addr;

    return read_hwaddr(page, start, off, count, eof, hwaddr);
}

static int read_permanent_hwaddr(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;
    unsigned char *hwaddr = bdp->bddp->perm_node_address;

    return read_hwaddr(page, start, off, count, eof, hwaddr);
}


static int read_part_number(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;
    sprintf(page, "%06lx-%03x\n",
        bdp->bddp->pwa_no >>8,
        bdp->bddp->pwa_no & 0xFF);
    return generic_read(page, start, off, count, eof);
}

static int read_link_status(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    e100_phy_check(bdp);
    if (bdp->flags & DF_LINK_UP)
        strncpy(page, "up", PAGE_SIZE);
    else
        strncpy(page, "down", PAGE_SIZE);

    return generic_read(page, start, off, count, eof);
}

static int read_speed(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    e100_FindPhySpeedAndDpx(bdp, bdp->bddp->PhyId);
    if (bdp->bddp->cur_line_speed)
        return read_ulong(page, start, off, count, eof,
            (unsigned long) (bdp->bddp->cur_line_speed));
    strncpy(page, "N/A", PAGE_SIZE);
    return generic_read(page, start, off, count, eof);
}

static int read_dplx_mode(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;
    char *dplx_mode;

    e100_FindPhySpeedAndDpx(bdp, bdp->bddp->PhyId);
    dplx_mode = bdp->bddp->cur_dplx_mode == FULL_DUPLEX ? "full" :
        ((bdp->bddp->cur_dplx_mode == NO_DUPLEX) ? "N/A" : "half");
    strncpy(page, dplx_mode, PAGE_SIZE);

    return generic_read(page, start, off, count, eof);
}

static int read_state(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    if (bdp->device->flags & IFF_UP)
        strncpy(page, "up", PAGE_SIZE);
    else
        strncpy(page, "down", PAGE_SIZE);

    return generic_read(page, start, off, count, eof);
}

static int read_rx_packets(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->perr_stats->gd_recvs);
}

static int read_tx_packets(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->perr_stats->gd_xmits);
}

static int read_rx_bytes(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->net_stats.rx_bytes);
}

static int read_tx_bytes(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->net_stats.tx_bytes);
}

static int read_rx_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;
        
    e100_get_stats(bdp->device);
    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->net_stats.rx_errors);
}

static int read_tx_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    e100_get_stats(bdp->device);
    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->net_stats.tx_errors);
}

static int read_rx_dropped(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->perr_stats->rcv_rsrc_err);
}

static int read_tx_dropped(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->net_stats.tx_dropped);
}

static int read_rx_multicast_packets(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->net_stats.multicast);
}

static int read_collisions(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->tx_tot_retries);
}

static int read_rx_length_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->perr_stats->rcv_runts);
}

static int read_rx_over_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->perr_stats->rcv_rsrc_err);
}

static int read_rx_crc_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->bddp->perr_stats->rcv_crc_err);
}

static int read_rx_frame_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->rcv_align_err);
}

static int read_rx_fifo_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->rcv_dma_orun);
}

static int read_rx_missed_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->net_stats.rx_missed_errors);
}

static int read_tx_aborted_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->tx_abrt_xs_col);
}

static int read_tx_carrier_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->tx_lost_csrs);
}

static int read_tx_fifo_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->tx_dma_urun);
}

static int read_tx_heartbeat_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    return read_ulong(page, start, off, count, eof, 
        (unsigned long)0); 
}

static int read_tx_window_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    return read_ulong(page, start, off, count, eof, 
        (unsigned long)0); 
}

static int read_rx_tcp_checksum_good(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    return read_ulong(page, start, off, count, eof, 
        (unsigned long)0); 
}

static int read_rx_tcp_checksum_bad(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    return read_ulong(page, start, off, count, eof, 
        (unsigned long)0); 
}

static int read_tx_tcp_checksum_good(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    return read_ulong(page, start, off, count, eof, 
        (unsigned long)0); 
}

static int read_tx_tcp_checksum_bad(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    return read_ulong(page, start, off, count, eof, 
        (unsigned long)0); 
}


static int read_tx_abort_late_coll(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->tx_late_col);
}

static int read_tx_deferred_ok(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->tx_ok_defrd);
}

static int read_tx_single_coll_ok(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->tx_one_retry);
}

static int read_tx_multi_coll_ok(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->tx_mt_one_retry);
}


static int read_rx_allign_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->rcv_align_err);
}


static int read_rx_long_length_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    return read_ulong(page, start, off, count, eof, 
        (unsigned long)0);
}

static int read_rx_fc_packets (char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->rcv_fc_pkts);
}

static int read_tx_fc_packets (char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->xmt_fc_pkts);
}

static int read_rx_fc_unsupported(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->rcv_fc_unsupported);
}

static int read_rx_tco_packets (char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->rcv_tco_pkts);
}

static int read_tx_tco_packets (char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)bdp->bddp->perr_stats->xmt_tco_pkts);
}



static int write_blink_led_timer(struct file *file, const char *buffer,
	unsigned long count, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;
    bdd_t   *bddp = bdp->bddp;


   	char s_seconds[count+1];
   	unsigned long i_seconds;

   	if( !buffer)
    	return -1;
	
	spin_lock(& (bdp->blink_led.blink_led_lock));

	if(bdp->blink_led.blink_is_running){
		spin_unlock(& (bdp->blink_led.blink_led_lock));
		return -1;
	}
	else
		bdp->blink_led.blink_is_running = B_TRUE;

	spin_unlock( &(bdp->blink_led.blink_led_lock) );

	
    copy_from_user(s_seconds,buffer,count);
	s_seconds[count] = '\0';
   	i_seconds= simple_strtoul(s_seconds,NULL,10);
   	

	
	
    e100_MdiWrite(bdp, PHY_82555_LED_SWITCH_CONTROL,
    	bddp->phy_addr,PHY_82555_LED_DRIVER_CONTROL);
    bdp->blink_led.blink_count  = i_seconds* BLINK_PER_SECOND* 2;
    bdp->blink_led.blink_state  = LED_OFF;

    init_timer( &(bdp->blink_led.blink_led_timer) );
    bdp->blink_led.blink_led_timer.data = (ulong_t) bdp->device;
    bdp->blink_led.blink_led_timer.function = (void *) &e100_blink_timer;

    e100_blink_timer(bdp->device);

    return count;
    
}

#if (DEBUG_LINK > 0)
static int read_link_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long) bdp->ErrorCounter);
}
#endif


static struct proc_dir_entry *create_proc_read(char *name,
    bd_config_t * bdp,
    struct proc_dir_entry *parent,
    read_proc_t * read_proc)
{
    struct proc_dir_entry *pdep;
        
    if (!(pdep = create_proc_entry(name, S_IFREG, parent)))
        return NULL;
    pdep->read_proc = read_proc;
    pdep->data = bdp;
    return pdep;
}


static struct proc_dir_entry *create_proc_write(char *name,
    bd_config_t * bdp,
    struct proc_dir_entry *parent,
    write_proc_t * write_proc)
{
    struct proc_dir_entry *pdep;
        
    if (!(pdep = create_proc_entry(name, S_IFREG, parent)))
        return NULL;
    pdep->write_proc = write_proc;
    pdep->data = bdp;
    return pdep;
}


int create_e100_proc_subdir(bd_config_t * bdp)
{
    struct proc_dir_entry *dev_dir ;
    char        info[256];
    int len;

    dev_dir = create_proc_entry(bdp->device->name, S_IFDIR, adapters_proc_dir);
    strncpy(info, bdp->device->name, sizeof(info));
    len = strlen(info);
    strncat(info + len, ".info", sizeof(info) - len);
        
    /* info */
    if (!(create_proc_read(info, bdp, adapters_proc_dir, read_info)))
        return -1;
        
    /* description */
    if (!(create_proc_read(DESCRIPTION_TAG, bdp, dev_dir, read_descr)))
        return -1;
    /* driver name */
    if (!(create_proc_read(DRVR_NAME_TAG, bdp, dev_dir, read_drvr_name)))
        return -1;
    /* driver version */
    if (!(create_proc_read(DRVR_VERSION_TAG, bdp, dev_dir, 
        read_drvr_ver))) 
        return -1;
    /* pci vendor */
    if (!(create_proc_read(PCI_VENDOR_TAG, bdp, dev_dir, 
        read_pci_vendor))) 
        return -1;
    /* pci device id */
    if (!(create_proc_read(PCI_DEVICE_ID_TAG, bdp, dev_dir, 
        read_pci_device))) 
        return -1;
    /* pci sub vendor */
    if (!(create_proc_read(PCI_SUBSYSTEM_VENDOR_TAG, bdp, dev_dir,
        read_pci_sub_vendor))) 
        return -1;
    /* pci sub device id */
    if (!(create_proc_read(PCI_SUBSYSTEM_ID_TAG, bdp, dev_dir,
        read_pci_sub_device))) 
        return -1;
    /* pci revision id */
    if (!(create_proc_read(PCI_REVISION_ID_TAG, bdp, dev_dir,
        read_pci_revision))) 
        return -1;
    /* device name */
    if (!(create_proc_read(SYSTEM_DEVICE_NAME_TAG, bdp, dev_dir, 
        read_dev_name))) 
        return -1;
    /* pci bus */
    if (!(create_proc_read(PCI_BUS_TAG, bdp, dev_dir, read_pci_bus)))
        return -1;
    /* pci slot */
    if (!(create_proc_read(PCI_SLOT_TAG, bdp,
        dev_dir, read_pci_slot))) 
        return -1;
    /* irq */
    if (!(create_proc_read(IRQ_TAG, bdp, dev_dir, read_irq)))
        return -1;
    /* current hwaddr */
    if (!(create_proc_read(CURRENT_HWADDR_TAG, bdp, dev_dir, 
        read_current_hwaddr))) 
        return -1;
    /* permanent hwaddr */
    if (!(create_proc_read(PERMANENT_HWADDR_TAG, bdp, dev_dir, 
        read_permanent_hwaddr))) 
        return -1;
    /* part number */
    if (!(create_proc_read(PART_NUMBER_TAG, bdp, dev_dir, 
        read_part_number))) 
        return -1;

    /* link status */
    if (!(create_proc_read(LINK_TAG, bdp, dev_dir, read_link_status))) 
        return -1;
    /* speed */
    if (!(create_proc_read(SPEED_TAG, bdp, dev_dir, read_speed)))
        return -1;
    /* duplex mode */
    if (!(create_proc_read(DUPLEX_TAG, bdp, dev_dir, read_dplx_mode)))
        return -1;
    /* state */
    if (!(create_proc_read(STATE_TAG, bdp, dev_dir, read_state)))
        return 1;

    /* rx packets */
    if (!(create_proc_read(RX_PACKETS_TAG, bdp, dev_dir, read_rx_packets)))
        return 1;
    /* tx packets */
    if (!(create_proc_read(TX_PACKETS_TAG, bdp, dev_dir, read_tx_packets)))
        return 1;
    /* rx bytes */
    if (!(create_proc_read(RX_BYTES_TAG, bdp, dev_dir, read_rx_bytes)))
        return 1;
    /* tx bytes */
    if (!(create_proc_read(TX_BYTES_TAG, bdp, dev_dir, read_tx_bytes)))
        return 1;
    /* rx errors */
    if (!(create_proc_read(RX_ERRORS_TAG, bdp, dev_dir, read_rx_errors)))
        return 1;
    /* tx errors */
    if (!(create_proc_read(TX_ERRORS_TAG, bdp, dev_dir, read_tx_errors)))
        return 1;
    /* rx dropped */
    if (!(create_proc_read(RX_DROPPED_TAG, bdp, dev_dir, read_rx_dropped)))
        return 1;
    /* tx dropped */
    if (!(create_proc_read(TX_DROPPED_TAG, bdp, dev_dir, read_tx_dropped)))
        return 1;
    /* multicast packets */
    if (!(create_proc_read(MULTICAST_TAG, bdp, dev_dir, 
        read_rx_multicast_packets)))
        return 1;
    /* collisions */
    if (!(create_proc_read(COLLISIONS_TAG, bdp, dev_dir, read_collisions)))
        return 1;
    /* rx length errors */
    if (!(create_proc_read(RX_LENGTH_ERRORS_TAG, bdp, dev_dir, 
        read_rx_length_errors)))
        return 1;
    /* rx over errors */
    if (!(create_proc_read(RX_OVER_ERRORS_TAG, bdp, dev_dir, 
        read_rx_over_errors)))
        return 1;
    /* rx crc errors */
    if (!(create_proc_read(RX_CRC_ERRORS_TAG, bdp, dev_dir, 
        read_rx_crc_errors)))
        return 1;
    /* rx frame errors */
    if (!(create_proc_read(RX_FRAME_ERRORS_TAG, bdp, dev_dir, 
        read_rx_frame_errors)))
        return 1;
    /* rx fifo errors */
    if (!(create_proc_read(RX_FIFO_ERRORS_TAG, bdp, dev_dir, 
        read_rx_fifo_errors)))
        return 1;
    /* rx missed errors */
    if (!(create_proc_read(RX_MISSED_ERRORS_TAG, bdp, dev_dir, 
        read_rx_missed_errors)))
        return 1;
    /* tx aborted errors */
    if (!(create_proc_read(TX_ABORTED_ERRORS_TAG, bdp, dev_dir, 
        read_tx_aborted_errors)))
        return 1;
    /* tx carrier errors */
    if (!(create_proc_read(TX_CARRIER_ERRORS_TAG, bdp, dev_dir, 
        read_tx_carrier_errors)))
        return 1;
    /* tx fifo errors */
    if (!(create_proc_read(TX_FIFO_ERRORS_TAG, bdp, dev_dir, 
        read_tx_fifo_errors)))
        return 1;
    /* tx heartbeat errors */
    if (!(create_proc_read(TX_HEARTBEAT_ERRORS_TAG, bdp, dev_dir, 
        read_tx_heartbeat_errors)))
        return 1;
    /* tx window errors */
    if (!(create_proc_read(TX_WINDOW_ERRORS_TAG, bdp, dev_dir, 
        read_tx_window_errors)))
        return 1;
        
    /* rx tcp checksum good */
    if (!(create_proc_read(RX_TCP_CHECKSUM_GOOD_TAG, bdp, dev_dir, 
        read_rx_tcp_checksum_good)))
        return 1;
    /* rx tcp checksum bad */
    if (!(create_proc_read(RX_TCP_CHECKSUM_BAD_TAG, bdp, dev_dir, 
        read_rx_tcp_checksum_bad)))
        return 1;
    /* tx tcp checksum good */
    if (!(create_proc_read(TX_TCP_CHECKSUM_GOOD_TAG, bdp, dev_dir, 
        read_tx_tcp_checksum_good)))
        return 1;
    /* tx tcp checksum bad */
    if (!(create_proc_read(TX_TCP_CHECKSUM_BAD_TAG, bdp, dev_dir, 
        read_tx_tcp_checksum_bad)))
        return 1;

    /* */
    if (!(create_proc_read(TX_ABORT_LATE_COLL_TAG, bdp, dev_dir, 
        read_tx_abort_late_coll)))
        return 1;
    /* */
    if (!(create_proc_read(TX_DEFERRED_OK_TAG, bdp, dev_dir, 
        read_tx_deferred_ok)))
        return 1;
    /* */
    if (!(create_proc_read(TX_SINGLE_COLL_OK_TAG, bdp, dev_dir, 
        read_tx_single_coll_ok)))
        return 1;
    /* */
    if (!(create_proc_read(TX_MULTI_COLL_OK_TAG, bdp, dev_dir, 
        read_tx_multi_coll_ok)))
        return 1;
    /* */
    if (!(create_proc_read(RX_LONG_LENGTH_ERRORS_TAG, bdp, dev_dir, 
        read_rx_long_length_errors)))
        return 1;
    /* */
    if (!(create_proc_read(RX_ALIGN_ERRORS_TAG, bdp, dev_dir, 
        read_rx_allign_errors)))
        return 1;
    /* */
    if (!(create_proc_read(TX_FCTL_PACKETS_TAG, bdp, dev_dir, 
        read_tx_fc_packets)))
        return 1;
    /* */
    if (!(create_proc_read(RX_FCTL_PACKETS_TAG, bdp, dev_dir, 
        read_rx_fc_packets)))
        return 1;
    /* */
    if (!(create_proc_read(RX_FCTL_UNSUPPOORTED_TAG, bdp, dev_dir, 
        read_rx_fc_unsupported)))
        return 1;
    /* */
    if (!(create_proc_read(TX_TCO_PACKETS_TAG, bdp, dev_dir, 
        read_tx_tco_packets)))
        return 1;
    /* */
    if (!(create_proc_read(RX_TCO_PACKETS_TAG, bdp, dev_dir, 
        read_rx_tco_packets)))
        return 1;
		if (!(create_proc_write(IDENTIFY_ADAPTER, bdp,dev_dir, 
	        write_blink_led_timer)))
	        return 1;
	    	
#if     (DEBUG_LINK > 0)
    if (!(create_proc_read("Link_Errors", bdp, dev_dir, 
        read_link_errors)))
        return 1;
#endif

  
    return 0;
}

void remove_e100_proc_subdir(device_t * dev)
{
    struct proc_dir_entry *de;
    char        info[256];
    int         len;

    len = strlen(dev->name);
    strncpy(info, dev->name, sizeof(info));
    strncat(info + len, ".info", sizeof(info) - len);

    for (de = adapters_proc_dir->subdir; de; de = de->next) {
        if ((de->namelen == len) && (!memcmp(de->name, dev->name, len)))
            break;
    }

    if (de) {
        remove_proc_entry(DESCRIPTION_TAG, de);
        remove_proc_entry(DRVR_NAME_TAG, de);
        remove_proc_entry(DRVR_VERSION_TAG, de);
        remove_proc_entry(PCI_VENDOR_TAG, de);
        remove_proc_entry(PCI_DEVICE_ID_TAG, de);
        remove_proc_entry(PCI_SUBSYSTEM_VENDOR_TAG, de);
        remove_proc_entry(PCI_SUBSYSTEM_ID_TAG, de);
        remove_proc_entry(PCI_REVISION_ID_TAG, de);
        remove_proc_entry(SYSTEM_DEVICE_NAME_TAG, de);
        remove_proc_entry(PCI_BUS_TAG, de);
        remove_proc_entry(PCI_SLOT_TAG, de);
        remove_proc_entry(IRQ_TAG, de);
        remove_proc_entry(CURRENT_HWADDR_TAG, de);
        remove_proc_entry(PERMANENT_HWADDR_TAG, de);
        remove_proc_entry(PART_NUMBER_TAG, de);

        remove_proc_entry(LINK_TAG, de);
        remove_proc_entry(SPEED_TAG, de);
        remove_proc_entry(DUPLEX_TAG, de);
        remove_proc_entry(STATE_TAG, de);

        remove_proc_entry(RX_PACKETS_TAG, de);
        remove_proc_entry(TX_PACKETS_TAG, de);
        remove_proc_entry(RX_BYTES_TAG, de);
        remove_proc_entry(TX_BYTES_TAG, de);
        remove_proc_entry(RX_ERRORS_TAG, de);
        remove_proc_entry(TX_ERRORS_TAG, de);
        remove_proc_entry(RX_DROPPED_TAG, de);
        remove_proc_entry(TX_DROPPED_TAG, de);
        remove_proc_entry(MULTICAST_TAG, de);
        remove_proc_entry(COLLISIONS_TAG, de);
        remove_proc_entry(RX_LENGTH_ERRORS_TAG, de);
        remove_proc_entry(RX_OVER_ERRORS_TAG, de);
        remove_proc_entry(RX_CRC_ERRORS_TAG, de);
        remove_proc_entry(RX_FRAME_ERRORS_TAG, de);
        remove_proc_entry(RX_FIFO_ERRORS_TAG, de);
        remove_proc_entry(RX_MISSED_ERRORS_TAG, de);
        remove_proc_entry(TX_ABORTED_ERRORS_TAG, de);
        remove_proc_entry(TX_CARRIER_ERRORS_TAG, de);
        remove_proc_entry(TX_FIFO_ERRORS_TAG, de);
        remove_proc_entry(TX_HEARTBEAT_ERRORS_TAG, de);
        remove_proc_entry(TX_WINDOW_ERRORS_TAG, de);

        remove_proc_entry(RX_TCP_CHECKSUM_GOOD_TAG, de);
        remove_proc_entry(RX_TCP_CHECKSUM_BAD_TAG, de);
        remove_proc_entry(TX_TCP_CHECKSUM_GOOD_TAG, de);
        remove_proc_entry(TX_TCP_CHECKSUM_BAD_TAG, de);
        
        remove_proc_entry(TX_ABORT_LATE_COLL_TAG, de);
        remove_proc_entry(TX_DEFERRED_OK_TAG, de);
        remove_proc_entry(TX_SINGLE_COLL_OK_TAG, de);
        remove_proc_entry(TX_MULTI_COLL_OK_TAG, de);
        remove_proc_entry(RX_LONG_LENGTH_ERRORS_TAG,de);
        remove_proc_entry(RX_ALIGN_ERRORS_TAG,de);
        remove_proc_entry(TX_FCTL_PACKETS_TAG,de);
        remove_proc_entry(RX_FCTL_PACKETS_TAG,de);
        remove_proc_entry(RX_FCTL_UNSUPPOORTED_TAG,de);
        remove_proc_entry(TX_TCO_PACKETS_TAG,de);
        remove_proc_entry(RX_TCO_PACKETS_TAG,de);
        remove_proc_entry(IDENTIFY_ADAPTER,de);

    
#if     (DEBUG_LINK > 0)
        remove_proc_entry("Link_Errors",de);
#endif

    }
    remove_proc_entry(info, adapters_proc_dir);
    remove_proc_entry(dev->name, adapters_proc_dir);
}
#endif /* CONFIG_PROC_FS */

