/*****************************************************************************
 *****************************************************************************
 Copyright (c) 1999-2001, Intel Corporation 

 All rights reserved.

 Redistribution and use in source and binary forms, with or without 
 modification, are permitted provided that the following conditions are met:

 1. Redistributions of source code must retain the above copyright notice, 
 this list of conditions and the following disclaimer.

 2. Redistributions in binary form must reproduce the above copyright notice,
 this list of conditions and the following disclaimer in the documentation 
 and/or other materials provided with the distribution.

 3. Neither the name of Intel Corporation nor the names of its contributors 
 may be used to endorse or promote products derived from this software 
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 

 *****************************************************************************
****************************************************************************/

/**********************************************************************
*                                                                     *
* INTEL CORPORATION                                                   *
*                                                                     *
* This software is supplied under the terms of the license included   *
* above.  All use of this driver must be in accordance with the terms *
* of that license.                                                    *
*                                                                     *
* Module Name:  e100_config.c                                         *
*                                                                     *
* Abstract:     Functions for configuring the network adapter.        *
*                                                                     *
* Environment:  This file is intended to be specific to the Linux     *
*               operating system.                                     *
*                                                                     *
**********************************************************************/

#include "e100_config.h"

/* global variables */
static uint8_t e100_cfg_parm6 = CFG_BYTE_PARM6;

static uint8_t e100_rx_fifo_lmt =
(RX_FIFO_LMT < 0) ? DEFAULT_RX_FIFO_LIMIT : ((RX_FIFO_LMT >
                                              15) ? DEFAULT_RX_FIFO_LIMIT : RX_FIFO_LMT);

static uint8_t e100_tx_fifo_lmt =
(TX_FIFO_LMT < 0) ? DEFAULT_TX_FIFO_LIMIT : ((TX_FIFO_LMT >
                                              7) ? DEFAULT_TX_FIFO_LIMIT : TX_FIFO_LMT);

static uint8_t e100_rx_dma_cnt =
(RX_DMA_CNT < 0) ? CB_557_CFIG_DEFAULT_PARM4 : ((RX_DMA_CNT >
                                                 63) ? CB_557_CFIG_DEFAULT_PARM4 : RX_DMA_CNT);

static uint8_t e100_tx_dma_cnt =
(TX_DMA_CNT < 0) ? CB_557_CFIG_DEFAULT_PARM5 : ((TX_DMA_CNT >
                                                 63) ? CB_557_CFIG_DEFAULT_PARM5 : TX_DMA_CNT);

static uint8_t e100_urun_retry =
(TX_UR_RETRY < 0) ? CB_557_CFIG_DEFAULT_PARM5 : ((TX_UR_RETRY >
                                                  3) ? CB_557_CFIG_DEFAULT_PARM5 : TX_UR_RETRY);

/* e100_MWI_enable
 * 
 * Enable/Disable use of Memory Write and Invalidate
 * 0 - Disable
 * 1 - Enable
 *
 * Note: This should be enabled only for an 82558 based adapter (PRO/100+)
 * and only on systems in which the PCI bus supports MWI. If enabled on a 
 * system that does not support MWI, performance might be affected.
 */       
static int e100_MWI_enable = 1;

/* 
 * e100_read_align_enable
 * 
 * Enable/Disable use of Cache Line Read/Write Termination
 * 0 - Disable
 * 1 - Enable
 *
 * Note: This should be enabled only for an 82558 based adapter (PRO/100+)
 * and only on cache line oriented systems. If enabled on a
 * system that is not cache line oriented, performance might be affected.
 */
static int e100_read_align_enable = 0;


/* extern variables */
extern int e100_enhanced_tx_enable;
extern int e100_flow_control_enable;
extern int e100_speed_duplex[MAX_NIC];
extern int XsumRX[MAX_NIC];


static void e100_config_vlan_drop(bd_config_t *bdp, boolean_t enable);
static void e100_config_long_rx(bd_config_t *bdp, boolean_t enable);

/*
 * Procedure:   e100_config_init
 *
 * Description: This routine will initialize the card's configure block.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *
 * Returns:
 */
void e100_config_init(bd_config_t *bdp)
{
    bdd_t          *bddp = bdp->bddp;   /* stores all adapter specific info */
    
    /* initialize config block */
    memset(bdp->config, 0x00, sizeof(bdp->config));

    /* First fill in the static (end user can't change) config bytes */
    bdp->config[0] = CB_CFIG_BYTE_COUNT;
    bdp->config[2] = CB_557_CFIG_DEFAULT_PARM2;
    bdp->config[3] = CB_557_CFIG_DEFAULT_PARM3;
    /* Commented out byte 6 for CNA intr fix. */
    /* bdp->config[6]  = CB_557_CFIG_DEFAULT_PARM6; */
    bdp->config[9] = CB_557_CFIG_DEFAULT_PARM9;
    bdp->config[10] = CB_557_CFIG_DEFAULT_PARM10;
    bdp->config[11] = CB_557_CFIG_DEFAULT_PARM11;
    bdp->config[12] = CB_557_CFIG_DEFAULT_PARM12;
    bdp->config[13] = CB_557_CFIG_DEFAULT_PARM13;
    bdp->config[14] = CB_557_CFIG_DEFAULT_PARM14;
    bdp->config[18] = CB_557_CFIG_DEFAULT_PARM18;
    bdp->config[20] = CB_557_CFIG_DEFAULT_PARM20;
    bdp->config[21] = CB_557_CFIG_DEFAULT_PARM21;
    
    /* Now fill in the rest of the configuration bytes (the bytes that *
     * contain user configurable parameters). */

    /* Change for 82558 enhancement */
    /* Set the Tx and Rx Fifo limits */
    if ((bddp->flags & IS_BACHELOR) && (e100_tx_fifo_lmt < 8)) {
        e100_tx_fifo_lmt = 8;   /* set 8 as the minimum */
    }
    
    bdp->config[1] = 
        (uint8_t) (BIT_7 | (e100_tx_fifo_lmt << 4) | e100_rx_fifo_lmt);
    
    e100_config_ifs(bdp);
    
    /* Change for 82558 enhancement */
    /* MWI enable. This should be turned on only if enabled in * e100.c and
     * if the adapter is a 82558/9 and if the PCI command reg. * has enabled
     * the MWI bit. */
    if ((bddp->flags & IS_BACHELOR) && (e100_MWI_enable)) {
        bdp->config[3] |= CB_CFIG_MWI_EN;
    }
    
    /* Read Align/Write Terminate on cache line. This should be * turned on
     * only if enabled in e100.c and if the adapter is a 82558/9 * and if the 
     * system is cache line oriented. */
    if ((bddp->flags & IS_BACHELOR) && (e100_read_align_enable))
        bdp->config[3] |= CB_CFIG_READAL_EN | CB_CFIG_TERMCL_EN;
    
    /* Set the Tx and Rx DMA maximum byte count fields. */
    bdp->config[4] = e100_rx_dma_cnt;
    bdp->config[5] = e100_tx_dma_cnt;
    if ((e100_rx_dma_cnt) || (e100_tx_dma_cnt)) {
        bdp->config[5] |= CB_CFIG_DMBC_EN;
    }
    
    /* Change for 82558 enhancement */
    /* Extended TCB. Should be turned on only if enabled * in e100.c and if
     * the adapter is a 82558/9. */
    if ((e100_cfg_parm6 == 0x32) || (e100_cfg_parm6 == 0x3a)) {
        bdp->config[6] = e100_cfg_parm6;
    } else {
        bdp->config[6] = CB_557_CFIG_DEFAULT_PARM6;
    }
    
    if ((bddp->flags & IS_BACHELOR) && (e100_enhanced_tx_enable)) {
        bdp->config[6] &= ~CB_CFIG_EXT_TCB_DIS;
    }
    
    /*
     * Enable extended statistical counters (82558 and up) and TCO counters
     * (82559 and up) and set the statistical counters' mode in bddp 
     *  
     *  stat. mode      |    TCO stat. bit (2)  |  Extended stat. bit (5)
     * ------------------------------------------------------------------
     *  Basic (557)     |       0               |         1
     * ------------------------------------------------------------------
     *  Extended (558)  |       0               |         0
     * ------------------------------------------------------------------
     *  TCO (559)       |       1               |         1
     * ------------------------------------------------------------------
     *  Reserved        |       1               |         0
     * ------------------------------------------------------------------
     */
    if (bddp->flags & IS_BACHELOR) {  /* This is 558 and up - enable extended counters */
        
        if (bddp->rev_id >= D101MA_REV_ID) {   /* this is 559 and up - enable TCO counters */
            bdp->config[6] |= CB_CFIG_TCO_STAT;
            bdp->config[6] |= CB_CFIG_EXT_STAT_DIS;
            bddp->stat_mode = E100_TCO_STATS;
        
        } else {  /* this is 558 */
            bdp->config[6] &= ~CB_CFIG_TCO_STAT;
            bdp->config[6] &= ~CB_CFIG_EXT_STAT_DIS;
            bddp->stat_mode = E100_EXTENDED_STATS;
        }
        
    } else {  /* this is 557 */
        bdp->config[6] &= ~CB_CFIG_TCO_STAT;
       	bdp->config[6] |= CB_CFIG_EXT_STAT_DIS;
        bddp->stat_mode = E100_BASIC_STATS;
    }
    
    /* Set up number of retries after under run */
    bdp->config[7] = ((CB_557_CFIG_DEFAULT_PARM7 & (~CB_CFIG_URUN_RETRY)) |
                      (e100_urun_retry << 1));
    
    /* Change for 82558 enhancement */
    /* Dynamic TBD. Should be turned on only if enabled in e100.c and if
     * the adapter is a 82558/9. */
    if ((bddp->flags & IS_BACHELOR) && (e100_enhanced_tx_enable))
        bdp->config[7] |= CB_CFIG_DYNTBD_EN;
    
    /* Setup for MII or 503 operation.  The CRS+CDT bit should only be set */
    /* when operating in 503 mode. */
    if (bddp->phy_addr == 32) {
        bdp->config[8] = (CB_557_CFIG_DEFAULT_PARM8 & (~CB_CFIG_503_MII));
        bdp->config[15] = (CB_557_CFIG_DEFAULT_PARM15 | CB_CFIG_CRS_OR_CDT);
    } else {
        bdp->config[8] = (CB_557_CFIG_DEFAULT_PARM8 | CB_CFIG_503_MII);
        bdp->config[15] = (CB_557_CFIG_DEFAULT_PARM15 & (~CB_CFIG_CRS_OR_CDT));
    }
    
    /* Change for 82558 enhancement */
    /* enable flow control only if not 557 */
    if ((bddp->flags & IS_BACHELOR) && (e100_flow_control_enable)) {
        bdp->config[16] = DFLT_FC_DELAY_LSB;
        bdp->config[17] = DFLT_FC_DELAY_MSB;
        /* Removed CB_CFIG_TX_FC_EN frm line below. This bit has to be 0 to 
         * enable flow control. */
        bdp->config[19] = (CB_557_CFIG_DEFAULT_PARM19 | CB_CFIG_FC_RESTOP |
                           CB_CFIG_FC_RESTART | CB_CFIG_REJECT_FC);
    } else {
        bdp->config[16] = CB_557_CFIG_DEFAULT_PARM16;
        bdp->config[17] = CB_557_CFIG_DEFAULT_PARM17;
        /* Bit 2 has to be 'OR'd to disable flow control. */
        bdp->config[19] = CB_557_CFIG_DEFAULT_PARM19 | CB_CFIG_TX_FC_DIS;
    }
    
    e100_config_force_dplx(bdp);
    e100_config_promisc(bdp);
    e100_config_brdcst_dsbl(bdp);
    e100_config_mulcst_enbl(bdp);
    
    /* Enable checksum offloading if we are on a supported adapter. */
    if ((bddp->rev_id >= D101MA_REV_ID) && (bddp->rev_id < D102_REV_ID) &&
        (XsumRX[bddp->bd_number] == TRUE) && (bddp->dev_id != 0x1209)) {
        
#ifndef __ia64__
        bddp->checksum_offload_enabled = 1;
        bdp->config[9] |= 1;
#endif /* __ia64__ */

    } else if (bddp->rev_id >= D102_REV_ID) {
        /* The D102 chip allows for 32 config bytes.  This value is
           supposed to be in Byte 0.  Just add the extra bytes to
           what was already setup in the block. */
        bdp->config[0] += CB_CFIG_D102_BYTE_COUNT;
        
        /* now we need to enable the extended RFD.  When this is
           enabled, the immediated receive data buffer starts at offset
           32 from the RFD base address, instead of at offset 16. */
        bdp->config[7] |= CB_CFIG_EXTENDED_RFD;
        
        /* put the chip into D102 receive mode.  This is neccessary
           for any parsing and offloading features. */
        bdp->config[22] = CB_CFIG_RECEIVE_GAMLA_MODE;
        
        /* set the flag if checksum offloading was enabled */
        if (XsumRX[bddp->bd_number] == TRUE) {
            bddp->checksum_offload_enabled = 1;
        }
    }
    /*
      #ifdef IANS
      if (ANS_PRIVATE_DATA_FIELD(bdp)->iANS_status == IANS_COMMUNICATION_UP) {
      
      #ifdef IANS_BASE_VLAN_TAGGING
      if (ANS_PRIVATE_DATA_FIELD(bdp)->vlan_mode == IANS_VLAN_MODE_ON) {
      bd_ans_hw_ConfigEnableTagging(bdp->config, bddp->rev_id);
      }
      #endif
      }
      #endif
    */
}

/*
 * Procedure:   e100_config
 *
 * Description: This routine will issue a configure command to the 82557.
 *              This command will be executed in polled mode as interrupts
 *              are _disabled_ at this time.  The configuration parameters
 *              that are user configurable will have been set in "e100.c".
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *
 * Returns:
 *      B_TRUE - If the configure command was successfully issued and completed
 *      B_FALSE - If the configure command failed to complete properly
 */
boolean_t
e100_config(bd_config_t *bdp)
{
        bdd_t          *bddp = bdp->bddp;   /* stores all adapter specific info */
        pcb_header_t    pntcb_hdr;
        int             e100_retry, timeout;
        unsigned long   lock_flag_bd;
        unsigned long   lock_flag_tx;
        unsigned long   lock_flag_conf;
    
        spin_lock_irqsave(&bddp->bd_tx_lock, lock_flag_tx);
        spin_lock_irqsave(&bddp->bd_lock, lock_flag_bd);
        spin_lock_irqsave(&bddp->config_lock, lock_flag_conf);
   
        pntcb_hdr = (pcb_header_t) bddp->pntcb;   /* get hdr of non tcb cmd */

        /* Setup the non-transmit command block header for the configure command. */
        pntcb_hdr->cb_status = 0;
        pntcb_hdr->cb_cmd = __constant_cpu_to_le16(CB_CONFIGURE);
        /* Note: cb_lnk_ptr is a physical address stored in an unsigned long */
        pntcb_hdr->cb_lnk_ptr = 0;

        /* Copy the device's config block to the device's memory */
        memcpy(bddp->pntcb->ntcb.config.cfg_byte, bdp->config, bdp->config[0]);
        barrier();
    
        /* Wait for the SCB command word to clear before we set the * general
         * pointer */
        if (e100_wait_scb(bdp) != B_TRUE) {
                spin_unlock_irqrestore(&bddp->config_lock, lock_flag_conf);
                spin_unlock_irqrestore(&bddp->bd_lock, lock_flag_bd);
                spin_unlock_irqrestore(&bddp->bd_tx_lock, lock_flag_tx);
                return (B_FALSE);
        }

        if (bdp->bddp->last_tcbp) {
                timeout = 1000*500;
                E100_WAIT_FOR_CONDITION(timeout,
                                        (bdp->bddp->last_tcbp->tcb_hdr.cb_status & 
                                         __constant_cpu_to_le16(CB_STATUS_COMPLETE)));
                if(timeout <= 0) {
                        spin_unlock_irqrestore(&bddp->config_lock, lock_flag_conf);
                        spin_unlock_irqrestore(&bddp->bd_lock, lock_flag_bd);
                        spin_unlock_irqrestore(&bddp->bd_tx_lock, lock_flag_tx);
                        printk(KERN_WARNING "Config failed\n");
                        return B_FALSE;  
                }
        }       

        e100_retry = E100_CMD_WAIT;
        while (((readw(&bddp->scbp->scb_status) & SCB_CUS_MASK) == SCB_CUS_ACTIVE)
               && (e100_retry)) {
                mdelay(20);
                e100_retry--;
        }
        if(!e100_retry){
                printk("%s config: timeout waiting for CU\n",
                       bdp->device->name);
                spin_unlock_irqrestore(&bddp->config_lock, lock_flag_conf);
                spin_unlock_irqrestore(&bddp->bd_lock, lock_flag_bd);
                spin_unlock_irqrestore(&bddp->bd_tx_lock, lock_flag_tx);
                return B_FALSE;
        }                
    
        bddp->next_cu_cmd = START_WAIT;

        /* write the config buffer in the scb gen pointer */
        writel(bddp->nontx_paddr, &(bddp->scbp->scb_gen_ptr));

        /* Submit the configure command to the chip, and wait for it to complete. */
        if (!e100_exec_poll_cmd(bdp)) {
                spin_unlock_irqrestore(&bddp->config_lock, lock_flag_conf);
                spin_unlock_irqrestore(&bddp->bd_lock, lock_flag_bd);
                spin_unlock_irqrestore(&bddp->bd_tx_lock, lock_flag_tx);
                return B_FALSE;
        
        } else {
                bdp->config[0] = CB_CFIG_MIN_PARAMS;  /* reset number of bytes to config next time */
                spin_unlock_irqrestore(&bddp->config_lock, lock_flag_conf);
                spin_unlock_irqrestore(&bddp->bd_lock, lock_flag_bd);
                spin_unlock_irqrestore(&bddp->bd_tx_lock, lock_flag_tx);
                return B_TRUE;
        }
}

/*
 * Procedure:   e100_config_promisc
 *
 * Description: This routine will enable or disable promiscuous mode
 *              in the adapter's congif block, as indicated by the
 *              adapter's e100_bdconfig structure.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *
 * Returns:
 */
void
e100_config_promisc(bd_config_t *bdp)
{
    bdd_t          *bddp = bdp->bddp;   /* stores all adapter specific info */
    unsigned long   lock_flag_conf;
    
    spin_lock_irqsave(&bddp->config_lock, lock_flag_conf);
    
    /* if in promiscuous mode, save bad frames */
    if (bddp->promisc) {
        
        if (!(bdp->config[6] & CB_CFIG_SAVE_BAD_FRAMES)) {
            bdp->config[6] |= CB_CFIG_SAVE_BAD_FRAMES;
            E100_CONFIG(bdp, 6);
        }
        
        if (bdp->config[7] & (uint8_t)BIT_0) {
            bdp->config[7] &= (uint8_t) (~BIT_0);
            E100_CONFIG(bdp, 7);
        }
        
        if (!(bdp->config[15] & CB_CFIG_PROMISCUOUS)) {
            bdp->config[15] |= CB_CFIG_PROMISCUOUS;
            E100_CONFIG(bdp, 15);
        }
    
    } else { /* not in promiscuous mode */
        
        if (bdp->config[6] & CB_CFIG_SAVE_BAD_FRAMES) {
            bdp->config[6] &= ~CB_CFIG_SAVE_BAD_FRAMES;
            E100_CONFIG(bdp, 6);
        }
        
        if (!(bdp->config[7] & (uint8_t)BIT_0)) {
            bdp->config[7] |= (uint8_t) (BIT_0);
            E100_CONFIG(bdp, 7);
        }
        
        if (bdp->config[15] & CB_CFIG_PROMISCUOUS) {
            bdp->config[15] &= ~CB_CFIG_PROMISCUOUS;
            E100_CONFIG(bdp, 15);
        }
    }
    
    spin_unlock_irqrestore(&bddp->config_lock, lock_flag_conf);
}

/*
 * Procedure:   e100_config_brdcst_dsbl
 *
 * Description: This routine will enable or disable broadcast
 *              in the adapter's congif block, as indicated by the
 *              adapter's e100_bdconfig structure.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *
 * Returns:
 */
void
e100_config_brdcst_dsbl(bd_config_t *bdp)
{
    bdd_t          *bddp = bdp->bddp;   /* stores all adapter specific info */
    unsigned long   lock_flag_conf;
    
    spin_lock_irqsave(&bddp->config_lock, lock_flag_conf);
    
    /* disable broadcast if so desired */
    if (bddp->brdcst_dsbl) {
        if (!(bdp->config[15] & CB_CFIG_BROADCAST_DIS)) {
            bdp->config[15] |= CB_CFIG_BROADCAST_DIS;
            E100_CONFIG(bdp, 15);
        }            
        
    } else {
        if (bdp->config[15] & CB_CFIG_BROADCAST_DIS) {
            bdp->config[15] &= ~CB_CFIG_BROADCAST_DIS;
            E100_CONFIG(bdp, 15);
        }
    }
    
    spin_unlock_irqrestore(&bddp->config_lock, lock_flag_conf);
}

/*
 * Procedure:   e100_config_mulcst_enbl
 *
 * Description: This routine will enable or disable multicast packets
 *              in the adapter's congif block, as indicated by the
 *              adapter's e100_bdconfig structure.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *
 * Returns:
 */
void
e100_config_mulcst_enbl(bd_config_t *bdp)
{
    bdd_t          *bddp = bdp->bddp;   /* stores all adapter specific info */
    unsigned long   lock_flag_conf;
    
    spin_lock_irqsave(&bddp->config_lock, lock_flag_conf);
    
    /* this flag is used to enable receiving all multicast packet */
    if (bddp->mulcst_enbl) {
        if (!(bdp->config[21] & CB_CFIG_MULTICAST_ALL)) {
            bdp->config[21] |= CB_CFIG_MULTICAST_ALL;
            E100_CONFIG(bdp, 21);
        }
        
    } else {
        if (bdp->config[21] & CB_CFIG_MULTICAST_ALL) {
            bdp->config[21] &= ~CB_CFIG_MULTICAST_ALL;
            E100_CONFIG(bdp, 21);
        }
    }
    
    spin_unlock_irqrestore(&bddp->config_lock, lock_flag_conf);
}

/*
 * Procedure:   e100_config_ifs
 *
 * Description: This routine will configure the adaptive IFS value
 *              in the adapter's congif block, as indicated by the
 *              adapter's e100_bdconfig structure.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *
 * Returns:
 */
void
e100_config_ifs(bd_config_t *bdp)
{
    bdd_t          *bddp = bdp->bddp;   /* stores all adapter specific info */
    unsigned long   lock_flag_conf;
    
    spin_lock_irqsave(&bddp->config_lock, lock_flag_conf);
    
    /* IFS value is only needed to be specified at half-duplex mode */
    if (bddp->cur_dplx_mode == HALF_DUPLEX) {
        bdp->config[2] = (uint8_t) bdp->ifs_value;
    } else {
        bdp->config[2] = 0;
    }
    
    spin_unlock_irqrestore(&bddp->config_lock, lock_flag_conf);
}

/*
 * Procedure:   e100_config_force_dplx
 *
 * Description: This routine will enable or disable force full duplex
 *              in the adapter's congif block, as indicated by the
 *              adapter's e100_bdconfig structure.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *
 * Returns:
 */
void
e100_config_force_dplx(bd_config_t *bdp)
{
    bdd_t          *bddp = bdp->bddp;   /* stores all adapter specific info */
    unsigned long   lock_flag_conf;
    
    spin_lock_irqsave(&bddp->config_lock, lock_flag_conf);
    
    /* We must force full duplex on if we are using PHY 0, and we are */
    /* supposed to run in FDX mode.  We do this because the e100 has only */
    /* one FDX# input pin, and that pin will be connected to PHY 1. */
    /* Changed the 'if' condition below to fix performance problem * at 10
     * full. The Phy was getting forced to full duplex while the MAC * was
     * not, because the cur_dplx_mode was not being set to 2 by SetupPhy. *
     * This is how the condition was, initially. * This has been changed so
     * that the MAC gets forced to full duplex * simply if the user has
     * forced full duplex. * * if (( bddp->phy_addr == 0 ) && (
     * bddp->cur_dplx_mode == 2 )) */
    /* The rest of the fix is in the PhyDetect code.          */
    if ((e100_speed_duplex[bdp->bd_number] == 2) ||
        (e100_speed_duplex[bdp->bd_number] == 4) ||
        ((bddp->phy_addr == 32) && (bddp->cur_dplx_mode == FULL_DUPLEX))) {
        if (!(bdp->config[19] & CB_CFIG_FORCE_FDX)) {
            bdp->config[19] |= CB_CFIG_FORCE_FDX;
            E100_CONFIG(bdp, 19);
        }
        
    } else {
        if (bdp->config[19] & CB_CFIG_FORCE_FDX) {
            bdp->config[19] &= ~CB_CFIG_FORCE_FDX;
            E100_CONFIG(bdp, 19);
        }
    }
    
    spin_unlock_irqrestore(&bddp->config_lock, lock_flag_conf);
}

/*
 * Procedure:   e100_config_long_rx
 *
 * Description: This routine will enable or disable reception of larger packets.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *      enable - should we enable this option or not
 *
 * Returns:
 */
static void
e100_config_long_rx(bd_config_t *bdp, boolean_t enable)
{
    if (enable == B_TRUE) {
        if (!(bdp->config[18] & CB_CFIG_LONG_RX_OK)) {
            bdp->config[18] |= CB_CFIG_LONG_RX_OK;
            E100_CONFIG(bdp, 18);
        }
        
    } else {
        if ((bdp->config[18] & CB_CFIG_LONG_RX_OK)) {
            bdp->config[18] &= ~CB_CFIG_LONG_RX_OK;
            E100_CONFIG(bdp, 18);
        }
    }
}

/*
 * Procedure:   e100_config_vlan_drop
 *
 * Description: This routine will enable or disable vlan tag stripping.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *      enable - should we enable this option or not
 *
 * Returns:
 */
static void
e100_config_vlan_drop(bd_config_t *bdp, boolean_t enable)
{
    if (enable == B_TRUE) {
        if (!(bdp->config[22] & CB_CFIG_VLAN_DROP_ENABLE)) {
            bdp->config[22] |= CB_CFIG_VLAN_DROP_ENABLE;
            E100_CONFIG(bdp, 22);
        }
        
    } else {
        if ((bdp->config[22] & CB_CFIG_VLAN_DROP_ENABLE)) {
            bdp->config[22] &= ~CB_CFIG_VLAN_DROP_ENABLE;
            E100_CONFIG(bdp, 22);
        }
    }
}

/*
 * Procedure:   e100_config_enable_tagging
 *
 * Description: This routine will enable or disable vlan tagging.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *      enable - should we enable this option or not
 *
 * Returns:
 */
void
e100_config_enable_tagging(bd_config_t *bdp, boolean_t enable)
{
    bdd_t          *bddp = bdp->bddp;   /* stores all adapter specific info */
    unsigned long   lock_flag_conf;
    
    spin_lock_irqsave(&bddp->config_lock, lock_flag_conf);
    
    if (bddp->rev_id >= D101B0_REV_ID)
        e100_config_long_rx(bdp, enable);
    
    if (bddp->rev_id >= D102_REV_ID)
        e100_config_vlan_drop(bdp, enable);
    
    spin_unlock_irqrestore(&bddp->config_lock, lock_flag_conf);
}
