#!/bin/sh
###############################################################################
#
#          Dell Inc. PROPRIETARY INFORMATION
# This software is supplied under the terms of a license agreement or
# nondisclosure agreement with Dell Inc. and may not
# be copied or disclosed except in accordance with the terms of that
# agreement.
#
# Copyright (c) 2000-2004 Dell Inc. All Rights Reserved.
#
# Module Name:
#
#   cli_ini_modifier.sh
#
# Abstract/Purpose:
#
#   This Script is used by OpenManage rpm packages to dynamically
#   register and unregister with OpenManage Command Line Interface 
#   at %post and %preun install sections of rpm packages.
#
# Environment:
#
#   Linux
#
###############################################################################

###############################################################################
#
#  Global Variables
#
###############################################################################
. /opt/dell/srvadmin/lib64/srvadmin-omilcore/Funcs.sh

# omsa cli ini file path
OMSA_CLI_INI_FILEPATH=/opt/dell/srvadmin/etc/openmanage/oma/ini/omcli.ini

# omreg keys
OMREG_READ_FILE_PATH=/opt/dell/srvadmin/etc/omreg.cfg
OMREG_WRITE_FILE_PATH=/opt/dell/srvadmin/etc/omreg.d/omreg-clireg.cfg
OMREG_POST_WRITE=$(cat /opt/dell/srvadmin/etc/omreg.d/*.cfg /etc/compat-omreg.cfg > /opt/dell/srvadmin/etc/omreg.cfg 2>/dev/null ||:)

OMREG_OM_OMCLIREGISTRATION_KEY=openmanage.omacore.clireg
OMREG_OM_OMACORE_INSTPATH_KEY=openmanage.omacore.installpath
OMREG_SYSLISTFILE=openmanage.syslistfile
ADDCLITOKEN="AddOMSACLI="
ADDCLITOKENBLADE="AddOMSACLIOnBlade="
DELCLITOKEN="DeleteOMSACLI="
DELCLITOKENBLADE="DeleteOMSACLIOnBlade="
REPLACESPACEWITH="REPLACINGSPACE"
FEATURE_INI_FILEPATH=
ACTION=
PLATFORMID=

###############################################################################
#
#  Function : AddToOMSACLI
#
#  Adds the given cli entry to omcli32.ini
#
###############################################################################
function CheckBlade 
{
    pattern=$1
    [ -z "${pattern}" ] && return 1
   
    SYSLIST_PATH=`GetRegVal "${OMREG_READ_FILE_PATH}" "${OMREG_SYSLISTFILE}"`

    awk -v found=0 -v pat="$pattern" -F= '
    /\[BladeSystems\]/ {found = 1} 
    { if((found == "1") && ($1 ~ pat)) { print "1";found=0 }}
    ' "${SYSLIST_PATH}"

    return 0
}

function AddToOMSACLI {
  # $1 : cli string

  # replace the REPLACESPACEWITH varibale with space
  CLIENTRY=`echo "$1" | sed "s/$REPLACESPACEWITH/ /"`
  TMP_OMSA_FILE=$OMSA_CLI_INI_FILEPATH".tmp"

  awk -v val_str="$CLIENTRY" -v tfile=$TMP_OMSA_FILE '
    BEGIN{
      # initialize variables
      tmpfile = tfile
      found_section = 0
      inside_section = 0
      added_new_kv = 0

      # split the clientry string into section, key and value strings
      # clientry string is formatted as: "section;key=value"
      # example: omreport storage;controllers=sngl,cdll,ssclp

      numsectoks = split(val_str, secarr, ";")
      section_str = "["secarr[1]"]"
      numkvtoks = split(secarr[2], kvarr, "=")
      key_str = kvarr[1]
      value_str = kvarr[2]
    }
    {
      if (found_section == 1 && inside_section == 1) {
        # we are in the correct section
        # there are 2 possibilites, the next line can be a
        # key-value pair or a new section.

        if (index($0, "[") == 1) {
          # check if key found
          if (added_new_kv == 0) {
            # key not found in the section
            # create for the first time.
            printf("%s=%s\n", key_str, value_str) >> tmpfile
            added_new_kv = 1;
          }

          # this line is a new section
          inside_section = 0
          printf("%s\n", $0) >> tmpfile
        }
        else {
          # this line is a key-value pair
          # search for the correct key
          if (index($0, key_str) == 1) {
            # found the key
            added_new_kv = 1

            # split the string using delimiter (=)
            arrlen = split($0, tokenarr, "=")
            if (arrlen > 0) {
              printf("%s=", tokenarr[1]) >> tmpfile
              printf("%s%s\n", tokenarr[2], value_str) >> tmpfile
            }
          }
          else {
            printf("%s\n", $0) >> tmpfile
          }
        }
      }
      else {
        if (index ($0, section_str) != 0) {
          found_section = 1
          inside_section = 1
        }
        printf("%s\n", $0) >> tmpfile
      }
    }
    END{
      if (found_section == 0) {
        # create section and add key-value
        printf("%s\n", section_str) >> tmpfile
        printf("%s=%s\n", key_str, value_str) >> tmpfile
      }
      else {
        if (added_new_kv == 0) {
          # key not found in the section
          # create for the first time.
          printf("%s=%s\n", key_str, value_str) >> tmpfile
        }
      }
      close(tmpfile)
    }
  ' $OMSA_CLI_INI_FILEPATH

  # rename the tmp file to the original file name
  mv $TMP_OMSA_FILE $OMSA_CLI_INI_FILEPATH
  chmod 644 ${OMSA_CLI_INI_FILEPATH}
}

###############################################################################
#
#  Function : IsComponentRegistered
#
#  Reads /etc/omreg file for confirmation
#
###############################################################################

function ExitIfComponentRegisteredWithOmreg {
    if grep "^$OMREG_OM_OMCLIREGISTRATION_KEY=" $OMREG_READ_FILE_PATH | grep -q $FEATURE_INI_FILEPATH; then
      exit 0
    fi
}

###############################################################################
#
#  Function : RegisterOMSACLI
#
#  Reads all the add cli entries from the feature ini file and invokes
#  AddToOMSACLI for each cli entry
#
###############################################################################

function RegisterOMSACLI {
  # $1 : path to feature ini file

  # Any cli entry with a space should be replaced with some unique string to
  # prevent it from being split into 2 tokens by "for" loop, if not done this
  # way, a tmp file is required. The Unique string used here is "REPLACINGSPACE" but
  # it will be changed.

  # do not register if already registered
  ExitIfComponentRegisteredWithOmreg

  FEATURE_INI_FILE=$1
  SYSID=$2

  for CLIENTRY in `grep "$ADDCLITOKEN" $FEATURE_INI_FILE | sed "s/^$ADDCLITOKEN//g" | sed "s/ /$REPLACESPACEWITH/g"`;
  do
    AddToOMSACLI "$CLIENTRY"
  done

  IS_BLADE=`CheckBlade "$SYSID"`
  if [ "$IS_BLADE" = "1" ]
  then
     for CLIENTRY in `grep "$ADDCLITOKENBLADE" $FEATURE_INI_FILE | sed "s/^$ADDCLITOKENBLADE//g" | sed "s/ /$REPLACESPACEWITH/g"`;
     do
        AddToOMSACLI "$CLIENTRY"
     done
 fi

  # register with /etc/omreg file
  echo "$OMREG_OM_OMCLIREGISTRATION_KEY=$FEATURE_INI_FILEPATH" >> $OMREG_WRITE_FILE_PATH
  $OMREG_POST_WRITE
}

###############################################################################
#
#  Function : RemoveEmptySection
#
#  For any given section, if sections is empty, remove it
#
###############################################################################

function RemoveEmptySection {
  # replace the REPLACESPACEWITH varibale with space
  CLIENTRY=`echo "$1" | sed "s/$REPLACESPACEWITH/ /"`
  TMP_OMSA_FILE=$OMSA_CLI_INI_FILEPATH".tmp"

  awk -v val_str="$CLIENTRY" -v tfile=$TMP_OMSA_FILE '
    BEGIN{
      # initialize variables
      tmpfile = tfile
      found_section = 0
      is_section_empty = 0

      # split the clientry string into section, key and value strings
      # clientry string is formatted as: "section;key=value"
      # example: omreport storage;controllers=sngl,cdll,ssclp

      numsectoks = split(val_str, secarr, ";")
      section_str = "["secarr[1]"]"
    }
    {
      if (found_section == 1) {
        # we are in the correct section
        # there are 2 possibilites, the next line can be a
        # key-value pair or a new section.

        if (index($0, "[") == 1) {
          # this line is a new section
          printf("%s\n", $0) >> tmpfile
          found_section = 0
        }
        else {
          # section is not empty,
          if ( is_section_empty == 0 ){
            printf("%s\n", section_str) >> tmpfile
            is_section_empty = 1
          }
          printf("%s\n", $0) >> tmpfile
        }
      }
      else {
        if (index ($0, section_str) != 0) {
          found_section = 1          
        }
        else {
          printf("%s\n", $0) >> tmpfile
        }
      }
    }
    END{
      close(tmpfile)
    }
  ' $OMSA_CLI_INI_FILEPATH

  # rename the tmp file to the original file name
  mv $TMP_OMSA_FILE $OMSA_CLI_INI_FILEPATH
  chmod 644 ${OMSA_CLI_INI_FILEPATH}
}


###############################################################################
#
#  Function : DelFromOMSACLI
#
#  Deletes the given cli entry from omcli32.ini
#
###############################################################################

function DelFromOMSACLI {

  # replace the REPLACESPACEWITH varibale with space
  CLIENTRY=`echo "$1" | sed "s/$REPLACESPACEWITH/ /"`
  TMP_OMSA_FILE=$OMSA_CLI_INI_FILEPATH".tmp"

  awk -v val_str="$CLIENTRY" -v tfile=$TMP_OMSA_FILE '
    BEGIN{
      # initialize variables
      tmpfile = tfile
      found_section = 0
      added_new_kv = 0

      # split the clientry string into section, key and value strings
      # clientry string is formatted as: "section;key=value"
      # example: omreport storage;controllers=sngl,cdll,ssclp

      numsectoks = split(val_str, secarr, ";")
      section_str = "["secarr[1]"]"
      numkvtoks = split(secarr[2], kvarr, "=")
      key_str = kvarr[1]
      value_str = kvarr[2]
    }
    {
      if (found_section == 1) {
        # we are in the correct section
        # there are 2 possibilites, the next line can be a
        # key-value pair or a new section.

        if (index($0, "[") == 1) {
          # this line is a new section, so key not present
          # reset section to not found
          found_section = 0
          printf("%s\n", $0) >> tmpfile
        }
        else {
          # this line is a key-value pair, search for the correct key
          if (index($0, key_str) == 1) {
            # found the key, split the entry using delimiter (=)
            arrlen = split($0, tokenarr, "=")
            if (arrlen > 0) {
              keyentry = tokenarr[1]
              valentry = tokenarr[2]

              if (index (valentry, value_str) != 0) {
                # current value contains value_str
                # remove value_str from current value
                newvaluehead=substr(valentry, 0, index (valentry, value_str) - 1)
                newvaluetail=substr(valentry, index (valentry, value_str) + length(value_str) , length(valentry))
                newvaluestring=sprintf("%s%s",newvaluehead, newvaluetail)
                if (length(newvaluestring) > 0) {
                  # otherwise, write the updated key=value to the file
                  printf("%s=%s\n", keyentry, newvaluestring) >> tmpfile
                }
              }
              else {
                printf("%s\n", $0) >> tmpfile
              }
            }
          }
          else {
            printf("%s\n", $0) >> tmpfile
          }
        }
      }
      else {
        if (index ($0, section_str) != 0) {
          found_section = 1
        }
        printf("%s\n", $0) >> tmpfile
      }
    }
    END{
      close(tmpfile)
    }
  ' $OMSA_CLI_INI_FILEPATH

  # rename the tmp file to the original file name
  mv $TMP_OMSA_FILE $OMSA_CLI_INI_FILEPATH
  chmod 644 ${OMSA_CLI_INI_FILEPATH}
}

###############################################################################
#
#  Function : UnRegisterOMSACLI
#
#  Reads all the delete cli entries from the feature ini file and invokes
#  DelFromOMSACLI for each cli entry.
#
###############################################################################

function UnRegisterOMSACLI {
  # $1 : path to feature ini file

  FEATURE_INI_FILE=$1
  for CLIENTRY in `grep "$DELCLITOKEN" $FEATURE_INI_FILE | sed "s/^$DELCLITOKEN//g" | sed "s/ /$REPLACESPACEWITH/g"`;
  do
    DelFromOMSACLI "$CLIENTRY"
    RemoveEmptySection "$CLIENTRY"
  done

  #delete if any blade specific entries
  for CLIENTRY in `grep "$DELCLITOKENBLADE" $FEATURE_INI_FILE | sed "s/^$DELCLITOKENBLADE//g" | sed "s/ /$REPLACESPACEWITH/g"`;
  do
    DelFromOMSACLI "$CLIENTRY"
    RemoveEmptySection "$CLIENTRY"
  done

  perl -n -i -e "print if ! m|^$OMREG_OM_OMCLIREGISTRATION_KEY=|;" $OMREG_WRITE_FILE_PATH
  $OMREG_POST_WRITE
}

###############################################################################
#
#  Function : Main
#
#  This is the starting point for the script, it invokes other functions
#  in the required order
#
###############################################################################

function Main {
  # Add to OMSA CLI or Delete from OMSA CLI depending on the request
  if [ "$ACTION" = "add" ];
  then
      RegisterOMSACLI $FEATURE_INI_FILEPATH $PLATFORMID
  elif [ "$ACTION" = "del" ];
  then
      UnRegisterOMSACLI $FEATURE_INI_FILEPATH $PLATFORMID
  else
      echo "Invalid Parameters to OMSA CLI Registration"
      echo "Second parameter has to be 'add' or 'del', but wasnt. It had '$ACTION'."
      echo "OMSA CLI Registration failed"
      exit 1
  fi
}

###############################################################################
#
#  Script execution begins here with the invocation
#  of Main function
#
###############################################################################
if [ $# -lt 2 ];
then
    echo "Invalid Parameters to OMSA CLI Registration"
    echo "OMSA CLI Registration failed!"
    exit 1
else
    FEATURE_INI_FILEPATH=$1
    FEATURE_INI=`basename "$FEATURE_INI_FILEPATH"`
    OMREG_OM_OMCLIREGISTRATION_KEY="${OMREG_OM_OMCLIREGISTRATION_KEY}.${FEATURE_INI}"

    ACTION=$2
    #Note: $3 is optional. current ussage is only for blades.added in om5.3
    PLATFORMID=$3 
fi

Main
