#!/bin/sh 
###############################################################################
#
#          Dell Inc. PROPRIETARY INFORMATION
# This software is supplied under the terms of a license agreement or
# nondisclosure agreement with Dell Inc. and may not
# be copied or disclosed except in accordance with the terms of that
# agreement.
#
# Copyright (c) 2000-2009 Dell Inc. All Rights Reserved.
#
# Module Name:
#
#   custom_install.sh
#
#
# Abstract/Purpose:
#
#   Interactive Custom Install Script to cutomize
#   Systems Management Managed Node RPM Package install.
#   This interactive script will enable the user to choose the optional
#   Systems Management Packages and make them available on the local machine.
#   It can also install the packages if the users chooses the scripted install.
#
#
# Environment:
#
#   Linux
#
###############################################################################

#set -x

###############################################################################
# Function : SetErrorAndInterrupt
#
#   The "tee" command used to write log, continues execution after
#   the "tee" when any part of the utility exits. Additionally, global
#   variables do not seem to be updated when execution resumes. So, all
#   exit errors are mapped to the signal HUP. kill works very quickly
#   AFTER exit is called, so to allow user messages to output, sleep
#   was added.
###############################################################################
function SetErrorAndInterrupt
{
    FF_EXIT=$1
    sleep 1
    exit $FF_EXIT
    kill -HUP $$
}

function CheckMediaMessage {
       echo "Error: Check whether the Server Administrator media is corrupted."
       echo "       If you are using the Web pack to install Server Administrator,"
       echo "       ensure that you use the Web pack corresponding to the"
       echo "       architecture and operating system on your system."
       echo ""
}

###############################################################################
# Function : Usage
#
## Display the usage messages
###############################################################################
function Usage {
cat <<EOF
Usage: ${0} [OPTION]...
Server Administrator Custom Install Utility.
This Utility will run in the interactive mode if no options
are given and runs silently if one or more options are given.

Components:

[-x|--express]    installs all applicable components. Any other 
                  options passed will be ignored.
                    1. Server Instrumentation and Default 
                       Management Interfaces
                    2. Storage Management
                    3. RAC, if applicable
                    4. Server Administrator Web Server

[-d|--dellagent]  installs 'Server Instrumentation and 
                  Default Management Interfaces':
                    1. Server Instrumentation
                    2. Server Administrator CLI Interface
                    3. Server Administrator SNMP Interface
                    4. Server Administrator Operating System Log Interface

[-s|--storage]    installs 'Storage Management' and 'Server Instrumentation 
                  and Default Management Interfaces'. 

[-r|--rac]        installs applicable 'RAC' components and 'Server 
                  Instrumentation and Default Management Interfaces'. 

[-g|--agent]      installs 'Server Instrumentation' only. This requires 
                  at least one management interface option in combination.

[-t|--stragent]   installs 'Server Instrumentation' and 'Storage Management'. 
                  This requires at least one management interface option 
                  in combination.

Management Interfaces:

[-w|--web]        installs 'Server Administrator Web Server' management 
                  interface. This can also be used independent of other options.

[-c|--cimagent]   installs 'Remote Enablement' Interface.

[-i|--cli]        installs 'Server Administrator CLI Interface'. 

[-l|--oslog]      installs 'Server Administrator Operating System Log 
                  Interface'.

[-m|--snmp]       installs 'Server Administrator SNMP Interface'.

[-z|--corecim]    installs core CIM Interface.

These can be used in combination with above options:
[-a|--autostart]  start the installed services after components have 
                  been installed.

[-p|--preserve]   preserve the screen without clearing off.

Additional Options:
[-h|--help]       displays this help
[-u|--update]     update applicable Server Administrator components.

EOF
  
  if [ -n "$1" ] && [ "$1" = "Help" ]
  then
	exit 0
  fi
  SetErrorAndInterrupt 10
}

# check for help
    echo "${*}" | egrep "^.*--help.*$" >/dev/null 2>&1
    if [ $? == 0 ]; then
      Usage Help
    fi

    echo "${*}" | egrep "^.*-h.*$" >/dev/null 2>&1
    if [ $? == 0 ]; then
      Usage Help
    fi

# ensure sbin utilities are available in path, so that su will also work
export PATH=/usr/kerberos/sbin:/usr/local/sbin:/sbin:/usr/sbin:$PATH

# install log file
SRVADMIN_INSTALL_LOG="/var/log/srvadmin.log"

# Global Variables
OMI_LOCK_FILE=/var/lock/LCK..srvadmin-install.sh.0
KEEP_LOCK=false
LOCAL_DEST_PATH=
PATH_TO_RPMS=
PATH_TO_RPMS_UPGRADE=
PATH_TO_RPMS_SUFFIX=
DISTRO=
PATH_TO_SCRIPT=
PATH_TO_OPENSOURCE_RPMS=
FIRST_SCREEN=0

SYSIDFILEPATH=
SYSIDFILE=prereqcheck/syslist.txt

LICENSEFILEPATH=
LICENSEFILE=prereqcheck/license.txt

EIGHTGSYSIDFILEPATH=
EIGHTGSYSIDFILE=prereqcheck/8gsyslist.txt

NINEGSYSIDFILEPATH=
NINEGSYSIDFILE=prereqcheck/9gsyslist.txt

IDRACSYSIDFILEPATH=
IDRACSYSIDFILE=prereqcheck/idracsyslist.txt

SYSREPORTPATH=
SYSREPORT=prereqcheck/smbios-sys-info-lite

PKG_PREFIX=srvadmin
DESTINATION_DIR=

DRAC4PRESENT=0
DRAC5PRESENT=0
IDRACPRESENT=0
APPS_PRESENT=0

# Operating system ("OS") types.
# These are the return values for the following function:
#     GetOSType
#
GBL_OS_TYPE_ERROR=0
GBL_OS_TYPE_UKNOWN=1
GBL_OS_TYPE_RHEL3=2
GBL_OS_TYPE_RHEL4=3
GBL_OS_TYPE_SLES10=4
GBL_OS_TYPE_RHEL5=5
GBL_OS_TYPE_SLES11=6
GBL_OS_TYPE_ESX40=7
GBL_OS_TYPE_ESX41=8
GBL_OS_TYPE_RHEL6=9
GBL_OS_TYPE_RHEL7=10

# Keys to the array subscripts
#INST_KEY="INST_KEY"
AGENT_KEY="AGENT_KEY"
CLI_KEY="CLI_KEY"
SNMP_KEY="SNMP_KEY"
LOG_KEY="LOG_KEY"
STD_AGENT_KEY="SSA_KEY"
SAWB_KEY="SAWEB_KEY"
OMSM_KEY="STOR_KEY"

RAC3_KEY="RAC_3_KEY"

RAC4_KEY="RAC_4_KEY"
RAC4SA_KEY="RAC_SA_4_KEY"

RAC5_KEY="RAC_5_KEY"
RAC5SA_KEY="RAC_SA_5_KEY"

IDRAC_KEY="IDRAC_KEY"
IDRACSA_KEY="IDRAC_SA_KEY"

ALL_KEY="ALL_KEY"

# Package group description array
SERVER_ADMINISTRATOR_AGENT_DESC="$AGENT_KEY=Server Instrumentation"
SERVER_ADMINISTRATOR_STANDARD_AGENT_DESC="$STD_AGENT_KEY=Remote Enablement Interface"
STORAGE_MANAGEMENT_DESC="$OMSM_KEY=Storage Management"
SERVER_ADMINISTRATOR_WEB_DESC="$SAWB_KEY=Server Administrator Web Server Interface"
CLI_DESC="$CLI_KEY=Server Administrator CLI Interface"
SNMP_DESC="$SNMP_KEY=Server Administrator SNMP Interface"
LOG_DESC="$LOG_KEY=Server Administrator Operating System Log Interface"

REMOTE_ACCESS_DESC="$RAC3_KEY=Remote Access"
REMOTE_ACCESS_CORE_DESC="$RAC4_KEY=Remote Access Core"
REMOTE_ACCESS_CORE_DESC5="$RAC5_KEY=Remote Access Core Components"
REMOTE_ACCESS_CORE_DESC_IDRAC="$IDRAC_KEY=Remote Access Core Component"

REMOTE_ACCESS_SA_PLUGINS_DESC="$RAC4SA_KEY=Remote Access SA Plugins"
REMOTE_ACCESS_SA_PLUGINS_DESC5="$RAC5SA_KEY=Remote Access SA Plugin Components"
REMOTE_ACCESS_SA_PLUGINS_DESC_IDRAC="$IDRACSA_KEY=Remote Access SA Plugin Component"

SRVADMIN_ALL_DESC="${ALL_KEY}=All"

COMP_PKGS_DESC_ARR=([0]="${SERVER_ADMINISTRATOR_AGENT_DESC}" [1]="${SERVER_ADMINISTRATOR_STANDARD_AGENT_DESC}" [2]="${SERVER_ADMINISTRATOR_WEB_DESC}" [3]="${STORAGE_MANAGEMENT_DESC}" [4]="${REMOTE_ACCESS_DESC}" [5]="${REMOTE_ACCESS_CORE_DESC}" [6]="${REMOTE_ACCESS_CORE_DESC5}" [7]="${REMOTE_ACCESS_CORE_DESC_IDRAC}" [8]="${REMOTE_ACCESS_SA_PLUGINS_DESC}" [9]="${REMOTE_ACCESS_SA_PLUGINS_DESC5}" [10]="${REMOTE_ACCESS_SA_PLUGINS_DESC_IDRAC}"  [11]="${CLI_DESC}" [12]="${SNMP_DESC}" [13]="${LOG_DESC}"  [14]="${SRVADMIN_ALL_DESC}")

# Packages for group array 
BASE_OM_PKG_LIST="$PKG_PREFIX-deng $PKG_PREFIX-hapi $PKG_PREFIX-ominst $PKG_PREFIX-omacs $PKG_PREFIX-isvc $PKG_PREFIX-omilcore $PKG_PREFIX-xmlsup"

SILENT_BASE_OM_PKG_LIST="$PKG_PREFIX-xmlsup $PKG_PREFIX-deng $PKG_PREFIX-hapi $PKG_PREFIX-omacore $PKG_PREFIX-isvc $PKG_PREFIX-omilcore $PKG_PREFIX-smcommon $PKG_PREFIX-omcommon $PKG_PREFIX-idrac-snmp $PKG_PREFIX-deng-snmp $PKG_PREFIX-isvc-snmp $PKG_PREFIX-omacs $PKG_PREFIX-ominst $PKG_PREFIX-oslog"

WEB_OM_PKG_LIST="$PKG_PREFIX-omilcore $PKG_PREFIX-smcommon $PKG_PREFIX-omcommon $PKG_PREFIX-omacs $PKG_PREFIX-smweb $PKG_PREFIX-tomcat $PKG_PREFIX-jre"
STORAGE_OM_PKG_LIST="$PKG_PREFIX-storelib $PKG_PREFIX-storage $PKG_PREFIX-sysfsutils $PKG_PREFIX-smcommon"
CLI_OM_PKG_LIST="$PKG_PREFIX-omacore $PKG_PREFIX-omcommon"
SNMP_OM_PKG_LIST="$PKG_PREFIX-deng-snmp $PKG_PREFIX-idrac-snmp $PKG_PREFIX-isvc-snmp"
LOG_OM_PKG_LIST="$PKG_PREFIX-oslog"
ITP_OM_PKG_LIST="$PKG_PREFIX-itunnelprovider"

# Legacy Instrumentation.
#INST_AGENT_PKGS="$INST_KEY=$BASE_INST_PKG"
DELL_AGENT_PKGS="$AGENT_KEY=$BASE_OM_PKG_LIST"
DELL_AGENT_PKGS_RAC5="$AGENT_KEY=$BASE_OM_PKG_LIST $PKG_PREFIX-rac-components"
DELL_AGENT_PKGS_IDRAC="$AGENT_KEY=$BASE_OM_PKG_LIST $PKG_PREFIX-rac-components"

# Standard Insturmentation.
CIM_AGENT_PKGS="$STD_AGENT_KEY=$BASE_OM_PKG_LIST $ITP_OM_PKG_LIST"
CIM_AGENT_PKGS_RAC5="$STD_AGENT_KEY=$BASE_OM_PKG_LIST $ITP_OM_PKG_LIST $PKG_PREFIX-rac-components"
CIM_AGENT_PKGS_IDRAC="$STD_AGENT_KEY=$BASE_OM_PKG_LIST $ITP_OM_PKG_LIST $PKG_PREFIX-rac-components"
SA_WEB_PKGS="$SAWB_KEY=$WEB_OM_PKG_LIST"
CLI_PKGS="$CLI_KEY=$BASE_OM_PKG_LIST $CLI_OM_PKG_LIST $PKG_PREFIX-rac-components"
SNMP_PKGS="$SNMP_KEY=$BASE_OM_PKG_LIST $SNMP_OM_PKG_LIST $PKG_PREFIX-rac-components"
LOG_PKGS="$LOG_KEY=$BASE_OM_PKG_LIST $LOG_OM_PKG_LIST $PKG_PREFIX-rac-components"
OMSM_PKGS="$OMSM_KEY=$BASE_OM_PKG_LIST $STORAGE_OM_PKG_LIST"
OMSM_PKGS_RAC5="$OMSM_KEY=$BASE_OM_PKG_LIST $STORAGE_OM_PKG_LIST $PKG_PREFIX-rac-components"
OMSM_PKGS_IDRAC="$OMSM_KEY=$BASE_OM_PKG_LIST $STORAGE_OM_PKG_LIST $PKG_PREFIX-rac-components"

RAC_3_PKGS="$RAC3_KEY=$BASE_OM_PKG_LIST $PKG_PREFIX-rac3-components $PKG_PREFIX-racdrsc3 $PKG_PREFIX-racser-devel $PKG_PREFIX-racvnc"
RAC_4_CORE="$RAC4_KEY=$PKG_PREFIX-racsvc $PKG_PREFIX-rac4-populator $PKG_PREFIX-rac-components $PKG_PREFIX-racdrsc $PKG_PREFIX-racadm4 $PKG_PREFIX-omilcore $PKG_PREFIX-omcommon $PKG_PREFIX-deng $PKG_PREFIX-isvc $PKG_PREFIX-hapi"
RAC_5_CORE="$RAC5_KEY=$PKG_PREFIX-argtable2 $PKG_PREFIX-deng $PKG_PREFIX-hapi $PKG_PREFIX-isvc $PKG_PREFIX-racadm5 $PKG_PREFIX-omilcore $PKG_PREFIX-racadm4 $PKG_PREFIX-racdrsc $PKG_PREFIX-rac-components $PKG_PREFIX-omacs"
IDRAC_CORE="$IDRAC_KEY=$PKG_PREFIX-argtable2 $PKG_PREFIX-idracadm $PKG_PREFIX-idrac7 $PKG_PREFIX-idracadm7 $PKG_PREFIX-racadm4 $PKG_PREFIX-hapi $PKG_PREFIX-omilcore $PKG_PREFIX-racdrsc $PKG_PREFIX-rac-components $PKG_PREFIX-idrac-ivmcli $PKG_PREFIX-idrac-vmcli $PKG_PREFIX-deng $PKG_PREFIX-isvc $PKG_PREFIX-idrac-snmp $PKG_PREFIX-omacs"

RAC_4_SA_PKGS="$RAC4SA_KEY=$BASE_OM_PKG_LIST $PKG_PREFIX-racdrsc $PKG_PREFIX-rac4-populator $PKG_PREFIX-racadm4 $PKG_PREFIX-racsvc $PKG_PREFIX-rac-components"
RAC_5_SA_PKGS="$RAC5SA_KEY=$BASE_OM_PKG_LIST $PKG_PREFIX-racdrsc $PKG_PREFIX-isvc $PKG_PREFIX-rac-components $PKG_PREFIX-racadm5 $PKG_PREFIX-racadm4 $PKG_PREFIX-argtable2"
IDRAC_SA_PKGS="$IDRACSA_KEY=$BASE_OM_PKG_LIST $PKG_PREFIX-racdrsc $PKG_PREFIX-rac-components $PKG_PREFIX-isvc $PKG_PREFIX-racadm4 $PKG_PREFIX-idracadm $PKG_PREFIX-idrac-vmcli $PKG_PREFIX-idrac-ivmcli $PKG_PREFIX-idrac7 $PKG_PREFIX-idracadm7 $PKG_PREFIX-argtable2 $PKG_PREFIX-idrac-snmp"

ALL_SA_NORAC_PKGS="${ALL_KEY}=$BASE_OM_PKG_LIST $WEB_OM_PKG_LIST $STORAGE_OM_PKG_LIST $ITP_OM_PKG_LIST $CLI_OM_PKG_LIST $SNMP_OM_PKG_LIST $LOG_OM_PKG_LIST"
ALL_SA_RAC3_PKGS="${ALL_KEY}=$BASE_OM_PKG_LIST $WEB_OM_PKG_LIST $STORAGE_OM_PKG_LIST $ITP_OM_PKG_LIST $PKG_PREFIX-rac3-components $PKG_PREFIX-racdrsc3 $PKG_PREFIX-racser-devel $PKG_PREFIX-racvnc"
ALL_SA_RAC4_PKGS="${ALL_KEY}=$BASE_OM_PKG_LIST $WEB_OM_PKG_LIST $STORAGE_OM_PKG_LIST $ITP_OM_PKG_LIST $CLI_OM_PKG_LIST $SNMP_OM_PKG_LIST $LOG_OM_PKG_LIST $PKG_PREFIX-racadm4 $PKG_PREFIX-racdrsc $PKG_PREFIX-rac-components $PKG_PREFIX-racsvc $PKG_PREFIX-rac4-populator"
ALL_SA_RAC5_PKGS="${ALL_KEY}=$BASE_OM_PKG_LIST $WEB_OM_PKG_LIST $STORAGE_OM_PKG_LIST $ITP_OM_PKG_LIST $CLI_OM_PKG_LIST $SNMP_OM_PKG_LIST $LOG_OM_PKG_LIST $PKG_PREFIX-argtable2 $PKG_PREFIX-racadm4 $PKG_PREFIX-racadm5 $PKG_PREFIX-racdrsc $PKG_PREFIX-rac-components"
ALL_SA_IDRAC_PKGS="${ALL_KEY}=$BASE_OM_PKG_LIST $WEB_OM_PKG_LIST $STORAGE_OM_PKG_LIST $ITP_OM_PKG_LIST $CLI_OM_PKG_LIST $SNMP_OM_PKG_LIST $LOG_OM_PKG_LIST $PKG_PREFIX-argtable2 $PKG_PREFIX-racadm4 $PKG_PREFIX-idracadm $PKG_PREFIX-racdrsc $PKG_PREFIX-rac-components $PKG_PREFIX-idrac-vmcli $PKG_PREFIX-idrac-ivmcli $PKG_PREFIX-idrac7 $PKG_PREFIX-idracadm7 $PKG_PREFIX-idrac-snmp"

SA_ALL="${ALL_KEY}="

COMP_PKGS_ARR=""

# optional packages for this system
# this variable is populated by the prereq function, depending on
# what is applicable on this system
APPLICABLE_OPTIONAL_PKGS=

# Selected Packages
SELECTED_COMP_DESC=
SELECTED_PKGS=

# users input variable
USER_SELECTED_INDEX=
USER_SELECTED_PKG=

# Preupgrade script name
PRE_UPGRADE_SCRIPT=supportscripts/preupgrade.sh

# standard shell return codes
ERROR_CODE_FILE_OPEN_ERROR=17
ERROR_CODE_FILE_WRITE_ERROR=19

# previous instrumentation packages
INST_3X_PREFIX=instsvc
DENG_3X_PREFIX=dataeng
SUPTLIB_3X_PREFIX=suptlib
INST_2X_PREFIX=dellomsa
OM_MISC_3X="ServerAdministrator"
DRAC_3X_RPM_LIST="racadm racfirmware racser racser-devel racvnc RACOMComponents drscfirmware-rh70 drscfirmware-rh71-up drsc-devel-rh70-up drsc-devel-rh71-up drsc-rh70-up drsc-rh71-up drsvnc-rh70-up drsvnc-rh71-up"
PRE_SM_RPM_LIST="$INST_3X_PREFIX $DENG_3X_PREFIX $SUPTLIB_3X_PREFIX $INST_2X_PREFIX $OM_MISC_3X $DRAC_3X_RPM_LIST"

# custom install path
SA_INSTALL_PATH=/opt/dell/srvadmin
DEFAULT_PREFIX=$SA_INSTALL_PATH
INSTALL_PREFIX=""
RPM_INSTALL="rpm"
SERVICES_SCRIPT=$SA_INSTALL_PATH/sbin/${PKG_PREFIX}-services.sh

# install mode
CURRENTLY_INSTALLED_KEYS=""
INSTALL_MODE=""
SKIP_UPGRADE="1"
OMCOMMON_NOT_PRESENT="0"
CURRENTLY_INSTALLED_PKGS=""
TMP_APP_PKGS=""
CHOSEN_PKGS=""
AUTO_START=0
PRESERVE_SCREEN=0

# install flags
START_SERVICES_DEF="no"

# silent install options
SHORT_OPTS=([0]="x" [1]="d" [2]="c" [3]="s" [4]="r" [5]="w" [6]="u" [7]="a" [8]="p" [9]="g" [10]="i" [11]="m"  [12]="l" [13]="t" [14]="z")
LONG_OPTS=([0]="express" [1]="dellagent" [2]="cimagent" [3]="storage" [4]="rac" [5]="web" [6]="update" [7]="autostart" [8]="preserve"  [9]="agent"  [10]="cli"  [11]="snmp" [12]="oslog" [13]="stragent" [14]="corecim")
USER_OPTS=""
SILENT_PKGS_LIST=""
SILENT_AGENT="agent=$BASE_OM_PKG_LIST $PKG_PREFIX-rac-components"
SILENT_DELL_AGENT="dellagent=$SILENT_BASE_OM_PKG_LIST"
SILENT_DELL_AGENT_RAC5="dellagent=$SILENT_BASE_OM_PKG_LIST $PKG_PREFIX-rac-components"
SILENT_DELL_AGENT_IDRAC="dellagent=$SILENT_BASE_OM_PKG_LIST $PKG_PREFIX-rac-components"
SILENT_CLI="cli=$BASE_OM_PKG_LIST $CLI_OM_PKG_LIST $PKG_PREFIX-rac-components"
SILENT_SNMP="snmp=$BASE_OM_PKG_LIST $SNMP_OM_PKG_LIST $PKG_PREFIX-rac-components"
SILENT_OSLOG="oslog=$BASE_OM_PKG_LIST $LOG_OM_PKG_LIST $PKG_PREFIX-rac-components"
SILENT_CORE_STORAGE="stragent=$BASE_OM_PKG_LIST $STORAGE_OM_PKG_LIST $PKG_PREFIX-rac-components"
SILENT_CORE_CIM="corecim=$BASE_OM_PKG_LIST $ITP_OM_PKG_LIST $PKG_PREFIX-rac-components"
SILENT_CIM_AGENT="cimagent=$SILENT_BASE_OM_PKG_LIST $ITP_OM_PKG_LIST"
SILENT_CIM_AGENT_RAC5="cimagent=$SILENT_BASE_OM_PKG_LIST $ITP_OM_PKG_LIST $PKG_PREFIX-rac-components"
SILENT_CIM_AGENT_IDRAC="cimagent=$SILENT_BASE_OM_PKG_LIST $ITP_OM_PKG_LIST $PKG_PREFIX-rac-components"

SILENT_WEB="web=$WEB_OM_PKG_LIST"

SILENT_STORAGE="storage=$SILENT_BASE_OM_PKG_LIST $STORAGE_OM_PKG_LIST"
SILENT_STORAGE_RAC5="storage=$SILENT_BASE_OM_PKG_LIST $STORAGE_OM_PKG_LIST $PKG_PREFIX-rac-components"
SILENT_STORAGE_IDRAC="storage=$SILENT_BASE_OM_PKG_LIST $STORAGE_OM_PKG_LIST $PKG_PREFIX-rac-components"

SILENT_RAC3="rac=$BASE_OM_PKG_LIST $PKG_PREFIX-rac3-components $PKG_PREFIX-racdrsc3 $PKG_PREFIX-racser-devel $PKG_PREFIX-racvnc"
SILENT_RAC4="rac=$BASE_OM_PKG_LIST $PKG_PREFIX-racdrsc $PKG_PREFIX-racadm4 $PKG_PREFIX-rac-components $PKG_PREFIX-racsvc $PKG_PREFIX-rac4-populator"
SILENT_RAC5="rac=$SILENT_BASE_OM_PKG_LIST $PKG_PREFIX-argtable2 $PKG_PREFIX-racdrsc $PKG_PREFIX-racadm5 $PKG_PREFIX-rac-components $PKG_PREFIX-racadm4 $PKG_PREFIX-isvc"
SILENT_IDRAC="rac=$SILENT_BASE_OM_PKG_LIST $PKG_PREFIX-argtable2 $PKG_PREFIX-racdrsc $PKG_PREFIX-idracadm $PKG_PREFIX-idrac7 $PKG_PREFIX-idracadm7 $PKG_PREFIX-rac-components $PKG_PREFIX-racadm4 $PKG_PREFIX-deng $PKG_PREFIX-omcommon $PKG_PREFIX-idrac-vmcli $PKG_PREFIX-idrac-ivmcli"
SELECTED_SILENT_PKGS=""

ENABLE_CONSOLE_ONLY_INSTALL=1
UNSUPPORTED_OS_FOR_SSA=0

# Need to populate the correct install status.
UPGRADE_STATUS=0
INSTALL_STATUS=0
INSTALL_FLAG=0

#stores commandline with --prefix removed
MOD_CM=""

###############################################################################
# Function : InterruptOMI
#
#   Process and clean up from INT, QUIT or TERM signal
###############################################################################
function InterruptOMI
{
    echo " "
    echo "Interrupting Install of $0 due to signal"
    exit -1
}

###############################################################################
# Function : InterruptOMIHUP
#
#   Process and clean up from an application error, as "tee" command used
#   to write log, continues execution after the "tee" when any part of the
#   utility exits.
###############################################################################
function InterruptOMIHUP
{
    exit -1
}

###############################################################################
# Function : InterruptOMIExit
#
#   Process and clean up from utilty exit
###############################################################################
function InterruptOMIExit
{
    THEEXITVALUE=$?
    #SLEEP_TIME=2
    [ ${KEEP_LOCK} == false ] && rm -f ${OMI_LOCK_FILE} > /dev/null 2>&1
    #sleep ${SLEEP_TIME} # gives users a chance to read output on temporary terminals
    [ -n "${OMIDEBUG}" ] && Log "Stop"
    exit $THEEXITVALUE
}

###############################################################################
# Function : SetSuccessAndExit
#
#   The "tee" command used to write log, continues execution after
#   the "tee" when any part of the utility exits. Additionally, global
#   variables do not seem to be updated when execution resumes. So, all
#   non error exits are mapped to the signal EXIT. kill works very quickly
#   AFTER exit is called, so to allow user messages to output, sleep
#   was added.
###############################################################################
function SetSuccessAndExit
{
    kill -EXIT $$
    sleep 1

    if [[ $OMCOMMON_NOT_PRESENT -eq 1 ]]; then

         echo "$ARG_CHECK" | grep "\-x" >/dev/null 2>&1
         if [ $? == 0 ]; then
             exit 22
         fi

         echo "$ARG_CHECK" | grep "\-u" >/dev/null 2>&1
         if [ $? == 0 ]; then
             exit 23
         fi

    fi

    exit 0
}

###############################################################################
#
# Function:    CheckForSupportedSystem
#
# Description: Check if the system is supported by doing a system id check.
#              Moved from "srvadmin-openipmi.sh" to "srvadmin-install.sh"
#              Check CR33667 for more details.
#
# Returns:     0 = False. System is not supported.
#              1 = True.  System is supported.
#
###############################################################################
CheckForSupportedSystem()
{
    # Check for override.
    if [ -n "${OMIIGNORESYSID}" ]; then
        echo " - System supported check: Override is set. Skipping check."
        return 1
    fi

    # $1 is the path to the script, inluding the script name
    SPACES_REPLMT=`echo "$1" | sed "s/ /\\\ /"`
    SCRIPT_PATH=`dirname "$SPACES_REPLMT"`
    # Locate some directories and files.
    # First get path to the "srvadmin/linux" directory using the path to
    # the script.
    CURRENT_DIR=`pwd | sed "s/ /\\\ /"`
    cd "$SCRIPT_PATH"
    cd ..

    # Set some file paths.
    SYSREPORT1=supportscripts/prereqcheck/smbios-sys-info-lite
    SYSIDFILE1=supportscripts/prereqcheck/syslist.txt
    SYSIDFILEPATH1="$LINUXROOT/$SYSIDFILE1"
    SYSREPORTPATH1="$LINUXROOT/$SYSREPORT1"

    # Set default value.
    SUPPORTED_SYSTEM=0

    # Get the system id.
    SYSID=`"${SYSREPORTPATH1}" | grep "System ID" | awk -F0x '{print $2}'`

    # Check for sysid Injection
    if [ -n "${OM_SYSTEM_ID}" ]; then
       SYSID=`echo ${OM_SYSTEM_ID} | sed "s/ //g"`
    fi

    if [ -n "${SYSID}" ]; then
        grep -qi "^${SYSID}=" "${SYSIDFILEPATH1}"
        if [ "$?" = "0" ]; then
            # System id supported.
            SUPPORTED_SYSTEM=1
        fi
    fi

    cd "${CURRENT_DIR}"

    return ${SUPPORTED_SYSTEM}
}

###############################################################################
# Function : FreshenSMBIOSPackages
# Check whether libsmbios2 and smbios-utils-bin are present on the system.
# If 32-bit bios packages are already present on the system then do a 
# freshen (-Fvh). If 64-bit bios packages are present on the system, then,
# install 32-bit bios packages (ivh) from the DVD.
## 
###############################################################################

function FreshenSMBIOSPackages {

  freshen_list=""
  install_list=""

  for rpm in `ls libsmbios* smbios-utils-bin*`
  do
     # query rpm for name and arch
     name_arch=$(rpm -qp --qf "%{name}.%{arch}" $rpm)

     # is a matching name/arch rpm installed on the system
     if rpm --quiet -q $name_arch; then
         # if so, freshen it
         freshen_list="$freshen_list $rpm"
     else
         #install it
         install_list="$install_list $rpm"
     fi
  done

  [ -n "$freshen_list" ] && rpm -Fvh $freshen_list
  [ -n "$install_list" ] && rpm -ivh $install_list
}


###############################################################################
# Function : InstallOpenWSManPackages
#
# Check whether libwsman1 and openwsman-client packages of 32-bit arch 
# are already installed on the box. If not installed already, then install them.
## 
###############################################################################

function InstallOpenWSManPackages {

  install_list=""

  for rpm in `ls $PATH_TO_OPENSOURCE_RPMS/libwsman1-* $PATH_TO_OPENSOURCE_RPMS/openwsman-client*`
  do
     # Query rpm for name and arch
     name_arch=$(rpm -qp --qf "%{name}.%{arch}" $rpm)

     # Is a matching name/arch rpm installed on the system
     if ! rpm --quiet -q $name_arch; then
         # Package of the correct arch is not installed.
         # Add it to the install list.
         install_list="$install_list $rpm"
     fi
  done

  [ -n "$install_list" ] && rpm -ivh $install_list
}


###############################################################################
# Function : StartServices
#
## Start OM services
###############################################################################
function StartServices {

  rpm -qa | grep "srvadmin-omilcore" > /dev/null
  if [ $? = 0 ]
  then
  # allow the answer to be passed through an existing environment variable
  sa_start="${START_SERVICES}"
  if [ -z "${sa_start}" ] ; then
    # else, prompt for the response
    cat <<EOF

   Do you want the Server Administrator services started?

EOF
    sa_start=`Prompt "   Press ('y' for yes | 'Enter' to exit): "`

    # use the default answer if no reply
    [ -z "${sa_start}" ] && sa_start="${START_SERVICES_DEF}"
  fi

  # now start if 'yes'
  #fix for DF 39092 - added a '$' in the regexp - BK
  if echo "${sa_start}" | grep -qi "^y$" ; then
    ldconfig
    "${SERVICES_SCRIPT}" start
    echo ""
  fi
  fi
}

##
## Prompt
##
function Prompt {
  MSG="${1}"

  # prompt and get response
  echo -n "${MSG}" 1>&2
  read RESP

  # remove leading/trailing spaces
  echo "${RESP}" | sed 's#^[[:space:]]*##; s#[[:space:]]*$##'
}

##
## Uniq
##
## to eliminate the possibility of the same rpm filename being
## present in the list to install more than once, this function
## returns a list of unique entries
##
function Uniq {
  for i in ${*} ; do
    echo $i
  done |
    sort |
      uniq
}



function GetSysId {
  # to assist in rac package detection, allow the sysid to be passed in the IGNORE variable
  if [ -n "${OMIIGNORESYSID}" ] ; then
    echo "${OMIIGNORESYSID}"
  elif [ -n "${OM_SYSTEM_ID}" ]; then
    echo ${OM_SYSTEM_ID} | sed "s/ //g"
  else
    "${SYSREPORTPATH}" | grep "System ID" | awk -F0x '{print $2}'
  fi
}

function IsSystem11GorHigher
{
	SYSID=`GetSysId`

	SYSID_HEX="0x0$SYSID"
	SYSID_DEC=`printf "%d" $SYSID_HEX`

# 0235 is the systemid for Bluefish (PER710), the first 11G server
        MIN_11G_SYSID_HEX=0x0235
        MIN_11G_SYSID_DEC=`printf "%d" $MIN_11G_SYSID_HEX`

	if [ $SYSID_DEC -ge $MIN_11G_SYSID_DEC ]; then
           if [ $SYSID != "023F" ] && [ $SYSID != "0240" ] && [ $SYSID != "0295" ] && [ $SYSID != "025C" ] && [ $SYSID != "027B" ]
           then
		return 0 
           fi
	fi

	return 1
}

function add_realssd
{
     RPM_LIST=${1}

     echo "${RPM_LIST}" | grep "srvadmin-realssd" > /dev/null
     if [ $? == 0 ] 
     then
        echo ${RPM_LIST}
     else
        #for srvadmin-realssd (check if this rpm is required)
        echo "${RPM_LIST}" | grep "srvadmin-storage-$CURRENT_VERSION" > /dev/null
        if [ $? == 0 ] 
        then
	     STORAGE_RPM_NAME=`ls srvadmin-storage-$CURRENT_VERSION*`
	     DEP_LIST=`rpm -qp --requires "${STORAGE_RPM_NAME}"` 
	     echo "${DEP_LIST}" | grep "srvadmin-realssd" > /dev/null
             if [ $? == 0 ] 
             then
	         REALSSD_RPM_NAME=`ls srvadmin-realssd-*`
                 RPM_LIST="${RPM_LIST} ${REALSSD_RPM_NAME}"
             fi 
        fi 
        echo ${RPM_LIST}
     fi
 }


function add_nvme
{
     RPM_LIST=${1}

     echo "${RPM_LIST}" | grep "srvadmin-nvme" > /dev/null
     if [ $? == 0 ] 
     then
        echo ${RPM_LIST}
     else
        #for srvadmin-nvme (check if this rpm is required)
        echo "${RPM_LIST}" | grep "srvadmin-storage-$CURRENT_VERSION" > /dev/null
        if [ $? == 0 ] 
        then
	     STORAGE_RPM_NAME=`ls srvadmin-storage-$CURRENT_VERSION*`
	     DEP_LIST=`rpm -qp --requires "${STORAGE_RPM_NAME}"` 
	     echo "${DEP_LIST}" | grep "srvadmin-nvme" > /dev/null
             if [ $? == 0 ] 
             then
	         REALSSD_RPM_NAME=`ls srvadmin-nvme-*`
                 RPM_LIST="${RPM_LIST} ${REALSSD_RPM_NAME}"
             fi 
        fi 
        echo ${RPM_LIST}
     fi
 }

function add_rnasoap
{
     RPM_LIST=${1}

     echo "${RPM_LIST}" | grep "srvadmin-rnasoap" > /dev/null
     if [ $? == 0 ] 
     then
        echo ${RPM_LIST}
     else
        #for srvadmin-rnasoap(check if this rpm is required)
        echo "${RPM_LIST}" | grep "srvadmin-storage-$CURRENT_VERSION" > /dev/null
        if [ $? == 0 ] 
        then
	     STORAGE_RPM_NAME=`ls srvadmin-storage-$CURRENT_VERSION*`
	     DEP_LIST=`rpm -qp --requires "${STORAGE_RPM_NAME}"` 
	     echo "${DEP_LIST}" | grep "srvadmin-rnasoap" > /dev/null
             if [ $? == 0 ] 
             then
	         REALSSD_RPM_NAME=`ls srvadmin-rnasoap-*`
                 RPM_LIST="${RPM_LIST} ${REALSSD_RPM_NAME}"
             fi 
        fi 
        echo ${RPM_LIST}
     fi
 }
##
## OMIUpgrade
##
function OMIUpgrade {
    UPGRADE_OMAUTH=0
    # rpm to rpm upgrade
    INSTALLED_RPMS=`rpm -q --queryformat "%{NAME} " \`rpm -qa | grep ${PKG_PREFIX}\``
    TO_BE_INSTALLED_LIST=""

    ARCH_FLAG="i386"

    for PKG in ${INSTALLED_RPMS};
    do
       #query all rpms
       TMP_FLAG=`rpm -q --queryformat "%{ARCH}" $PKG` 
       #ignore noarch
       if [ $TMP_FLAG != "i386" ] && [ $TMP_FLAG != "x86_64" ] 
       then
         continue
	   else
		ARCH_FLAG=$TMP_FLAG
       fi
       #if arch is not i386 for any rpm, we have a 64 bit install
       if [ ${ARCH_FLAG} != "i386" ]
       then
          break
       fi
    done
    
    PATH_TO_RPMS_UPGRADE="$LINUXROOT/RPMS/supportRPMS/srvadmin/${PATH_TO_RPMS_SUFFIX}/$ARCH_FLAG"

    if [ ! -d "${PATH_TO_RPMS_UPGRADE}" ]
    then
       CheckMediaMessage
       SetErrorAndInterrupt 4
    fi


    cd "${PATH_TO_RPMS_UPGRADE}"
    for PKG in ${INSTALLED_RPMS};
    do
       #exclude IPMI, wsmanclient & omauth packages here as later on, they will be 
       #correctly added to the list.
          echo "$PKG" | grep "srvadmin-omauth" >/dev/null 2>&1
          if [ $? != 0 ]; then
             echo "$PKG" | grep "srvadmin-wsmanclient" >/dev/null 2>&1
             if [ $? != 0 ]; then
                PATH_TO_PKG=`ls ${PKG}* 2>/dev/null`
                TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} ${PATH_TO_PKG}"
             fi
       #post 7.0 iws is no longer available. replace with tomcat
             echo "$PKG" | grep "srvadmin-iws" >/dev/null 2>&1
             if [ $? == 0 ]; then
                PATH_TO_PKG=`ls srvadmin-tomcat-* 2>/dev/null`
                TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} ${PATH_TO_PKG}"
             fi
          else
             UPGRADE_OMAUTH=1
          fi
    done
     #for 7.2 upgrades
     
       echo "$TO_BE_INSTALLED_LIST" | grep "srvadmin-deng-" >/dev/null 2>&1
       if [ $? == 0 ]; then
          TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} srvadmin-deng-snmp-* srvadmin-isvc-snmp-* srvadmin-idrac-snmp-* srvadmin-omacs-* srvadmin-ominst-* srvadmin-oslog-*"
       fi

    #for 6.0.1 iDRAC and RAC5 clash
    echo "${STR_INSTALLED_VER}" | grep "6.0.[1-9]" > /dev/null
    if [ $? = 0 ] 
      then
      RACADM=`rpm -qa | grep "srvadmin-racadm5"`
      if [ ! -z "${RACADM}" ];
      then
       TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} srvadmin-idracadm* srvadmin-racadm4* srvadmin-argtable2* srvadmin-idrac-vmcli* $PKG_PREFIX-idrac-ivmcli*"
      fi
    fi 	

    # srvadmin-omcommon & smcommon RPM have to be installed in all upgrade scenarios where
    IWS_PRESENT=1
    OMACORE_PRESENT=1

    # srvadmin-omacore or srvadmin-tomcat package is scheduled for install.
    echo "$TO_BE_INSTALLED_LIST" | grep "srvadmin-omacore" >/dev/null 2>&1
    if [ $? == 0 ]; then
       OMACORE_PRESENT=0
    fi

    echo "$TO_BE_INSTALLED_LIST" | grep "srvadmin-tomcat" >/dev/null 2>&1
    if [ $? == 0 ]; then
	   IWS_PRESENT=0
    fi
	
    if [ $OMACORE_PRESENT == 0 ] || [ $IWS_PRESENT == 0 ]; then
       
       echo "$TO_BE_INSTALLED_LIST" | grep "srvadmin-omcommon-" >/dev/null 2>&1
       if [ $? != 0 ]; then
          TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} srvadmin-omcommon-*"
       fi

       #add new rpm srvadmin-omacs
       TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} srvadmin-omacs-*"

       echo "$TO_BE_INSTALLED_LIST" | grep "srvadmin-smcommon-" >/dev/null 2>&1
       if [ $? != 0 ]; then
          TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} srvadmin-smcommon-*"
       fi
    fi

    # srvadmin-xmlsup has to be installed in all upgrade scenarios where
    # srvadmin-omacore package is scheduled for install.
    if [ $OMACORE_PRESENT == 0 ]; then
       echo "$TO_BE_INSTALLED_LIST" | grep "srvadmin-xmlsup-" >/dev/null 2>&1
       if [ $? != 0 ]; then
          if [ $GBL_OS_TYPE_STRING = "ESX400" ] || [ $GBL_OS_TYPE_STRING = "ESX410" ]
        then
          TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} srvadmin-xmlsup-* srvadmin-libxslt*"
         else
          TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} srvadmin-xmlsup-*"
         fi
       fi
    fi
 
    # srvadmin-smweb has to be installed in all upgrade scenarios where
    # srvadmin-tomcat package is scheduled for install.
    if [ $IWS_PRESENT == 0 ]; then
       echo "$TO_BE_INSTALLED_LIST" | grep "srvadmin-smweb-" >/dev/null 2>&1
       if [ $? != 0 ]; then
          TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} srvadmin-smweb-*"
       fi
    fi
   
    #to be installed for rac and idrac components

     RACOMP=`rpm -qa | grep "srvadmin-racsvc"`
      if [ ! -z "${RACOMP}" ];
      then
       TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} srvadmin-rac4-populator*"
      fi
     RACOMP=""

     RACOMP=`rpm -qa | grep "srvadmin-idrac-components"`
      if [ ! -z "${RACOMP}" ];
      then
       TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} srvadmin-rac-components*"
      fi
     RACOMP=""

     RACOMP=`rpm -qa | grep "srvadmin-rac4-components"`
      if [ ! -z "${RACOMP}" ];
      then
       TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} srvadmin-rac-components*"
      fi
     RACOMP=""

     RACOMP=`rpm -qa | grep "srvadmin-rac5-components"`
      if [ ! -z "${RACOMP}" ];
      then
       TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} srvadmin-rac-components*"
      fi

     RACDRSC=`rpm -qa | grep "racdrsc"`
      if [ ! -z "${RACDRSC}" ];
      then
       TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} srvadmin-racdrsc*"
      fi

    echo "$TO_BE_INSTALLED_LIST" | grep "srvadmin-idracadm" >/dev/null 2>&1
    if [ $? == 0 ]; then
       TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} srvadmin-argtable* srvadmin-racadm4* srvadmin-idrac-* srvadmin-idrac-vmcli* $PKG_PREFIX-idrac-ivmcli* $PKG_PREFIX-idrac-snmp*"
    fi

    echo "$TO_BE_INSTALLED_LIST" | grep "srvadmin-racadm5" >/dev/null 2>&1
    if [ $? == 0 ]; then
       TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} srvadmin-argtable* srvadmin-racadm4*"
    fi

#on a 64bit OS, srvadmin-cm is not present, install srvadmin-cm
        if [ "$ARCH_FLAG" == "x86_64" ] 
        then
            echo $INSTALLED_SRVADM_PKGS | grep "srvadmin-cm" > /dev/null 2>&1
	    if [ $? != 0 ]
	    then
               echo $INSTALLED_SRVADM_PKGS | grep "srvadmin-ominst" > /dev/null 2>&1
	       if [ $? == 0 ]
	       then
                   TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} srvadmin-cm*"
               fi
	    fi
        fi

    # Storage rpms have to be installed in all upgrade scenarios where
    # srvadmin-storage package is scheduled for install.
    echo "$TO_BE_INSTALLED_LIST" | grep "srvadmin-storage-" >/dev/null 2>&1
    if [ $? == 0 ]; then
       CURRENT_VERSION=`rpm -qp --queryformat "%{VERSION}" $PATH_TO_RPMS_UPGRADE/${PKG_PREFIX}-omilcore* 2>/dev/null`
       TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} $PKG_PREFIX-storelib-${CURRENT_VERSION}* $PKG_PREFIX-sysfsutils-* $PKG_PREFIX-storage-cli-* $PKG_PREFIX-storage-snmp-*"
    
       if [ ${ARCH_FLAG} = "i386" ]
       then
          TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} $PKG_PREFIX-megalib-*"
       fi

       [ $GBL_OS_TYPE_STRING != "ESX400" ] && [ $GBL_OS_TYPE_STRING != "ESX410" ] && TO_BE_INSTALLED_LIST="${TO_BE_INSTALLED_LIST} $PKG_PREFIX-storelib-sysfs*"
    fi
 
    STOR_SYSFS=`rpm -qa | grep "$PKG_PREFIX-storelib-sysfs*"`

    # Remove discontinued RPMs from the upgrade list.
    CLEANED_INSTALL_LIST=""
    for THIS_PKG in ${TO_BE_INSTALLED_LIST};
    do
       #exclude IPMI, wsmanclient & omauth packages here as later on, they will be
       #correctly added to the list.
       echo "$THIS_PKG" | grep "srvadmin-omauth" >/dev/null 2>&1
       if [ $? != 0 ]; then
          echo "$THIS_PKG" | grep "srvadmin-wsmanclient" >/dev/null 2>&1
          if [ $? != 0 ]; then
             echo "$THIS_PKG" | grep "srvadmin-omhip" >/dev/null 2>&1
             if [ $? != 0 ]; then
                echo "$THIS_PKG" | grep "srvadmin-ipmi" >/dev/null 2>&1
                if [ $? != 0 ]; then
                   echo "$THIS_PKG" | grep "srvadmin-syscheck" >/dev/null 2>&1
   #                if [ $? != 0 ]; then
   #                   echo "$THIS_PKG" | grep "srvadmin-storelib-libpci" >/dev/null 2>&1
   #                   if [ $? != 0 ]; then
   #                      echo "$THIS_PKG" | grep "srvadmin-storelib-sysfs" >/dev/null 2>&1
                         if [ $? != 0 ]; then
                             CLEANED_INSTALL_LIST="${CLEANED_INSTALL_LIST} ${THIS_PKG}"
   #                      else
   #                          [ -z $STOR_SYSFS ] && CLEANED_INSTALL_LIST="${CLEANED_INSTALL_LIST} ${THIS_PKG}"
   #                      fi
   #                   else
   #                       [ -z $STOR_PCI ] && CLEANED_INSTALL_LIST="${CLEANED_INSTALL_LIST} ${THIS_PKG}"
   #                   fi
                   fi
                fi
             fi
          fi
       fi
    done

    TO_BE_INSTALLED_LIST="$CLEANED_INSTALL_LIST"

    TO_BE_INSTALLED_LIST=`add_realssd "${TO_BE_INSTALLED_LIST}"`
    TO_BE_INSTALLED_LIST=`add_rnasoap "${TO_BE_INSTALLED_LIST}"`
    TO_BE_INSTALLED_LIST=`add_nvme "${TO_BE_INSTALLED_LIST}"`

    


    if [ -n "${TO_BE_INSTALLED_LIST}" ]; then
        #"${SERVICES_SCRIPT}" stop >/dev/null 2>&1

        # check installed package version, omilcore must be installed.
        # is omilcore present? look for any version
        PREVIOUS_INSTALLED_VER=`rpm -q ${PKG_PREFIX}-omilcore | sed -e s,.*[0-9].*,,`
        if [ ! -n "$PREVIOUS_INSTALLED_VER" ];
        then
            # omilcore is installed
            PREVIOUS_INSTALL_DIR=`rpm -q --queryformat "%{INSTALLPREFIX}\t" ${PKG_PREFIX}-omilcore 2>/dev/null`
        fi
        PREVIOUS_INSTALL_DIR=`echo "${PREVIOUS_INSTALL_DIR}" | awk -F" " '{print $1}'`
        #Install/Freshen libsmbios packages
        FreshenSMBIOSPackages

        # Update openwsman if Webserver is selected for install.
        echo "$TO_BE_INSTALLED_LIST" | grep "srvadmin-tomcat-" >/dev/null 2>&1
        if [ $? == 0 ] 
        then
          if [ $GBL_OS_TYPE_STRING != "RHEL6" -a $GBL_OS_TYPE_STRING != "RHEL7" ]
          then
           PATH_TO_OPENSOURCE_RPMS="$PATH_TO_RPMS_UPGRADE/../../../opensource-components/$PATH_TO_RPMS_SUFFIX/$ARCH_FLAG"
           InstallOpenWSManPackages
          fi
        fi


        if [ -n "${PREVIOUS_INSTALL_DIR}" -a "${PREVIOUS_INSTALL_DIR}" != "/opt/dell/srvadmin" ]; then
            # default install
            ${RPM_INSTALL} -Uhv `Uniq ${TO_BE_INSTALLED_LIST}`
            UPGRADE_STATUS=$?
            INSTALL_FLAG=1
            if [ ${UPGRADE_STATUS} == 0 -a -z "${1}" ]; then
              ldconfig
              StartServices
            elif [ ${UPGRADE_STATUS} == 0 -a ${AUTO_START} == 1 -a ${INSTALL_FLAG} == 1 ]; then
              echo ""
              echo "Auto Start is Enabled and thus Starting the Services."
              echo ""
              ldconfig
              "${SERVICES_SCRIPT}" start
            fi
            ldconfig
        else
            # install with prefix
            ${RPM_INSTALL} -Uhv `Uniq ${TO_BE_INSTALLED_LIST}`
            UPGRADE_STATUS=$?
            INSTALL_FLAG=1
            if [ ${UPGRADE_STATUS} == 0 -a -z "${1}" ]; then
              StartServices
            elif [ ${UPGRADE_STATUS} == 0 -a ${AUTO_START} == 1 -a ${INSTALL_FLAG} == 1 ]; then
              echo ""
              echo "Auto Start is Enabled and thus Starting the Services."
              echo ""
              "${SERVICES_SCRIPT}" start
            fi
            ldconfig
        fi
    else
        echo "Error: unable to locate rpm packages to upgrade"
        SetErrorAndInterrupt 4
    fi
}

###############################################################################
#UpdateSelectedPkgsDescArr
#  Function : UpdateSelectedPkgsDescArr
#
#
###############################################################################
function UpdateSelectedPkgsDescArr {
    MODE="$1"
    LIST_OF_COMP_GROUP_KEYS=""
    NUM_COMP_GROUPS=${#COMP_PKGS_ARR[@]}
    ARRINDEX=0

    while [ $ARRINDEX -lt $NUM_COMP_GROUPS ];
    do
        PKGS_IN_THIS_COMP_GROUP=`echo "${COMP_PKGS_ARR[$ARRINDEX]}" | sed "s/^.*=//"`
        PRESENT_IN_SELECTED_PKGS=0

        for PKG_IN_GROUP in $PKGS_IN_THIS_COMP_GROUP;
        do
            # strip off any trailing "-" character
            PKG_IN_GROUP=`echo "${PKG_IN_GROUP}" | sed "s/-$//"`
            PRESENT_IN_SELECTED_PKGS=0
            ALL_PKGS=""
            if [ "${MODE}" == "INIT" ]; then
              ALL_PKGS="${SELECTED_PKGS}"
	    elif [ "${MODE}" == "UPDATE" ]; then
		ALL_PKGS="${SELECTED_PKGS}"
            else
              ALL_PKGS="${SELECTED_PKGS} ${CURRENTLY_INSTALLED_PKGS}"
            fi

	     for SLCT_PKG in ${ALL_PKGS};
        	    do
                if [ "$SLCT_PKG" == "$PKG_IN_GROUP" ];
                then
                    PRESENT_IN_SELECTED_PKGS=1
                    break
                fi
            done

            # all packages in a particular group should be present
            [ ${PRESENT_IN_SELECTED_PKGS} == 0 ] && break
        done

        # all the packages in this group are present
        if [ ${PRESENT_IN_SELECTED_PKGS} == 1 ]; then
            THIS_COMP_GROUP_KEY=`echo "${COMP_PKGS_ARR[$ARRINDEX]}" | sed "s/=.*$//"`
	    LIST_OF_COMP_GROUP_KEYS="$LIST_OF_COMP_GROUP_KEYS $THIS_COMP_GROUP_KEY"
        fi
        ARRINDEX=`expr $ARRINDEX + 1`
    done

    # now update the SELECTED_COMP_DESC
    ARRLENGTH=${#APPLICABLE_OPTIONAL_PKGS[@]}

    for (( i=0 ; i<${ARRLENGTH}; i++ ));
    do
        OPTPKGKEY=`echo "${APPLICABLE_OPTIONAL_PKGS[${i}]}" | sed "s/=.*$//"`
        for COMP_GROUP_KEY in ${LIST_OF_COMP_GROUP_KEYS}
        do
            if [ "${COMP_GROUP_KEY}" == "${OPTPKGKEY}" ]; then
                # add only if not already present
                echo "$SELECTED_COMP_DESC" | grep "$OPTPKGKEY" >/dev/null
                if [ $? != 0 ]; then
                    SELECTED_COMP_DESC="${SELECTED_COMP_DESC} ${APPLICABLE_OPTIONAL_PKGS[${i}]}"
                fi
                break
            fi
        done
    done
    if [ "${MODE}" == "INIT" ]; then
        CURRENTLY_INSTALLED_KEYS="${LIST_OF_COMP_GROUP_KEYS}"
    fi

    if [ "${MODE}" == "UPDATE" ]; then #for update mode

        SEL_PKGS=""
        PKGS_TO_INSTALL=""

	#echo " selected pkgs :" $SELECTED_PKGS

        for  SEL_PKGS in $SELECTED_PKGS
        do
 	       FOUND_FLAG=0
 	       for F_PKGS in ${CURRENTLY_INSTALLED_PKGS}
                do
                 if [ $SEL_PKGS == $F_PKGS ]; then
                        #PKGS_TO_INSTALL="$PKGS_TO_INSTALL $SEL_PKGS"
                        FOUND_FLAG=1
			break
                 fi
                done

        	if [ $FOUND_FLAG -eq 0 ]; then
                 PKGS_TO_INSTALL="$PKGS_TO_INSTALL $SEL_PKGS"

       		 fi

	done
        
	SELECTED_PKGS=$PKGS_TO_INSTALL
    fi

}

###############################################################################
#
#  Function : DetectPreviousInstall
#
#
###############################################################################
function DetectPreviousInstall {

#    for THIRD_PTY_PKG in libsmbios2 smbios-utils-bin libcmpiCppImpl0
#    do
#        INSTALLED_THIRD_PTY_PKG=`rpm -qa | grep $THIRD_PTY_PKG`
#        if [ -n "$INSTALLED_THIRD_PTY_PKG" ]; then
#           FMT_THIRD_PTY_PKG=`rpm -q --queryformat "%{NAME}\t" ${INSTALLED_THIRD_PTY_PKG}`
#           FMT_THIRD_PTY_PKG=`echo "${FMT_THIRD_PTY_PKG}" | awk -F" " '{print $1}'`
#           CURRENTLY_INSTALLED_PKGS="$CURRENTLY_INSTALLED_PKGS $FMT_THIRD_PTY_PKG"
#           INSTALL_MODE="tpty_add"
#        fi
#    done

    # Get the list of all packages that are currently installed.
    rpm -qa | grep $PKG_PREFIX >/dev/null
    if [ $? == 0 ]; then
        # some packages are already installed.
        INSTALLED_SRVADM_PKGS=`rpm -qa | grep $PKG_PREFIX`
        CURRENTLY_INSTALLED_PKGS="$CURRENTLY_INSTALLED_PKGS $INSTALLED_SRVADM_PKGS"

        if [ ! -z "${INSTALLED_SRVADM_PKGS}" ];
        then
            INSTALLED_PKG_FOR_VERSION_CHECK=${PKG_PREFIX}-omilcore
            echo "$INSTALLED_SRVADM_PKGS" | grep "${PKG_PREFIX}-omilcore" >/dev/null 2>&1
            if [ $? != 0 ]; then
                # omilcore is not installed. 
                # Some other package that is not dependent 
                # on omilcore is installed on the system.
                INSTALLED_PKG_FOR_VERSION_CHECK=`echo $INSTALLED_SRVADM_PKGS | awk -F" " '{print $1}'`
            fi

            INSTALLED_VER=`rpm -q ${INSTALLED_PKG_FOR_VERSION_CHECK} | sed -e s,.*[0-9].*,,`
            if [ ! -n "$INSTALLED_VER" ];
            then
                # check version and release numbers
                INSTALLED_VER=`rpm -q --queryformat "%{VERSION}" ${INSTALLED_PKG_FOR_VERSION_CHECK}`
                INSTALLED_REL=`rpm -q --queryformat "%{RELEASE}" ${INSTALLED_PKG_FOR_VERSION_CHECK}`
            fi

            FORMATTED_INSTALLED_VER=`echo "${INSTALLED_VER}" | sed "s/\.//g" | sed "s/ //g"`
            if [ $FORMATTED_INSTALLED_VER -lt 801 ]; then
               #disable upgrade on supported server only, allow on clients      
	       if [ $ENABLE_CONSOLE_ONLY_INSTALL == 0 ]; then
               cat <<EOF

  OpenManage ${INSTALLED_VER} is already installed. Please uninstall existing version and install OpenManage 8.0.1.
  Error: Upgrade cannot continue. 

EOF
               SetErrorAndInterrupt 3
	       fi
              
            fi

            STR_INSTALLED_VER=`echo "${INSTALLED_VER}" | awk -F" " '{print $1}' | sed "s/ //g"`

            # Remove the .Priv part (if any) from Release
            #INSTALLED_REL=`echo ${INSTALLED_REL} | awk -F"." '{print $1}' | sed "s/ //g"`
            INSTALLED_REL=" "
            INSTALLED_VER="${INSTALLED_VER}-${INSTALLED_REL}"

            if [ -n "${INSTALLED_VER}" ]; then
                INSTALLED_VER=`echo "${INSTALLED_VER}" | sed "s/\.//g" | sed "s/-//g" | sed "s/ //g"`
            fi

            # check about to install package version
            CURR_DIR=`pwd | sed "s/ /\\\ /"`
            ALL_OMILCORES=`find "${LINUXROOT}"/RPMS/supportRPMS/srvadmin -name ${PKG_PREFIX}-omilcore*` 
            FIRST_OMILCORE=`echo ${ALL_OMILCORES} | awk -F" " '{print $1}' | sed "s/ //g"`
            PATH_TO_OMILCORE=`dirname "${FIRST_OMILCORE}"`
         NEW_VER=""

         
         if [ -n "${PATH_TO_OMILCORE}" ]
         then
            cd "${PATH_TO_OMILCORE}"
            NEW_VER=`rpm -qp --queryformat "%{VERSION}" ${PKG_PREFIX}-omilcore* 2>/dev/null`
            #NEW_REL=`rpm -qp --queryformat "%{RELEASE}" ${PKG_PREFIX}-omilcore* 2>/dev/null`
            NEW_REL=" "
            cd "${CURR_DIR}"
            STR_NEW_VER="${NEW_VER}"

            if [ -n "${NEW_VER}" -a  -n "${NEW_REL}" ]; then
                NEW_VER=`echo "${NEW_VER}" | sed "s/\.//g" | sed "s/-//g" | sed "s/ //g"`
                NEW_REL=`echo "${NEW_REL}" | awk -F"." '{print $1}' | sed "s/ //g"`
                NEW_VER="${NEW_VER}${NEW_REL}"
            fi
          fi
            if [ -z "${INSTALLED_VER}" -o -z "${NEW_VER}" ]; then
                cat <<EOF

  Unable to retrieve following information:
   - Installed version: ${STR_INSTALLED_VER}
   - Current version  : ${STR_NEW_VER}

EOF
                SetErrorAndInterrupt 5
            fi

            if [ ${INSTALLED_VER} == ${NEW_VER} ]; then
              if [ "${INSTALL_MODE}" != "update" ]; then
                # already installed, mode=add
                # check the applicable packages group descriptions array based on
                # the installed packages.
                INSTALL_MODE="add"
                for CUR_PKG in ${CURRENTLY_INSTALLED_PKGS};
                do
                    SLT_PKG=`rpm -q --queryformat "%{NAME}\t" ${CUR_PKG}`
                    SLT_PKG=`echo "${SLT_PKG}" | awk -F" " '{print $1}'`
                    SELECTED_PKGS="${SELECTED_PKGS} ${SLT_PKG}"
                done
                UpdateSelectedPkgsDescArr "INIT"
                CURRENTLY_INSTALLED_PKGS="${SELECTED_PKGS}"
                SELECTED_PKGS=""
              else
     echo "Selected Server Administrator components are currently installed."
              fi
            elif [ ${INSTALLED_VER} -lt ${NEW_VER} ]; then
                # upgrade
                cat <<EOF
  Server Administrator version ${STR_INSTALLED_VER} is currently installed.
  Installed components are:
EOF

                # Check whether omacore RPM is installed or not.
                # If not installed, it could be a Diags install instance.
                echo "$CURRENTLY_INSTALLED_PKGS" | grep "srvadmin-omcommon" >/dev/null 2>&1
                if [ $? != 0 ]; then
                    OMCOMMON_NOT_PRESENT=1 
                fi

                for PKG in ${CURRENTLY_INSTALLED_PKGS};
                do
                   echo "   - ${PKG}"
                done
                upgrade_input=""

                if [ -n "${1}" -a "${1}" == "silent" ]; then
                  upgrade_input="y"
                  cat <<EOF
  Upgrading Server Administrator to ${STR_NEW_VER}

EOF
                  OMIUpgrade "silent"
                else
                  cat <<EOF

  Do you want to upgrade Server Administrator to ${STR_NEW_VER}?
EOF
                  upgrade_input=`Prompt "  Press ('y' for yes | 'Enter' to exit): "`
                #fix for DF 39092 - added a '$' in the regexp - BK
                  echo "${upgrade_input}" | grep -iq "^y$" ||
                    SetErrorAndInterrupt 13

                  OMIUpgrade ""
                fi
                [ $UPGRADE_STATUS -eq 0 ] && SetSuccessAndExit 0
                let UPGRADE_STATUS=$UPGRADE_STATUS+200
                SetErrorAndInterrupt  $UPGRADE_STATUS
            else
                # downgrade
                cat <<EOF
  Server Administrator version ${STR_INSTALLED_VER} is currently installed.
  Installed components are:
EOF
                for PKG in ${CURRENTLY_INSTALLED_PKGS};
                do
                    echo "   - ${PKG}"
                done
                cat <<EOF

  You are attempting to install Server Administrator version ${STR_NEW_VER}
  ${STR_NEW_VER} is lower than the installed version ${STR_INSTALLED_VER}.
  Error: install cannot continue.

EOF
                SetErrorAndInterrupt 3
            fi
       fi
       else

         if [ "${INSTALL_MODE}" == "update" ]; then
          cat <<EOF
  There are no Server Administrator components available for upgrade. 
  No actions taken.
EOF
          SKIP_UPGRADE=0
        fi
    fi

}
###############################################################################
#
#  Function : RunPreUpgradeScript
#
#
#
###############################################################################
function RunPreUpgradeScript {
    if [ -e $PRE_UPGRADE_SCRIPT ];
    then
        if [ -x $PRE_UPGRADE_SCRIPT ];
        then
            $PRE_UPGRADE_SCRIPT --force  >/dev/null 2>&1 || SetErrorAndInterrupt 1
        else
            echo "$PRE_UPGRADE_SCRIPT has no execute permissions."
            SetErrorAndInterrupt $ERROR_CODE_FILE_OPEN_ERROR
        fi
    else
        echo "$PRE_UPGRADE_SCRIPT does not exist"
        SetErrorAndInterrupt $ERROR_CODE_FILE_OPEN_ERROR
    fi
}

function ExitNonRoot {
  MSGSTR="Install"
  if [ -n "${1}" ]; then
    MSGSTR="${1}"
  fi

  if [ ${UID} != 0 ]; then
    cat <<EOF

  ${MSGSTR} requires root privileges

EOF
    SetErrorAndInterrupt 6
  fi
}

###############################################################################
#
#  Function : PrepareForInstall
#
#
#
###############################################################################
function PrepareForInstall {
    # check for root privileges
    ExitNonRoot
}

function BlockOnOldOM {
    PREVIOUS_VER_OF_SM_PRESENT="no"
    for OLD_SM_PKG in ${PRE_SM_RPM_LIST}
    do
        rpm -q ${OLD_SM_PKG} >/dev/null 2>&1
       if [ $? == 0 ]; then
           PREVIOUS_VER_OF_SM_PRESENT="yes"
           break
        fi
    done

    if [ "${PREVIOUS_VER_OF_SM_PRESENT}" == "yes" ]; then
cat << EOF
  A previous version of the Server Administrator software is
  installed on this system.
  You must uninstall that version before install can continue.

EOF
      SetErrorAndInterrupt 8
    fi
}

###############################################################################
#
#  Function : InstallPackages
#
#
#
###############################################################################
function InstallPackages {
    PrepareForInstall "${1}"
    CURRENT_DIR=`pwd | sed "s/ /\\\ /"`

    if [ ! -d "${PATH_TO_RPMS}" ]
    then
            CheckMediaMessage
     	    SetErrorAndInterrupt 4
    fi

    cd "$PATH_TO_RPMS"

    CURRENT_VERSION=`rpm -qp --queryformat "%{VERSION}" $PATH_TO_RPMS/${PKG_PREFIX}-omilcore* 2>/dev/null`
    FULLY_QUALIFIED_RPMS_LIST=

    #Do not install cm on a 64 bit OS. srvadmin-cm is 32 bit.

    #For megalib 32bit
    if [ "${DISTRO}" == "i386" ] 
    then
        echo "${SELECTED_PKGS}" | grep "srvadmin-storage" >/dev/null 2>&1
        if [ $? == 0 ]
        then
            SELECTED_PKGS="${SELECTED_PKGS} srvadmin-megalib"
        fi
    fi

    echo "${SELECTED_PKGS}" | grep "srvadmin-ominst" >/dev/null 2>&1
    if [ $? == 0 ]
    then
        SELECTED_PKGS="${SELECTED_PKGS} srvadmin-cm"
    fi

    #for CLI and SNMP granular for storage
    echo "${SELECTED_PKGS}" | grep "srvadmin-storage" >/dev/null 2>&1
    if [ $? == 0 ]
    then
       echo "${SELECTED_PKGS}" | grep "srvadmin-deng-snmp" >/dev/null 2>&1
       if [ $? == 0 ]
       then
         SELECTED_PKGS="${SELECTED_PKGS} srvadmin-storage-snmp"
       fi

       echo "${CURRENTLY_INSTALLED_PKGS}" | grep "srvadmin-deng-snmp" >/dev/null 2>&1
       if [ $? == 0 ]
       then
         SELECTED_PKGS="${SELECTED_PKGS} srvadmin-storage-snmp"
       fi

       echo "${SELECTED_PKGS}" | grep "srvadmin-omacore" >/dev/null 2>&1
       if [ $? == 0 ]
       then 
         SELECTED_PKGS="${SELECTED_PKGS} srvadmin-storage-cli"
       fi

       echo "${CURRENTLY_INSTALLED_PKGS}" | grep "srvadmin-omacore" >/dev/null 2>&1
       if [ $? == 0 ]
       then 
         SELECTED_PKGS="${SELECTED_PKGS} srvadmin-storage-cli"
       fi
    fi

    #If storage is already installed, install cli or snmp accordingly 
       echo "${CURRENTLY_INSTALLED_PKGS}" | grep "srvadmin-storage" >/dev/null 2>&1
       if [ $? == 0 ]
       then 
          echo "${SELECTED_PKGS}" | grep "srvadmin-omacore" >/dev/null 2>&1
          if [ $? == 0 ]
          then 
             SELECTED_PKGS="${SELECTED_PKGS} srvadmin-storage-cli"
          fi

          echo "${SELECTED_PKGS}" | grep "srvadmin-deng-snmp" >/dev/null 2>&1
          if [ $? == 0 ]
          then 
             SELECTED_PKGS="${SELECTED_PKGS} srvadmin-storage-snmp"
          fi
       fi
    
    if [ $GBL_OS_TYPE_STRING != "ESX400" ] && [ $GBL_OS_TYPE_STRING != "ESX410" ] 
    then 
        echo "${SELECTED_PKGS}" | grep "srvadmin-storage" >/dev/null 2>&1
        if [ $? == 0 ]
        then
            SELECTED_PKGS="${SELECTED_PKGS} $PKG_PREFIX-storelib-sysfs"
        fi
    fi

    for SELECTED_PKG in $SELECTED_PKGS
    do
        THIS_RPM_NAME=`ls $SELECTED_PKG-${CURRENT_VERSION}* 2>/dev/null`
        FULLY_QUALIFIED_RPMS_LIST="$FULLY_QUALIFIED_RPMS_LIST $THIS_RPM_NAME"
    done

    if [ ! -z "$FULLY_QUALIFIED_RPMS_LIST" ];
    then
        if [ "${INSTALL_MODE}" == "add" ];
        then

            OMIL_PKG=`rpm -q ${PKG_PREFIX}-omilcore | sed -e s,.*[0-9].*,,`
            if [ ! -n "$OMIL_PKG" ];
            then
                # omilcore is installed
                OMIL_PKG="${PKG_PREFIX}-omilcore"
            fi

        fi
   

        if [ $GBL_OS_TYPE_STRING = "ESX400" ] || [ $GBL_OS_TYPE_STRING = "ESX410" ]
        then
           echo "$FULLY_QUALIFIED_RPMS_LIST" | grep "srvadmin-xmlsup-" >/dev/null 2>&1
           if [ $? == 0 ]
             then
                 XSLT_RPM_NAME=`ls srvadmin-libxslt* 2>/dev/null`
                 FULLY_QUALIFIED_RPMS_LIST="${FULLY_QUALIFIED_RPMS_LIST} $XSLT_RPM_NAME"
             fi 
        fi

        # final screening of duplicates, if any
        FINAL_INSTALL_LIST=""
        for PF_PKG in ${FULLY_QUALIFIED_RPMS_LIST};
        do
          if [ "${FINAL_INSTALL_LIST}" != "" ]; then
            FOUND_PKG=0

            for F_PKG in ${FINAL_INSTALL_LIST};
            do
              if [ "${F_PKG}" == "${PF_PKG}" ]; then
                FOUND_PKG=1
                break
              fi
            done

            if [ ${FOUND_PKG} == 0 ]; then
              FINAL_INSTALL_LIST="${FINAL_INSTALL_LIST} ${PF_PKG}"
            fi
          else
            FINAL_INSTALL_LIST="${FINAL_INSTALL_LIST} ${PF_PKG}"
          fi
        done

        # stop the services before installing more components
        #if [ "${INSTALL_MODE}" == "add" ]; then
        #  ${SERVICES_SCRIPT} stop >/dev/null 2>&1
        #fi
        "${SERVICES_SCRIPT}" stop >/dev/null 2>&1

        # clear screen in preparation of installing
        if [ $PRESERVE_SCREEN == 0 ]; then 
            clear
        fi

        # start installing
        cat <<EOF
Installing the selected packages.

EOF

         #Install/Freshen libsmbios packages
         FreshenSMBIOSPackages

         # Update openwsman if Webserver is selected for install.
         echo "$FINAL_INSTALL_LIST" | grep "srvadmin-tomcat-" >/dev/null 2>&1
         if [ $? == 0 ] 
         then
            if [ $GBL_OS_TYPE_STRING != "RHEL6"  -a  $GBL_OS_TYPE_STRING != "RHEL7" ]; then

            PATH_TO_OPENSOURCE_RPMS="$PATH_TO_RPMS/../../../opensource-components/$PATH_TO_RPMS_SUFFIX/$DISTRO"
            InstallOpenWSManPackages
            fi
         fi


         if [ -n "${CURRENTLY_INSTALLED_PKGS}" ]
         then
            for component in ${CURRENTLY_INSTALLED_PKGS}
            do
                comp_name=`rpm -q --qf "%{name}" $component`
                for install_comp in ${FINAL_INSTALL_LIST}
                do
                    echo "${install_comp}" | grep "${comp_name}-$CURRENT_VERSION" > /dev/null
                    if [ $? == 0 ] 
                    then
                     FINAL_INSTALL_LIST=`echo "${FINAL_INSTALL_LIST}" | sed "s/${install_comp}//"`
                    fi 
                done 
            done
         fi
         FINAL_INSTALL_LIST=`add_realssd "${FINAL_INSTALL_LIST}"`
         FINAL_INSTALL_LIST=`add_rnasoap "${FINAL_INSTALL_LIST}"`
         FINAL_INSTALL_LIST=`add_nvme "${FINAL_INSTALL_LIST}"`
         # Install srvadmin packages
         ${RPM_INSTALL} -Uhv `Uniq ${FINAL_INSTALL_LIST}`
         INSTALL_STATUS=$?
         ldconfig

        INSTALL_FLAG=1
    fi

    # back to the current directory
    cd "$CURRENT_DIR"

#For servers below 11G display message on a 64 bit OS

    SYSTEM11G="false"
    IsSystem11GorHigher

    if [ $? == 0 ]
    then
       SYSTEM11G="true"
    fi

    ARCH_FLAG=`uname -i`

    if [ "$1" != "silent" ] && [ $SYSTEM11G == "false" ]
    then
        if [ "$ARCH_FLAG" == "x86_64" ]
        then
            echo "${FINAL_INSTALL_LIST}" | grep "srvadmin-cm" > /dev/null 2>&1
           if [ $? == 0 ]
           then
               cat <<EOF

 Note: The Inventory Collector RPM (srvadmin-cm) requires the following RPMs
       to be preinstalled on servers prior to 11G running supported 64-bit
       Linux operating system.

         glibc.i686
         compat-libstdc++.i686
         libstdc++.i686
         zlib.i686
         libxml2.i686

       If the dependent RPMs are not installed then Inventory Collector will
       not function as expected and exits with error messages.

EOF
           fi
        fi
    fi

    if [ "${1}" != "silent" -a -n "${INSTALL_STATUS}" -a ${INSTALL_STATUS} == 0 ]; then
      StartServices
    elif [ ${AUTO_START} == 1 -a -n "${INSTALL_STATUS}" -a ${INSTALL_STATUS} == 0  -a ${INSTALL_FLAG} == 1 ]; then
      echo ""
      echo "Auto Start is Enabled and thus Starting the Services."
      echo ""
      "${SERVICES_SCRIPT}" start
    fi
}

###############################################################################
#
#  Function : CleanUpAndQuit
#
#
#
###############################################################################
function CleanUpAndQuit {
    if [ ! -z "$LOCAL_DEST_PATH" ] && [ -d "$LOCAL_DEST_PATH" ];
    then
        rm -f $LOCAL_DEST_PATH/* &>/dev/null
      [ -d "$LOCAL_DEST_PATH" ] &&
        rmdir --ignore-fail-on-non-empty "$LOCAL_DEST_PATH" &>/dev/null
    fi

    SetSuccessAndExit 0
}

###############################################################################
#
#  Function : ResetAndStartOver
#
#
#
###############################################################################
function ResetAndStartOver {
    LOCAL_DEST_PATH=""
    SELECTED_COMP_DESC=""
    SELECTED_PKGS=""
    USER_SELECTED_INDEX=""
    USER_SELECTED_PKG=""
    CHOSEN_PKGS=""
    S_DESC_LIST=""
}

###############################################################################
#
#  Function : ExitMessage
#
#
#
###############################################################################

function ExitMessage {
  cat <<EOF

  Please find the Server Administrator packages at: $LOCAL_DEST_PATH

  Use rpm install commands to install the packages

  Install at default locaton with : rpm -Uhv *.rpm

EOF

  SetSuccessAndExit 0
}

###############################################################################
#
#  Function : CopyPkgsToDestDirectory
#
#
#
###############################################################################

function CopyPkgsToDestDirectory {

    if [ ! -d "${PATH_TO_RPMS}" ]
    then
       CheckMediaMessage
       SetErrorAndInterrupt 4
    fi

  cat <<EOF
  Please enter the destination directory to
  copy the Server Administrator RPM packages.

EOF

  # prompt for dest path
  ReadAndValidateDestPath

  cat <<EOF

  Copying packages to $LOCAL_DEST_PATH
  This may take a while .......
EOF

  echo "$SELECTED_PKGS" | grep "srvadmin-xmlsup" >/dev/null 2>&1
  if [ $? == 0 ]
  then
     if [ $GBL_OS_TYPE_STRING = "ESX400" ] || [ $GBL_OS_TYPE_STRING = "ESX410" ]
     then
         SELECTED_PKGS="${SELECTED_PKGS} srvadmin-libxslt"
     fi
  fi

  #Get file names and clean the list for copy
  for selected_pkg in $SELECTED_PKGS
  do
    #get version number to create the pattern
    CURRENT_COPY_VERSION=`rpm -qp --queryformat "%{VERSION}" $PATH_TO_RPMS/${PKG_PREFIX}-omilcore* 2>/dev/null`

    #create pattern
    PATTERN="${selected_pkg}*"

    echo $selected_pkg | grep "libxslt" > /dev/null
    if [ $? == 0 ]
    then
       FILE=`ls $PATH_TO_RPMS/$selected_pkg* 2> /dev/null`
       FILES_TO_COPY="${FILE} ${FILES_TO_COPY}"
       continue
    fi

    #use pattern to get actual file name
    FILE=`ls $PATH_TO_RPMS/$PATTERN 2> /dev/null`

    #add to list
    FILES_TO_COPY="${FILE} ${FILES_TO_COPY}"
  done
#---------------------- customization for copy----------------

    CURRENT_VERSION=`rpm -qp --queryformat "%{VERSION}" $PATH_TO_RPMS/${PKG_PREFIX}-omilcore* 2>/dev/null`
    FULLY_QUALIFIED_RPMS_LIST=""

    #Do not install cm on a 64 bit OS. srvadmin-cm is 32 bit.

    #For megalib 32bit
    if [ "${DISTRO}" == "i386" ] 
    then
        echo "${FILES_TO_COPY}" | grep "srvadmin-storage" >/dev/null 2>&1
        if [ $? == 0 ]
        then
            TEMP=`ls $PATH_TO_RPMS/srvadmin-megalib*  2> /dev/null`
            FILES_TO_COPY="${FILES_TO_COPY} $TEMP"
        fi
    fi

    #For srvadmin-cm 
    echo "${FILES_TO_COPY}" | grep "srvadmin-ominst" >/dev/null 2>&1
    if [ $? == 0 ]
    then
       TEMP=`ls $PATH_TO_RPMS/srvadmin-cm* 2> /dev/null`
       SELECTED_PKGS="${FILES_TO_COPY} $TEMP"
    fi
    
    #for CLI and SNMP granular for storage
    echo "${SELECTED_PKGS}" | grep "srvadmin-storage" >/dev/null 2>&1
    if [ $? == 0 ]
    then
       echo "${SELECTED_PKGS}" | grep "srvadmin-deng-snmp" >/dev/null 2>&1
       if [ $? == 0 ]
       then
         FILES_TO_COPY="${FILES_TO_COPY} srvadmin-storage-snmp"
       fi

       echo "${CURRENTLY_INSTALLED_PKGS}" | grep "srvadmin-deng-snmp" >/dev/null 2>&1
       if [ $? == 0 ]
       then
         FILES_TO_COPY="${FILES_TO_COPY} srvadmin-storage-snmp"
       fi

       echo "${SELECTED_PKGS}" | grep "srvadmin-omacore" >/dev/null 2>&1
       if [ $? == 0 ]
       then 
         FILES_TO_COPY="${FILES_TO_COPY} srvadmin-storage-cli"
       fi

       echo "${CURRENTLY_INSTALLED_PKGS}" | grep "srvadmin-omacore" >/dev/null 2>&1
       if [ $? == 0 ]
       then 
         FILES_TO_COPY="${FILES_TO_COPY} srvadmin-storage-cli"
       fi
    fi

    #If storage is already installed, install cli or snmp accordingly 
       echo "${CURRENTLY_INSTALLED_PKGS}" | grep "srvadmin-storage" >/dev/null 2>&1
       if [ $? == 0 ]
       then 
          echo "${SELECTED_PKGS}" | grep "srvadmin-omacore" >/dev/null 2>&1
          if [ $? == 0 ]
          then 
             FILES_TO_COPY="${FILES_TO_COPY} srvadmin-storage-cli"
          fi

          echo "${SELECTED_PKGS}" | grep "srvadmin-deng-snmp" >/dev/null 2>&1
          if [ $? == 0 ]
          then 
             FILES_TO_COPY="${FILES_TO_COPY} srvadmin-storage-snmp"
          fi
       fi
    
    if [ $GBL_OS_TYPE_STRING != "ESX400" ] && [ $GBL_OS_TYPE_STRING != "ESX410" ] 
    then 
        echo "${FILES_TO_COPY}" | grep "srvadmin-storage" >/dev/null 2>&1
        if [ $? == 0 ]
        then
	    TEMP=`ls $PATH_TO_RPMS/$PKG_PREFIX-storelib-sysfs* 2> /dev/null`
            FILES_TO_COPY="${FILES_TO_COPY} $TEMP"
        fi
    fi

    #for SELECTED_PKG in $FILES_TO_COPY
    #do
    #    THIS_RPM_NAME=`ls $SELECTED_PKG-${CURRENT_VERSION}* 2>/dev/null`
    #    FULLY_QUALIFIED_RPMS_LIST="$FULLY_QUALIFIED_RPMS_LIST $THIS_RPM_NAME"
    #done
    
        if [ $GBL_OS_TYPE_STRING = "ESX400" ] || [ $GBL_OS_TYPE_STRING = "ESX410" ]
        then
           echo "$FILES_TO_COPY" | grep "srvadmin-xmlsup-" >/dev/null 2>&1
           if [ $? == 0 ]
             then
                 XSLT_RPM_NAME=`ls $PATH_TO_RPMS/srvadmin-libxslt* 2>/dev/null`
                 FILES_TO_COPY="${FILES_TO_COPY} $XSLT_RPM_NAME"
             fi 
        fi

        # final screening of duplicates, if any
        FINAL_INSTALL_LIST=""
       
        for PF_PKG in ${FILES_TO_COPY};
        do
          if [ "${FINAL_INSTALL_LIST}" != "" ]; then
            FOUND_PKG=0

            for F_PKG in ${FINAL_INSTALL_LIST};
            do
              if [ "${F_PKG}" == "${PF_PKG}" ]; then
                FOUND_PKG=1
                break
              fi
            done

            if [ ${FOUND_PKG} == 0 ]; then
              FINAL_INSTALL_LIST="${FINAL_INSTALL_LIST} ${PF_PKG}"
            fi
          else
            FINAL_INSTALL_LIST="${FINAL_INSTALL_LIST} ${PF_PKG}"
          fi
        done

         #FreshenSMBIOSPackages

         # Update openwsman if Webserver is selected for install.
         echo "$FINAL_INSTALL_LIST" | grep "srvadmin-tomcat-" >/dev/null 2>&1
         if [ $? == 0 ] 
         then
         if [ $GBL_OS_TYPE_STRING != "RHEL6" ] || [ $GBL_OS_TYPE_STRING != "RHEL7" ]; then

           PATH_TO_OPENSOURCE_RPMS="$PATH_TO_RPMS/../../../opensource-components/$PATH_TO_RPMS_SUFFIX/$DISTRO"
          
           for rpm in `ls $PATH_TO_OPENSOURCE_RPMS/libwsman1-* $PATH_TO_OPENSOURCE_RPMS/openwsman-client*`
           do
     # Query rpm for name and arch
     name_arch=$(rpm -qp --qf "%{name}.%{arch}" $rpm 2> /dev/null) 

           # Is a matching name/arch rpm installed on the system
           if ! rpm --quiet -q $name_arch; then
           # Package of the correct arch is not installed.
           # Add it to the install list.
              FINAL_INSTALL_LIST="$FINAL_INSTALL_LIST $rpm"
           fi
           done
           fi
         fi

         #detect whether any other OM rpms are already installed
         ALREADY_INSTALLED_LIST=`rpm -qa | grep srvadmin`

         if [ -n "${ALREADY_INSTALLED_LIST}" ]
         then
            for component in ${ALREADY_INSTALLED_LIST}
            do
                comp_name=`rpm -q --qf "%{name}" $component`
                for install_comp in ${FINAL_INSTALL_LIST}
                do
                    echo "${install_comp}" | grep "${comp_name}" > /dev/null
                    if [ $? == 0 ] 
                    then
                     FINAL_INSTALL_LIST=`echo "${FINAL_INSTALL_LIST}" | sed "s/${install_comp}//"`
                    fi 
                done 
            done
         fi
         REAL_SSD_RPM=`ls $PATH_TO_RPMS/$PKG_PREFIX-realssd* 2>/dev/null`
         FINAL_INSTALL_LIST="$FINAL_INSTALL_LIST $REAL_SSD_RPM"
         RNASOAP_RPM=`ls $PATH_TO_RPMS/$PKG_PREFIX-rnasoap* 2>/dev/null`
         FINAL_INSTALL_LIST="$FINAL_INSTALL_LIST $RNASOAP_RPM"
         NVME_RPM=`ls $PATH_TO_RPMS/$PKG_PREFIX-nvme* 2>/dev/null`
         FINAL_INSTALL_LIST="$FINAL_INSTALL_LIST $NVME_RPM"
         FILES_TO_COPY="$FINAL_INSTALL_LIST"

#----------------------------------------------------------------
     FILES_TO_COPY=`Uniq $FILES_TO_COPY`

     #Finally copy files
     cp $FILES_TO_COPY "$LOCAL_DEST_PATH"
  cat <<EOF
  done copying.

EOF

  cd "$LOCAL_DEST_PATH"
}

###############################################################################
#
#  Function : DisplayCurrentSelection
#
#  Display the current selected package list
#
###############################################################################

function DisplayCurrentSelection {
  DISPLAY_PKGS=""

  if [ -n "${SELECTED_COMP_DESC}" ]; then
      echo "  Selected Options"
      ARRLENGTH=${#APPLICABLE_OPTIONAL_PKGS[@]}
      ARRINDEX=0
      while [ $ARRINDEX -lt $ARRLENGTH ]
      do
          OPTPKGKEY=`echo "${APPLICABLE_OPTIONAL_PKGS[$ARRINDEX]}" | sed "s/=.*$//"`
          OPTPKGDESC=`echo "${APPLICABLE_OPTIONAL_PKGS[$ARRINDEX]}" | sed "s/^.*=//"`
          echo "${S_DESC_LIST}" | grep "$OPTPKGKEY" >/dev/null 2>&1 && (echo "   - $OPTPKGDESC")
          ARRINDEX=`expr $ARRINDEX + 1`
      done
  fi

  if [ -n "${SELECTED_COMP_DESC}" ]; then
      cat <<EOF

  Dependencies
EOF
      ARRLENGTH=${#APPLICABLE_OPTIONAL_PKGS[@]}
      ARRINDEX=0
      while [ $ARRINDEX -lt $ARRLENGTH ]
      do
          OPTPKGKEY=`echo "${APPLICABLE_OPTIONAL_PKGS[$ARRINDEX]}" | sed "s/=.*$//"`
          OPTPKGDESC=`echo "${APPLICABLE_OPTIONAL_PKGS[$ARRINDEX]}" | sed "s/^.*=//"`
          echo "${S_DESC_LIST}" | grep "$OPTPKGKEY" >/dev/null 2>&1 || (
            echo "$SELECTED_COMP_DESC" | grep "$OPTPKGKEY" >/dev/null 2>&1 && [ "${OPTPKGDESC}" != "All" ] && (echo "   - $OPTPKGDESC")
          )
          ARRINDEX=`expr $ARRINDEX + 1`
      done
      echo ""
  fi
}

###############################################################################
#
#  Function : PullDependencies
#
#  for a given package name, this function queries the rpm package
#  and includes all the packages THIS package  depends on recursively
#
###############################################################################

function PullDependencies {
  # get the required packages for the selected package

  CURDIR=`pwd | sed "s/ /\\\ /"`

    if [ ! -d "${PATH_TO_RPMS}" ]
    then
       CheckMediaMessage
       SetErrorAndInterrupt 4
    fi
  cd "$PATH_TO_RPMS"

  CURRENT_VERSION=`rpm -qp --queryformat "%{VERSION}" ${PKG_PREFIX}-omilcore* 2>/dev/null`

  NSLTPKG=$1
  # echo " "
  # echo "NSLTPKG=$1"
  NSLTPKGNAME=`ls $NSLTPKG-$CURRENT_VERSION* 2>/dev/null`
  # echo "NSLTPKGNAME=$NSLTPKGNAME"
  NSLTPKGREQUIRES=`rpm -qp "$NSLTPKGNAME" --requires 2>/dev/null`
  # echo "NSLTPKGREQUIRES=$NSLTPKGREQUIRES"
  NSLTPKGREQUIRES=`echo "${NSLTPKGREQUIRES}" | grep $PKG_PREFIX | sort | uniq`
  # echo "NSLTPKGREQUIRES=$NSLTPKGREQUIRES"

  for REQUIREPKG in `echo "$NSLTPKGREQUIRES"`;
  do
      # this check is required to filter out tokens like ">=, 3.x" in "pkg_name >= 3.x"
      echo $REQUIREPKG | grep $PKG_PREFIX >/dev/null
      if [ $? == 0 ];
      then
          PRESENT_IN_SELECTED_PKGS=0
          if [ "${INSTALL_MODE}" == "add" ];  #filtered in other functions
          then
              for INSTALLED_PKG in ${CURRENTLY_INSTALLED_PKGS};
              do
                  if [ ${INSTALLED_PKG} == ${REQUIREPKG} ];
                  then
                      PRESENT_IN_SELECTED_PKGS=1
                      break
                  fi
              done
          fi

          if [ ${PRESENT_IN_SELECTED_PKGS} == 0 ];
          then
              for SLCT_PKG in ${SELECTED_PKGS};
              do
                  if [ ${SLCT_PKG} == ${REQUIREPKG} ];
                  then
                      PRESENT_IN_SELECTED_PKGS=1
                      break
                  fi
              done
          fi

          if [ $PRESENT_IN_SELECTED_PKGS == 0 ];
          then
              SELECTED_PKGS="$SELECTED_PKGS $REQUIREPKG"
              # echo "SELECTED_PKGS=$SELECTED_PKGS"

              # recursively call PullDependencies on required package
              PullDependencies $REQUIREPKG
          fi
      fi
  done

  # back to the current directory
  cd "$CURDIR"
}
################################################################################
#function : PullDependenciesUpdate
#Dependencies for package
#
###############################################################################
function PullDependenciesUpdate {
  # get the required packages for the selected package

  CURDIR=`pwd | sed "s/ /\\\ /"`

    if [ ! -d "${PATH_TO_RPMS}" ]
    then
       CheckMediaMessage
       SetErrorAndInterrupt 4
    fi
  cd "$PATH_TO_RPMS"

  NSLTPKG=$1
  # echo " "
  # echo "NSLTPKG=$1"
  NSLTPKGNAME=`ls $NSLTPKG* 2>/dev/null`
  # echo "NSLTPKGNAME=$NSLTPKGNAME"
  NSLTPKGREQUIRES=`rpm -qp "$NSLTPKGNAME" --requires 2>/dev/null`
  # echo "NSLTPKGREQUIRES=$NSLTPKGREQUIRES"
  NSLTPKGREQUIRES=`echo "${NSLTPKGREQUIRES}" | grep $PKG_PREFIX | sort | uniq`
  # echo "NSLTPKGREQUIRES=$NSLTPKGREQUIRES"


  for REQUIREPKG in `echo "$NSLTPKGREQUIRES"`;
  do
      # this check is required to filter out tokens like ">=, 3.x" in "pkg_name >= 3.x"
      echo $REQUIREPKG | grep $PKG_PREFIX >/dev/null
      if [ $? == 0 ];
      then
          if [ ${PRESENT_IN_SELECTED_PKGS} == 0 ];
          then
              for SLCT_PKG in ${SELECTED_PKGS};
              do
                  if [ ${SLCT_PKG} == ${REQUIREPKG} ];
                  then
                      PRESENT_IN_SELECTED_PKGS=1
                      break
                  fi
              done
          fi

          if [ $PRESENT_IN_SELECTED_PKGS == 0 ];
          then
              SELECTED_PKGS="$SELECTED_PKGS $REQUIREPKG"
              # echo "SELECTED_PKGS=$SELECTED_PKGS"

              # recursively call PullDependencies on required package
              PullDependenciesUpdate $REQUIREPKG
          fi
      fi
  done

  # back to the current directory
  cd "$CURDIR"
}

###############################################################################
#
#  Function : UpdatePkgSlection
#
#  Add the user selected package to the Selected Packages list.
#
###############################################################################
function UpdatePkgSlection {
  # local variable
  NEW_PKGS_ADDED=""

  # add the selected component description if it is not in the list

  echo "$SELECTED_COMP_DESC" | grep "$USER_SELECTED_PKG" >/dev/null
  if [ $? != 0 ];
  then
      # component description is not in the list, add it
      SELECTED_COMP_DESC="$SELECTED_COMP_DESC $USER_SELECTED_PKG"

      # extract the key from the description
      SELECTED_COMP_KEY=`echo "$USER_SELECTED_PKG" | sed "s/=.*$//"`
      S_DESC_LIST="${S_DESC_LIST} ${SELECTED_COMP_KEY}"

      # match the key, get the real package name component group array and update the
      # selected packages array
      NUM_COMP_GROUPS=${#COMP_PKGS_ARR[@]}
      ARRINDEX=0
      while [ $ARRINDEX -lt $NUM_COMP_GROUPS ];
      do
       	echo "${COMP_PKGS_ARR[$ARRINDEX]}" | grep $SELECTED_COMP_KEY >/dev/null

		 if [ $? == 0 ]; then
        	      # key found, get the real package list for the selected group
        	      PKG_IN_THIS_COMP_GROUP=`echo "${COMP_PKGS_ARR[$ARRINDEX]}" | sed "s/^.*=//"`
        	      # filter already existing packages

		     for NEW_PKG in $PKG_IN_THIS_COMP_GROUP;
        	      do
        	          PRESENT_IN_SELECTED_PKGS=0
             		  for SLCT_PKG in ${SELECTED_PKGS};
                            do
        	                  if [ "${SLCT_PKG}" == "${NEW_PKG}" ];
        	                  then
        	                      PRESENT_IN_SELECTED_PKGS=1
        	                      break
        	                  fi
        	            done

		             if [ $PRESENT_IN_SELECTED_PKGS == 0 ]; then
        	                 SELECTED_PKGS="$SELECTED_PKGS $NEW_PKG"
        	                NEW_PKGS_ADDED="${NEW_PKGS_ADDED} ${NEW_PKG}"
        	             fi
        	      done
        	      break
        	  fi

	ARRINDEX=`expr $ARRINDEX + 1`
done

      # automatically pull any dependencies that these packages have
      if [ "${NEW_PKGS_ADDED}" != "" ];
      then
          # reset the golbal variable
          for PKG in `echo "${NEW_PKGS_ADDED}"`
          do
              PullDependenciesUpdate "$PKG"
          done
      fi
      # update selected description array
      UpdateSelectedPkgsDescArr "UPDATE" # added

fi
}

###############################################################################
#
#  Function : ValidateUserSelection
#
#  Validate the user selection
#
###############################################################################

function ValidateUserSelection {
  USER_SELECTED_PKG=""
  OPT_PKG_INDEX=0
  USER_SELECTED_INDEX="bad"
  SELECTED_INDEX=$1

  OPTPKGARRLEN=${#APPLICABLE_OPTIONAL_PKGS[@]}

  if [ $SELECTED_INDEX -gt 0 -a $SELECTED_INDEX -lt $OPTPKGARRLEN -o $SELECTED_INDEX == $OPTPKGARRLEN ]; then
      SELECTED_INDEX=`expr $SELECTED_INDEX - 1`
      USER_SELECTED_PKG="${APPLICABLE_OPTIONAL_PKGS[$SELECTED_INDEX]}"
      USER_SELECTED_INDEX="good"
  else
      echo "Invalid selection."
  fi
  echo "$SELECTED_COMP_DESC" | grep "$USER_SELECTED_PKG" >/dev/null
  if [ $? == 0 ];
  then
      # This component already selected
      USER_SELECTED_INDEX="bad"
  fi
}
###############################################################################
#
#  function CheckForMultipleSelection
#
#
#
###############################################################################

function CheckForMultipleSelection {
    # remove any space characters
    STRIPPED_INPUT=`echo "$opt_pkg_index" | sed "s/ //g"`

    OPTPKGARRLEN=${#APPLICABLE_OPTIONAL_PKGS[@]}
    ARR_IDX=1
    # only process multiple selection less or eq to arr len
    while [ ! $ARR_IDX -gt $OPTPKGARRLEN ] && [ ! -z "$STRIPPED_INPUT" ];
    do
        INDEX=`expr index "$STRIPPED_INPUT" ","`
        LEN=`expr length $STRIPPED_INPUT`
        if [ $INDEX == 0 ];
        then
            ValidateUserSelection $STRIPPED_INPUT
            if [ $USER_SELECTED_INDEX == "good" ];
            then
                UpdatePkgSlection
            fi
            break
        elif [ $INDEX -lt $LEN ] || [ $INDEX == $LEN ];
        then
            INDEX=`expr $INDEX - 1`
            NUM_INPUT=`expr substr "$STRIPPED_INPUT" 1 $INDEX`
            if [ ! -z "$NUM_INPUT" ];
            then
                ValidateUserSelection $NUM_INPUT
                if [ $USER_SELECTED_INDEX == "good" ];
                then
                    UpdatePkgSlection
                fi
                INDEX=`expr $INDEX + 2`
                STRIPPED_INPUT=`expr substr "$STRIPPED_INPUT" $INDEX $LEN`
                ARR_IDX=`expr $ARR_IDX + 1`
            else
                break
            fi
        else
           break
        fi
    done
}

function CheckForInterfaces
{
     echo $S_DESC_LIST | grep "RAC" > /dev/null
     if [ $? == 0 ]
     then
        echo $S_DESC_LIST | grep "STOR" > /dev/null
        if [ $? != 0 ]
        then
           return 0
        fi 
     fi

     local PACKAGES_TO_CHECK="$INSTALLED_SRVADM_PKGS $SELECTED_PKGS"
     echo $PACKAGES_TO_CHECK | grep "srvadmin-jre" >/dev/null 
     if [ $? != 0 ]
     then
        echo $PACKAGES_TO_CHECK | grep "srvadmin-deng-snmp" >/dev/null 
        if [ $? != 0 ]
        then
             echo $PACKAGES_TO_CHECK | grep "srvadmin-omacore" >/dev/null 
             if [ $? != 0 ]
             then
                 echo $PACKAGES_TO_CHECK | grep "srvadmin-oslog" >/dev/null 
                 if [ $? != 0 ]
                 then
                    echo $PACKAGES_TO_CHECK | grep "srvadmin-itunnelprovider" >/dev/null 
                    if [ $? != 0 ]
                    then
                        return 1
                    fi
                 fi
             fi
          fi 
       fi
     return 0 
}
###############################################################################
#
#  Function : TakeUsersInputOnOptions
#
#
#
###############################################################################

function TakeUsersInputOnOptions
{
  if [ "${INSTALL_MODE}" == "add" -a ${APPS_PRESENT} == 0 ]; then
    cat <<EOF
  All Server Administrator components are currently installed.

EOF
  else
    if [ $FIRST_SCREEN == 0 ]
    then
    cat <<EOF
  Enter the number to select a component from the above list.
  Enter q to quit.

EOF
    else
    cat <<EOF
  Enter the number to select a component from the above list.
  Enter c to copy selected components to destination folder.
  Enter i to install the selected components.
  Enter r to reset selection and start over.
  Enter q to quit.

EOF
    fi
    opt_pkg_index=`Prompt "  Enter : "`
    CLEAR_CONTINUE="yes"
    # check if number
    if [ `echo $opt_pkg_index | egrep '^[0-9]+$' ` ]; then
        ValidateUserSelection $opt_pkg_index
        if [ $USER_SELECTED_INDEX == "good" ]; then
            FIRST_SCREEN=1
            UpdatePkgSlection
        fi
    else
        opt_pkg_index=`echo "$opt_pkg_index" | tr 'a-z' 'A-Z'`
        if [ "$opt_pkg_index" == "Q" ]; then
            CleanUpAndQuit
            CLEAR_CONTINUE="no"
        elif [ "$opt_pkg_index" == "R" ]; then
            FIRST_SCREEN=0
            ResetAndStartOver
        elif [ "$opt_pkg_index" == "I" ]; then
            if [ ! -z "$SELECTED_PKGS" ]; then
               #check for interface here
               CheckForInterfaces
               if [ $? != 0 ]
		then

#    -----------------------------------------
                clear
  cat <<EOF





     At least one management interface component is required to be 
     installed from the following:

     * Remote Enablement Interface
     * Server Administrator Web Server Interface
     * Server Administrator CLI Interface
     * Server Administrator SNMP Interface
     * Server Administrator Operating System Log Interface
           
     Press any key to continue...

EOF
     read
               ResetAndStartOver

#    -----------------------------------------
             else  
              InstallPackages
                CLEAR_CONTINUE="no"
             fi
            else
                echo "No packages are selected to install."
            fi
        elif [ "$opt_pkg_index" == "C" ]; then
            if [ ! -z "$SELECTED_PKGS" ]; then
                CopyPkgsToDestDirectory
                ExitMessage
                CLEAR_CONTINUE="no"
            else
                echo "No packages are selected to copy."
            fi
        else
            # check for multiple selection, comma separated
            opt_pkg_index=`echo "$opt_pkg_index" | sed "s/ //g"`
            if [ `echo $opt_pkg_index | egrep '^[0-9,]+$' ` ]; then
                CheckForMultipleSelection $opt_pkg_index
                if [ $? != 0 ]; then
                    USER_SELECTED_INDEX="bad"
                    FIRST_SCREEN=0
                 else
                    FIRST_SCREEN=1
                fi
            else
                echo "Invalid selection."
                USER_SELECTED_INDEX="bad"
            fi
        fi
    fi

    if [ $CLEAR_CONTINUE == "yes" ]; then
        # clear the screen and start over
        clear
        PrintGreetings
        DisplayCurrentSelection
        PresentOptionalPkgsToUser
        TakeUsersInputOnOptions
    fi
  fi
}

###############################################################################
#
#  Function : PresentOptionalPkgsToUser
#
#  Dislay the applicable optional packages to the user depending on the
#  results on prereqcheck. User can select one or more of these
#  optional packages.
#
###############################################################################

function PresentOptionalPkgsToUser {
  cat <<EOF
  Components for Server Administrator Managed Node Software:

EOF

  if [ "${INSTALL_MODE}" == "add" ];
  then
      echo "  Installed Components:"
      for INSTALLED_PKG in ${CURRENTLY_INSTALLED_KEYS}
      do
          NUM_COMP_GROUPS=${#COMP_PKGS_DESC_ARR[@]}
          ARRINDEX=0
          for (( i=0 ; i<${NUM_COMP_GROUPS} ; i++ ));
          do
              OPTPKGKEY=`echo "${COMP_PKGS_DESC_ARR[${i}]}" | sed "s/=.*$//"`
              OPTPKGDESC=`echo "${COMP_PKGS_DESC_ARR[${i}]}" | sed "s/^.*=//"`
              echo "${INSTALLED_PKG}" | grep "${OPTPKGKEY}" >/dev/null 2>&1 && [ "${OPTPKGDESC}" != "All" ] && echo "    [x] $OPTPKGDESC"
          done
      done
      cat <<EOF

  Add Components
EOF
  fi

  ARRLENGTH=${#APPLICABLE_OPTIONAL_PKGS[@]}
  ARRINDEX=0
  INDEX=1

  for (( i=0, j=1; i<${ARRLENGTH} ; i++, j++ ));
  do
      OPTPKGKEY=`echo "${APPLICABLE_OPTIONAL_PKGS[${i}]}" | sed "s/=.*$//"`
      OPTPKGDESC=`echo "${APPLICABLE_OPTIONAL_PKGS[${i}]}" | sed "s/^.*=//"`
      if [ -n "${OPTPKGKEY}" -a -n "${OPTPKGDESC}" ]; then
          APPS_PRESENT=1

          echo "$SELECTED_COMP_DESC" | grep "$OPTPKGKEY" >/dev/null 2>&1
          if [ $? == 0 ]; then
              echo "    [x] ${j}. $OPTPKGDESC"
          else
              echo "    [ ] ${j}. $OPTPKGDESC"
          fi
      fi
  done
  if [ ${APPS_PRESENT} == 0 ]; then
      echo "    - None"
  fi
  echo ""
}

###############################################################################
#
#  Function : UpdateApplicablePkgs
#
#  Update Applicable packages group based on already installed packages
#
#
###############################################################################

function UpdateApplicablePkgs {
  INDEX=0
  APP_INDEX=0
  TMP_ARR_LEN=${#TMP_APP_PKGS[@]}
  while [ ${INDEX} -lt ${TMP_ARR_LEN} ]
  do
      IGNORE=""
      for RM_PKG in ${CURRENTLY_INSTALLED_KEYS}
      do
          echo "${TMP_APP_PKGS[${INDEX}]}" | grep "$RM_PKG" >/dev/null
          if [ $? == 0 ];
          then
              IGNORE="true"
              break
          fi
      done

      if [ "${IGNORE}" != "true" ];
      then
          APPLICABLE_OPTIONAL_PKGS[${APP_INDEX}]="${TMP_APP_PKGS[${INDEX}]}"
          APP_INDEX=`expr ${APP_INDEX} + 1`
      fi

      INDEX=`expr ${INDEX} + 1`
  done
}

###############################################################################
# Function : PopulateApplicablePkgsArr
#
## Determine proper packages to install for this system
###############################################################################
function PopulateApplicablePkgsArr {

  if [ $ENABLE_CONSOLE_ONLY_INSTALL == 1 ]; then
      # Not a PE Server; Console only can be installed.
      if [ "${INSTALL_MODE}" == "add" ]; then
          TMP_APP_PKGS=([0]="$SERVER_ADMINISTRATOR_WEB_DESC")
          UpdateApplicablePkgs
      else
          APPLICABLE_OPTIONAL_PKGS=([0]="$SERVER_ADMINISTRATOR_WEB_DESC")
      fi

  elif [ $DRAC4PRESENT == 1 ]; then

      if [ "${UNSUPPORTED_OS_FOR_SSA}" == 0 ]; then
          if [ "${INSTALL_MODE}" == "add" ]; then
              TMP_APP_PKGS=([1]="$SERVER_ADMINISTRATOR_AGENT_DESC" [3]="$SERVER_ADMINISTRATOR_STANDARD_AGENT_DESC" [0]="$SERVER_ADMINISTRATOR_WEB_DESC" [2]="$STORAGE_MANAGEMENT_DESC" [4]="$REMOTE_ACCESS_CORE_DESC" [5]="$REMOTE_ACCESS_SA_PLUGINS_DESC" [6]="${SRVADMIN_ALL_DESC}")
              UpdateApplicablePkgs
          else
              APPLICABLE_OPTIONAL_PKGS=([1]="$SERVER_ADMINISTRATOR_AGENT_DESC" [3]="$SERVER_ADMINISTRATOR_STANDARD_AGENT_DESC" [0]="$SERVER_ADMINISTRATOR_WEB_DESC" [2]="$STORAGE_MANAGEMENT_DESC" [4]="$REMOTE_ACCESS_CORE_DESC" [5]="$REMOTE_ACCESS_SA_PLUGINS_DESC" [6]="${SRVADMIN_ALL_DESC}")
          fi

      elif [ "${UNSUPPORTED_OS_FOR_SSA}" == 1 ]; then
          if [ "${INSTALL_MODE}" == "add" ]; then
              TMP_APP_PKGS=([1]="$SERVER_ADMINISTRATOR_AGENT_DESC"  [0]="$SERVER_ADMINISTRATOR_WEB_DESC" [2]="$STORAGE_MANAGEMENT_DESC" [3]="$REMOTE_ACCESS_CORE_DESC" [4]="$REMOTE_ACCESS_SA_PLUGINS_DESC" [5]="${SRVADMIN_ALL_DESC}")
              UpdateApplicablePkgs
          else
              APPLICABLE_OPTIONAL_PKGS=([1]="$SERVER_ADMINISTRATOR_AGENT_DESC" [0]="$SERVER_ADMINISTRATOR_WEB_DESC" [2]="$STORAGE_MANAGEMENT_DESC" [3]="$REMOTE_ACCESS_CORE_DESC" [4]="$REMOTE_ACCESS_SA_PLUGINS_DESC" [5]="${SRVADMIN_ALL_DESC}")
          fi
      fi

  elif [ $DRAC5PRESENT == 1 ]; then
      if [ "${UNSUPPORTED_OS_FOR_SSA}" == 0 ]; then
          if [ "${INSTALL_MODE}" == "add" ]; then
              TMP_APP_PKGS=([0]="$SERVER_ADMINISTRATOR_AGENT_DESC" [4]="$SERVER_ADMINISTRATOR_STANDARD_AGENT_DESC" [5]="$SERVER_ADMINISTRATOR_WEB_DESC" [1]="$STORAGE_MANAGEMENT_DESC" [2]="$REMOTE_ACCESS_CORE_DESC5" [3]="$REMOTE_ACCESS_SA_PLUGINS_DESC5" [9]="${SRVADMIN_ALL_DESC}" [6]="${CLI_DESC}" [7]="${SNMP_DESC}" [8]="${LOG_DESC}")
              UpdateApplicablePkgs
          else
              APPLICABLE_OPTIONAL_PKGS=([0]="$SERVER_ADMINISTRATOR_AGENT_DESC" [4]="$SERVER_ADMINISTRATOR_STANDARD_AGENT_DESC" [5]="$SERVER_ADMINISTRATOR_WEB_DESC" [1]="$STORAGE_MANAGEMENT_DESC" [2]="$REMOTE_ACCESS_CORE_DESC5" [3]="$REMOTE_ACCESS_SA_PLUGINS_DESC5" [9]="${SRVADMIN_ALL_DESC}" [6]="${CLI_DESC}" [7]="${SNMP_DESC}" [8]="${LOG_DESC}")
          fi

      elif [ "${UNSUPPORTED_OS_FOR_SSA}" == 1 ]; then
          if [ "${INSTALL_MODE}" == "add" ]; then
              TMP_APP_PKGS=([0]="$SERVER_ADMINISTRATOR_AGENT_DESC" [4]="$SERVER_ADMINISTRATOR_WEB_DESC" [1]="$STORAGE_MANAGEMENT_DESC" [2]="$REMOTE_ACCESS_CORE_DESC5" [3]="$REMOTE_ACCESS_SA_PLUGINS_DESC5" [5]="${CLI_DESC}" [6]="${SNMP_DESC}" [7]="${LOG_DESC}" [8]="${SRVADMIN_ALL_DESC}")
              UpdateApplicablePkgs
          else
              APPLICABLE_OPTIONAL_PKGS=([0]="$SERVER_ADMINISTRATOR_AGENT_DESC" [4]="$SERVER_ADMINISTRATOR_WEB_DESC" [1]="$STORAGE_MANAGEMENT_DESC" [2]="$REMOTE_ACCESS_CORE_DESC5" [3]="$REMOTE_ACCESS_SA_PLUGINS_DESC5" [5]="${CLI_DESC}" [6]="${SNMP_DESC}" [7]="${LOG_DESC}" [8]="${SRVADMIN_ALL_DESC}")
          fi
      fi

  elif [ $IDRACPRESENT == 1 ]; then

      if [ "${UNSUPPORTED_OS_FOR_SSA}" == 0 ]; then
          if [ "${INSTALL_MODE}" == "add" ]; then
              TMP_APP_PKGS=([0]="$SERVER_ADMINISTRATOR_AGENT_DESC" [4]="$SERVER_ADMINISTRATOR_STANDARD_AGENT_DESC" [5]="$SERVER_ADMINISTRATOR_WEB_DESC" [1]="$STORAGE_MANAGEMENT_DESC" [2]="$REMOTE_ACCESS_CORE_DESC_IDRAC" [3]="$REMOTE_ACCESS_SA_PLUGINS_DESC_IDRAC" [9]="${SRVADMIN_ALL_DESC}" [6]="${CLI_DESC}" [7]="${SNMP_DESC}" [8]="${LOG_DESC}")
              UpdateApplicablePkgs
          else
              APPLICABLE_OPTIONAL_PKGS=([0]="$SERVER_ADMINISTRATOR_AGENT_DESC" [4]="$SERVER_ADMINISTRATOR_STANDARD_AGENT_DESC" [5]="$SERVER_ADMINISTRATOR_WEB_DESC" [1]="$STORAGE_MANAGEMENT_DESC" [2]="$REMOTE_ACCESS_CORE_DESC_IDRAC" [3]="$REMOTE_ACCESS_SA_PLUGINS_DESC_IDRAC" [9]="${SRVADMIN_ALL_DESC}" [6]="${CLI_DESC}" [7]="${SNMP_DESC}" [8]="${LOG_DESC}")
          fi

      elif [ "${UNSUPPORTED_OS_FOR_SSA}" == 1 ]; then
          if [ "${INSTALL_MODE}" == "add" ]; then
              TMP_APP_PKGS=([0]="$SERVER_ADMINISTRATOR_AGENT_DESC" [4]="$SERVER_ADMINISTRATOR_WEB_DESC" [1]="$STORAGE_MANAGEMENT_DESC" [2]="$REMOTE_ACCESS_CORE_DESC_IDRAC" [3]="$REMOTE_ACCESS_SA_PLUGINS_DESC_IDRAC" [8]="${SRVADMIN_ALL_DESC}" [5]="${CLI_DESC}" [6]="${SNMP_DESC}" [7]="${LOG_DESC}")
              UpdateApplicablePkgs
          else
              APPLICABLE_OPTIONAL_PKGS=([0]="$SERVER_ADMINISTRATOR_AGENT_DESC" [4]="$SERVER_ADMINISTRATOR_WEB_DESC" [1]="$STORAGE_MANAGEMENT_DESC" [2]="$REMOTE_ACCESS_CORE_DESC_IDRAC" [3]="$REMOTE_ACCESS_SA_PLUGINS_DESC_IDRAC" [8]="${SRVADMIN_ALL_DESC}" [5]="${CLI_DESC}" [6]="${SNMP_DESC}" [7]="${LOG_DESC}")
          fi
      fi

  else
      if [ "${UNSUPPORTED_OS_FOR_SSA}" == 0 ]; then
          if [ "${INSTALL_MODE}" == "add" ]; then
              # No DRAC, just standard optional packages
              TMP_APP_PKGS=([1]="$SERVER_ADMINISTRATOR_AGENT_DESC" [3]="$SERVER_ADMINISTRATOR_STANDARD_AGENT_DESC" [0]="$SERVER_ADMINISTRATOR_WEB_DESC" [2]="$STORAGE_MANAGEMENT_DESC" [4]="${SRVADMIN_ALL_DESC}")
              UpdateApplicablePkgs
          else
              # No DRAC, just standard optional packages
              APPLICABLE_OPTIONAL_PKGS=([1]="$SERVER_ADMINISTRATOR_AGENT_DESC" [3]="$SERVER_ADMINISTRATOR_STANDARD_AGENT_DESC" [0]="$SERVER_ADMINISTRATOR_WEB_DESC" [2]="$STORAGE_MANAGEMENT_DESC" [4]="${SRVADMIN_ALL_DESC}")
          fi

      elif [ "${UNSUPPORTED_OS_FOR_SSA}" == 1 ]; then
          if [ "${INSTALL_MODE}" == "add" ]; then
              # No DRAC, just standard optional packages
              TMP_APP_PKGS=([1]="$SERVER_ADMINISTRATOR_AGENT_DESC" [0]="$SERVER_ADMINISTRATOR_WEB_DESC" [2]="$STORAGE_MANAGEMENT_DESC" [3]="${SRVADMIN_ALL_DESC}")
              UpdateApplicablePkgs
          else
              # No DRAC, just standard optional packages
              APPLICABLE_OPTIONAL_PKGS=([1]="$SERVER_ADMINISTRATOR_AGENT_DESC" [0]="$SERVER_ADMINISTRATOR_WEB_DESC" [2]="$STORAGE_MANAGEMENT_DESC" [3]="${SRVADMIN_ALL_DESC}")
          fi
      fi
  fi

}

###############################################################################
#  Function : RunPrereqCheck
#
#  Run the Prerequisite check engine to determine the applicability
#  of some of the Optional packages, for example (RAC3 vs RAC4 vs RAC5 vs iDRAC)
#
###############################################################################

function RunPrereqCheck {
  # IsRACHardwarePresent should have already set the rac present variables
  if [ $DRAC4PRESENT == 1 ];
  then
    COMP_PKGS_ARR=([0]="$DELL_AGENT_PKGS" [1]="$CIM_AGENT_PKGS" [2]="$SA_WEB_PKGS" [3]="$OMSM_PKGS" [4]="$RAC_4_CORE" [5]="$RAC_4_SA_PKGS" [6]="${ALL_SA_RAC4_PKGS}")

  elif [ $DRAC5PRESENT == 1 ];
  then
    COMP_PKGS_ARR=([0]="$DELL_AGENT_PKGS_RAC5" [1]="$CIM_AGENT_PKGS_RAC5" [2]="$SA_WEB_PKGS" [3]="$OMSM_PKGS_RAC5" [4]="$RAC_5_CORE" [5]="$RAC_5_SA_PKGS" [6]="${ALL_SA_RAC5_PKGS}" [7]="$CLI_PKGS" [8]="$SNMP_PKGS" [9]="$LOG_PKGS")

  elif [ $IDRACPRESENT == 1 ];
  then
    COMP_PKGS_ARR=([0]="$DELL_AGENT_PKGS_IDRAC" [1]="$CIM_AGENT_PKGS_IDRAC" [2]="$SA_WEB_PKGS" [3]="$OMSM_PKGS_IDRAC" [4]="$IDRAC_CORE" [5]="$IDRAC_SA_PKGS" [6]="${ALL_SA_IDRAC_PKGS}" [7]="$CLI_PKGS" [8]="$SNMP_PKGS" [9]="$LOG_PKGS")

  else
    COMP_PKGS_ARR=([0]="$DELL_AGENT_PKGS" [1]="$CIM_AGENT_PKGS" [2]="$SA_WEB_PKGS" [3]="$OMSM_PKGS" [4]="${ALL_SA_NORAC_PKGS}" [5]="$CLI_PKGS" [6]="$SNMP_PKGS" [7]="$LOG_PKGS")
  fi
}

###############################################################################
#  Function : ReadAndValidateDestPath
#
#  Reads users input on destination directory and validates it.
#
###############################################################################

function ReadAndValidateDestPath {

  destpath=`Prompt "  Enter absolute destination path here ('q' to quit) : "`
  # Fix for DF35851 - Added a '$' in the pattern - BK
  if echo "${destpath}" | grep -iq "^q$" ; then
      CleanUpAndQuit
  else

      if echo "${destpath}" | grep -iq "^/" ; then

        if [ -d "$destpath" ];
        then
            cat <<EOF

  $destpath directory already exists!
  Please enter a new one.
EOF
            ReadAndValidateDestPath
        else
            LOCAL_DEST_PATH="$destpath"
            mkdir -p "$LOCAL_DEST_PATH"

            if [ $? != 0 ];
            then
                cat <<EOF
  Please try again.

EOF
                ReadAndValidateDestPath
            fi
        fi
      else
        ReadAndValidateDestPath
      fi
  fi
}

###############################################################################
#  Function : PrintGreetings
#
###############################################################################

function PrintGreetings {
  cat <<EOF

##############################################

  Server Administrator Custom Install Utility

##############################################

EOF
}

###############################################################################
# Function : GetPathToScript
#
# extracts the path to the script, this path will be used to locate
# the rpms repository on the CD or on the system
#
###############################################################################

function GetPathToScript {
    # $1 is the path to the script, inluding the script name
    REPLACE_ANY_SPACES=`echo "$1" | sed "s/ /\\\ /"`
    PATH_TO_SCRIPT=`dirname "$REPLACE_ANY_SPACES"`
    SCRIPTNAME=`basename $1`

    #Flag for itaomsa.sh script.
    ITA_OMSA_SCRIPT_FLAG=0
    
    #Flag for differntiating tarball types
    DISTRIBUTION_SPECIFIC=0

    # locate the rpms directory

    CURDIR=`pwd | sed "s/ /\\\ /"`
    cd "$PATH_TO_SCRIPT"
    SYSIDFILEPATH="`pwd | sed "s/ /\\\ /"`"
    CHECK_SRVADMIN_DIR=`pwd | sed 's#[[:space:]]*$##; s/\/$//' | awk -F"/" '{print $NF}'`

    if [ ! -f "$SYSIDFILEPATH/prereqcheck/smbios-sys-info-lite" ]; then
       if [ -d "$SYSIDFILEPATH/linux/supportscripts" ]; then
          SYSIDFILEPATH="$SYSIDFILEPATH/linux/supportscripts"
       else
          SYSIDFILEPATH="$SYSIDFILEPATH/supportscripts"
          DISTRIBUTION_SPECIFIC=1
       fi
       ITA_OMSA_SCRIPT_FLAG=1
    fi

    # Include CheckSystemType so that type/generation of system can be checked later.
    . "${SYSIDFILEPATH}/prereqcheck/CheckSystemType" dvd "${SYSIDFILEPATH}/prereqcheck"

    EIGHTGSYSIDFILEPATH="$SYSIDFILEPATH/$EIGHTGSYSIDFILE"
    NINEGSYSIDFILEPATH="$SYSIDFILEPATH/$NINEGSYSIDFILE"
    IDRACSYSIDFILEPATH="$SYSIDFILEPATH/$IDRACSYSIDFILE"
    LICENSEFILEPATH="$SYSIDFILEPATH/$LICENSEFILE"
    SYSREPORTPATH="$SYSIDFILEPATH/$SYSREPORT"
    SYSIDFILEPATH="$SYSIDFILEPATH/$SYSIDFILE"

    cd ..
    LINUXROOT=`pwd | sed "s/ /\\\ /"`

    if [ $ITA_OMSA_SCRIPT_FLAG == 1 ]; then
       if [ $CHECK_SRVADMIN_DIR != supportscripts  ]; then
          if [ $DISTRIBUTION_SPECIFIC == 0 ]; then
             LINUXROOT="$LINUXROOT/$CHECK_SRVADMIN_DIR/linux"
          else
             LINUXROOT="$LINUXROOT/$CHECK_SRVADMIN_DIR"
          fi
       fi
    fi

    PRE_UPGRADE_SCRIPT="$LINUXROOT/$PRE_UPGRADE_SCRIPT"
    #SERVICES_SCRIPT="$LINUXROOT/${SERVICES_SCRIPT}"
    PATH_TO_RPMS="$LINUXROOT/RPMS/supportRPMS/srvadmin/${PATH_TO_RPMS_SUFFIX}/$DISTRO"
    
    if [ ! -d "$PATH_TO_RPMS" ] || [ $ITA_OMSA_SCRIPT_FLAG != 0 ]; then
        if [ $DISTRIBUTION_SPECIFIC == 1 ]; then
           PATH_TO_RPMS="$LINUXROOT"
        fi
    fi

    cd "$CURDIR"
}

###############################################################################
# Function : EnterOMI
###############################################################################
function EnterOMI {
    SCRIPT_NAME=`basename $1`
    LOCK_UTIL=$(which lockfile 2>/dev/null)

    if [ -z "${LOCK_UTIL}" ]; then
        LOCK_UTIL=$(which /usr/bin/lockfile 2>/dev/null)

       #look for the bundled lockfile
        if [ -z "${LOCK_UTIL}" ]; then

	    REPLACE_ANY_SPACES=`echo "$1" | sed "s/ /\\\ /"`
    	    PATH_TO_SCRIPT=`dirname "$REPLACE_ANY_SPACES"`

       		# locate the lockfile directory

	    	CURDIR=`pwd | sed "s/ /\\\ /"`
    		cd "$PATH_TO_SCRIPT"
	    	LOCKFILEPATH="`pwd | sed "s/ /\\\ /"`"

    		if [ -f "$LOCKFILEPATH/prereqcheck/lockfile" ]; 
		then
       		  LOCKFILEPATH="$LOCKFILEPATH/prereqcheck"
            elif [ -d "$LOCKFILEPATH/linux/supportscripts/prereqcheck" ]; 
            then
         	  LOCKFILEPATH="$LOCKFILEPATH/linux/supportscripts/prereqcheck"
       		fi
        fi

        LOCK_UTIL=$(which "$LOCKFILEPATH"/lockfile 2>/dev/null)
        if [ -z "${LOCK_UTIL}" ]; then
          cat <<EOF
Warning: The lockfile utility is not found in PATH or /usr/bin.
         This utility prevents concurrent executions of ${SCRIPT_NAME}
         which can lead to unexpected or invalid installation results.
EOF
          return 0
        fi
    fi

    if ! eval ${LOCK_UTIL} -r0 ${OMI_LOCK_FILE} 2>/dev/null; then
        cat <<EOF
Error: ${SCRIPT_NAME} is already executing.
       Wait for ${SCRIPT_NAME} to complete and try again.
       Concurrent executions of ${SCRIPT_NAME} can lead to unexpected
       or invalid installation results.

       If you want to continue anyway, please delete '${OMI_LOCK_FILE}' and try again.
EOF
        KEEP_LOCK=true
        exit 15
    fi
}



###############################################################################
# Function : IsRACHardwarePresent
#
## Check for RAC
###############################################################################
function IsRACHardwarePresent {
  # including rac based on RAC hardware presence
  DRAC4PRESENT=0
  DRAC5PRESENT=0
  IDRACPRESENT=0

  SYSID=`GetSysId`
  IsThisSupportedGeneration
  # Check whether a valid system ID is available and is a supported DELL server
  if [ -n "${SYSID}" ] && [ $? = '0' ]; then
      # 9G System detected?
      if grep -iq "^${SYSID}=" "${NINEGSYSIDFILEPATH}" ; then
          # system is 9G, always install
          DRAC5PRESENT=1
          return
      else
          for X in `lspci -n`;
          do

           # DRAC4 found?
           grep $X=DRAC4 "${SYSIDFILEPATH}" > /dev/null;
           if [ $? = 0 ];
           then
               DRAC4PRESENT=1
               return
           fi
          done
      fi

      SYSID_HEX="0x$SYSID"
      SYSID_DEC=`printf "%d" $SYSID_HEX`

      MIN_11G_SYSID_HEX=0x0235
      MIN_11G_SYSID_DEC=`printf "%d" $MIN_11G_SYSID_HEX`

      if [ $SYSID_DEC -ge $MIN_11G_SYSID_DEC ]; then
          # system is with iDRAC, always install
          IDRACPRESENT=1
      fi
  fi
}

###############################################################################
# Function : GetSilentInstallPkgs
#
## Create a list of packages to install silently
###############################################################################
function GetSilentInstallPkgs {
    # Silent Express install will not install RAC3 or RAC4 on
    # a system without a RAC card

    # findout if option rac is chosen without rac hardware and set flag
    INCLUDE_RAC=""
    for OPT in ${USER_OPTS};
    do
        echo "${OPT}" | grep "rac" >/dev/null 2>&1
        if [ $? == 0 -a ${DRAC4PRESENT} == 0 -a ${DRAC5PRESENT} == 0 -a ${IDRACPRESENT} == 0 ]; then
            INCLUDE_RAC="true"
            break
        fi
    done

  # if rac option is chosen with rac hardware, determine the generation
  # of the server and include appropriate RAC software
  # 9G        --> RAC5 Always install with SA
  # 8G        --> RAC4
  # below 8G  --> RAC3
  if [ -n "${INCLUDE_RAC}" ]; then
    # run the prerequisite engine, include the correct RAC, based on server generation
    RunPrereqCheck
  fi
 
  if [ ${DRAC4PRESENT} == 1 ]; then
    SILENT_PKGS_LIST=([0]="${SILENT_CIM_AGENT}" [1]="${SILENT_DELL_AGENT}" [2]="${SILENT_WEB}" [3]="${SILENT_STORAGE}" [4]="${SILENT_RAC4}")

  elif [ ${DRAC5PRESENT} == 1 ]; then
    SILENT_PKGS_LIST=([0]="${SILENT_CIM_AGENT_RAC5}" [1]="${SILENT_DELL_AGENT_RAC5}" [2]="${SILENT_WEB}" [3]="${SILENT_STORAGE_RAC5}" [4]="${SILENT_RAC5}" [5]="${SILENT_AGENT}" [6]="${SILENT_CLI}" [7]="${SILENT_SNMP}" [8]="${SILENT_OSLOG}" [9]="${SILENT_CORE_STORAGE}" [10]="${SILENT_CORE_CIM}")

  elif [ ${IDRACPRESENT} == 1 ]; then
    SILENT_PKGS_LIST=([0]="${SILENT_CIM_AGENT_IDRAC}" [1]="${SILENT_DELL_AGENT_IDRAC}" [2]="${SILENT_WEB}" [3]="${SILENT_STORAGE_IDRAC}" [4]="${SILENT_IDRAC}" [5]="${SILENT_AGENT}" [6]="${SILENT_CLI}" [7]="${SILENT_SNMP}" [8]="${SILENT_OSLOG}" [9]="${SILENT_CORE_STORAGE}" [10]="${SILENT_CORE_CIM}")

  else
    SILENT_PKGS_LIST=([0]="${SILENT_CIM_AGENT}" [1]="${SILENT_DELL_AGENT}" [2]="${SILENT_WEB}" [3]="${SILENT_STORAGE}" [4]="${SILENT_AGENT}" [5]="${SILENT_CLI}" [6]="${SILENT_SNMP}" [7]="${SILENT_OSLOG}" [8]="${SILENT_CORE_STORAGE}" [10]="${SILENT_CORE_CIM}")
  fi


  SELECTED_SILENT_PKGS_TMP=""
  for OPT in ${USER_OPTS}; do
    ARRLEN=${#SILENT_PKGS_LIST[*]}
    for (( i=0 ; i<ARRLEN ; i++ )); do
      OPT_KEY=`echo "${SILENT_PKGS_LIST[i]}" | sed "s/=.*$//"`
      OPT_VAL=`echo "${SILENT_PKGS_LIST[i]}" | sed "s/^.*=//"`
      if [ "${OPT_KEY}" == "${OPT}" ]; then
        SELECTED_SILENT_PKGS_TMP="${SELECTED_SILENT_PKGS_TMP} ${OPT_VAL}"
        break
      fi
    done
  done
  if [ "${INSTALL_MODE}" == "add" ] || [ "${INSTALL_MODE}" == "tpty_add" ]; then
    for TMP_PKG in ${SELECTED_SILENT_PKGS_TMP}; do
      NOT_INSTALLED=0
      for INSTALLED_PKG in ${CURRENTLY_INSTALLED_PKGS}; do
        if [ "${INSTALLED_PKG}" == ${TMP_PKG} ]; then
          NOT_INSTALLED=1
          break
        fi
      done
      if [ ${NOT_INSTALLED} == 0 ]; then
        SELECTED_SILENT_PKGS="${SELECTED_SILENT_PKGS} ${TMP_PKG}"
      fi
    done
  else
    SELECTED_SILENT_PKGS="${SELECTED_SILENT_PKGS_TMP}"
  fi

  if [ -n "${SELECTED_SILENT_PKGS}" ]; then
    SELECTED_PKGS="${SELECTED_SILENT_PKGS}"

    # now, pull the dependencies
    for SLT_PKG in ${SELECTED_PKGS}; do
      PullDependencies ${SLT_PKG}
    done
  else
    echo "  Selected Server Administrator components are currently installed."
  fi
}


###############################################################################
# Function : AddLongOpt
#
## Process long command line options
###############################################################################
function AddLongOpt {
  if [ ${1} == "update" ] 
  then
     INSTALL_MODE="update"
  fi

  if [ ${1} == "express" -o ${1} == "update" ]; then
    ALL_LONG_OPTS=`echo "${LONG_OPTS[*]}" | sed "s/express//"`

    # if RAC hardware is not detected, remove rac from list of components to install
    if [ ${DRAC4PRESENT} == 0 -a ${DRAC5PRESENT} == 0 -a ${IDRACPRESENT} == 0 ]; then
      ALL_LONG_OPTS=`echo "${LONG_OPTS[*]}" | sed "s/rac//"`
    fi

    # SSA should not get installed in an express install.
    if [ ${1} == "express" ]; then
       ALL_LONG_OPTS=`echo "${LONG_OPTS[*]}" | sed "s/cimagent//"`
       ALL_LONG_OPTS=`echo "$ALL_LONG_OPTS" | sed "s/corecim//"`
    fi

    for LONG_OPT in ${ALL_LONG_OPTS}; do
      echo "${USER_OPTS}" | grep ${LONG_OPT} >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} ${LONG_OPT}"
    done

  elif [ ${1} == "storage" -o ${1} == "rac" ]; then
    echo "${USER_OPTS}" | grep ${1} >/dev/null 2>&1
    if [ $? != 0 ]; then
      echo "${USER_OPTS}" | grep "dellagent" >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} dellagent"
      USER_OPTS="${USER_OPTS} ${1}"
    fi
  elif [ ${1} == "stragent" ]; then
    echo "${USER_OPTS}" | grep ${1} >/dev/null 2>&1
    if [ $? != 0 ]; then
      echo "${USER_OPTS}" | grep "agent" >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} agent"
      USER_OPTS="${USER_OPTS} ${1}"
    fi
  elif [ ${1} == "cli" -o ${1} == "snmp" -o ${1} == "oslog" -o ${1} == "corecim" ]; then
    echo "${USER_OPTS}" | grep ${1} >/dev/null 2>&1
    if [ $? != 0 ]; then
      echo "${USER_OPTS}" | grep "agent" >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} agent"
      USER_OPTS="${USER_OPTS} ${1}"
    fi
  elif [ ${1} == "dellagent" ]; then
    echo "${USER_OPTS}" | grep "dellagent" >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} dellagent"
  elif [ ${1} == "agent" ]; then
    echo "${USER_OPTS}" | grep "agent" >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} agent"
  elif [ ${1} == "cimagent" ]; then
    echo "${USER_OPTS}" | grep "cimagent" >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} cimagent"
  elif [ ${1} == "web" ]; then
    echo "${USER_OPTS}" | grep "web" >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} web"
  elif [ ${1} == "autostart" ]; then
    AUTO_START=1
  elif [ ${1} == "preserve" ]; then
    PRESERVE_SCREEN=1
  fi
}

###############################################################################
# Function : AddShortOpt
#
## Process short command line options
###############################################################################
function AddShortOpt {
  if [ ${1} == "u" ] 
  then
     INSTALL_MODE="update"
  fi

  for SHORT_OPT in `echo "${1}" | sed "s/[a-z]/& /g"`; do
    if [ ${SHORT_OPT} == "x" -o ${SHORT_OPT} == "u" ]; then
      ALL_LONG_OPTS=`echo "${LONG_OPTS[*]}" | sed "s/express//"`

      # if RAC hardware is not detected, remove rac from list of components to install
      if [ ${DRAC4PRESENT} == 0 -a ${DRAC5PRESENT} == 0 -a ${IDRACPRESENT} == 0 ]; then
        ALL_LONG_OPTS=`echo "${LONG_OPTS[*]}" | sed "s/rac//"`
      fi

      # SSA should not get installed in an express install.
      if [ ${SHORT_OPT} == "x" ]; then
        ALL_LONG_OPTS=`echo "${LONG_OPTS[*]}" | sed "s/cimagent//"`
        ALL_LONG_OPTS=`echo "$ALL_LONG_OPTS" | sed "s/corecim//"`
      fi

      for LONG_OPT in ${ALL_LONG_OPTS}; do
        echo "${USER_OPTS}" | grep ${LONG_OPT} >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} ${LONG_OPT}"
      done
    elif [ ${SHORT_OPT} == "d" ]; then
      echo "${USER_OPTS}" | grep "dellagent" >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} dellagent"
    elif [ ${SHORT_OPT} == "g" ]; then
      echo "${USER_OPTS}" | grep "agent" >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} agent"
    elif [ ${SHORT_OPT} == "c" ]; then
      echo "${USER_OPTS}" | grep "cimagent" >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} cimagent"
    elif [ ${SHORT_OPT} == "s" ]; then
      echo "${USER_OPTS}" | grep "storage" >/dev/null 2>&1
      if [ $? != 0 ]; then
        echo "${USER_OPTS}" | grep "dellagent" >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} dellagent"
        USER_OPTS="${USER_OPTS} storage"
      fi
    elif [ ${SHORT_OPT} == "t" ]; then
      echo "${USER_OPTS}" | grep "stragent" >/dev/null 2>&1
      if [ $? != 0 ]; then
        echo "${USER_OPTS}" | grep "agent" >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} agent"
        USER_OPTS="${USER_OPTS} stragent"
      fi
    elif [ ${SHORT_OPT} == "z" ]; then
      echo "${USER_OPTS}" | grep "corecim" >/dev/null 2>&1
      if [ $? != 0 ]; then
        echo "${USER_OPTS}" | grep "agent" >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} agent"
        USER_OPTS="${USER_OPTS} corecim"
      fi
    elif [ ${SHORT_OPT} == "r" ]; then
      echo "${USER_OPTS}" | grep "rac" >/dev/null 2>&1
      if [ $? != 0 ]; then
        echo "${USER_OPTS}" | grep "dellagent" >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} dellagent"
        USER_OPTS="${USER_OPTS} rac"
      fi
    elif [ ${SHORT_OPT} == "i" ]; then
      echo "${USER_OPTS}" | grep "cli" >/dev/null 2>&1
      if [ $? != 0 ]; then
        echo "${USER_OPTS}" | grep "agent" >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} agent"
        USER_OPTS="${USER_OPTS} cli"
      fi
    elif [ ${SHORT_OPT} == "m" ]; then
      echo "${USER_OPTS}" | grep "snmp" >/dev/null 2>&1
      if [ $? != 0 ]; then
        echo "${USER_OPTS}" | grep "agent" >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} agent"
        USER_OPTS="${USER_OPTS} snmp"
      fi
    elif [ ${SHORT_OPT} == "l" ]; then
      echo "${USER_OPTS}" | grep "oslog" >/dev/null 2>&1
      if [ $? != 0 ]; then
        echo "${USER_OPTS}" | grep "agent" >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} agent"
        USER_OPTS="${USER_OPTS} oslog"
      fi
    elif [ ${SHORT_OPT} == "w" ]; then
      echo "${USER_OPTS}" | grep "web" >/dev/null 2>&1 || USER_OPTS="${USER_OPTS} web"
    elif [ ${SHORT_OPT} == "a" ]; then
      AUTO_START=1
    elif [ ${SHORT_OPT} == "p" ]; then
      PRESERVE_SCREEN=1
    fi
  done
}

###############################################################################
##
## Function:    GetOSType
##
## Decription:  Determines the operating system type.
##
## Returns:     0 = No error
##
## Return Vars: GBL_OS_TYPE=[GBL_OS_TYPE_ERROR|
##                           GBL_OS_TYPE_UKNOWN|
##                           GBL_OS_TYPE_RHEL5|
##			     GBL_OS_TYPE_SLES10|		       
##			     GBL_OS_TYPE_SLES11|	       
##			     GBL_OS_TYPE_ESX40|
##			     GBL_OS_TYPE_ESX41]		       ]
##                           GBL_OS_TYPE_RHEL6|
##                           GBL_OS_TYPE_RHEL7|
##              GBL_OS_TYPE_STRING=[RHEL5|SLES10|SLES11|ESX40|ESX41||RHEL6||RHEL7|UNKNOWN]
##
###############################################################################
GetOSType()
{
    # Set default values for return variables.
    GBL_OS_TYPE=${GBL_OS_TYPE_UKNOWN}
    GBL_OS_TYPE_STRING="UKNOWN"

   #for piecemeal install on existing 32bit OM components
   #########################################################
    # Get the list of all packages that are currently installed.
    rpm -qa | grep $PKG_PREFIX >/dev/null
    if [ $? == 0 ]; then
        # some packages are already installed.
        INSTALLED_SRVADM_PKGS=`rpm -qa | grep $PKG_PREFIX`
        if [ ! -z "${INSTALLED_SRVADM_PKGS}" ];
        then
            echo "$INSTALLED_SRVADM_PKGS" | grep "${PKG_PREFIX}-omcommon" >/dev/null 2>&1
            if [ $? != 0 ]; then
                # omcommon is not installed. Check for hapi then.
                echo "$INSTALLED_SRVADM_PKGS" | grep "${PKG_PREFIX}-hapi" >/dev/null 2>&1
                if [ $? != 0 ]; then
                    # hapi is not installed. Set it for any random rpm.
                    INSTALLED_PKG_FOR_VERSION_CHECK=`echo $INSTALLED_SRVADM_PKGS | awk -F" " '{print $1}'`
                else
                    # hapi is installed.
                    INSTALLED_PKG_FOR_VERSION_CHECK=${PKG_PREFIX}-hapi
                fi
            else
                # omcommon is installed.
                INSTALLED_PKG_FOR_VERSION_CHECK=${PKG_PREFIX}-omcommon
            fi

            INSTALLED_VER=`rpm -q ${INSTALLED_PKG_FOR_VERSION_CHECK} | sed -e s,.*[0-9].*,,`
            if [ ! -n "$INSTALLED_VER" ];
            then
                # check version and release numbers
                INSTALLED_VER=`rpm -q --queryformat "%{VERSION}" ${INSTALLED_PKG_FOR_VERSION_CHECK}`
            fi

            FORMATTED_INSTALLED_VER=`echo "${INSTALLED_VER}" | sed "s/\.//g" | sed "s/ //g"`
            #if [ $FORMATTED_INSTALLED_VER -lt 620 ]
            #then
            DISTRO=`rpm -q --qf "%{ARCH}" $INSTALLED_PKG_FOR_VERSION_CHECK`
            #else
            #    DISTRO=`rpm -q --qf "%{ARCH}" ${PKG_PREFIX}-omcommon`
            #fi

            if [ "${DISTRO}" = "noarch" ]; then
               DISTRO=`uname -i`
            fi
            
        fi
      else #if no srvadmin packages exist
            DISTRO=`uname -i`
      fi
######################################################### 


    if [ -z "${DISTRO}" ] || [ "${DISTRO}" != "i386" ] && [ "${DISTRO}" != "x86_64" ] 
    then
    	return 0
    fi

    # Local variable.
    local LOC_VERSION=""
    ESX_CHECK=0

    # Check if operating system is ESX.
    if [ -x /usr/bin/vmware ]; then
       /usr/bin/vmware -v  | grep "^VMware ESX 4" >/dev/null 2>&1
       [ $? = 0 ] && ESX_CHECK=1
    fi

    if [ $ESX_CHECK = 1 ]; then
       DISTRO="i386"
       case "$( /usr/bin/vmware -v )" in
         'VMware ESX 4.0.0'* )
                GBL_OS_TYPE=${GBL_OS_TYPE_ESX40}
                GBL_OS_TYPE_STRING="ESX400"
	        PATH_TO_RPMS_SUFFIX=ESX40
              ;;

          'VMware ESX 4.1.0'* )
                GBL_OS_TYPE=${GBL_OS_TYPE_ESX41}
                GBL_OS_TYPE_STRING="ESX410"
	        PATH_TO_RPMS_SUFFIX=ESX41
              ;;
       esac

    # Else check if operating system is RHEL5.
    elif [ -f /etc/redhat-release ] && [ `grep -c "Tikanga" /etc/redhat-release` -gt 0 ]; then
        GBL_OS_TYPE=${GBL_OS_TYPE_RHEL5}
        GBL_OS_TYPE_STRING="RHEL5"
        PATH_TO_RPMS_SUFFIX=RHEL5
    elif [ -f /etc/redhat-release ] && [ `grep -c "Santiago" /etc/redhat-release` -gt 0 ]; then
        if [ "${DISTRO}" = "x86_64" ]; then # 32bit is not supported for rhel6.
           GBL_OS_TYPE=${GBL_OS_TYPE_RHEL6}
           GBL_OS_TYPE_STRING="RHEL6"
           PATH_TO_RPMS_SUFFIX=RHEL6
        fi
    elif [ -f /etc/redhat-release ] && [ `grep -c "Maipo" /etc/redhat-release` -gt 0 ]; then
        if [ "${DISTRO}" = "x86_64" ]; then # 32bit is not supported for rhel7.
           GBL_OS_TYPE=${GBL_OS_TYPE_RHEL7}
           GBL_OS_TYPE_STRING="RHEL7"
           PATH_TO_RPMS_SUFFIX=RHEL7
        fi

    # Else check if operating system is SLES.
    elif [ -f /etc/SuSE-release ]; then
        LOC_VERSION=`cat /etc/SuSE-release | grep "VERSION" | sed  -e 's#[^0-9]##g'`

        # Check if operating system is SLES10.
        if [ "${LOC_VERSION}" = "10" ]; then
            GBL_OS_TYPE=${GBL_OS_TYPE_SLES10}
            GBL_OS_TYPE_STRING="SLES10"
            PATH_TO_RPMS_SUFFIX=SLES10
        # Check if operating system is SLES11.
        elif [ "${LOC_VERSION}" = "11" ]; then
            GBL_OS_TYPE=${GBL_OS_TYPE_SLES11}
            GBL_OS_TYPE_STRING="SLES11"
            PATH_TO_RPMS_SUFFIX=SLES11
        fi
    fi

    return 0
}

###############################################################################
# Function : ValidateOpts
#
## Validate command line options
###############################################################################
function ValidateOpts {
  if [ $# -gt 0 ]; then
    # invald characters
    #BY  added //,.,_,0-9,A-Z
    echo "${*}" | sed "s/ //g" | egrep "^[-/._a-z0-9//]+$" >/dev/null 2>&1
    if [ $? != 0 ]; then
      echo "Invalid Options, please see the usage below"
      echo ""
      Usage
    fi
    
    MOD_CM=$*

# replace  $* by $MOD_CM which does not contain --prefix <path>
for param in $MOD_CM; do
      # check for long option
      echo "${param}" | egrep "^--[a-z\-]+$" >/dev/null 2>&1
      if [ $? == 0 ]; then
        GOOD_LONG_OPT=1
        for (( i=0 ; i<${#LONG_OPTS[*]} ; i++ )); do
          if [ ${param} == "--${LONG_OPTS[i]}" ]; then
            GOOD_LONG_OPT=0
            AddLongOpt ${LONG_OPTS[i]}
            break
          fi
        done

        if [ ${GOOD_LONG_OPT} != 0 ]; then
          echo "Invalid Option ${param}, please see the usage below"
          Usage
        fi
      else
        # check for short option
        VALID_SHORT_OPTS=`echo "${SHORT_OPTS[*]}" | sed "s/ //g"`
        echo "${param}" | egrep "^-[${VALID_SHORT_OPTS}]+$" >/dev/null 2>&1
        if [ $? == 0 ]; then
          TEMP_OPT=`echo "${param}" | sed "s/-//"`
          AddShortOpt ${TEMP_OPT}
        else
          echo "Invalid Option ${param}, please see the usage below"
          Usage
        fi
      fi
    done

    ARG_CHECK=`echo ${MOD_CM} | sed 's#^[[:space:]]*##; s#[[:space:]]*$##'`

    VALID_PARAMS=0
    for i in $ARG_CHECK; do
       if [ "$i" != "-a" -a "$i" != "--autostart" -a "$i" != "-p" -a "$i" != "--preserve" ]; then
          VALID_PARAMS=1
       fi
    done

    if [ $VALID_PARAMS != 1 ]; then
       echo "Invalid usage. \"${ARG_CHECK}\" option(s) can be used only along with other options"
       echo ""
       Usage
    fi
  fi
}

###############################################################################
# Function : CheckSystemID
#
## exit on upsupported system
###############################################################################
function CheckSystemID {

  [ -n "${OMIIGNORESYSID}" ] && return

  SYSID=`GetSysId`

  # Check for sysid Injection
  if [ -n "${OM_SYSTEM_ID}" ]; then
     SYSID=`echo ${OM_SYSTEM_ID} | sed "s/ //g"`
  fi

  IsThisSupportedGeneration
  retVal=$?
  if [ $retVal = '0' ] 
  then
         ENABLE_CONSOLE_ONLY_INSTALL=0
  else
    if [ $retVal = '3' ]
    then
        ENABLE_CONSOLE_ONLY_INSTALL=1
    else
       echo "Unsupported Server... OpenManage 8.0.1 installation cannot continue.. Exiting !!"
       exit 1
    fi
  fi

  if [ -n "${SYSID}" ]; then 
     if [ "${SYSID}" = "0124" ] || [ "${SYSID}" = "0134" ] || [ "${SYSID}" = "0135" ] || [ "${SYSID}" = "0141" ]; then
          #Servers that are not supported.
          ENABLE_CONSOLE_ONLY_INSTALL=1
     fi
  fi

  return 0
}

###############################################################################
# Function : ShowLicense 
# Function will show license
#
###############################################################################
function ShowLicense()
{
   REQRPM=`rpm -qa | grep srvadmin-omilcore`
   
   if [ -z $REQRPM ]
   then
       if [ -f "$LICENSEFILEPATH" ]
       then
           more "$LICENSEFILEPATH"
           echo ""
           echo -n "Do you agree to the above license terms? ('y' for yes | 'Enter' to exit): "
           read ANSWER 
           answer=`echo "${ANSWER}" | sed 's#^[[:space:]]*##; s#[[:space:]]*$##'`
           if echo "${answer}" | grep -qi "^y$" 
           then
               clear
           else
               exit 0
           fi
       fi
   fi
}
###############################################################################
# Function : Main
#
# This is the starting point for the script, this
# function invokes other functions in the required
# order
###############################################################################
function Main {

    # operating system check
    GetOSType
    if [ "${GBL_OS_TYPE}" = "${GBL_OS_TYPE_UKNOWN}" ] || [ "${GBL_OS_TYPE}" = "${GBL_OS_TYPE_ERROR}" ]; then
        # Operating system type is unknown, or an error occurred trying to
        # determine the operating system type. Exit with Error 2
   cat <<EOF
     Unrecognized Operating System or Architecture. This script cannot 
     continue with the installation. Select rpms from the OS folder in 
     the media that closely matches this Operating System to continue 
     with the manual install.
EOF
        SetErrorAndInterrupt 2
    fi

    # get the path and the script name
    GetPathToScript "$0"

    #Show License if there are no options passed
    if [ -z "$1" ]
    then
       ShowLicense
    fi

    # detect and set drac present variables
    IsRACHardwarePresent

    # process any options passed
    if [ $# -gt 0 ];
    then
      # process options
     # ValidateOpts $*
    ValidateOpts $*
    fi

    # block older version
    BlockOnOldOM

    # system id check
    CheckSystemID

    # process any options passed
    #as MOD_CM Contains filtered command line options i.e other than --prefix
    #if [ $# -gt 0 ]; then

    if [  -n "${MOD_CM}" ]; then

      # if already installed, should be able to add comps OR upgrade
      DetectPreviousInstall "silent"

      if [ ! $SKIP_UPGRADE -eq 0 ] && [ "${INSTALL_MODE}" != "update" ]
      then
      # gather silent install packages
      GetSilentInstallPkgs
      CheckForInterfaces
      if [ $? != 0 ]
      then
         Log "$0 : You have chosen to install Server Instrumentation. To proceed with installation, choose either Server Administrator Web Server or one of the Management Interfaces. This will enable you to manage the server."
         echo "No interface selected for install. No Management interface found installed." 
         SetErrorAndInterrupt 10
      fi
      # install
      InstallPackages "silent"
      fi

    else
      # clear screen and print greetings

      clear
      PrintGreetings

      # run the prerequisite engine
      RunPrereqCheck

      # if already installed, should be able to add comps
      DetectPreviousInstall

      # run the prerequisite engine
      PopulateApplicablePkgsArr

      # list the optional packages that user can choose from
      PresentOptionalPkgsToUser

      # read users input on the optional packages
      TakeUsersInputOnOptions
    fi

}

function CheckDependenciesForSSA
{
   rpm --test -ivh "$PATH_TO_RPMS"/${PKG_PREFIX}-ssa* >/tmp/ssaCheck.$$ 2>&1

   if cat /tmp/ssaCheck.$$ | grep -iq "wsman" ; then
      return 1
   elif cat /tmp/ssaCheck.$$ | grep -iq "sfcb" ; then
      return 1
   elif cat /tmp/ssaCheck.$$ | grep -iq "sfcc" ; then
      return 1
   else
      return 0
   fi
}

###############################################################################
#
#  Log
#
###############################################################################
function Log {
  MSG="${1}"

  cat <<EOF >> ${SRVADMIN_INSTALL_LOG}

------------------------------------------------------------------------
Server Administrator Custom Install ${MSG}
`date`
------------------------------------------------------------------------
EOF
}

###############################################################################
#
#  Script execution begins here with the invocation
#  of Main function
#
###############################################################################
# set traps to handle signals
#  signals are needed due to the way this utility was originally written,
#  on encountering an error - exit was called.
#  Clean up still needed to be done, e.g. clean up the lock file.
trap 'InterruptOMI' INT QUIT TERM
trap 'InterruptOMIHUP' HUP
trap 'InterruptOMIExit' exit

# exit if not root
ExitNonRoot "This Utility"

# lock omi
EnterOMI "$0"

# Allow output to a log file if OMIDEBUG variable is set.
if [ -n "${OMIDEBUG}" ];
then
    Log "Start"
    Main $* 2>&1 | tee -a ${SRVADMIN_INSTALL_LOG}
    Log "Stop"
else
    Main $* 2>&1
fi

if [[ $INSTALL_STATUS != 0 ]]; then
   echo ""
   echo " -   Installation of some of the packages have failed."
   
   echo "$FINAL_INSTALL_LIST" | grep "srvadmin-ssa" >/dev/null 2>&1
   if [ $? = 0 ]; then
      CheckDependenciesForSSA
      if [ "$?" == "1" ]; then
         echo " -   Missing WSMan/SFCB/SFCC Packages are available from 
                                srvadmin/linux/RPMS/supportRPMS folder."
         echo " -   Automation of WSMan/SFCB can be done with scripts available from 
                                srvadmin/linux/supportscripts/opensource-conf folder."
         echo ""
      fi

      [ -f /tmp/ssaCheck.$$ ] && rm -f /tmp/ssaCheck.$$ >/dev/null 2>&1
   fi
   
   echo " -   Please refer to 'readme_ins.txt' file for more details."
   echo " -   The readme_ins.txt can be found in srvadmin/docs/readme folder"
   echo ""
   let INSTALL_STATUS=$INSTALL_STATUS+100
   exit $INSTALL_STATUS
fi

if [[ $UPGRADE_STATUS != 0 ]]; then
   echo ""
   echo " -   Upgrade of some of the packages have failed."
   
   echo "$TO_BE_INSTALLED_LIST" | grep "srvadmin-ssa" >/dev/null 2>&1
   if [ $? = 0 ]; then
      CheckDependenciesForSSA
      if [ "$?" == "1" ]; then
         echo " -   Missing WSMan/SFCB/SFCC Packages are available from 
                                srvadmin/linux/RPMS/supportRPMS folder."
         echo " -   Automation of WSMan/SFCB can be done with scripts available from 
                                srvadmin/linux/supportscripts/opensource-conf folder."
         echo ""
      fi

      [ -f /tmp/ssaCheck.$$ ] && rm -f /tmp/ssaCheck.$$ >/dev/null 2>&1
   fi
   
   echo " -   Please refer to 'readme_ins.txt' file for more details."
   echo " -   The readme_ins.txt can be found in srvadmin/docs/readme folder"
   echo ""
   let UPGRADE_STATUS =$UPGRADE_STATUS+200
   exit $UPGRADE_STATUS
fi

if [[ $SKIP_UPGRADE -eq 0 ]]; then
   exit 21
fi

exit 0
