#!/bin/sh
# -----------------------------------------------------------------------------
#
#          DELL COMPUTER CORPORATION PROPRIETARY INFORMATION
#
# This software is supplied under the terms of a license agreement or
# nondisclosure agreement with Dell Computer Corporation and may not
# be copied or disclosed except in accordance with the terms of that
# agreement.
#
# Copyright (c) 2007 Dell Computer Corporation. All Rights Reserved.
#
# Script Name: ivmdeploy.sh
#
# Purpose: sample script to illustrate OS/patch deployment via Virtual Media.
#    NOTE: the boot image supplied to this script performs the deployment.
#          [ie. the boot image determines what/how deployment is done]
# -----------------------------------------------------------------------------

SCRIPT=${0##*/}
RC=0
PASS=0			# count successes/failures, report later
FAIL=0
DBUG=0
VMDELAY=30				# expected max. Virtual Media connect time (seconds)
PID=$$
DATETIME=`date +%Y%m%d.%H%M%S`
LOGPATH=$(mktemp -d /tmp/.${SCRIPT%.sh}-XXXXXX)
trap 'rm -rf $LOGPATH' QUIT HUP INT TERM EXIT
EXITING=0
NOKILL=0
CLEANUP="$LOGPATH/.$PID-CLEANUP"		# to track things needing cleanup
CLEAN_TEMP=/tmp/clean_tmp
FLOPPY_FORMAT1=.img
FLOPPY_FORMAT2=.IMG
CDROM_FORMAT1=.iso
CDROM_FORMAT2=.ISO
FLOPPY=1
CDROM=2

# --- console/log printing
function printcon
{
	echo -e "$@"
	[ -n "$CONLOG" ] && echo -e "$@" >>$CONLOG
}

function usage
{	# -------------------------------------------------------------------------
	# if usage is passed an argument, it's an error message: print it & return.
	#  else, print the usage text and exit
	# -------------------------------------------------------------------------
   if [ $# -gt 0 ]; then
		FAIL=$((FAIL + 1))
		printcon "\aerror($FAIL): $@"
	else
		printcon "\nusage: $SCRIPT -r <RAC-IP> -u <RAC-userid> -p <RAC-password>"
		printcon "                 [ -f <floppy-image | FDDrive> | -c <ISO9660-image | CDDrive> ]"
   	printcon "where:"
		printcon "       <ISO9660-image> and <floppy-image> are bootable image files"
		printcon "	 <FDDrive> and <CDDrive> are Floppy and CD devices respectively"
		printcon "       <RAC-userid> = RAC user id, with 'virtual media' privilege"
	   printcon "       <RAC-password> = RAC user password"
   	printcon "       <RAC-IP> is either:"
	   printcon "         - a string of the form: 'RAC-IP'"
	   printcon "         - a file containing lines matching that form"
	   printcon "       In the latter case, the boot image is setup and booted"
		printcon "       for each host/RAC IP contained in the file.\n"
   	printcon "*Note: your boot image determines what is deployed, and how.\n"
		EXITING=1
   	exit $RC
	fi
}

# --- termination handler
function quit
{
	if [ $EXITING -eq 0 ]; then
		EXITING=1
		RC=99
	   [ $# -ge 1 ] && printcon "\a\n$@\n"
	fi
	# -------------------------------------------------------------------------
	# if this script is interrupted, any running tasks it spawned are terminated.
	#  to change this default behavior, set NOKILL=1 at the top of this script.
	# then, any work started prior to the interruption is allowed to complete.
	# -------------------------------------------------------------------------
	if [ -e $CLEANUP -a $NOKILL -eq 0 ]; then
		printcon "Cleaning up, please wait.."
		while read
		do												# we're dying unexpectedly..
			if [ -e "$REPLY" ]; then
				rm -f $REPLY						# cleanup: delete log file
			else
				kill $REPLY && sleep 3			# cleanup: kill process
			fi
		done <$CLEANUP
		rm -f $CLEANUP
	fi
	exit $RC
}
function clean
{
	find /tmp/ -name .${SCRIPT%.sh}* -type d > $CLEAN_TEMP

	while read
	do
		if [ -d $REPLY ]; then
			CLEAN1=`basename $REPLY/.*CLEANUP`
			#echo $CLEAN1
			CLEAN=$REPLY/$CLEAN1
			if [ -f $CLEAN ]; then
				echo "Cleaning old ivmdeploy, please wait.."
				while read
				do							
					echo $REPLY
					if [ -e "$REPLY" ]; then
						rm -f $REPLY						# cleanup: delete log file
					else
						kill -9 $REPLY && sleep 3			# cleanup: kill process
					fi
				done <$CLEAN
				rm -f $CLEAN
			fi
		fi
		
	done < $CLEAN_TEMP

	find /tmp/ -name .${SCRIPT%.sh}* -type d -exec rm -rf '{}' \; 2> /dev/null
	rm -f $CLEAN_TEMP
}

# --- Checks if proper input file is given with appropriate option
function check_validity
{
	if [ "$2" == "1" ]; then
		echo $1 | grep $CDROM_FORMAT1 || echo $1 | grep $CDROM_FORMAT2 > /dev/null
	else
		echo $1 | grep $FLOPPY_FORMAT1 || echo $1 | grep $FLOPPY_FORMAT2 > /dev/null
	fi

	return $?	
}

# --- hook for unplanned termination
trap "quit \"$SCRIPT interrupted!!\"" EXIT

# --- parse command arguments
if [ $# -gt 0 ]; then
	TEMP=`getopt -o r:u:p:f:c:d -- "$@"`
	RC=$?
	if [ $RC -eq 0 ]; then
		eval set -- "$TEMP"
		while true
		do
			case "$1" in
				-d) set -x; DBUG=1; VMOPTS="-v"; shift 1
				;;
				-f) [ -n "$TYPE" ] && TYPE="?" || TYPE="-f"
				    check_validity $2 $FLOPPY
				    if [ "$?" == "0" ]; then
usage ; break;
				    else
FILE=$2
				    fi	
			            shift 2
				;;
				-c) [ -n "$TYPE" ] && TYPE="?" || TYPE="-c"
				    check_validity $2 $CDROM
				    if [ "$?" == "0" ]; then
					usage
					break
				    else
				    	FILE=$2
				    fi
				    shift 2
				;;
				-p) RACP=$2; shift 2
				;;
				-u) RACU=$2; shift 2
				;;
				-r) LIST=$2; shift 2
				;;
				--) shift; break
				;;
				*) RC=8; break
				;;
			esac
		done
	fi
fi

# --- validate command arguments
[ $RC -gt 0 ] && usage "invalid options found"
[ -z "$LIST" ] && usage "<RAC-IP> missing"
[ -z "$RACU" ] && usage "<RAC-userid> missing"
[ -z "$RACP" ] && usage "<RAC-password> missing"
[ ".$TYPE" = ".?" ] && usage "use [<floppy-image> | FDDrive] or [<ISO9660-image> | CDDrive], not both"

#if [ -z "$FILE" ]; then
#	usage "an image file is required"
#elif [ ! -e "$FILE" ]; then
#	usage "no such image file: '$FILE'"
#elif [ ! -f "$FILE" ]; then
#	usage "the image file must be a regular file"
#fi

# --- ensure RAC utilities available
if [ -z "`which ipmitool 2>/dev/null`" ]; then
	usage "ipmitool doesn't appear to be installed"
fi
if [ -z "`which ./iVMCLI 2>/dev/null`" ]; then
	usage "iVMCLI doesn't appear to be installed"
fi

[ $FAIL -gt 0 ] && usage				# error(s) ? quit now

# --- check for writable floppy image
if [ ".$TYPE" = ".-f" -a -w $FILE ]; then
	printcon "\n**warning: <floppy-image> appears to be writable."
	printcon "  Enter 'q' now to quit, any other key to continue.."
	read RESP
   if [ ".$RESP" = ".q" -o ".$RESP" = ".Q" ]; then
		printcon "\nOK, quitting.. (good choice)"
		EXITING=1
		exit 0
	fi
fi
#Cleaning old deploy if any
clean
# --- OK, "let's roll"

mkdir $LOGPATH >/dev/null 2>&1		# create directory for this run
CONLOG=$LOGPATH/console.log			# setup the console log file

if [ ! -e "$LIST" ]; then				# single RAC IP provided..
	tmpfile="$LOGPATH/.iplist"			# ..create a temp file
	echo "$LIST" >$tmpfile				# ..add the single RAC IP
	echo "$tmpfile" >>$CLEANUP			# ..mark it for cleanup
	LIST=$tmpfile
fi

# RAC 'boot once' property
BOPROP="raw 0x00 0x08 0x05 0x80 0x20 0x00 0x00 0x00"

while read									# for each RAC IP/hostname:
do
	IP=${REPLY%%[[:space:]]*}
	[ -z "$IP" -o -z "${IP##\#*}" ] && continue # (skip blank/comment lines)

	printcon "\nIP = $IP"
	tmpfile="$LOGPATH/log.$IP"			# VM process logfile name (includes RAC IP)

	# ideally, at this point we'd check for the appropriate boot device 
	# at top of boot list for the target server (or set it thus).
	# currently, we can't :(

	# fire off a virtual media process for the target server
	./iVMCLI -r $IP -u $RACU -p $RACP ${TYPE} ${FILE} $VMOPTS >$tmpfile 2>&1 &
	VMPID=$!
	printcon "\tVM PID = $VMPID"
	echo "$VMPID" >>$CLEANUP		# ..mark the VM process ID for cleanup
	# ensure the virtual media process connects successfully.
	#  (check periodically, so we don't always incur the max delay)
	CONN=0
	for ((i=0; i<$VMDELAY; i++))
	{
		sleep 1
		grep "Enter the Media to be stopped from redirecting" $tmpfile >/dev/null 2>&1
		if [ $? -eq 0 ]; then
			CONN=1							# we're connected
			printcon "\tconnected"
			break;
		fi
	}
	if [ $CONN -eq 1 ]; then
	
		sleep 7
		ipmitool -I lanplus -H $IP -U $RACU -P $RACP $BOPROP TRUE 2>&1
		sleep 1	
		ipmitool -I lanplus -H $IP -U $RACU -P $RACP chassis power cycle 2>&1
		sleep 20
		ipmitool -I lanplus -H $IP -U $RACU -P $RACP chassis power up 2>&1
	fi
	#CONN=1
	if [ $CONN -eq 1 ]; then			# deployment is underway / host booting up
		echo $tmpfile >>$CLEANUP		# ..ditto for the VM process logfile
		PASS=$((PASS + 1))
		printcon "\tOK."
	else
		# ----------------------------------------------------------------------
		# deployment failed, for this RAC host:
		#  - kill the Virtual Media process
		#  - save the RAC IP in the 'failed-hosts' file
		# NOTE: we never clean up logs for failed hosts -- must be done manually
		# ----------------------------------------------------------------------
		kill $VMPID >/dev/null 2>&1
		echo "$IP" >>$LOGPATH/failed-hosts
		FAIL=$((FAIL + 1))
		printcon "\tFAILED -- see '$tmpfile'"
	fi
done < $LIST

# --- normal completion: if we get here, don't terminate any work we started.
#     (all files in the LOGPATH directory are left alone, for admin perusal)

NOKILL=1
EXITING=1
printcon "\n$PASS deployment tasks started successfully."
printcon "$FAIL deployment tasks failed."
if [ $FAIL -gt 0 ]; then
	printcon "\nNote: the failed RAC IPs are in file '$LOGPATH/failed-hosts'"
	printcon "      Check the individual process logfiles for more information"
	printcon "      (for IP=XX, it's logfile is '$LOGPATH/log.XX')"
fi
printcon "\nThe console log was saved as '$CONLOG'."
exit 0
