/*******************************************************************************

  
  Copyright(c) 2003 - 2005 Intel Corporation. All rights reserved.
  
  This program is free software; you can redistribute it and/or modify it 
  under the terms of the GNU General Public License as published by the Free 
  Software Foundation; either version 2 of the License, or (at your option) 
  any later version.
  
  This program is distributed in the hope that it will be useful, but WITHOUT 
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
  more details.
  
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 
  Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
  The full GNU General Public License is included in this distribution in the
  file called LICENSE.
  
  Contact Information:
  Linux NICS <linux.nics@intel.com>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497
  
  file: iamt_main.c
  part of "Intel(R) Active Management Technology - KCS" Linux driver 

*******************************************************************************/

#include <linux/kdev_t.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/reboot.h>
#include <asm/uaccess.h>
#include <linux/poll.h>
#include <linux/init.h>

#include "iamt.h"

#define MAX_KCS_MESSAGE_SIZE 4160

char iamt_driver_name[] = "iamt";
char iamt_driver_string[] = "Intel(R) Active Management Technology - KCS";
char iamt_driver_version[] = "1.0.9";
char iamt_copyright[] = "Copyright (c) 2003 - 2005 Intel Corporation.";

/* major number for device */
int major = 0;

/* The device pointer */
struct pci_dev *kcs_device = NULL;

/* lock for the device */
spinlock_t device_lock = SPIN_LOCK_UNLOCKED;
spinlock_t extra_lock = SPIN_LOCK_UNLOCKED;

/* WD timeout parameter */
unsigned short wd_timeout = DEAFULT_WD_TIMER;

MODULE_PARM(wd_timeout, "h");
MODULE_PARM_DESC(wd_timeout, "Timeout value for WD commands in seconds (min=40 max=65535)");

/**
 * alloc_message - initialize a kcs message structure and set it up.
 * @msg: a pointer to the message structure
 * @size: the size of the message.
 * 
 * Returns 0 on success, non-zero on failure
 */

int alloc_message(struct iamt_kcs_msg *msg, unsigned int size) 
{
    msg->msg = kmalloc(size, GFP_KERNEL);
    if (!msg->msg) {
        return -ENOMEM;
    }
    msg->size = size;
    return 0;
}

/**
 * free_msg - frees a kcs message.
 * @msg: a pointer to the message structure
 */

void free_msg(struct iamt_kcs_msg *msg) 
{
    kfree(msg->msg);
    msg->msg = NULL;
}

/**
 * alloc_priv - allocates a private file structure and set it up.
 * @file: the file structure
 * 
 * Returns the allocated file or NULL on failure
 */

struct iamt_kcs_file_private *alloc_priv(struct file *file) 
{
    struct iamt_kcs_file_private *priv = kmalloc(sizeof(struct iamt_kcs_file_private), GFP_KERNEL);
    if (!priv) {
        return NULL;
    }
    spin_lock_init(&priv->file_lock);
    init_waitqueue_head(&priv->wait);
    priv->file = file;
    priv->state = NO_MESSAGE;
    priv->res.msg = NULL;
    priv->req.msg = NULL;
    return priv;
}

/* kcs_pci_tbl - PCI Device ID Table */
static struct pci_device_id kcs_pci_tbl[] = {
    {PCI_DEVICE(PCI_VENDOR_ID_INTEL, PCI_KCS_DEVICE_ID)},
	/* required last entry */
	{0,}
};

MODULE_DEVICE_TABLE(pci, kcs_pci_tbl);


/* Local Function Prototypes */
static int __init kcs_init_module(void);
static void __exit kcs_exit_module(void);
static int __devinit kcs_probe(struct pci_dev *pdev, const struct pci_device_id *ent);
static void __devexit kcs_remove(struct pci_dev *pdev);

/* PCI driver struct */
static struct pci_driver kcs_driver = {
	.name     = iamt_driver_name,
	.id_table = kcs_pci_tbl,
	.probe    = kcs_probe,
	.remove   = kcs_remove,
};

/**
 * kcs_open - the open function
 */

static int kcs_open(struct inode *inode, struct file *file)
{
	int if_num = MINOR(inode->i_rdev);
    unsigned long flags;

    spin_lock_bh_irqsave(&device_lock, flags);
    if ((if_num != 0) || !kcs_device) {
        spin_unlock_bh_irqrestore(&device_lock, flags);
        return -ENODEV;
    }
    spin_unlock_bh_irqrestore(&device_lock, flags);
    
    file->private_data = alloc_priv(file);
    if (!file->private_data) {
        return -ENOMEM;
    }
 
	return 0;
}

/**
 * kcs_release - the release function
 */

static int kcs_release(struct inode *inode, struct file *file)
{
    struct iamt_kcs_file_private *priv = file->private_data;
	int if_num = MINOR(inode->i_rdev);
    
    if (if_num != 0) {
        return -ENODEV;
    }
   
    /* remove message */
    remove_message_from_queue(priv);

    file->private_data = NULL;
    if (priv->req.msg) {
        kfree(priv->req.msg);
    }
    if (priv->res.msg) {
        kfree(priv->res.msg);
    }

    kfree(priv);
    return 0;
}

/**
 * kcs_ioctl - the IOCTL function
 */

static int kcs_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long data)
{
    int rv = 0, if_num = MINOR(inode->i_rdev);
    
    struct iamt_kcs_file_private *priv = file->private_data;
    struct iamt_kcs_msg *u_msg =(struct iamt_kcs_msg *)data;  // in user space
    struct iamt_kcs_msg k_msg, req_msg, res_msg; // all in kernel on the stack
    struct iamt_kcs_device *device;
    unsigned long operation, flags;
    
    if (if_num != 0) {
        return -ENODEV;
    }

    switch (cmd) {    
    case IAMT_KCS_SEND_MESSAGE_COMMAND:
        /* we first copy all what we need from the user and allocate everything
        we need before taking any locks */

        /* copy all the struct to kernel space */
        if (copy_from_user(&k_msg, u_msg, sizeof(k_msg))) {
            rv = -EFAULT;
            break;
        }

        if ((k_msg.size > MAX_KCS_MESSAGE_SIZE) || (k_msg.size < sizeof(struct pthi_header))) {
            rv = -EMSGSIZE;
        }

        rv = alloc_message(&req_msg, k_msg.size);
        if (rv) {
            break;
        }
        rv = alloc_message(&res_msg, MAX_KCS_MESSAGE_SIZE);
        if (rv) {
            free_msg(&req_msg);
            break;
        }

        /* copy the message to kernel space - use a pointer already copied into
        kernel space */
        if (copy_from_user(req_msg.msg, k_msg.msg, k_msg.size)) {
			rv = -EFAULT;
            free_msg(&req_msg);
            free_msg(&res_msg);
			break;
		}

        /* WD operations are not allowed from user */
        operation = ((struct pthi_header *)(req_msg.msg))->command.cmd.value;
        if ((operation == START_WATCHDOG_REQUEST_CMD) || 
            (operation == STOP_WATCHDOG_REQUEST_CMD)) {
            rv = -EINVAL;
            free_msg(&req_msg);
            free_msg(&res_msg);
            break;                
        }

        spin_lock_bh_irqsave(&device_lock, flags);
        spin_lock(&priv->file_lock);

        if (priv->state != NO_MESSAGE) {
             rv = -EBUSY;
             spin_unlock(&priv->file_lock);
             spin_unlock_bh_irqrestore(&device_lock, flags);
             free_msg(&req_msg);
             free_msg(&res_msg);
             break;
        }
        
        if (!kcs_device) {
            rv = -ENODEV;
            spin_unlock(&priv->file_lock);
            spin_unlock_bh_irqrestore(&device_lock, flags);
            free_msg(&req_msg);
            free_msg(&res_msg);
            break;
        }
        
        device = pci_get_drvdata(kcs_device);
        
        if (!device->active) {
            rv = -ENODEV;
            spin_unlock(&priv->file_lock);
            spin_unlock_bh_irqrestore(&device_lock, flags);
            free_msg(&req_msg);
            free_msg(&res_msg);
            break;
        }

        memcpy(&priv->req, &req_msg, sizeof(req_msg));
        memcpy(&priv->res, &res_msg, sizeof(res_msg));

        add_message_to_queue(device, priv);
        spin_unlock(&priv->file_lock);
        spin_unlock_bh_irqrestore(&device_lock, flags);
        break;

    case IAMT_KCS_RECEIVE_MESSAGE_COMMAND:

        /* first copy from user all data needed */
        if (copy_from_user(&k_msg, u_msg, sizeof(k_msg))) {
            rv = -EFAULT;
            break;
        }

        spin_lock(&priv->file_lock);
        if (priv->state != MESSAGE_WAITING) {
            rv = -EAGAIN;
            spin_unlock(&priv->file_lock);
            break;
        }

        if (k_msg.size < priv->res.size) {
            rv = -EMSGSIZE;
            spin_unlock(&priv->file_lock);
            break;
        }
        
        /* copy data to the stack so we can release the lock */
        memcpy(&req_msg, &priv->req, sizeof(req_msg));
        memcpy(&res_msg, &priv->res, sizeof(res_msg));
      
        rv = priv->status;

        priv->status = 0;
        priv->state = NO_MESSAGE;

        priv->req.msg = NULL;
        priv->req.size = 0;
        priv->res.msg = NULL;
        priv->res.size = 0;
        
        spin_unlock(&priv->file_lock);
        
        /* now copy the data to user space */
        if (copy_to_user(k_msg.msg, res_msg.msg, res_msg.size)) {
            rv = -EFAULT;
            break;
        }
        if (put_user(res_msg.size, &u_msg->size)) {
            rv = -EFAULT;
            break;
        }

        free_msg(&req_msg);
        free_msg(&res_msg);
    
        break;

    default:
        rv = -EINVAL;
    }        
    return rv;
}

/**
 * kcs_poll - the poll function
 */

static unsigned int kcs_poll(struct file *file, poll_table *wait)
{
	struct iamt_kcs_file_private *priv = file->private_data;
	unsigned int mask = 0;
	    
	spin_lock(&priv->file_lock);

	poll_wait(file, &priv->wait, wait);

	if (priv->state == MESSAGE_WAITING)
		mask |= (POLLIN | POLLRDNORM);

	spin_unlock(&priv->file_lock);

	return mask;
}

/* file operations structure */
static struct file_operations kcs_fops = {
	.owner   = THIS_MODULE,
	.ioctl   = kcs_ioctl,
	.open    = kcs_open,
	.release = kcs_release,
    .poll    = kcs_poll,
};

MODULE_AUTHOR("Intel Corporation");
MODULE_DESCRIPTION("Intel(R) Active Management Technology - KCS");
MODULE_LICENSE("GPL");

/**
 * kcs_init_module - Driver Registration Routine
 *
 * kcs_init_module is the first routine called when the driver is
 * loaded. All it does is register with the PCI subsystem.
 **/

static int __init kcs_init_module(void)
{
	int ret;
	printk(KERN_INFO "%s - version %s\n",
	       iamt_driver_string, iamt_driver_version);

	printk(KERN_INFO "%s\n", iamt_copyright);

    if ((wd_timeout < DEAFULT_WD_TIMER) || (wd_timeout > 0xffff)) {
        printk(KERN_WARNING "iamt: wd_timeout invalid, using default value\n");
        wd_timeout = DEAFULT_WD_TIMER;
    }

	ret = pci_module_init(&kcs_driver);
	if(ret < 0) {
        return ret;
    }
    if ((major = register_chrdev(0, iamt_driver_name, &kcs_fops)) < 0) {
        pci_unregister_driver(&kcs_driver);
        return major;
    }

	return 0;
}

module_init(kcs_init_module);

/**
 * kcs_exit_module - Driver Exit Cleanup Routine
 *
 * kcs_exit_module is called just before the driver is removed
 * from memory.
 **/

static void __exit kcs_exit_module(void)
{
    unregister_chrdev(major, iamt_driver_name);
	pci_unregister_driver(&kcs_driver);
}

module_exit(kcs_exit_module);


/**
 * kcs_probe - Device Initialization Routine
 * @pdev: PCI device information struct
 * @ent: entry in kcs_pci_tbl
 *
 * Returns 0 on success, negative on failure
 **/

static int __devinit kcs_probe(struct pci_dev *pdev,
            const struct pci_device_id *ent)
{
    struct iamt_kcs_device *device = NULL; 
	int i, err = 0;
    unsigned long flags;
	
    if((err = pci_enable_device(pdev))) {
        printk(KERN_ERR "iamt: failed to enable device\n");
		return err;
    }

	if((err = pci_request_regions(pdev, iamt_driver_name))) {
        printk(KERN_ERR "iamt: failed to get regions\n");
        goto disable_device;
    }

    device = init_iamt_kcs_device(pdev);
    if (!device) {
        err = -ENOMEM;
        goto release_regions;
    }

    if (setup_wd(device) != 0) {
        goto free_device;
    }


 	for (i = 0; i <= 5; i++) {
		if (pci_resource_len(pdev, i) == 0)
			continue;
		if (pci_resource_flags(pdev, i) & IORESOURCE_IO) {
            if (device->io_base) {
                printk(KERN_ERR "iamt: ERROR: too many IO ports.\n");
                goto free_wd;

            }
			device->io_base = pci_resource_start(pdev, i);
            device->io_length = pci_resource_len(pdev, i);
		}
        else if (pci_resource_flags(pdev, i) & IORESOURCE_MEM) {
            if (device->mem_base) {
                printk(KERN_ERR "iamt: ERROR: too many mem addresses.\n");
                goto free_wd;

            }
			device->mem_base = pci_resource_start(pdev, i);
            device->mem_length = pci_resource_len(pdev, i);
		}
	}

    if (!device->io_base && !device->mem_base) {
        printk(KERN_ERR "iamt: ERROR: no address to use.\n");
        err = -ENODEV;
        goto free_wd;
    } else if (!device->mem_base) {
        device->use_io = 1;
    } else {
        device->mem_addr = ioremap(device->mem_base, device->mem_length);
        if (!device->mem_addr) {
            if (!device->io_base) {
                printk(KERN_ERR "iamt: ERROR: remap failure.\n");
                err = -ENOMEM;
                goto free_wd;
            }
            device->use_io = 1;
        }
    }

    device->irq = pdev->irq;

    err = request_irq(device->irq, iamt_kcs_intr, SA_SHIRQ, iamt_driver_name, device);
    if (err) {
        printk(KERN_ERR "iamt: ERROR: request_irq failure.\n");
        goto unmap_mem;
    }

    if (enable_interrupts(device)) {
        printk(KERN_ERR "iamt: ERROR: enable interrupts failure.\n");
        goto release_irq;
    }
    device->interrupts_enabled = 1;

    spin_lock_bh_irqsave(&device_lock, flags);
    if (kcs_device) {
        spin_unlock_bh_irqrestore(&device_lock, flags);
        err = -EEXIST;
        goto disable_intr;
    }
    
    kcs_device = pdev;
    pci_set_drvdata(pdev, device);

    spin_unlock_bh_irqrestore(&device_lock, flags);

    send_wd((unsigned long)device);

    return 0;

disable_intr:
    disable_interrupts(device);
release_irq:
    free_irq(pdev->irq, device);    
unmap_mem:
    if (device->mem_addr) {
        iounmap(device->mem_addr);   
    }
free_wd:
    free_wd(device);
free_device:
    kfree(device);
release_regions:
    pci_release_regions(pdev);
disable_device:
    pci_disable_device(pdev);
    
  	return err;
}

/**
 * kcs_remove - Device Removal Routine
 * @pdev: PCI device information struct
 *
 * kcs_remove is called by the PCI subsystem to alert the driver
 * that it should release a PCI device.
 **/

static void __devexit kcs_remove(struct pci_dev *pdev)
{
    struct iamt_kcs_device *device = pci_get_drvdata(pdev);
    struct iamt_kcs_file_private *priv;
    struct list_head *pos, *tmp;
    unsigned long flags;
    
    spin_lock_bh_irqsave(&device_lock, flags);
    if (kcs_device != pdev) {
        spin_unlock_bh_irqrestore(&device_lock, flags);
        return;
    }

    kcs_device = NULL;
    spin_lock(&extra_lock);
    device->active = 0;
    spin_unlock(&extra_lock);
    spin_unlock_bh_irqrestore(&device_lock, flags);

    /* return messages from queue */
    list_for_each_safe(pos, tmp, &device->message_list) {
        priv = list_entry(pos, struct iamt_kcs_file_private, link);
        remove_message_from_queue(device->curr_file);
    }

    del_timer_sync(&device->wd_timer);
    del_timer_sync(&device->timer);
    flush_scheduled_tasks();

    stop_wd(device);
    free_wd(device);
    
    disable_interrupts(device);
    free_irq(pdev->irq, device);
    pci_set_drvdata(pdev, NULL);

    if (device->mem_addr) {          
        iounmap(device->mem_addr);
    }
    kfree(device);
    
    pci_release_regions(pdev);
    pci_disable_device(pdev);
}

