/*******************************************************************************

  
  Copyright(c) 2003 - 2005 Intel Corporation. All rights reserved.
  
  This program is free software; you can redistribute it and/or modify it 
  under the terms of the GNU General Public License as published by the Free 
  Software Foundation; either version 2 of the License, or (at your option) 
  any later version.
  
  This program is distributed in the hope that it will be useful, but WITHOUT 
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
  more details.
  
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 
  Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
  The full GNU General Public License is included in this distribution in the
  file called LICENSE.
  
  Contact Information:
  Linux NICS <linux.nics@intel.com>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497
  
  file: iamt_kcs.c
  part of "Intel(R) Active Management Technology - KCS" Linux driver 

*******************************************************************************/

#include <linux/delay.h>
#include <linux/sched.h>
#include "iamt.h"

/**
 * read_kcs_register - Read a byte from the kcs device
 * @devcie: the device structure
 * @offset: offset from which to read the byte (0-1)
 * Returns the byte read.
 */
 
static inline unsigned char read_kcs_register(struct iamt_kcs_device *device, 
                                              unsigned long offset)
{
    if (device->use_io) {
        return (inb(device->io_base + offset));
    }
    return (*(device->mem_addr + offset));
}

/**
 * write_kcs_register - Write a byte to the kcs device
 * @devcie: the device structure
 * @offset: offset from which to read the byte (0-1)
 * @value: the byte to write
 */

static inline void write_kcs_register(struct iamt_kcs_device *device, 
                                      unsigned long offset, unsigned char value)
{
    if (device->use_io) {
        outb(value, device->io_base + offset);
    }
    else {
        *(device->mem_addr + offset) = value;
    }
}

/**
 * read_kcs_data - Read a data byte from the kcs device
 * @device: the device structure
 * 
 * Returns the data read
 * 
 * NOTE: this function isn't inline since if it is inline the compiler will 
 * optimize out the clear_obf.
 */

static unsigned char read_kcs_data(struct iamt_kcs_device *device)
{
    return read_kcs_register(device, KCS_DATAOUT_REGISTER);
}

/**
 * read_kcs_status - Read the status byte from the kcs device
 * @device: the device structure
 * 
 * Returns the status read
 */

static inline unsigned char read_kcs_status(struct iamt_kcs_device *device)
{
    return read_kcs_register(device, KCS_STATUS_REGISTER);
}

/**
 * write_kcs_data - Write a data byte to the kcs device
 * @device: the device structure
 * @data: the byte to write
 */

static inline void write_kcs_data(struct iamt_kcs_device *device, unsigned char data)
{
    write_kcs_register(device, KCS_DATAIN_REGISTER, data);
}

/**
 * write_kcs_cmd - Write a command to the kcs device
 * @device: the device structure
 * @cmd: the command to write
 */

static inline void write_kcs_cmd(struct iamt_kcs_device *device, unsigned char cmd)
{
    write_kcs_register(device, KCS_COMMAND_REGISTER, cmd);
}

/**
 * OBFset - Check if the OBF bit is set
 * @device: the device structure
 * 
 * Returns 1 if OBF is set, 0 otherwise
 */

static inline int OBFset(struct iamt_kcs_device *device)
{
    unsigned char status = read_kcs_status(device);

    return ((status & KCS_OBF) == KCS_OBF);
}

/**
 * clear_obf - clear the OBF bit
 * @device: the device structure
 */         

static inline void clear_obf(struct iamt_kcs_device *device)
{
    read_kcs_data(device);
}

/**
 * iamt_transaction_state - The state machine of the KCS transaction
 * @device: the device structure
 * 
 * NOTE: This function is called with the device lock taken.
 * NOTE: This function may be called in irq context.
 */

void iamt_transaction_state(struct iamt_kcs_device *device)
{
    unsigned char status;

    switch (device->state)
    {
        /* This case is not called in irq context */
        case TRANSFER_INIT:
            device->byte_count = 0;
            clear_obf(device);
            write_kcs_cmd(device, WRITE_START);
            device->state = TRANSFER;
            break;
            
        case TRANSFER:
            status = read_kcs_status(device);
            clear_obf(device);

            if (((status & KCS_STATE_MASK) != KCS_WRITE_STATE) || 
                ((status & KCS_IBF) != 0)) {
                device->state = TRANSACTION_ERROR;
                break;
            }
 
            /* check for end of Request, transition to the END state which will transfer
               our last UCHAR */
            if (device->byte_count < (device->curr_file->req.size - 1)) {
                write_kcs_data(device, 
                               device->curr_file->req.msg[device->byte_count++]);
            }
            else {
                write_kcs_cmd(device, WRITE_END);
                device->state = TRANSFER_END;
            }
            break;

        case TRANSFER_END:
            status = read_kcs_status(device);
            clear_obf(device);

            if (((status & KCS_STATE_MASK) != KCS_WRITE_STATE) || 
                ((status & KCS_IBF) != 0)) {
                device->state = TRANSACTION_ERROR;
                break;
            }

            write_kcs_data(device, device->curr_file->req.msg[device->byte_count++]);

            device->byte_count = 0;
            device->state = RECEIVE;
            break;
            
        case RECEIVE:
            status = read_kcs_status(device);
            
            switch (status & KCS_STATE_MASK)
            {
                case KCS_IDLE_STATE:
                    clear_obf(device);
                    device->state = MACHINE_END;
                    device->curr_file->res.size = device->byte_count; 
                    break;

                case KCS_READ_STATE:
                    /* check for overrun */
                    if (device->byte_count >= device->curr_file->res.size)
                    {
                        clear_obf(device);
                        device->state = TRANSACTION_ERROR;
                        break;
                    }
                    device->curr_file->res.msg[device->byte_count++] = read_kcs_data(device);                 
                    write_kcs_data(device, KCS_READ);
                    break;

                default:
                    clear_obf(device);
                    device->state = TRANSACTION_ERROR;
                    break;
            }

            break;

        case MACHINE_END:        
        case TRANSACTION_ERROR:
            clear_obf(device);
            break;

        default:
            clear_obf(device);
            device->state = TRANSACTION_ERROR;
            break;
    }    
}

/**
 * transaction_end - function called after the end of a transaction.
 * @data: pointer to the device structure
 * 
 * NOTE: This function is called by a task (by the KEVENTD process)
 */

void transaction_end(void *data) 
{
    struct iamt_kcs_device *device = (struct iamt_kcs_device *)data;
    struct iamt_kcs_file_private *priv;
    int timed_out;
    unsigned long flags;


    spin_lock_bh_irqsave(&device_lock, flags);

    /* We don't check the active parameter since this function has to work 
       correctly also during removal of driver */
    timed_out = device->timeout_exceeded;

    if (device->state != MACHINE_END && device->state != TRANSACTION_ERROR && 
        !device->timeout_exceeded) {
        printk(KERN_ERR "iamt: transaction end without being completed\n");
    }

    device->timeout_exceeded = 1;
    spin_unlock_bh_irqrestore(&device_lock, flags);

    /* In case that this is after a time out it may have been caused by a FW reset so 
       we may need to reenable interupts. */
    if (timed_out) {
        spin_lock(&extra_lock);
        if (device->active) {        
            if (device->interrupts_enabled && 
                ((read_kcs_status(device) & KCS_CIMV) != KCS_CIMV)) {
                enable_interrupts(device);
            }
        }
        spin_unlock(&extra_lock);
    }

    spin_lock_bh_irqsave(&device_lock, flags); 
    
    priv = device->curr_file;
    if (priv == NULL) {
        spin_unlock_bh_irqrestore(&device_lock, flags);   
        return;
    }

    spin_lock(&priv->file_lock);

    if (!timed_out) {
        if (device->state == TRANSACTION_ERROR) {
            printk(KERN_WARNING "iamt: Transaction error\n");
            priv->status = -EIO;
            priv->res.size = 0;
        }
        else {
            priv->status = 0;
        }
    }
    else {
        printk(KERN_WARNING "iamt: Transaction timeout\n");
        priv->status = -ETIMEDOUT;
        priv->res.size = 0;
    }

    priv->state = MESSAGE_WAITING;
    wake_up_interruptible(&priv->wait);
    list_del(&priv->link);
    device->curr_file = NULL;
    device->state = MACHINE_END;


    if (!device->active) {
        spin_unlock(&priv->file_lock);
        spin_unlock_bh_irqrestore(&device_lock, flags);
        return;
    }

    if ((priv->file == NULL)) { // WD command run timer again
        device->wd_timer.function = send_wd;
        device->wd_timer.data = (unsigned long)device;
        device->wd_timer.expires = jiffies + 
            ((wd_timeout - device->message_timeout) * HZ);
        add_timer(&device->wd_timer);        
    }
    spin_unlock(&priv->file_lock);

    if (device->wd_pending) {
        list_add(&device->wd_file->link, &device->message_list);
        device->wd_pending = 0;
    }

    if (!list_empty(&device->message_list)) {
        device->curr_file = list_entry(device->message_list.next, 
                                       struct iamt_kcs_file_private, link);
        spin_lock(&device->curr_file->file_lock);
        iamt_start_transaction(device);
        spin_unlock(&device->curr_file->file_lock);
    }

    spin_unlock_bh_irqrestore(&device_lock, flags);
}

/**
 * timer_func - timeout function for the KCS device.
 * @data: pointer to the device structure
 * 
 * NOTE: This function is called by a timer
 */

void timer_func(unsigned long data) 
{
    struct iamt_kcs_device *device = (struct iamt_kcs_device *)data;
    unsigned long flags;

    spin_lock_irqsave(&device_lock, flags);

    if (device->timeout_exceeded || (device->state == MACHINE_END) || 
        (device->state == TRANSACTION_ERROR)) {
        spin_unlock_irqrestore(&device_lock, flags);
        return;
    }
    device->timeout_exceeded = 1;
    
    // schedule task
    device->task.routine = transaction_end;
    device->task.data = device;

    if (!schedule_task(&device->task)) {
        printk(KERN_ERR "iamt: schedule of task failed.\n");
    }
    spin_unlock_irqrestore(&device_lock, flags);
}

/**
 * iamt_start_transaction - Transaction Initialization Routine
 * @device: device structure
 *
 * NOTE: this function should be called with device->curr_file->file_lock
 * and device_lock taken.
 **/ 

int iamt_start_transaction(struct iamt_kcs_device *device)
{
    int rv = 0;

    device->timeout_exceeded = 0; 

    device->timer.expires = jiffies + (device->message_timeout * HZ);
    device->timer.function = timer_func;
    device->timer.data = (unsigned long)device;

    add_timer(&device->timer);

    device->state = TRANSFER_INIT;

    device->curr_file->state = MESSAGE_TRANSACTION;

    iamt_transaction_state(device);
 
    return rv;
}

/**
 * iamt_kcs_intr - The ISR of the KCS device
 * @irq: The irq number 
 * @dev_id: pointer to the device structure
 * @regs: the register values
 **/ 

void iamt_kcs_intr(int irq, void* dev_id, struct pt_regs *regs)
{
    struct iamt_kcs_device *device = (struct iamt_kcs_device *)dev_id;

    if (OBFset(device)) {
        spin_lock(&device_lock);
        if (device->timeout_exceeded) {
            device->interrupt_flag = 1;
            clear_obf(device);
            spin_unlock(&device_lock);
            return;
        }
        iamt_transaction_state(device);
        if ((device->state == MACHINE_END) || 
            (device->state == TRANSACTION_ERROR)) {
            del_timer(&device->timer);
            
            // schedule task
            device->task.routine = transaction_end;
            device->task.data = device;
            if (!schedule_task(&device->task)) {
                printk(KERN_ERR "iamt: schedule of task failed.\n");
            }           
        }
        spin_unlock(&device_lock);
    }
}

/**
 * change_interrupt_state - Sets the interrupt state
 * @device: The device structure
 * @state: The new intrrupt state
 * 
 * Returns 0 on success, negetive value on failure.
 **/ 

int change_interrupt_state(struct iamt_kcs_device *device, unsigned char state)
{
    unsigned char status, retry = 1;

    device->interrupt_flag = 0;
    
    do
    {
        write_kcs_cmd(device , state);

        mdelay(OBF_SET_TIMEOUT);

        status = read_kcs_status(device);

        if (!device->interrupt_flag) {

            if ((status & KCS_OBF) != KCS_OBF)
            {
                printk(KERN_DEBUG "iamt: OBF not set.\n");
                return -1;
            }
            if ((status & KCS_IBF) != 0) {
                printk(KERN_ERR "iamt: IBF is set.\n");
            }
            clear_obf(device);
        }
        if (((status & KCS_STATE_MASK) != KCS_IDLE_STATE) && (retry == IDLE_STATE_RETRY))
        {
            printk(KERN_DEBUG "iamt: Not IDLE state.\n");
            return -1;
        }
        ++retry;
    } while ((status & KCS_STATE_MASK) != KCS_IDLE_STATE);
    return 0;
}

/**
 * enable_interrupts - Sets the interrupt state to enabled
 * @device: The device structure
 * 
 * Returns 0 on success, negetive value on failure.
 **/ 

int enable_interrupts(struct iamt_kcs_device *device)
{
    int status = change_interrupt_state(device, ENABLE_HOST_INT);

    if((status == 0) && ((read_kcs_status(device) & KCS_CIMV) != KCS_CIMV)) {
        printk(KERN_ERR "iamt: CIMV bit is not on.\n");
    }
    return status;
}

/**
 * disable_interrupts - Sets the interrupt state to disabled
 * @device: The device structure
 * 
 * Returns 0 on success, negetive value on failure.
 **/ 

int disable_interrupts(struct iamt_kcs_device *device)
{
    return change_interrupt_state(device, DISABLE_HOST_INT);        
}

/**
 * init_iamt_kcs_device - allocates and initializes the kcs device structure
 * @pdev: The pci device structure
 * 
 * Returns the iamt_kcs_device pointer on success, NULL on failure.
 **/ 

struct iamt_kcs_device *init_iamt_kcs_device(struct pci_dev *pdev)
{
    struct iamt_kcs_device *device = kmalloc(sizeof(struct iamt_kcs_device), GFP_KERNEL);
    if (device) {
        device->pdev = pdev;
        INIT_LIST_HEAD(&device->message_list);
        init_timer(&device->timer);
        init_timer(&device->wd_timer);
        device->curr_file = NULL;
        device->state = MACHINE_END;
        device->io_base = 0;
        device->io_length = 0;
        device->mem_base = 0;
        device->mem_length = 0;
        device->mem_addr = NULL;
        device->use_io = 0;
        device->irq = 0;
        device->timeout_exceeded = 1;

        device->message_timeout = TRANSACTION_TIMEOUT;

        device->interrupts_enabled = 0;
        device->byte_count = 0;

        device->task.sync = 0;
        device->wd_pending = 0;
        device->active = 1;

    }
    return device;
}

/**
 * add_message_to_queue - adds a message to the kcs device queue
 * @priv: file private structure with info on message
 * 
 * Note: This function should be called with device_lock and priv->file_lock 
 * taken.
 */

void add_message_to_queue(struct iamt_kcs_device *device, 
                          struct iamt_kcs_file_private *priv) 
{
    int first_item = 0;

    priv->state = MESSAGE_RECEIVED;

    first_item = list_empty(&device->message_list);

    list_add_tail(&priv->link, &device->message_list);   

    if (first_item) {
        device->curr_file = priv;
        iamt_start_transaction(device);
    }
}

/**
 * remove_message_from_queue - removes a message to the kcs device queue
 *  If the message is currentlly in the middle of a transaction it will be aborted.
 * @priv: file private structure with info on message
 */

void remove_message_from_queue(struct iamt_kcs_file_private *priv) 
{
    struct iamt_kcs_device *device;
    unsigned long flags;
    int deleted = 0;

    spin_lock_bh_irqsave(&device_lock,flags);
    
    if (!kcs_device) {
        spin_unlock_bh_irqrestore(&device_lock, flags);
        return;
    }

    device = pci_get_drvdata(kcs_device);
    
    spin_lock(&priv->file_lock);

    switch (priv->state) {
    case NO_MESSAGE:
        break;
    case MESSAGE_RECEIVED:
        list_del(&priv->link);
        break;
    case MESSAGE_TRANSACTION:
        write_kcs_cmd(device, GET_STATUS_ABORT);
        list_del(&priv->link);
        del_timer(&device->timer);
        deleted = 1;
        break;
    case MESSAGE_WAITING:
        /* nothing to do message will be freed.*/
        break;
    }

    if (deleted) {
        device->timeout_exceeded = 1;
        device->curr_file = NULL;
        device->state = MACHINE_END;
    }

    priv->state = MESSAGE_WAITING;
    priv->res.size = 0;
    priv->status = -ENODEV;

    spin_unlock(&priv->file_lock);
    spin_unlock_bh_irqrestore(&device_lock,flags);

    if (!deleted) {
        return;
    }
    
    /* wait for all interrupts and tasks to clear */
    mdelay(OBF_SET_TIMEOUT); 
    flush_scheduled_tasks();

    spin_lock_bh_irqsave(&device_lock, flags);

    if (!kcs_device || !device->active) {
        spin_unlock_bh_irqrestore(&device_lock,flags);
        return;
    }

    /* if a task was available it may have run a diffrent transaction */
    if (device->curr_file) {
        spin_unlock_bh_irqrestore(&device_lock,flags);
        return;
    }
    
    if (device->wd_pending) {
        list_add(&device->wd_file->link, &device->message_list);
        device->wd_pending = 0;
    }

    if (!list_empty(&device->message_list)) {
        device->curr_file = list_entry(device->message_list.next, 
                                       struct iamt_kcs_file_private, link);
        spin_lock(&device->curr_file->file_lock);
        iamt_start_transaction(device);
        spin_unlock(&device->curr_file->file_lock);
    }
    
    spin_unlock_bh_irqrestore(&device_lock, flags);
    return;
}

