/*******************************************************************************

  
  Copyright(c) 2003 - 2005 Intel Corporation. All rights reserved.
  
  This program is free software; you can redistribute it and/or modify it 
  under the terms of the GNU General Public License as published by the Free 
  Software Foundation; either version 2 of the License, or (at your option) 
  any later version.
  
  This program is distributed in the hope that it will be useful, but WITHOUT 
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
  more details.
  
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 
  Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
  The full GNU General Public License is included in this distribution in the
  file called LICENSE.
  
  Contact Information:
  Linux NICS <linux.nics@intel.com>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497
  
  file: iamt.h
  part of "Intel(R) Active Management Technology - KCS" Linux driver 
*******************************************************************************/

#ifndef _IAMT_H_
#define _IAMT_H_

#include <linux/spinlock.h>
#include <linux/list.h>
#include <linux/pci.h>
#include <linux/tqueue.h>
#include <linux/timer.h>

#define spin_lock_bh_irqsave(lock, flags)   \
    do { local_bh_disable(); local_irq_save(flags); spin_lock(lock); } while (0)

#define spin_unlock_bh_irqrestore(lock, flags)   \
    do { spin_unlock(lock); local_irq_restore(flags); local_bh_enable(); } while (0)


#ifndef PCI_DEVICE
#define PCI_DEVICE(vend,dev) \
	.vendor = (vend), .device = (dev), \
	.subvendor = PCI_ANY_ID, .subdevice = PCI_ANY_ID
#endif


#define PCI_KCS_DEVICE_ID 0x108E

/* IOCTL commands */
#define IAMT_IOC_MAGIC 'i'
#define IAMT_KCS_SEND_MESSAGE_COMMAND _IOR(IAMT_IOC_MAGIC, 1, struct iamt_kcs_msg)
#define IAMT_KCS_RECEIVE_MESSAGE_COMMAND _IOW(IAMT_IOC_MAGIC, 2, struct iamt_kcs_msg) 

/* transaction timeout */
#define TRANSACTION_TIMEOUT         10
#define OBF_SET_TIMEOUT             10              //10 miliseconds
#define IDLE_STATE_RETRY            2

/* Keyboard Controller Style Interface addresses and defines */
#define KCS_COMMAND_REGISTER    1 //(KCS_BASE_ADDRESS + 1)
#define KCS_STATUS_REGISTER     1 //(KCS_BASE_ADDRESS + 1)
#define KCS_DATAIN_REGISTER     0 //KCS_BASE_ADDRESS
#define KCS_DATAOUT_REGISTER    0 //KCS_BASE_ADDRESS

/* Status register bit flags */
#define KCS_OBF             0x01
#define KCS_IBF             0x02
#define KCS_CD              0x04
#define KCS_CIMV            0x10
#define KCS_STATE_MASK      0xc0

/* KCS state */
#define KCS_IDLE_STATE      0x00
#define KCS_READ_STATE      0x40
#define KCS_WRITE_STATE     0x80
#define KCS_ERROR_STATE     0xc0

/* KCS commands */
#define GET_STATUS_ABORT    0x60
#define WRITE_START         0x61
#define WRITE_END           0x62
#define ENABLE_HOST_INT     0x63
#define DISABLE_HOST_INT    0x64
#define KCS_READ            0x68

/* Deafault WD timer - 40 seconds */
#define DEAFULT_WD_TIMER (4 * TRANSACTION_TIMEOUT) 

/* Transactinon state machine */
enum kcs_machine_state
{
    MACHINE_END,
    TRANSFER_INIT,
    TRANSFER,
    TRANSFER_END,
    RECEIVE,
    TRANSACTION_ERROR
}; 

/* File state */
enum file_state 
{
    NO_MESSAGE,
    MESSAGE_RECEIVED,
    MESSAGE_TRANSACTION,
    MESSAGE_WAITING
};

/* KCS message struct */
struct iamt_kcs_msg 
{
	uint32_t size;
	char* msg;
};

/* Private file struct */
struct iamt_kcs_file_private 
{
    struct list_head link;
    struct file *file;
    enum file_state state;
    wait_queue_head_t wait;
    struct iamt_kcs_msg req;
    struct iamt_kcs_msg res;
    spinlock_t file_lock;
    int status;
};

/* Private device struct */
struct iamt_kcs_device 
{
    struct pci_dev *pdev;
    struct list_head message_list;
    struct iamt_kcs_file_private *curr_file;
    struct iamt_kcs_file_private *wd_file;
    enum kcs_machine_state state;
    unsigned int io_base;
    unsigned int io_length;
    unsigned int mem_base;
    unsigned int mem_length;
    char *mem_addr;
    int use_io;
    int irq;

    int message_timeout;
    int timeout_exceeded;

    struct timer_list timer;
    struct timer_list wd_timer;

    int wd_pending;

    int interrupts_enabled;
    int interrupt_flag;
    unsigned short byte_count;   

    struct tq_struct task;

    int active;
};

/* WD struct definitions */
#define HOST_IF_MAJOR 1
#define HOST_IF_MINOR 1

struct pthi_version
{
    unsigned char major;
    unsigned char minor;
};

struct command
{
    union {
        unsigned long value;
        struct {
            unsigned long operation : 23;
            unsigned long response  : 1;
            unsigned long cmd_class : 8;
        } fields;
    } cmd;
};

struct pthi_header
{
    struct pthi_version version;
    unsigned short reserved;
    struct command command;
    unsigned long length;
};

#define START_WATCHDOG_REQUEST_CMD 0x5000000
struct wd_start_request
{
    struct pthi_header header;
    unsigned short timeout;
};

#define START_WATCHDOG_RESPONSE_CMD 0x5800000
struct wd_start_response
{
    struct pthi_header header;
    unsigned long status;
};

#define STOP_WATCHDOG_REQUEST_CMD 0x5000001
struct wd_stop_request
{
    struct pthi_header header;
};

#define STOP_WATCHDOG_RESPONSE_CMD 0x5800001
struct wd_stop_response
{
    struct pthi_header header;
    unsigned long status;
};

extern struct pci_dev *kcs_device;
extern spinlock_t device_lock;
extern spinlock_t extra_lock;
extern unsigned short wd_timeout;

/* general functions */
void remove_message_from_queue(struct iamt_kcs_file_private *priv);
void add_message_to_queue(struct iamt_kcs_device *device, 
                          struct iamt_kcs_file_private *priv);
struct iamt_kcs_device *init_iamt_kcs_device(struct pci_dev *pdev);
int iamt_start_transaction(struct iamt_kcs_device  *device);
void iamt_kcs_intr(int irq, void* dev_id, struct pt_regs *regs);
int enable_interrupts(struct iamt_kcs_device *device);
int disable_interrupts(struct iamt_kcs_device *device);
int alloc_message(struct iamt_kcs_msg *msg, unsigned int size);
struct iamt_kcs_file_private *alloc_priv(struct file *file);
void free_msg(struct iamt_kcs_msg *msg);

/* WD functions */
void send_wd(unsigned long data);
int setup_wd(struct iamt_kcs_device *device);
void stop_wd(struct iamt_kcs_device *device);
void free_wd(struct iamt_kcs_device *device);

#endif /* _IAMT_H_ */
