/*
 *   ALSA driver for AZALIA controllers
 *
 *	Copyright (c) 2004 PeiSen Hou <pshou@realtek.com.tw>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#include <sound/driver.h>
#include <asm/io.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/slab.h>
#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/info.h>
#include <sound/sndmagic.h>
//#include <sound/memalloc.h>
#include <sound/azalia_codec.h>	
#define SNDRV_GET_ID
#include <sound/initval.h>

#include "azx_patch.h"

MODULE_AUTHOR("pshou <pshou@realtek.com.tw>");
MODULE_DESCRIPTION("AZALIA controller");
MODULE_LICENSE("GPL");
MODULE_CLASSES("{sound}");
MODULE_DEVICES("{{AZALIA}}");

static int index[SNDRV_CARDS] = SNDRV_DEFAULT_IDX;	/* Index 0-MAX */
static char *id[SNDRV_CARDS] = SNDRV_DEFAULT_STR;	/* ID for this card */
static int enable[SNDRV_CARDS] = SNDRV_DEFAULT_ENABLE_PNP;	/* Enable this card */
static int azx_clock[SNDRV_CARDS] = {[0 ... (SNDRV_CARDS - 1)] = 48000};

MODULE_PARM(index, "1-" __MODULE_STRING(SNDRV_CARDS) "i");
MODULE_PARM_DESC(index, "Index value for AZALIA controller.");
MODULE_PARM_SYNTAX(index, SNDRV_INDEX_DESC);
MODULE_PARM(id, "1-" __MODULE_STRING(SNDRV_CARDS) "s");
MODULE_PARM_DESC(id, "ID string for AZALIA controller.");
MODULE_PARM_SYNTAX(id, SNDRV_ID_DESC);
MODULE_PARM(enable, "1-" __MODULE_STRING(SNDRV_CARDS) "i");
MODULE_PARM_DESC(enable, "Enable audio part of AZALIA controller.");
MODULE_PARM_SYNTAX(enable, SNDRV_ENABLE_DESC);
MODULE_PARM(azx_clock, "1-" __MODULE_STRING(SNDRV_CARDS) "i");
MODULE_PARM_DESC(azx_clock, "Azalia codec clock (default 48000Hz).");
MODULE_PARM_SYNTAX(azx_clock, SNDRV_ENABLED ",default:48000");


/*
 */
#define ICH6_REG_GCAP			0
#define ICH6_REG_VMIN			2
#define ICH6_REG_VMAJ			3
#define ICH6_REG_OUTPAY			4
#define ICH6_REG_INPAY			6
#define ICH6_REG_GCTL			8
#define ICH6_REG_WAKEEN			0x0c
#define ICH6_REG_STATESTS		0x0e
#define ICH6_REG_GSTS			0x10
#define ICH6_REG_INTCTL			0x20
#define ICH6_REG_INTSTS			0x24
#define ICH6_REG_WALCLK			0x30
#define ICH6_REG_SYNC			0x34	
#define ICH6_REG_CORBLBASE		0x40
#define ICH6_REG_CORBUBASE		0x44
#define ICH6_REG_CORBWP			0x48
#define ICH6_REG_CORBRP			0x4A
#define ICH6_REG_CORBCTL		0x4c
#define ICH6_REG_CORBSTS		0x4d
#define ICH6_REG_CORBSIZE		0x4e

#define ICH6_REG_RIRBLBASE		0x50
#define ICH6_REG_RIRBUBASE		0x54
#define ICH6_REG_RIRBWP			0x58
#define ICH6_REG_RINTCNT		0x5a
#define ICH6_REG_RIRBCTL		0x5c
#define ICH6_REG_RIRBSTS		0x5d
#define ICH6_REG_RIRBSIZE		0x5e

#define ICH6_REG_IC			0x60
#define ICH6_REG_IR			0x64
#define ICH6_REG_IRS			0x68
#define   ICH6_IRS_VALID	(1<<1)
#define   ICH6_IRS_BUSY		(1<<0)

#define ICH6_REG_DPLBASE		0x70
#define ICH6_REG_DPUBASE		0x74
#define   ICH6_DPLBASE_ENABLE	0x1	/* Enable position buffer */

//	stream discription registers
#define ICH6_REG_ISDCTL			0x80
#define ICH6_REG_ISDSTS			0x83
#define ICH6_REG_ISDLPIB		0x84
#define ICH6_REG_ISDCBL			0x88
#define ICH6_REG_ISDLVI			0x8c
#define ICH6_REG_ISDFIFOW		0x8e
#define ICH6_REG_ISDFIFOS		0x90
#define ICH6_REG_ISDFMT			0x92
#define ICH6_REG_ISDBDPL		0x98
#define ICH6_REG_ISDBDPU		0x9c

#define ICH6_REG_OSDCTL			0x100
#define ICH6_REG_OSDSTS			0x103
#define ICH6_REG_OSDLPIB		0x104
#define ICH6_REG_OSDCBL			0x108
#define ICH6_REG_OSDLVI			0x10c
#define ICH6_REG_OSDFIFOW		0x10e
#define ICH6_REG_OSDFIFOS		0x110
#define ICH6_REG_OSDFMT			0x112
#define ICH6_REG_OSDBDPL		0x118
#define ICH6_REG_OSDBDPU		0x11c

#define ICH6_DMA_REGSIZE		0x20

// stream register offsets from stream base
#define ICH6_REG_SD_CTL			0x0
#define ICH6_REG_SD_STS			0x3
#define ICH6_REG_SD_LPIB		0x4
#define ICH6_REG_SD_CBL			0x8
#define ICH6_REG_SD_LVI			0xC
#define ICH6_REG_SD_FIFOSIZE		0x10
#define ICH6_REG_SD_FORMAT		0x12
#define ICH6_REG_SD_BDLPL		0x18
#define ICH6_REG_SD_BDLPU		0x1C
#define ICH6_REG_SD_LPIBA		0x2004

#define GCTL_CONTROLLER_RESET_BIT_MASK 1

/* PCI space */
#define ICH6_PCIREG_TCSEL	0x44

#define ICH6_MEM_REGION		1024*16 //384	/* i/o memory size */

// for pcm support
#define get_azx_dev(substream) (azx_dev_t*)(substream->runtime->private_data)
#define AZX_START_DMA 0x02

/*
 * Use CORB/RIRB for communication from/to codecs.
 * This is the way recommended by Intel (see below).
 */
//#define USE_CORB_RIRB

enum { SDI0, SDI1, SDI2, SDI3, SDO0, SDO1, SDO2, SDO3 };
/*
 */

static int snd_azx_codec_info_volume(snd_kcontrol_t * kcontrol, snd_ctl_elem_info_t * uinfo);
static int snd_azx_codec_get_volume(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol);
static int snd_azx_codec_put_volume(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol);

int snd_azx_codec_info_mute(snd_kcontrol_t* kcontrol, snd_ctl_elem_info_t* uinfo);
int snd_azx_codec_mute_get(snd_kcontrol_t* kcontrol, snd_ctl_elem_value_t* ucontrol);
int snd_azx_codec_mute_put(snd_kcontrol_t* kcontrol, snd_ctl_elem_value_t* ucontrol);

int snd_azx_info_enum(snd_kcontrol_t* kcontrol, snd_ctl_elem_info_t* uinfo);

// 2, 6 channel select
static int snd_azx_mux_info(snd_kcontrol_t* kcontrol, snd_ctl_elem_info_t* uinfo);
static int snd_azx_mux_get(snd_kcontrol_t* kcontrol, snd_ctl_elem_value_t* ucontrol);
static int snd_azx_mux_put(snd_kcontrol_t* kcontrol, snd_ctl_elem_value_t* ucontrol);

// 6, 8 channel select
static int snd_azx_mux1_info(snd_kcontrol_t* kcontrol, snd_ctl_elem_info_t* uinfo);
static int snd_azx_mux1_get(snd_kcontrol_t* kcontrol, snd_ctl_elem_value_t* ucontrol);
static int snd_azx_mux1_put(snd_kcontrol_t* kcontrol, snd_ctl_elem_value_t* ucontrol);

static int snd_azx_mux_digital_info(snd_kcontrol_t* kcontrol, snd_ctl_elem_info_t* uinfo);
static int snd_azx_mux_digital_get(snd_kcontrol_t* kcontrol, snd_ctl_elem_value_t* ucontrol);
static int snd_azx_mux_digital_put(snd_kcontrol_t* kcontrol, snd_ctl_elem_value_t* ucontrol);
/*
 */
static struct pci_device_id snd_azx_ids[] = {
	{ 0x8086, 0x2668, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 },
	{ 0x8086, 0x27d8, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 },
	{ 0x10b9, 0x5461, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 }, 
};

MODULE_DEVICE_TABLE(pci, snd_azx_ids);

/*
	Mixer define
 */

typedef struct {
	unsigned int id;
	unsigned int mask;
	const char *name;
	void (*patch)(azx_codec_t *azx_codec);
} azx_codec_id_t;

static const azx_codec_id_t snd_azx_codec_id_vendors[] = {
{ 0x10ec0000, 0xffff0000, "Realtek", NULL},
};

static const azx_codec_id_t snd_azx_codec_ids[] = {
{ 0x10ec0260, 0xffffffff, "ALC260", patch_alc260},
{ 0x10ec0880, 0xffffffff, "ALC880", patch_alc880},
{ 0x10ec0860, 0xffffffff, "ALC861VS", patch_alc880},
};

struct codec_driver codec_callback_info;

typedef struct {
	unsigned short 		sub_vendor;
	unsigned short		sub_device;
	unsigned short		board_config;	
} azx_codec_mixer_t;
	
static const azx_codec_mixer_t alc_mixer_index[] = {
// subsysytem_vendor, subsysytem_device, index >> 4 | max_channels (0 = 2, 1= 6, 2=8, 4=hp)
// Back 3 jack, front 2 jack
{ 0x8086, 0xe200, 0x0004}, { 0x8086, 0xe201, 0x0004}, { 0x8086, 0xe202, 0x0004}, { 0x8086, 0xe203, 0x0004}, 
{ 0x8086, 0xe204, 0x0004}, { 0x8086, 0xe205, 0x0004}, { 0x8086, 0xe206, 0x0004}, { 0x8086, 0xe207, 0x0004}, 
{ 0x8086, 0xe208, 0x0004}, { 0x8086, 0xe209, 0x0004}, { 0x8086, 0xe20a, 0x0004}, { 0x8086, 0xe20b, 0x0004}, 
{ 0x8086, 0xe20c, 0x0004}, { 0x8086, 0xe20d, 0x0004}, { 0x8086, 0xe20e, 0x0004}, { 0x8086, 0xe20f, 0x0004}, 
{ 0x8086, 0xe210, 0x0004}, { 0x8086, 0xe211, 0x0004}, { 0x8086, 0xe214, 0x0004}, { 0x8086, 0xe302, 0x0004},
{ 0x8086, 0xe303, 0x0004}, { 0x8086, 0xe304, 0x0004}, { 0x8086, 0xe306, 0x0004}, { 0x8086, 0xe307, 0x0004},
{ 0x8086, 0xe404, 0x0004}, { 0x8086, 0xa101, 0x0004}, { 0x107b, 0x3031, 0x0004}, { 0x107b, 0x4036, 0x0004},   	
{ 0x107b, 0x4037, 0x0004}, { 0x107b, 0x4038, 0x0004}, { 0x107b, 0x4040, 0x0004}, { 0x107b, 0x4041, 0x0004}, { 0x104d, 0x81d6, 0x0004}, 


// Back 3 jack, front 2 jack (Internal add Aux-In)
{ 0x1025, 0xe310, 0x0004},

// Back 3 jack plus 1 SPDIF out jack, front 2 jack
{ 0x8086, 0xe308, 0x0014},

// Back 3 jack plus 1 SPDIF out jack, front 2 jack (Internal add Aux-In)
{ 0x8086, 0xe305, 0x0014}, { 0x8086, 0xd402, 0x0014}, { 0x1025, 0xe309, 0x0014},

// Back 5 jack, front 2 jack
 { 0x107b, 0x3033, 0x0025}, { 0x107b, 0x4039, 0x0025},{ 0x107b, 0x3032, 0x0025},

// Back 5 jack plus 1 SPDIF out jack, front 2 jack
{ 0x8086, 0xe224, 0x0035}, { 0x8086, 0xe400, 0x0035}, { 0x8086, 0xe401, 0x0035}, { 0x8086, 0xe402, 0x0035},
{ 0x8086, 0xd400, 0x0035}, { 0x8086, 0xd401, 0x0035}, { 0x8086, 0xa100, 0x0035},{ 0x1565, 0x8202, 0x0035},
{0x1019, 0xa880, 0x0035}, {0x1695, 0x400d, 0x0035}, 
{ 0x0000, 0x8086, 0x0035},

// alc 861
{0x1043, 0x814e, 0x1025},

// ULi
{ 0x10b9, 0x5461, 0x0045}, 

// Cyberlink
{0x161f, 0x203d, 0x0054}, 

// Fuji
{0x1734, 0x107c, 0x0064},

//alc 260 for HP
{0x103c, 0x3010, 0x0010}, {0x103c, 0x3011, 0x0010}, {0x103c, 0x3012, 0x0010}, {0x103c, 0x3013, 0x0010},
{0x103c, 0x3014, 0x0010}, {0x103c, 0x3015, 0x0010}, {0x103c, 0x3016, 0x0010},

{0, 0, 0},
}; 

static mixer_t mixer_values = {
	.current_mux_select_val = 0,
	.current_mux_digital_select_val = 0,
	.channel_setting = 2,
};

static struct snd_dma_device dev_dma;

static u32 stat_board_config;

/*
 * macros for easy use
 */
#define azx_writel(chip,reg,value) \
	writel(value, chip->remap_addr + ICH6_REG_##reg)
#define azx_readl(chip,reg) \
	readl(chip->remap_addr + ICH6_REG_##reg)
#define azx_writew(chip,reg,value) \
	writew(value, chip->remap_addr + ICH6_REG_##reg)
#define azx_readw(chip,reg) \
	readw(chip->remap_addr + ICH6_REG_##reg)
#define azx_writeb(chip,reg,value) \
	writeb(value, chip->remap_addr + ICH6_REG_##reg)
#define azx_readb(chip,reg) \
	readb(chip->remap_addr + ICH6_REG_##reg)

#define azx_dma_writel(chip, reg, index, value) \
	writel(value, chip->remap_addr + (ICH6_REG_##reg + ICH6_DMA_REGSIZE * index))
#define azx_dma_readl(chip, reg, index) \
	readl(chip->remap_addr + (ICH6_REG_##reg + ICH6_DMA_REGSIZE * index))
#define azx_dma_writew(chip,reg,index,value) \
	writew(value, chip->remap_addr + (ICH6_REG_##reg + ICH6_DMA_REGSIZE * index))
#define azx_dma_readw(chip,reg,index) \
	readw(chip->remap_addr + (ICH6_REG_##reg + ICH6_DMA_REGSIZE * index))
#define azx_dma_writeb(chip,reg,index,value) \
	writeb(value, chip->remap_addr + (ICH6_REG_##reg + ICH6_DMA_REGSIZE * index))
#define azx_dma_readb(chip,reg,index) \
	readb(chip->remap_addr + (ICH6_REG_##reg + ICH6_DMA_REGSIZE * index))

#define azx_sd_writel(dev,reg,value) \
	writel(value, (dev)->sd_addr + ICH6_REG_##reg)
#define azx_sd_readl(dev,reg) \
	readl((dev)->sd_addr + ICH6_REG_##reg)
#define azx_sd_writew(dev,reg,value) \
	writew(value, (dev)->sd_addr + ICH6_REG_##reg)
#define azx_sd_readw(dev,reg) \
	readw((dev)->sd_addr + ICH6_REG_##reg)
#define azx_sd_writeb(dev,reg,value) \
	writeb(value, (dev)->sd_addr + ICH6_REG_##reg)
#define azx_sd_readb(dev,reg) \
	readb((dev)->sd_addr + ICH6_REG_##reg)

/* Get the upper 32bit of the given dma_addr_t
 * Compiler should optimize and eliminate the code if dma_addr_t is 32bit
 */
#define upper_32bit(addr) (sizeof(addr) > 4 ? (u32)((addr) >> 32) : (u32)0)


/* delay for one tick */
#define do_delay() do { \
	set_current_state(TASK_UNINTERRUPTIBLE); \
	schedule_timeout(1); \
} while (0)

#ifdef USE_CORB_RIRB
/*
 * CORB / RIRB interface
 */
static int azx_alloc_cmd_io(azx_t *chip)
{
	int err;

	/* single page (at least 4096 bytes) must suffice for both ringbuffes */
	err = snd_dma_alloc_pages(&dev_dma, PAGE_SIZE, &chip->rb);
	if (err < 0) {
		snd_printk("cannot allocate CORB/RIRB\n");
		return err;
	}
	return 0;
}

static void azx_init_cmd_io(azx_t *chip)
{
	/* CORB set up */
	chip->corb.addr = chip->rb.addr;
	chip->corb.buf = (u32 *)chip->rb.area;
	azx_writel(chip, CORBLBASE, (u32)chip->corb.addr);
	azx_writel(chip, CORBUBASE, upper_32bit(chip->corb.addr));

	/* set the corb write pointer to 0 */
	azx_writew(chip, CORBWP, 0);
	/* reset the corb hw read pointer */
	azx_writew(chip, CORBRP, ICH6_RBRWP_CLR);
	/* enable corb dma */
	azx_writeb(chip, CORBCTL, ICH6_RBCTL_DMA_EN);

	/* RIRB set up */
	chip->rirb.addr = chip->rb.addr + 2048;
	chip->rirb.buf = (u32 *)(chip->rb.area + 2048);
	azx_writel(chip, RIRBLBASE, (u32)chip->rirb.addr);
	azx_writel(chip, RIRBUBASE, upper_32bit(chip->rirb.addr));

	/* reset the rirb hw write pointer */
	azx_writew(chip, RIRBWP, ICH6_RBRWP_CLR);
	/* enable rirb dma */
	azx_writeb(chip, RIRBCTL, ICH6_RBCTL_DMA_EN);
}

static void azx_free_cmd_io(azx_t *chip)
{
	/* disable ringbuffer DMAs */
	azx_writeb(chip, RIRBCTL, 0);
	azx_writeb(chip, CORBCTL, 0);
}
/* receive a response */
static int snd_azx_getresponse(azx_t *chip)
{
	unsigned int wp, corbwp;
	int timeout = 2;

	corbwp = azx_readb(chip, CORBWP);
	while ((wp = azx_readb(chip, RIRBWP)) != corbwp) {
		if (! --timeout) {
			snd_printk("azx_get_response timeout\n");
			break;
		}
		msleep(1);
	}
	wp <<= 1; /* an RIRB entry is 8-bytes */
	/* read the low value only */
	return le32_to_cpu(chip->rirb.buf[wp]);
}

/* send a command */
static int snd_azx_codec_sendcorb(azx_t *chip, u16 nid, u16 direct, u16 verb, u16 para) 
{
	unsigned int wp;
	u32 val;

	val = (u32)(chip->caindex & 0x0f) << 28;
	val |= (u32)direct << 27;
	val |= (u32)nid << 20;
	val |= verb << 8;
	val |= para;

	/* add command to corb */
	wp = azx_readb(chip, CORBWP);
	wp++;
	wp %= ICH6_MAX_CORB_ENTRIES;

	chip->corb.buf[wp] = cpu_to_le32(val);
	azx_writel(chip, CORBWP, wp);

	return 0;
}
#else
/*
 * Use the single immediate command instead of CORB/RIRB for simplicity
 *
 * Note: according to Intel, this is not preferred use.  The command was
 *       intended for the BIOS only, and may get confused with unsolicited
 *       responses.  So, we shouldn't use it for normal operation from the
 *       driver.
 *       I left the codes, however, for debugging/testing purposes.
 */

#define azx_alloc_cmd_io(chip)	0
#define azx_init_cmd_io(chip)
#define azx_free_cmd_io(chip)

static int snd_azx_getresponse(azx_t *chip)
{
	unsigned int i = 200;

	while (i--) {
		if ((azx_readw(chip, IRS) & 2)) // check IRV busy bit
			return azx_readl(chip, IR);
	}
	return -1;
}

static int snd_azx_codec_sendcorb(azx_t *chip, u16 nid, u16 direct, u16 verb, u16 para) 
{
	u32	val = 0;
	unsigned int i = 200;

	val |= (u32)((chip->caindex & 0x0f) << 28);
	val |= (u32)(nid << 20) | (u32)(direct << 27);
	val |= (u32)(verb << 8) | (u32)(para);
	while (i--) {
		if (!(azx_readw(chip, IRS) & 1)) {	// check ICB busy bit
			// Clear IRV valid bit
			azx_writew(chip, IRS, azx_readw(chip, IRS) | 2);
			azx_writel(chip,IC,val);
			azx_writew(chip, IRS, azx_readw(chip, IRS) | 1);
//			printk("ok val = 0x%8x.\n", val);
			return 0;
		}
	}
//	printk("fail 0x%4x val 0x%8x.\n", azx_readw(chip, IRS), val);
	return -1;
}
#endif

static int snd_azx_codec_write(azx_codec_t *azx_codec, u16 nid, u16 direct, u16 verb, u16 para) 
{
	azx_t *chip = snd_magic_cast(azx_t, azx_codec->private_data, return -1);
	unsigned int	data;
	
	spin_lock(&chip->azx_codec_lock);
	data = snd_azx_codec_sendcorb(chip, nid, direct, verb, para);
	spin_unlock(&chip->azx_codec_lock);
	return data;
}

static int snd_azx_codec_read(azx_codec_t *azx_codec, u16 nid, u16 direct, u16 verb, u16 para) 
{
	azx_t *chip = snd_magic_cast(azx_t, azx_codec->private_data, return -1);
	unsigned int	data = 0;

	spin_lock(&chip->azx_codec_lock);
	if (!snd_azx_codec_sendcorb(chip, nid, direct, verb, para)) { 
		data = snd_azx_getresponse(chip);
//		printk("data = 0x%8x.\n", data);
	} else
		data = -1;
	spin_unlock(&chip->azx_codec_lock);	
	return data;
}
 

void snd_azx_int_enable(azx_t* chip)
{
#if 0
	u8	val;
	// enable SDO 0 in stream descriptor
	// bit 4 = DEIE - Descriptor Error Interrupt
	// bit 3 = FIFO - First in First out interrupt
	// bit 2 = IOCE - Interrupt on completion
	val = azx_readl(chip, OSDCTL) | 0x1c;
	azx_writeb(chip, OSDCTL, val);

	// enable SIE for SDO 0
	azx_writeb(chip, INTCTL, 0xff);
#endif
	/* enable controller CIE and GIE */
	azx_writel(chip, INTCTL, azx_readl(chip, INTCTL) |
		   ICH6_INT_CTRL_EN | ICH6_INT_GLOBAL_EN);

}

void snd_azx_int_disable(azx_t* chip)
{
#if 0
	void __iomem *azx_base = chip->remap_addr;
	u8 val8;
	u32 val32;

	// disable SDO 0 in stream descriptor
	// bit 4 = DEIE - Descriptor Error Interrupt
	// bit 3 = FIFO - First in First out interrupt
	// bit 2 = IOCE - Interrupt on completion
	val8 = readl(azx_base + ICH6_REG_OSDCTL) & (~0x1c);
	writeb(val8, azx_base + ICH6_REG_OSDCTL);

	// disable SIE for SDO 0
	val8 = 0x10;
	writeb(val8, azx_base + ICH6_REG_INTCTL);
#endif
	int i;

	/* disable interrupts in stream descriptor */
	for (i = 0; i < ICH6_MAX_DEV; i++) {
		azx_dev_t *azx_dev = &chip->azx_dev[i];
		azx_sd_writeb(azx_dev, SD_CTL,
			      azx_sd_readb(azx_dev, SD_CTL) & ~SD_INT_MASK);
	}

	/* disable SIE for all streams */
	azx_writeb(chip, INTCTL, 0);

/* disable controller CIE and GIE */
	azx_writel(chip, INTCTL, azx_readl(chip, INTCTL) &
		   ~(ICH6_INT_CTRL_EN | ICH6_INT_GLOBAL_EN));
}

void snd_azx_int_clear(azx_t* chip)
{
	int i;

	// clear stream SDO status
#if 0
	azx_writeb(chip, OSDSTS, 0x1c);

	// clear stream SDI status
	azx_writeb(chip, ISDSTS, 0x1c);
	// clear STATESTS
	azx_writeb(chip, STATESTS, 0x7f);
#else
	for (i = 0; i < ICH6_MAX_DEV; i++) {
		azx_dev_t *azx_dev = &chip->azx_dev[i];
		azx_sd_writeb(azx_dev, SD_STS, SD_INT_MASK);
	}
	
	azx_writeb(chip, STATESTS, STATESTS_INT_MASK); //7
#endif
	// clear rirb status
	azx_writeb(chip, RIRBSTS, RIRB_INT_MASK);

	// clear int status
#if 0	
	azx_writel(chip, INTSTS, 0x7fffffff);
#else
	azx_writel(chip, INTSTS, ICH6_INT_CTRL_EN | ICH6_INT_ALL_STREAM);
#endif
}

static void azx_stream_start(azx_t *chip, azx_dev_t *azx_dev)
{
	/* enable SIE */
	azx_writeb(chip, INTCTL,
		   azx_readb(chip, INTCTL) | (1 << azx_dev->index));
	/* set DMA start and interrupt mask */
	azx_sd_writeb(azx_dev, SD_CTL, azx_sd_readb(azx_dev, SD_CTL) |
		      SD_CTL_DMA_START | SD_INT_MASK);
}

/* stop a stream */
static void azx_stream_stop(azx_t *chip, azx_dev_t *azx_dev)
{
	/* stop DMA */
	azx_sd_writeb(azx_dev, SD_CTL, azx_sd_readb(azx_dev, SD_CTL) &
		      ~(SD_CTL_DMA_START | SD_INT_MASK));
	azx_sd_writeb(azx_dev, SD_STS, SD_INT_MASK); /* to be sure */
	/* disable SIE */
	azx_writeb(chip, INTCTL,
		   azx_readb(chip, INTCTL) & ~(1 << azx_dev->index));
}

/*
 * reset AC link
 */
static int snd_azx_reset(azx_t *chip)
{
	int count = 0, alc_index = 0;
	int data = 0;
	u16 statests;
	u16 codec_ack;

/* reset controller */
	azx_writel(chip, GCTL, azx_readl(chip, GCTL) & ~(GCTL_CONTROLLER_RESET_BIT_MASK));

	count = 50;
	while (azx_readb(chip, GCTL) && --count)
		udelay(1);
	
	// delay for >= 100us for codec PLL to settle per spec
	// Rev 0.9 section 5.5.1
	udelay(100);

	// Bring controller out of reset
	azx_writeb(chip, GCTL, azx_readb(chip, GCTL) | GCTL_CONTROLLER_RESET_BIT_MASK);

	mdelay(10);
	count = 0;
	while ((count < 500) && (!(azx_readb(chip, GCTL)))) {
		count++;
	}

	// Brent Chartrand said to wait >= 540us for codecs to intialize
	mdelay(1);

	chip->num_codecs = 0;
	statests = azx_readw(chip, STATESTS);
	for (count = 0; count < 16; count++) {
		codec_ack = statests & (1 << count);
		if (codec_ack) {
			printk("Codec found on SDI[%i]\n", count);
			chip->num_codecs++;
			// 02/22/05
			chip->caindex = count;
			spin_lock(&chip->azx_codec_lock);
			if (!snd_azx_codec_sendcorb(chip, AC_NODE_ROOT, 0, AC_VERB_PARAMETERS, AC_PRAR_VENDOR_ID)) { 
				data = snd_azx_getresponse(chip);
			} else {
				printk("no data");
				return -1;
			}
			spin_unlock(&chip->azx_codec_lock);
//			id = snd_azx_codec_read(chip->azx_codec, AC_NODE_ROOT, 0, AC_VERB_PARAMETERS, AC_PRAR_VENDOR_ID);
//			if (1 == chip->num_codecs) {
			if ((data & 0xffff0000) == 0x10ec0000) {
				alc_index = count;
			}
		}
	}
	chip->caindex = alc_index;	
	chip->codec_mask = statests & 0x7F;

	return 0;
}

/*
 * PCM section
 */

/*
 * pointer callback simplly reads XXX_DMA_DT_CUR register as the current
 * position.  when SG-buffer is implemented, the offset must be calculated
 * correctly...
 */
static snd_pcm_uframes_t snd_azx_pcm_pointer(snd_pcm_substream_t *substream)
{
//	azx_t* chip = snd_pcm_substream_chip(substream);
	azx_dev_t* azx_dev = get_azx_dev(substream);

	unsigned int pos;

#ifdef USE_POSBUF
	/* use the position buffer */
	pos = *azx_dev->posbuf;
#else
	/* read LPIB */
	pos = azx_sd_readl(azx_dev, SD_LPIB) + azx_dev->fifo_size;
#endif

	if (pos >= azx_dev->bufsize)
		pos = 0;
	
	return bytes_to_frames(substream->runtime, pos);
}

/* common trigger callback
 * calling the lowlevel callbacks in it
 */
static int snd_azx_pcm_trigger(snd_pcm_substream_t *substream, int cmd)
{
	azx_t* chip = snd_pcm_substream_chip(substream);
	azx_dev_t* azx_dev = get_azx_dev(substream);

	spin_lock(&chip->reg_lock);
	switch (cmd) {
	case SNDRV_PCM_TRIGGER_PAUSE_RELEASE:
	case SNDRV_PCM_TRIGGER_RESUME:
	case SNDRV_PCM_TRIGGER_START:
		azx_stream_start(chip, azx_dev);
		azx_dev->running = 1;
		// gonna wait a little for data to be buffered
		break;
	case SNDRV_PCM_TRIGGER_PAUSE_PUSH:
	case SNDRV_PCM_TRIGGER_STOP:
		azx_stream_stop(chip, azx_dev);
		azx_dev->running = 0;
		break;
	default:
		return -EINVAL;
	}
	spin_unlock(&chip->reg_lock);

	if (cmd == SNDRV_PCM_TRIGGER_PAUSE_PUSH ||
	    cmd == SNDRV_PCM_TRIGGER_STOP) {
		int timeout = 5000;
		while (azx_sd_readb(azx_dev, SD_CTL) & SD_CTL_DMA_START && --timeout)
			;
	}

	return 0;
}

static void snd_azx_setup_periods(azx_t *chip, azx_dev_t *azx_dev)
{
	u32 *bdl = azx_dev->bdl;
	dma_addr_t dma_addr = azx_dev->substream->runtime->dma_addr;
	int idx;

	/* reset BDL address */
	azx_sd_writel(azx_dev, SD_BDLPL, 0);
	azx_sd_writel(azx_dev, SD_BDLPU, 0);

	/* program the initial BDL entries */
	for (idx = 0; idx < azx_dev->frags; idx++) {
		unsigned int off = idx << 2; /* 4 dword step */
		dma_addr_t addr = dma_addr + idx * azx_dev->fragsize;
		/* program the address field of the BDL entry */
		bdl[off] = cpu_to_le32((u32)addr);
		bdl[off+1] = cpu_to_le32(upper_32bit(addr));

		/* program the size field of the BDL entry */
		bdl[off+2] = cpu_to_le32(azx_dev->fragsize);

		/* program the IOC to enable interrupt when buffer completes */
		bdl[off+3] = cpu_to_le32(0x01);
	}
}

void snd_azx_pcm_setup_controller(azx_t* chip, azx_dev_t* azx_dev)
{
	snd_pcm_runtime_t *runtime = azx_dev->substream->runtime;

	u32 rate = runtime->rate;
	u32 num_channels = runtime->channels;
	u32 bits_per_sample = runtime->sample_bits;

	u32 val = 0;
	int timeout;

//	printk("stream info:\n");
//	printk("rate = %i\n", rate);
//	printk("num_channels = %i\n", num_channels);
//	printk("bits per sample = %i\n", bits_per_sample);


	// set the format
	if (44100 == rate) {
		val = 0x4000;
	}
	if (48000 == rate) {
		val = 0x0000;
	}
	if (96000 == rate) {
		val = 0x0800;
	}
	if (192000 == rate) {
		val = 0x1800;
	}
	val = val | (num_channels - 1); 


	switch (bits_per_sample) {
		case 8:
			val = val | 0x00;
			break;
		case 16:
			val = val | 0x10;
			break;
		case 20:
			val = val | 0x20;
			break;
		case 24:
			val = val | 0x30;
			break;
		case 32:
			val = val | 0x40;
			break;
		default:
			printk("Error...not supported bits per sample!!!\n");
			printk("   defaulting to 8 bits\n");
	}
	azx_dev->format_val = val; // save the value to program the controller and the codec

	// **** The next thing is to setup the controller for streaming
	// make sure the run bit is zero for SDO 0
	azx_sd_writeb(azx_dev, SD_CTL, azx_sd_readb(azx_dev, SD_CTL) & ~SD_CTL_DMA_START);
	/* reset stream */
	azx_sd_writeb(azx_dev, SD_CTL, azx_sd_readb(azx_dev, SD_CTL) | SD_CTL_STREAM_RESET);
	udelay(3);
	timeout = 300;
	while (!((val = azx_sd_readb(azx_dev, SD_CTL)) & SD_CTL_STREAM_RESET) &&
	       --timeout)
		;
	val &= ~SD_CTL_STREAM_RESET;
	azx_sd_writeb(azx_dev, SD_CTL, val);
	udelay(3);

	timeout = 300;
	/* waiting for hardware to report that the stream is out of reset */
	while (((val = azx_sd_readb(azx_dev, SD_CTL)) & SD_CTL_STREAM_RESET) &&
	       --timeout)
		;

	// program the stream_tag
	azx_sd_writel(azx_dev, SD_CTL,
		      (azx_sd_readl(azx_dev, SD_CTL) & ~SD_CTL_STREAM_TAG_MASK) |
//		      (azx_dev->stream_tag << SD_CTL_STREAM_TAG_SHIFT));
			(azx_dev->stream_tag));
	udelay(10);

	// program the length of samples in cyclic buffer in register offset 108h
	azx_sd_writel(azx_dev, SD_CBL, azx_dev->bufsize);

	// this needs to be done during pcm_hw_param
	// program the stream format...this value needs to be the same as the one programmed
	// in the converter...which was 4010h
	azx_sd_writew(azx_dev, SD_FORMAT, azx_dev->format_val);

	// program the stream LVI (last valid index) of the BDL
	azx_sd_writew(azx_dev, SD_LVI, azx_dev->frags - 1);

	// program the BDL address
	// lower BDL address
	azx_sd_writel(azx_dev, SD_BDLPL, (u32)azx_dev->bdl_addr);
	// upper BDL address
	azx_sd_writel(azx_dev, SD_BDLPU, upper_32bit(azx_dev->bdl_addr));

#ifdef USE_POSBUF
	/* enable the position buffer */
	if (! (azx_readl(chip, DPLBASE) & ICH6_DPLBASE_ENABLE))
		azx_writel(chip, DPLBASE, (u32)chip->posbuf.addr | ICH6_DPLBASE_ENABLE);
#endif

	// set the interrupt enable bits in the descriptor control register
	azx_sd_writel(azx_dev, SD_CTL, azx_sd_readl(azx_dev, SD_CTL) | SD_INT_MASK);

}

void snd_azx_pcm_setup_codec(azx_codec_t* azx_codec, u8 nid, u32 stream_tag, u8 channel_id, u16 format)
{
	u8 direct = 0;
	u8 tag = stream_tag >> 16;

	// result is 0x00000010
	snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_CHANNEL_STREAMID, tag | channel_id);
	mdelay(1);

	snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_STREAM_FORMAT, format);

}
/* set up slots and formats for analog OUT */
static int snd_azx_pcm_prepare(snd_pcm_substream_t *substream)
{
	azx_t* chip = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;
	azx_dev_t* azx_dev = get_azx_dev(substream);
	u32 num_channels = runtime->channels;

	azx_dev->bufsize = snd_pcm_lib_buffer_bytes(substream);
	azx_dev->fragsize = snd_pcm_lib_period_bytes(substream);
	azx_dev->frags = azx_dev->bufsize / azx_dev->fragsize;

	// setup the controller
	// NOTE: since all the DACs are currently decoding the left/right channels,
	//       no programming of other DACs are needed, however when 5.1 surround is implemented
	//       other DACs will need to be programmed!
	// the BDL address and BDL entries are programmed in setup_periods()
	snd_azx_setup_periods(chip, azx_dev); // to be implemented
	snd_azx_pcm_setup_controller(chip, azx_dev);
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
		azx_dev->fifo_size = azx_sd_readw(azx_dev, SD_FIFOSIZE) + 1;
	else
		azx_dev->fifo_size = 0;

//	u8 table_index = (chip->board_config & 0xFF0) >> 4;
	
	codec_callback_info.pcm_codec_setup(chip, azx_dev, num_channels, chip->board_config);

#if 0
printk("2 0x%8x, 0x%8x.\n", snd_azx_codec_read(chip->azx_codec, 0x2, 0, 0xa00, 0), snd_azx_codec_read(chip->azx_codec, 0x2, 0, 0xf06, 0));
printk("3 0x%8x, 0x%8x.\n", snd_azx_codec_read(chip->azx_codec, 0x3, 0, 0xa00, 0), snd_azx_codec_read(chip->azx_codec, 0x3, 0, 0xf06, 0));
printk("4 0x%8x, 0x%8x.\n", snd_azx_codec_read(chip->azx_codec, 0x4, 0, 0xa00, 0), snd_azx_codec_read(chip->azx_codec, 0x4, 0, 0xf06, 0));
printk("5 0x%8x, 0x%8x.\n", snd_azx_codec_read(chip->azx_codec, 0x5, 0, 0xa00, 0), snd_azx_codec_read(chip->azx_codec, 0x5, 0, 0xf06, 0));	

printk("18 0x%8x, 10 0x%8x.\n", snd_azx_codec_read(chip->azx_codec, 0x18, 0, 0xf07, 0), snd_azx_codec_read(chip->azx_codec, 0x10, 0, 0xf01, 0));
printk("19 0x%8x, 11 0x%8x.\n", snd_azx_codec_read(chip->azx_codec, 0x19, 0, 0xf07, 0), snd_azx_codec_read(chip->azx_codec, 0x11, 0, 0xf01, 0));
printk("1a 0x%8x, 12 0x%8x.\n", snd_azx_codec_read(chip->azx_codec, 0x1a, 0, 0xf07, 0), snd_azx_codec_read(chip->azx_codec, 0x12, 0, 0xf01, 0));
printk("1b 0x%8x, 13 0x%8x.\n", snd_azx_codec_read(chip->azx_codec, 0x1b, 0, 0xf07, 0), snd_azx_codec_read(chip->azx_codec, 0x13, 0, 0xf01, 0));
printk("14 0x%8x, 0x%8x.\n", snd_azx_codec_read(chip->azx_codec, 0x14, 0, 0xf02, 0), snd_azx_codec_read(chip->azx_codec, 0x14, 0, 0xf07, 0));
printk("15 0x%8x, 0x%8x.\n", snd_azx_codec_read(chip->azx_codec, 0x15, 0, 0xf02, 0), snd_azx_codec_read(chip->azx_codec, 0x15, 0, 0xf07, 0));
printk("16 0x%8x, 0x%8x.\n", snd_azx_codec_read(chip->azx_codec, 0x16, 0, 0xf02, 0), snd_azx_codec_read(chip->azx_codec, 0x16, 0, 0xf07, 0));
printk("17 0x%8x, 0x%8x.\n", snd_azx_codec_read(chip->azx_codec, 0x17, 0, 0xf02, 0), snd_azx_codec_read(chip->azx_codec, 0x17, 0, 0xf07, 0));

printk(" C 0x%4x, D 0x%4x,\n", snd_azx_codec_read(chip->azx_codec, 0x0c, 0, AC_VERB_GET_AMP_GAIN_MUTE, 0x8000), snd_azx_codec_read(chip->azx_codec, 0x0d, 0, AC_VERB_GET_AMP_GAIN_MUTE, 0x8000));
printk(" E 0x%4x, F 0x%4x,\n", snd_azx_codec_read(chip->azx_codec, 0x0E, 0, AC_VERB_GET_AMP_GAIN_MUTE, 0x8000), snd_azx_codec_read(chip->azx_codec, 0x0F, 0, AC_VERB_GET_AMP_GAIN_MUTE, 0x8000));
printk(" 14 0x%4x, 15 0x%4x,\n", snd_azx_codec_read(chip->azx_codec, 0x014, 0, AC_VERB_GET_AMP_GAIN_MUTE, 0x8000), snd_azx_codec_read(chip->azx_codec, 0x015, 0, AC_VERB_GET_AMP_GAIN_MUTE, 0x8000));
printk(" 16 0x%4x, 17 0x%4x,\n", snd_azx_codec_read(chip->azx_codec, 0x016, 0, AC_VERB_GET_AMP_GAIN_MUTE, 0x8000), snd_azx_codec_read(chip->azx_codec, 0x17, 0, AC_VERB_GET_AMP_GAIN_MUTE, 0x8000));
printk(" 18 0x%4x, 19 0x%4x,\n", snd_azx_codec_read(chip->azx_codec, 0x18, 0, AC_VERB_GET_AMP_GAIN_MUTE, 0x8000), snd_azx_codec_read(chip->azx_codec, 0x19, 0, AC_VERB_GET_AMP_GAIN_MUTE, 0x8000));
printk(" 1a 0x%4x, 1b 0x%4x,\n", snd_azx_codec_read(chip->azx_codec, 0x1a, 0, AC_VERB_GET_AMP_GAIN_MUTE, 0x8000), snd_azx_codec_read(chip->azx_codec, 0x1b, 0, AC_VERB_GET_AMP_GAIN_MUTE, 0x8000));
#endif


	return 0;
}

/*
 * hw_params - allocate the buffer and set up buffer descriptors
 */
static int snd_azx_pcm_hw_params(snd_pcm_substream_t *substream,
				   snd_pcm_hw_params_t *hw_params)
{
	return snd_pcm_lib_malloc_pages(substream, params_buffer_bytes(hw_params));
}

static int snd_azx_pcm_hw_free(snd_pcm_substream_t * substream)
{
	snd_pcm_lib_free_pages(substream);
	return 0;
}


/*
 * pcm hardware definition, identical for all DMA types
 */

static snd_pcm_hardware_t snd_azx_playback_hw = {

	.info =			(SNDRV_PCM_INFO_MMAP | SNDRV_PCM_INFO_INTERLEAVED |
				 SNDRV_PCM_INFO_BLOCK_TRANSFER |
				 SNDRV_PCM_INFO_MMAP_VALID |
				 SNDRV_PCM_INFO_PAUSE |
				 SNDRV_PCM_INFO_RESUME),
	.formats =		SNDRV_PCM_FMTBIT_S16_LE,
	.rates =		SNDRV_PCM_RATE_48000,
	.rate_min =		48000,
	.rate_max =		48000,
	.channels_min =		2,
	.channels_max =		8, //2,
	.buffer_bytes_max =	AZX_MAX_BUF_SIZE,
	.period_bytes_min =	128,
	.period_bytes_max =	AZX_MAX_BUF_SIZE / 2,
	.periods_min =		2,
	.periods_max =		AZX_MAX_FRAG,
	.fifo_size =		0,
#if 0
	.rates =		(SNDRV_PCM_RATE_192000 | SNDRV_PCM_RATE_96000 |
				 SNDRV_PCM_RATE_48000 | SNDRV_PCM_RATE_44100),
	.rate_min =		44100,
	.rate_max =		192000,
	.channels_min =		2,
	.channels_max =		8, //6,
	.buffer_bytes_max =	64 * 1024,
	.period_bytes_min =	128,
	.period_bytes_max =	4096,
	.periods_min =		4,
	.periods_max =		16,
	.fifo_size =		0,
#endif
};

static snd_pcm_hardware_t snd_azx_capture_hw = {
	.info =			(SNDRV_PCM_INFO_MMAP | SNDRV_PCM_INFO_INTERLEAVED |
				 SNDRV_PCM_INFO_BLOCK_TRANSFER |
				 SNDRV_PCM_INFO_MMAP_VALID),
	.formats =		SNDRV_PCM_FMTBIT_S16_LE,
	.rates =		SNDRV_PCM_RATE_48000,
	.rate_min =		48000,
	.rate_max =		48000,
	.channels_min =		2,
	.channels_max =		2,
	.buffer_bytes_max =	AZX_MAX_BUF_SIZE,
	.period_bytes_min =	128,
	.period_bytes_max =	AZX_MAX_BUF_SIZE / 2,
	.periods_min =		2,
	.periods_max =		AZX_MAX_FRAG,
	.fifo_size =		0,
#if 0
	.rates =		(SNDRV_PCM_RATE_96000 |
				 SNDRV_PCM_RATE_48000 | SNDRV_PCM_RATE_44100),
	.rate_min =		44100,
	.rate_max =		96000,
	.channels_min =		2,
	.channels_max =		2,
	.buffer_bytes_max =	128 * 1024,
	.period_bytes_min =	32,
	.period_bytes_max =	128 * 1024,
	.periods_min =		1,
	.periods_max =		1024,
	.fifo_size =		0,
#endif
};

static int snd_azx_pcm_open(snd_pcm_substream_t *substream, azx_dev_t* azx_dev)
{
	snd_pcm_runtime_t *runtime = substream->runtime;
	azx_t* chip = snd_pcm_substream_chip(substream);

	int err;
//	int dev;
	unsigned long flags;

	down(&chip->open_mutex);
	
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
//		dev = 4;
		runtime->hw = snd_azx_playback_hw;
	} else {
//		dev = 0;
		runtime->hw = snd_azx_capture_hw;
	}
#if 0
	azx_dev = NULL
	for (i = 0; i < 4; i++, dev++)
		if (! chip->azx_dev[dev].opened) {
			chip->azx_dev[dev].opened = 1;
			azx_dev = &chip->azx_dev[dev];
			break;
		}
	if (azx_dev == NULL) {
		up(&chip->open_mutex);
		return -EBUSY;
	}
#endif
	// get the fifo size of the controller
	// I am making an assumption that the context/meaning of fifo size is the same for
	// Azalia (spec Rev. 0.9 section 3.3.38) and ALSA...
//	val = readw(azx_base + sd_base + ICH6_REG_SD_FIFOSIZE);
//	runtime->hw.fifo_size = val;	
	snd_pcm_limit_hw_rates(runtime);

	if ((err = snd_pcm_hw_constraint_integer(runtime, SNDRV_PCM_HW_PARAM_PERIODS)) < 0) {
		printk("Error: snd_pcm_hw_constraint_integer()\n");
		return err;
	}
	spin_lock_irqsave(&chip->reg_lock, flags);
	azx_dev->substream = substream;
	azx_dev->running = 0;
	spin_unlock_irqrestore(&chip->reg_lock, flags);

	runtime->private_data = azx_dev;
	up(&chip->open_mutex);

	return 0;
}

/*
 */
static int snd_azx_playback_open(snd_pcm_substream_t *substream)
{
	azx_t *chip = snd_pcm_substream_chip(substream);
	int err;

	// set the operation to playback
	chip->operation = 1;

	err = snd_azx_pcm_open(substream, &chip->azx_dev[SDO0]);
	if (err < 0) {
		printk("Error: snd_azx_pcm_open!!!\n");
		return err;
	}

	return 0;
}

static int snd_azx_playback_close(snd_pcm_substream_t *substream)
{
	azx_t *chip = snd_pcm_substream_chip(substream);
	unsigned long flags;

	down(&chip->open_mutex);
	spin_lock_irqsave(&chip->reg_lock, flags);
	chip->azx_dev[SDO0].substream = NULL;
	chip->azx_dev[SDO0].running = 0;
	chip->azx_dev[SDO0].opened = 0;
	spin_unlock_irqrestore(&chip->reg_lock, flags);
	up(&chip->open_mutex);
	return 0;
}

static int snd_azx_capture_open(snd_pcm_substream_t *substream)
{
	azx_t *chip = snd_pcm_substream_chip(substream);
	// set the operation to capture
	chip->operation = 0;

	return snd_azx_pcm_open(substream, &chip->azx_dev[SDI0]);
}

static int snd_azx_capture_close(snd_pcm_substream_t *substream)
{
	azx_t *chip = snd_pcm_substream_chip(substream);
	unsigned long flags;
//	chip->azx_dev[SDI0].substream = NULL;
	
	down(&chip->open_mutex);
	spin_lock_irqsave(&chip->reg_lock, flags);
	chip->azx_dev[SDI0].substream = NULL;
	chip->azx_dev[SDI0].running = 0;
	chip->azx_dev[SDI0].opened = 0;
	spin_unlock_irqrestore(&chip->reg_lock, flags);
	up(&chip->open_mutex);

	return 0;
}

/* AC97 playback */
static snd_pcm_ops_t snd_azx_playback_ops = {
	.open =		snd_azx_playback_open,
	.close =	snd_azx_playback_close,
	.ioctl =	snd_pcm_lib_ioctl,
	.hw_params =	snd_azx_pcm_hw_params,
	.hw_free =	snd_azx_pcm_hw_free,
	.prepare =	snd_azx_pcm_prepare,
	.trigger =	snd_azx_pcm_trigger,
	.pointer =	snd_azx_pcm_pointer,
};

/* AC97 capture */
static snd_pcm_ops_t snd_azx_capture_ops = {
	.open =		snd_azx_capture_open,
	.close =	snd_azx_capture_close,
	.ioctl =	snd_pcm_lib_ioctl,
	.hw_params =	snd_azx_pcm_hw_params,
	.hw_free =	snd_azx_pcm_hw_free,
	.prepare =	snd_azx_pcm_prepare,
	.trigger =	snd_azx_pcm_trigger,
	.pointer =	snd_azx_pcm_pointer,
};

static int __devinit snd_azx_pcm_new(azx_t *chip)
{
	snd_pcm_t* pcm;
	int err;

	if ((err = snd_pcm_new(chip->card, "ICH6 HDA", 0, 1, 1, &pcm)) < 0) {
		printk("Error: snd_pcm_new() failed!!!\n");
		return err;
	}

	pcm->private_data = chip;
	strcpy(pcm->name, "ICH6 HDA");
//	chip->pcm = pcm;  // don't think I need this

	snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_PLAYBACK, &snd_azx_playback_ops);

	snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_CAPTURE, &snd_azx_capture_ops);

	// size taken from AC'97
	// 1.0.3 ALSA
//	snd_pcm_lib_preallocate_pci_pages_for_all(chip->pci, pcm, 1024 * 128, 1024 * 128);

	// 1.0.5 ALSA
	snd_pcm_lib_preallocate_pages_for_all(pcm, SNDRV_DMA_TYPE_DEV, snd_dma_pci_data(chip->pci),
					      1024 * 64, 1024 * 128);

	return 0;
}
#if 0
static inline void snd_azx_update(azx_t *chip, azx_dev_t *azx_dev)
{
//	printk("*-*-* Entered update *-*-*\n");

	u32* bdl_v_addr = (u32*)azx_dev->bdl_v_addr;
	u32 new_addr;

	azx_dev->position += azx_dev->fragsize;
	azx_dev->position %= azx_dev->size;

	azx_dev->lvi %= ICH6_MAX_FRAG;
	azx_dev->lvi_frag %= azx_dev->frags;

	// update the BDL entry that just got processed with a new address
	new_addr = cpu_to_le32(azx_dev->playbuff_p_addr + azx_dev->lvi_frag * azx_dev->fragsize);
	bdl_v_addr[azx_dev->lvi * 4] = new_addr;

	azx_dev->lvi++;
	azx_dev->lvi_frag++;

	if (azx_dev->substream) {
		snd_pcm_period_elapsed(azx_dev->substream);
	}
}
#endif


/*
 * interrupt handler
 */
static irqreturn_t snd_azx_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	azx_t *chip = snd_magic_cast(azx_t, dev_id, return IRQ_NONE);
	azx_dev_t* azx_dev;
	u32 count;
	u32 status;

	spin_lock(&chip->reg_lock);

	status = azx_readl(chip, INTSTS);

	if (status == 0) {
		spin_unlock(&chip->reg_lock);
		return IRQ_NONE;
	}

	for (count = 0; count < ICH6_MAX_DEV; count++) {
		azx_dev = &chip->azx_dev[count];
		if (status & azx_dev->sd_int_sta_mask) {
			azx_sd_writeb(azx_dev, SD_STS, SD_INT_MASK);
//			snd_azx_update(chip, azx_dev);
			if (azx_dev->substream && azx_dev->running) {
				spin_unlock(&chip->reg_lock);
				snd_pcm_period_elapsed(azx_dev->substream);
				spin_lock(&chip->reg_lock);
			}
		}
	}

	// clear rirb int
	if (azx_readb(chip, RIRBSTS) & RIRB_INT_MASK)
		azx_writeb(chip, RIRBSTS, RIRB_INT_MASK);

	spin_unlock(&chip->reg_lock);
	return IRQ_HANDLED;
}

// mixer control configurations
#define VOLUME 1
#define MUTE 0

// ALC880
snd_kcontrol_new_t three_stack_rear_panel_mixer[] = {
//	AZX_CODEC_VOLUME("control name here", nid, index, direction, max_val, type)
//	AZX_CODEC_VOLUME("PCM Front Playback Volume", 0x0C, 0x0, OUTPUT, 0x40, VOLUME),
//	AZX_CODEC_MUTE("PCM Front Playback Switch", 0x14, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("PCM Playback Volume", 0x0C, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("PCM Playback Switch", 0x14, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("PCM Rear Playback Volume", 0x0F, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("PCM Rear Playback Switch", 0x1A, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("PCM C/LFE Playback Volume", 0x0E, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("PCM C/LFE Playback Switch", 0x18, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("CD Playback Volume", 0x0B, 0x04, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("CD Playback Switch", 0x0B, 0x04, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Line Playback Volume", 0x0B, 0x02, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Line Playback Switch", 0x0B, 0x02, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Mic-1 Playback Volume", 0x0B, 0x0, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Mic-1 Playback Switch", 0x0B, 0x0, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Mic-2 Playback Volume", 0x0B, 0x3, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Mic-2 Playback Switch", 0x0B, 0x3, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Headphone Playback Volume", 0x0D, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("Headphone Playback Switch", 0x19, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Capture Volume", 0x07, 0x0, INPUT, 0x23, VOLUME),
	AZX_CODEC_ENUM("Capture Source1", 0x07, 0x0),
	AZX_CODEC_CUSTOM("Channel Source", snd_azx_mux_info, snd_azx_mux_get, snd_azx_mux_put, 0x0)
};

snd_kcontrol_new_t base_panel_mixer[] = {
//	AZX_VOLUME("control name here", nid, index, direction, max_val, type)
	AZX_CODEC_VOLUME("MASTER Playback Volume", 0x0C, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("MASTER Playback Switch", 0x14, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("CD Playback Volume", 0x0B, 0x04, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("CD Playback Switch", 0x0B, 0x04, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Line Playback Volume", 0x0B, 0x02, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Line Playback Switch", 0x0B, 0x02, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Mic Playback Volume", 0x0B, 0x0, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Mic Playback Switch", 0x0B, 0x0, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Capture Volume", 0x07, 0x0, INPUT, 0x23, VOLUME),
	AZX_CODEC_ENUM("Capture Source", 0x07, 0x01),
};

snd_kcontrol_new_t two_stack_rear_panel_mixer[] = {
//	AZX_VOLUME("control name here", nid, index, direction, max_val, type)
	AZX_CODEC_VOLUME("HEADPHONE Playback Volume", 0x0C, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("HEADPHONE Playback Switch", 0x14, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Internal Speaker Playback Volume", 0x0D, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("Internal Speaker Playback Switch", 0x15, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("CD Playback Volume", 0x0B, 0x04, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("CD Playback Switch", 0x0B, 0x04, INPUT, 0x01, MUTE),
//	AZX_CODEC_VOLUME("Line Playback Volume", 0x0B, 0x02, INPUT, 0x41, VOLUME),
//	AZX_CODEC_MUTE("Line Playback Switch", 0x0B, 0x02, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Mic Playback Volume", 0x0B, 0x0, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Mic Playback Switch", 0x0B, 0x0, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Capture Volume", 0x07, 0x0, INPUT, 0x23, VOLUME),
//	AZX_CODEC_ENUM("Capture Source", 0x07, 0x01),
};


snd_kcontrol_new_t five_stack_rear_panel_mixer[] = {
//	AZX_CODEC_VOLUME("control name here", nid, index, direction, max_val, type)
//	AZX_CODEC_VOLUME("Front Playback Volume", 0x0C, 0x0, OUTPUT, 0x40, VOLUME),
//	AZX_CODEC_MUTE("Front Playback Switch", 0x14, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("PCM Playback Volume", 0x0C, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("PCM Playback Switch", 0x14, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Surround Playback Volume", 0x0F, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("Surround Playback Switch", 0x17, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("C/LFE Playback Volume", 0x0E, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("C/LFE Playback Switch", 0x16, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Side Playback Volume", 0x0D, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("Side Playback Switch", 0x15, 0x0, OUTPUT, 0x01, MUTE), //0x19
	AZX_CODEC_VOLUME("CD Playback Volume", 0x0B, 0x04, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("CD Playback Switch", 0x0B, 0x04, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Line Playback Volume", 0x0B, 0x02, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Line Playback Switch", 0x0B, 0x02, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Mic-1 Playback Volume", 0x0B, 0x0, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Mic-1 Playback Switch", 0x0B, 0x0, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Mic-2 Playback Volume", 0x0B, 0x3, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Mic-2 Playback Switch", 0x0B, 0x3, INPUT, 0x01, MUTE),
//	AZX_ENUM("control name", nid, index into the enum list where the NID is associated with the selection)
	AZX_CODEC_ENUM("Capture Source1", 0x07, 0x0),
	AZX_CODEC_VOLUME("Capture Volume", 0x07, 0x0, INPUT, 0x23, VOLUME),
	AZX_CODEC_CUSTOM("Channel Source", snd_azx_mux1_info, snd_azx_mux1_get, snd_azx_mux1_put, 0x0),
};

snd_kcontrol_new_t five_stack_rear_panel_digital_mixer[] = {
//	AZX_CODEC_VOLUME("control name here", nid, index, direction, max_val, type)
//	AZX_CODEC_VOLUME("PCM Front Playback Volume", 0x0C, 0x0, OUTPUT, 0x40, VOLUME),
//	AZX_CODEC_MUTE("PCM Front Playback Switch", 0x14, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("PCM Playback Volume", 0x0C, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("PCM Playback Switch", 0x14, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("PCM Rear Playback Volume", 0x0F, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("PCM Rear Playback Switch", 0x17, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("PCM C/LFE Playback Volume", 0x0E, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("PCM C/LFE Playback Switch", 0x16, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Side Playback Volume", 0x0D, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("Side Playback Switch", 0x19, 0x0, OUTPUT, 0x01, MUTE), 
	AZX_CODEC_VOLUME("CD Playback Volume", 0x0B, 0x04, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("CD Playback Switch", 0x0B, 0x04, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Line Playback Volume", 0x0B, 0x02, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Line Playback Switch", 0x0B, 0x02, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Mic-1 Playback Volume", 0x0B, 0x0, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Mic-1 Playback Switch", 0x0B, 0x0, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Mic-2 Playback Volume", 0x0B, 0x3, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Mic-2 Playback Switch", 0x0B, 0x3, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Capture Volume", 0x07, 0x0, INPUT, 0x23, VOLUME),
	AZX_CODEC_ENUM("Capture Source1", 0x07, 0x0),
	AZX_CODEC_CUSTOM("Channel Source", snd_azx_mux1_info, snd_azx_mux1_get, snd_azx_mux1_put, 0x0),
	AZX_CODEC_CUSTOM("Digital-Out", snd_azx_mux_digital_info, snd_azx_mux_digital_get, snd_azx_mux_digital_put, 0x0)
};

snd_kcontrol_new_t five_stack_uli_mixer[] = {
//	AZX_CODEC_VOLUME("control name here", nid, index, direction, max_val, type)
	AZX_CODEC_VOLUME("PCM Playback Volume", 0x0C, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("PCM Playback Switch", 0x14, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("PCM Rear Playback Volume", 0x0F, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("PCM Rear Playback Switch", 0x17, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("PCM C/LFE Playback Volume", 0x0E, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("PCM C/LFE Playback Switch", 0x16, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Side Playback Volume", 0x0D, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("Side Playback Switch", 0x19, 0x0, OUTPUT, 0x01, MUTE), 
	AZX_CODEC_VOLUME("Line Playback Volume", 0x0B, 0x02, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Line Playback Switch", 0x0B, 0x02, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Mic-1 Playback Volume", 0x0B, 0x0, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Mic-1 Playback Switch", 0x0B, 0x0, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Capture Volume", 0x07, 0x0, INPUT, 0x23, VOLUME),
	AZX_CODEC_ENUM("Capture Source1", 0x07, 0x0),
	AZX_CODEC_CUSTOM("Channel Source", snd_azx_mux1_info, snd_azx_mux1_get, snd_azx_mux1_put, 0x0),
	AZX_CODEC_CUSTOM("Digital-Out", snd_azx_mux_digital_info, snd_azx_mux_digital_get, snd_azx_mux_digital_put, 0x0)
};

snd_kcontrol_new_t three_stack_rear_panel_digital_mixer[] = {
//	AZX_CODEC_VOLUME("control name here", nid, index, direction, max_val, type)
//	AZX_CODEC_VOLUME("PCM Front Playback Volume", 0x0C, 0x0, OUTPUT, 0x40, VOLUME),
//	AZX_CODEC_MUTE("PCM Front Playback Switch", 0x14, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("PCM Playback Volume", 0x0C, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("PCM Playback Switch", 0x14, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("PCM Rear Playback Volume", 0x0F, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("PCM Rear Playback Switch", 0x1A, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("PCM C/LFE Playback Volume", 0x0E, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("PCM C/LFE Playback Switch", 0x18, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("CD Playback Volume", 0x0B, 0x04, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("CD Playback Switch", 0x0B, 0x04, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Line Playback Volume", 0x0B, 0x02, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Line Playback Switch", 0x0B, 0x02, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Mic-1 Playback Volume", 0x0B, 0x0, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Mic-1 Playback Switch", 0x0B, 0x0, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Mic-2 Playback Volume", 0x0B, 0x3, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Mic-2 Playback Switch", 0x0B, 0x3, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Headphone Playback Volume", 0x0D, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("Headphone Playback Switch", 0x19, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Capture Volume", 0x07, 0x0, INPUT, 0x23, VOLUME),
	AZX_CODEC_ENUM("Capture Source1", 0x07, 0x0),
	AZX_CODEC_CUSTOM("Channel Source", snd_azx_mux_info, snd_azx_mux_get, snd_azx_mux_put, 0x0),
	AZX_CODEC_CUSTOM("Digital-Out", snd_azx_mux_digital_info, snd_azx_mux_digital_get, snd_azx_mux_digital_put, 0x0)
};


// for Cyberlink
snd_kcontrol_new_t three_stack_digital_mixer[] = {
//	AZX_CODEC_VOLUME("control name here", nid, index, direction, max_val, type)
//	AZX_CODEC_VOLUME("PCM Front Playback Volume", 0x0C, 0x0, OUTPUT, 0x40, VOLUME),
//	AZX_CODEC_MUTE("PCM Front Playback Switch", 0x14, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("PCM Playback Volume", 0x0C, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("PCM Playback Switch", 0x14, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("PCM Surr Playback Volume", 0x0D, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("PCM Surr Playback Switch", 0x1A, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("PCM C/LFE Playback Volume", 0x0E, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("PCM C/LFE Playback Switch", 0x18, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("CD Playback Volume", 0x0B, 0x04, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("CD Playback Switch", 0x0B, 0x04, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Line Playback Volume", 0x0B, 0x02, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Line Playback Switch", 0x0B, 0x02, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Mic Playback Volume", 0x0B, 0x0, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Mic Playback Switch", 0x0B, 0x0, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Headphone Playback Volume", 0x0C, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("Headphone Playback Switch", 0x1B, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Capture Volume", 0x07, 0x0, INPUT, 0x23, VOLUME),
	AZX_CODEC_ENUM("Capture Source1", 0x07, 0x0),
	AZX_CODEC_CUSTOM("Channel Source", snd_azx_mux_info, snd_azx_mux_get, snd_azx_mux_put, 0x0),
	AZX_CODEC_CUSTOM("Digital-Out", snd_azx_mux_digital_info, snd_azx_mux_digital_get, snd_azx_mux_digital_put, 0x0)
};

static verb_t alc880_three_stack_volume[] = {
// format = {nid, direct, command, parameter}
// Line In pin widget(nid=0x14) for input
	{0x1A, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x20},
// CD pin widget(nid=0x1C) for input
	{0x1C, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x20},
// Mic1 (rear panel) pin widget(nid=0x18) for input and vref at 80%
	{0x18, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x24},
// Mic2 (front panel) pin widget(nid=0x1B) for input and vref at 80%
	{0x1B, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x24},
// unmute amp left and right
	{0x07, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x7000},
// set connection select to line in (default select for this ADC)
	{0x07, 0x0, AC_VERB_SET_CONNECT_SEL, 0x02},
// unmute front mixer amp left and right and set to max volume Line out
	{0x0C, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
// unmute pin widget amp left and right (no gain on this amp)
	{0x14, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
// unmute rear mixer amp left and right and set to max volume rear
	{0x0F, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
// unmute pin widget amp left and right (no gain on this amp)
	{0x1A, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
// unmute rear mixer amp left and right and set to max volume clfe
	{0x0E, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
// unmute pin widget amp left and right (no gain on this amp)
	{0x18, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},

// using rear surround as the path for headphone output
	// unmute rear surround mixer amp left and right and set to max volume
	{0x0D, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
	// PASD 3 stack boards use the Mic 2 as the headphone output
	// need to program the selector associated with the Mic 2 pin widget to
	// surround path (index 0x01) for headphone output
	{0x11, 0x0, AC_VERB_SET_CONNECT_SEL, 0x01},
	// unmute pin widget amp left and right (no gain on this amp)
	{0x19, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
	// need to retask the Mic 2 pin widget to output
	{0x19, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x40},

// Unmute input amps (CD, Line In, Mic 1 & Mic 2) for mixer widget(nid=0x0B) to support
	// the input path of analog loopback
	// Note: PASD motherboards uses the Line In 2 as the input for front panel mic (mic 2)
	// Amp Indexes: CD = 0x04, Line In 1 = 0x02, Mic 1 = 0x00 & Line In 2 = 0x03
	// unmute CD
	{0x0B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x04 << 8))},
	// unmute Line In
	{0x0B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x02 << 8))},
	// unmute Mic 1
	{0x0B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	// unmute Line In 2 (for PASD boards Mic 2)
	{0x0B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x03 << 8))},

	// Unmute input amps for the line out paths to support the output path of analog loopback
	// the mixers on the output path has 2 inputs, one from the DAC and one from the mixer
	// Amp Indexes: DAC = 0x01 & mixer = 0x00
	// Unmute Front out path
	{0x0C, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x0C, 0x0,AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},
	// Unmute Surround (used as HP) out path
	{0x0D, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x0D, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},
	// Unmute C/LFE out path
	{0x0E, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x0E, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},
	// Unmute rear Surround out path
	{0x0F, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x0F, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},

	{0x0, 0x0, 0x0, 0x0}
};

static verb_t alc880_three_cyber_stack_volume[] = {
// format = {nid, direct, command, parameter}
	{0x1A, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x20},
	{0x1B, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0xc0},
	{0x1C, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x20},
	{0x14, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0xc0},
// Mic1 (rear panel) pin widget(nid=0x18) for input and vref at 80%
	{0x18, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x24},
	{0x07, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x7000},
// set connection select to line in (default select for this ADC)
	{0x07, 0x0, AC_VERB_SET_CONNECT_SEL, 0x02},
	{0x0C, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
	{0x14, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
	{0x15, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
	{0x16, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
	{0x17, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
// unmute pin widget amp left and right (no gain on this amp)
	{0x1B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
	{0x1A, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
	{0x0E, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
	{0x18, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},

// using front as the path for headphone output
	{0x0D, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
	// surround path (index 0x01) for headphone output
	{0x10, 0x0, AC_VERB_SET_CONNECT_SEL, 0x02},
	{0x12, 0x0, AC_VERB_SET_CONNECT_SEL, 0x01},
	{0x13, 0x0, AC_VERB_SET_CONNECT_SEL, 0x00},

	// Amp Indexes: CD = 0x04, Line In 1 = 0x02, Mic 1 = 0x00 & Line In 2 = 0x03
	// unmute CD
	{0x0B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x04 << 8))},
	// unmute Line In
	{0x0B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x02 << 8))},
	// unmute Mic 1
	{0x0B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	// unmute Line In 2 (for PASD boards Mic 2)
	{0x0B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x03 << 8))},
	// Amp Indexes: DAC = 0x01 & mixer = 0x00
	// Unmute Front out path
	{0x0C, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x0C, 0x0,AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},
	// Unmute Surround (used as HP) out path
	{0x0D, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x0D, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},
	// Unmute C/LFE out path
	{0x0E, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x0E, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},

	{0x0, 0x0, 0x0, 0x0}
};

static verb_t alc880_two_stack_volume[] = {
// format = {nid, direct, command, parameter}
	{0x18, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x40},
	{0x19, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x21},
	{0x1A, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x40},
	{0x1B, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x40},
	{0x1C, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x20},
	{0x14, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0xc0},
	{0x15, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x40},
	{0x16, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x40},
	{0x17, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x40},

	{0x07, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x7000},
// set connection select to line in (default select for this ADC)
	{0x07, 0x0, AC_VERB_SET_CONNECT_SEL, 0x02},
	{0x0C, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
	{0x14, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
	{0x15, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
	{0x16, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
	{0x17, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
// unmute pin widget amp left and right (no gain on this amp)
	{0x1B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
	{0x1A, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
	{0x0D, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
	{0x18, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},

// using front as the path for headphone output
//	{0x0D, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
	// surround path (index 0x01) for headphone output
	{0x10, 0x0, AC_VERB_SET_CONNECT_SEL, 0x02},
	{0x11, 0x0, AC_VERB_SET_CONNECT_SEL, 0x00},
	{0x12, 0x0, AC_VERB_SET_CONNECT_SEL, 0x01},
	{0x13, 0x0, AC_VERB_SET_CONNECT_SEL, 0x00},

	// Amp Indexes: CD = 0x04, Line In 1 = 0x02, Mic 1 = 0x00 & Line In 2 = 0x03
	// unmute CD
	{0x0B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x04 << 8))},
	// unmute Line In
	{0x0B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x02 << 8))},
	// unmute Mic 1
	{0x0B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	// unmute Line In 2 (for PASD boards Mic 2)
	{0x0B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x03 << 8))},
	// Amp Indexes: DAC = 0x01 & mixer = 0x00
	// Unmute Front out path
	{0x0C, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x0C, 0x0,AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},
	// Unmute Surround (used as HP) out path
	{0x0D, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x0D, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},
	// Unmute C/LFE out path
	{0x0E, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x0E, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},

	{0x0, 0x0, 0x0, 0x0}
};


static verb_t alc880_five_stack_volume[] = {
// format = {nid, direct, command, parameter}
// Line In pin widget(nid=0x14) for input
	{0x1A, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x20},
// CD pin widget(nid=0x1C) for input
	{0x1C, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x20},
// Mic1 (rear panel) pin widget(nid=0x18) for input and vref at 80%
	{0x18, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x24},
// Mic2 (front panel) pin widget(nid=0x1B) for input and vref at 80%
	{0x1B, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x24},
// unmute amp left and right
	{0x07, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x7000},
// set connection select to line in (default select for this ADC)
	{0x07, 0x0, AC_VERB_SET_CONNECT_SEL, 0x02},
// unmute front mixer amp left and right and set to max volume Line out
	{0x0C, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
// unmute pin widget amp left and right (no gain on this amp)
	{0x14, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
// five rear and clfe
// unmute rear mixer amp left and right and set to max volume 
	{0x0F, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
// unmute pin widget amp left and right (no gain on this amp)
	{0x17, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
// unmute rear mixer amp left and right and set to max volume clfe
	{0x0E, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
// unmute pin widget amp left and right (no gain on this amp)
	{0x16, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},

// using rear surround as the path for headphone output
	// unmute rear surround mixer amp left and right and set to max volume
	{0x0D, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
	// PASD 3 stack boards use the Mic 2 as the headphone output
	// need to program the selector associated with the Mic 2 pin widget to
	// surround path (index 0x01) for headphone output
	{0x11, 0x0, AC_VERB_SET_CONNECT_SEL, 0x01},
	// unmute pin widget amp left and right (no gain on this amp)
	{0x19, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
	// need to retask the Mic 2 pin widget to output
	{0x19, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x40},

// Unmute input amps (CD, Line In, Mic 1 & Mic 2) for mixer widget(nid=0x0B) to support
	// the input path of analog loopback
	// Note: PASD motherboards uses the Line In 2 as the input for front panel mic (mic 2)
	// Amp Indexes: CD = 0x04, Line In 1 = 0x02, Mic 1 = 0x00 & Line In 2 = 0x03
	// unmute CD
	{0x0B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x04 << 8))},
	// unmute Line In
	{0x0B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x02 << 8))},
	// unmute Mic 1
	{0x0B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	// unmute Line In 2 (for PASD boards Mic 2)
	{0x0B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x03 << 8))},

	// Unmute input amps for the line out paths to support the output path of analog loopback
	// the mixers on the output path has 2 inputs, one from the DAC and one from the mixer
	// Amp Indexes: DAC = 0x01 & mixer = 0x00
	// Unmute Front out path
	{0x0C, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x0C, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},
	// Unmute Surround (used as HP) out path
	{0x0D, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x0D, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},
	// Unmute C/LFE out path
	{0x0E, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x0E, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},
	// Unmute rear Surround out path
	{0x0F, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x0F, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},

	{0x0, 0x0, 0x0, 0x0}
};

static verb_t alc880_uli_five_stack_volume[] = {
// format = {nid, direct, command, parameter}
// Line In pin widget(nid=0x14) for input
	{0x1A, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x20},
// Mic1 (rear panel) pin widget(nid=0x18) for input and vref at 80%
	{0x18, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x24},
// unmute amp left and right
	{0x07, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x7000},
// set connection select to line in (default select for this ADC)
	{0x07, 0x0, AC_VERB_SET_CONNECT_SEL, 0x02},
// unmute front mixer amp left and right and set to max volume Line out
	{0x0C, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
// unmute pin widget amp left and right (no gain on this amp)
	{0x14, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
// five rear and clfe
// unmute rear mixer amp left and right and set to max volume 
	{0x0F, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
// unmute pin widget amp left and right (no gain on this amp)
	{0x17, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
// unmute rear mixer amp left and right and set to max volume clfe
	{0x0E, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
// unmute pin widget amp left and right (no gain on this amp)
	{0x16, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},

// using rear surround as the path for headphone output
	// unmute rear surround mixer amp left and right and set to max volume
	{0x0D, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
	// PASD 3 stack boards use the Mic 2 as the headphone output
	// need to program the selector associated with the Mic 2 pin widget to
	// surround path (index 0x01) for headphone output
	{0x11, 0x0, AC_VERB_SET_CONNECT_SEL, 0x01},
	// unmute pin widget amp left and right (no gain on this amp)
	{0x19, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
	// need to retask the Mic 2 pin widget to output
	{0x19, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x40},

// Unmute input amps (CD, Line In, Mic 1 & Mic 2) for mixer widget(nid=0x0B) to support
	// the input path of analog loopback
	// Note: PASD motherboards uses the Line In 2 as the input for front panel mic (mic 2)
	// Amp Indexes: CD = 0x04, Line In 1 = 0x02, Mic 1 = 0x00 & Line In 2 = 0x03
	// unmute Line In
	{0x0B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x02 << 8))},
	// unmute Mic 1
	{0x0B, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},

	// Unmute input amps for the line out paths to support the output path of analog loopback
	// the mixers on the output path has 2 inputs, one from the DAC and one from the mixer
	// Amp Indexes: DAC = 0x01 & mixer = 0x00
	// Unmute Front out path
	{0x0C, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x0C, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},
	// Unmute Surround (used as HP) out path
	{0x0D, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x0D, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},
	// Unmute C/LFE out path
	{0x0E, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x0E, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},
	// Unmute rear Surround out path
	{0x0F, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x0F, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},

	{0x0, 0x0, 0x0, 0x0}
};

static codec_mixer_t alc880_mixer[] = {
	{"*-*- ALC880 Three Stack Implementation Mixer\n", 19, three_stack_rear_panel_mixer, 28, alc880_three_stack_volume},
{"*-*- ALC880 Three Stack Digital\n", 20, three_stack_rear_panel_digital_mixer, 28, alc880_three_stack_volume},
	{"*-*- ALC880 Five Stack Implementation Mixer\n", 19, five_stack_rear_panel_mixer, 28, alc880_five_stack_volume},
{"*-*- ALC880 Five Stack Digital\n", 20, five_stack_rear_panel_digital_mixer, 28, alc880_five_stack_volume},
{"*-*- ALC880 For Uli Five Stack Digital\n", 16, five_stack_uli_mixer, 24, alc880_uli_five_stack_volume},
{"*-*- ALC880 Three Stack Digital\n", 18, three_stack_digital_mixer, 30, alc880_three_cyber_stack_volume},
{"*-*- ALC880 Three Stack Digital\n", 9, two_stack_rear_panel_mixer, 34, alc880_two_stack_volume},
	{"",0, NULL}
};

static enum_list_t three_stack_rear_panel_capture_source = {
	.num_items = 4,
	.labels = {"Mic-1", "Mic-2", "Line", "CD"},
};

static enum_list_t base_capture_source = {
	.num_items = 2,
	.labels = {"Mic-1", "Line"},
};

static enum_table_t alc880_enum_config[] = {
	// format = {enum list string}, address of the enum list}
	{"ALC880 Three Stack Digital\n", &three_stack_rear_panel_capture_source},
	{"ALC880 Three Stack Implementation Capture Source\n", &three_stack_rear_panel_capture_source},
	{"ALC880 Five Stack Digital\n", &three_stack_rear_panel_capture_source},
	{"ALC880 Five Stack Implementation Capture Source\n", &three_stack_rear_panel_capture_source},
	{"ALC880 For ULi Capture Source\n", &base_capture_source},
	{"ALC880 Three Stack Implementation Capture Source\n", &base_capture_source},
	{"ALC880 Three Stack Implementation Capture Source\n", &base_capture_source},	
	{"", NULL}
};

static nid_list_t alc880_cap_select_table[] = {
	{{0x0, 0x3, 0x2, 0x4}},  // Three Stack Digital Implementation
	{{0x0, 0x3, 0x2, 0x4}},  // Three Stack Implementation
	{{0x0, 0x3, 0x2, 0x4}},  // Five Stack Digital and Three Stack is the same
	{{0x0, 0x3, 0x2, 0x4}},  // Five Stack and Three Stack is the same
	{{0x0, 0x3, 0x2, 0x4}},
	{{0x0, 0x3, 0x2, 0x4}},
	{{0x0, 0x3, 0x2, 0x4}},
};

static nid_converter_t alc880_converter_nids[] = {
	// format = {number of ADC, number of DAC, list of ADC nids {adc0, adc1, adc2, etc}, list of DAC nids {front, rear, clfe, rear_surr, etc.}
	{3, 4, {0x7, 0x8, 0x9}, {0x2, 0x5, 0x4, 0x3}},	// Three Stack Digital Implementation
	{3, 4, {0x7, 0x8, 0x9}, {0x2, 0x5, 0x4, 0x3}},	// Three Stack Implementation
	{3, 4, {0x7, 0x8, 0x9}, {0x2, 0x5, 0x4, 0x3}},	// Five Stack Digital Implementation
	{3, 4, {0x7, 0x8, 0x9}, {0x2, 0x5, 0x4, 0x3}},	// Five Stack Implementation
	{3, 4, {0x7, 0x8, 0x9}, {0x2, 0x5, 0x4, 0x3}},
	{3, 4, {0x7, 0x8, 0x9}, {0x2, 0x3, 0x4, 0x5}},	// Three Stack Implementation
	{3, 4, {0x7, 0x8, 0x9}, {0x2, 0x3, 0x4, 0x5}},	// Three Stack Implementation
	{0, 0, {0x0}, {0x0}},
};

int mixer_codec_setup_alc880(azx_codec_t* azx_codec, snd_card_t* card, u8 table_index)
{
	int err = 0;
	int temp;
	snd_kcontrol_new_t* mixer_controls = alc880_mixer[table_index].mixer_ctrls;
//printk("table_index = 0x%4x.\n", table_index);
	// set codec default volume and connect
	verb_t *def_ctrl = alc880_mixer[table_index].mixer_default_vol;
	for (temp = 0; temp < alc880_mixer[table_index].num_default; temp++) {
		snd_azx_codec_write(azx_codec, def_ctrl[temp].nid, def_ctrl[temp].direct, def_ctrl[temp].command, def_ctrl[temp].parameter);
	}

	// set mixer controller
	for (temp = 0; temp < alc880_mixer[table_index].num_controls; temp++) {
		mixer_controls[temp].index = 0;		//only support one codec
		if ((err = snd_ctl_add(card, snd_ctl_new1(&(mixer_controls[temp]), azx_codec))) < 0) {
			printk("Error adding %s to mixer\n", mixer_controls[temp].name);
		}
//		printk("adding %s to mixer\n", mixer_controls[temp].name);
	}
	if (table_index == 5) {
		snd_azx_codec_write(azx_codec, 0x1, 0, AC_VERB_SET_GPIO_MASK, 2);	//enable GPIO direction control
		snd_azx_codec_write(azx_codec, 0x1, 0, AC_VERB_SET_GPIO_DIRECTION, 2);	// enable output
		snd_azx_codec_write(azx_codec, 0x1, 0, AC_VERB_SET_GPIO_DATA, 2);
	} 

	return err;
}

int pcm_codec_setup_alc880(azx_t *chip, azx_dev_t *azx_dev, u32 num_channels, u32 board_config)
{
	u32 format = azx_dev->format_val;
	u32 stream_tag = azx_dev->stream_tag;
	u8 operation = chip->operation;		// 0 = capture, 1 = playback
	u8 nid = 0;
	u8 channel_id;
	u8 table_index = (board_config & 0xFF0) >> 4;
	u8 front_panel_support = (board_config & 0x4);

	if (operation) {
		// playback
		// stereo & headphone support (both uses the same DAC)
		nid = alc880_converter_nids[table_index].dac[FRONT];
		channel_id = 0x0;
		snd_azx_pcm_setup_codec(chip->azx_codec, nid, stream_tag, channel_id, format);

		// front panel output support
		if (front_panel_support) {
			if (table_index == 5) {
				// set Line-2 H-Phn enable, output enable 
				snd_azx_codec_write(chip->azx_codec, 0x1b, 0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0xc0);
				// set 13h to Front
				snd_azx_codec_write(chip->azx_codec, 0x13, 0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x00);
				// set 14h H-Phn enable, output enable
				snd_azx_codec_write(chip->azx_codec, 0x14, 0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0xc0);
			} else if (table_index == 6 ) {
				nid = 3;
				channel_id = 0x0;
				snd_azx_pcm_setup_codec(chip->azx_codec, nid, stream_tag, channel_id, format);
			} else {
			// currently using REAR SURROUND output as the headphone output
				nid = alc880_converter_nids[table_index].dac[REAR_SURR];
			// headphone out will just decode front left/right (stereo)
				channel_id = 0x0;
				snd_azx_pcm_setup_codec(chip->azx_codec, nid, stream_tag, channel_id, format);
			} 
			
		}

		// multi-channel support with 2 channel samples
		if (((6 == mixer_values.channel_setting) | (8 == mixer_values.channel_setting)) & (2 == num_channels))  {
			// setup the rear output
			nid = alc880_converter_nids[table_index].dac[REAR];
			// mapping stereo to rears
			channel_id = 0x0;
			snd_azx_pcm_setup_codec(chip->azx_codec, nid, stream_tag, channel_id, format);

			// setup the C/LFE output
			nid = alc880_converter_nids[table_index].dac[CLFE];
			// mapping stereo to C/LFE
			channel_id = 0x0;
			snd_azx_pcm_setup_codec(chip->azx_codec, nid, stream_tag, channel_id, format);
		}

		// multi-channel support with 4 channel samples
		else if (((6 == mixer_values.channel_setting) | (8 == mixer_values.channel_setting)) & (4 == num_channels))  {
			// setup the rear output
			nid = alc880_converter_nids[table_index].dac[REAR];
			// set DAC to decode rear
			channel_id = 0x2;
			snd_azx_pcm_setup_codec(chip->azx_codec, nid, stream_tag, channel_id, format);
			// nothing to do with c/lfe...
		}

		// multi-channel support with 6 channel samples
		else if (((6 == mixer_values.channel_setting) | (8 == mixer_values.channel_setting)) & (6 == num_channels))  {
			// setup the rear output
			nid = alc880_converter_nids[table_index].dac[REAR];
			// set DAC to decode rear
			channel_id = 0x4;
			snd_azx_pcm_setup_codec(chip->azx_codec, nid, stream_tag, channel_id, format);
			// setup the C/LFE output
			nid = alc880_converter_nids[table_index].dac[CLFE];
			// set DAC to decode c/lfe
			channel_id = 0x2;
			snd_azx_pcm_setup_codec(chip->azx_codec, nid, stream_tag, channel_id, format);
		}
		
		// multi-channel support with 8 channel samples
		else if ((8 == mixer_values.channel_setting) & (8 == num_channels))  {
			// setup the rear output
			nid = alc880_converter_nids[table_index].dac[REAR];
			// set DAC to decode rear
			channel_id = 4;
			snd_azx_pcm_setup_codec(chip->azx_codec, nid, stream_tag, channel_id, format);

			// setup the rear output
			nid = alc880_converter_nids[table_index].dac[REAR_SURR];
			// set DAC to decode rear
			channel_id = 0x6;
			snd_azx_pcm_setup_codec(chip->azx_codec, nid, stream_tag, channel_id, format);
			// setup the C/LFE output
			nid = alc880_converter_nids[table_index].dac[CLFE];
			// set DAC to decode c/lfe
			channel_id = 0x2;
			snd_azx_pcm_setup_codec(chip->azx_codec, nid, stream_tag, channel_id, format);
		}
#if 0
		else if ((mixer_values.channel_setting == 2) && (table_index == 5)) {
			nid = 3;
			channel_id = 0x0;
			snd_azx_pcm_setup_codec(chip->azx_codec, nid, stream_tag, channel_id, format);
			nid = 4;
			snd_azx_pcm_setup_codec(chip->azx_codec, nid, stream_tag, channel_id, format);
		}
#endif
		// for digital output
		if (mixer_values.current_mux_digital_select_val) {
			channel_id = 0x0;
			nid = 0x06;
			snd_azx_pcm_setup_codec(chip->azx_codec, nid, stream_tag, channel_id, format);
		}

	} else {
		// capture
		nid = alc880_converter_nids[table_index].adc[CAP0];
		channel_id = 0x0;
		snd_azx_pcm_setup_codec(chip->azx_codec, nid, stream_tag, channel_id, format);
	}
	return 0;
}

int azx_alc880_codec_driver_create(u32 board_config)
{
	u8 config_val = (board_config & 0x03);

	if (1 == config_val) {
		// board has a five stack back panel
		// set the value of channel to 6
		mixer_values.channel_setting = 6;
	} else {
		// board only has a 3 stack back panel
		// set the value of channel to 2 as default
		mixer_values.channel_setting = 2;
	}

	return 0;
}

int azx_alc880_codec_driver_free(void)
{
	return 0;
}

void patch_alc880(azx_codec_t *azx_codec) 
{
	u8 config_index = (azx_codec->board_config & 0xFF0) >> 4;

	azx_codec->cap_select_table = &alc880_cap_select_table[config_index];
	codec_callback_info.pcm_codec_setup = pcm_codec_setup_alc880;
	codec_callback_info.mixer_codec_setup = mixer_codec_setup_alc880;
	codec_callback_info.codec_driver_free = azx_alc880_codec_driver_free;
	codec_callback_info.codec_driver_create = azx_alc880_codec_driver_create;
	codec_callback_info.pcm_setup_info = NULL;
	codec_callback_info.model_num = 1;
}
// ************************************** MIXER/CONTROL ***************************
/****
 * snd_azx_mux_info - custom .info function for control that allows retasking
 * of the mic input and line input to outputs supporting 6channel output on
 * certain implementations.
 * @kcontrol: snd_kcontrol_t instance
 * @uinfo: snd_ctl_elem_info_t instance
 *
 * Returns error
****/
static int snd_azx_mux_info(snd_kcontrol_t* kcontrol, snd_ctl_elem_info_t* uinfo)
{
	static char *texts[2] = {
		"2ch", "6ch"
	};

	uinfo->type = SNDRV_CTL_ELEM_TYPE_ENUMERATED;
	uinfo->count = 1;
	uinfo->value.enumerated.items = 2;
	strcpy(uinfo->value.enumerated.name, texts[uinfo->value.enumerated.item]);
//printk("info mux item 0x%4x.\n", uinfo->value.enumerated.item);
	return 0;

}
int snd_azx_codec_setup_multi_channel_selector(azx_codec_t* azx_codec, u8 rear_index, u8 clfe_index)
{
//	u32 cad = 0x02;
	u32 nid;
	u32 direct = 0;
	u32 parameter;
	u8 table_index = (stat_board_config & 0xFF0) >> 4;

	// this is to set the selector for retasking the rear panel jacks to support 6 channel
	// output
	// for rear channel output using Line In 1
	// set select widget connection (nid = 0x12) - to summer node for front NID = 0x0C...offset 0 in connection list
	nid = 0x12;
	parameter = rear_index;
	snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_CONNECT_SEL, parameter);

	// Program PW for Line In pin NID = 0x1A;
	nid = 0x1A;
	// set select pin widget connection - to selector node NID = 0x10...offset 0 in connection list
	parameter = 0x0;
	snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_CONNECT_SEL, parameter);

	// Rear Pin Widget AMP SETUP (LEFT)

	// **** set the summer gain/mute to unmute **************************
//	parameter = 0xB080;
//	snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_AMP_GAIN_MUTE, parameter);

	// program R_PW controls to output instead of default input
	parameter = 0x40;
	snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_PIN_WIDGET_CONTROL, parameter);

	// for Mic1 - retask for center/lfe
	// set select widget connection (nid = 0x10) - to summer node for front NID = 0x0C...offset 0 in connection list
	nid = 0x10;
	parameter = clfe_index;
	snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_CONNECT_SEL, parameter);

	// Program PW for MIC1 pin NID = 18;
	nid = 0x18;

	// set select pin widget connection - to selector node NID = 0x10...offset 0 in connection list
	parameter = 0x0;
	snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_CONNECT_SEL, parameter);

	// Rear Pin Widget AMP SETUP (LEFT)
//	parameter = 0xB080;
//	snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_AMP_GAIN_MUTE, parameter);

	// program R_PW controls to output instead of default input
	parameter = 0x40;
	snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_PIN_WIDGET_CONTROL, parameter);

#if 1
	if (table_index == 5) {
		snd_azx_codec_write(azx_codec, 0x13, direct, AC_VERB_SET_CONNECT_SEL, 0);
		snd_azx_codec_write(azx_codec, 0x1b, direct, AC_VERB_SET_CONNECT_SEL, 0);
//		parameter = 0xB080;
//		snd_azx_codec_write(azx_codec, 0x1b, direct, AC_VERB_SET_AMP_GAIN_MUTE, parameter);
		snd_azx_codec_write(azx_codec, 0x1b, direct, AC_VERB_SET_PIN_WIDGET_CONTROL, 0xc0);

	}
#endif
//	printk(" 12 select 0x%8x.\n", snd_azx_codec_write(azx_codec, 0x12, direct, AC_VERB_GET_CONNECT_SEL, 0));
	return 0;	
}

/****
 * snd_azx_mux_get - custom .get function for control that allows retasking
 * of the mic input and line input to outputs supporting 6channel output on
 * certain implementations.
 * @kcontrol: snd_kcontrol_t instance
 * @ucontrol: snd_ctl_elem_value_t instance
 *
 * Returns error
****/
static int snd_azx_mux_get(snd_kcontrol_t* kcontrol, snd_ctl_elem_value_t* ucontrol)
{
	ucontrol->value.enumerated.item[0] = mixer_values.current_mux_select_val;
//printk("get mux item 0x%4x.\n", mixer_values.current_mux_select_val);
	return 0;
}

static int snd_azx_mux_put(snd_kcontrol_t* kcontrol, snd_ctl_elem_value_t* ucontrol)
{
	azx_codec_t* azx_codec = _snd_kcontrol_chip(kcontrol);
	u32 nid = 0x07;
//	u32 cad = 0x02;
	u32 direct = 0x0;
	u32 parameter = 0;
	u8 table_index = (stat_board_config & 0xFF0) >> 4;
	
	int changed;
	mixer_values.current_mux_select_val = ucontrol->value.enumerated.item[0];
//printk("put mux item 0x%4x.\n", mixer_values.current_mux_select_val);
	switch (ucontrol->value.enumerated.item[0]) {
		case 0 :
			// set the path ways for 2 channel output
			// need to set the codec line out and mic 1 pin widgets to inputs
			// set pin widget 1Ah (line in) for input
			parameter = 0x20;
			nid = 0x1A;
			snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_PIN_WIDGET_CONTROL, parameter);

			if (table_index == 5) {
			nid = 0x1B;
			snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_PIN_WIDGET_CONTROL, parameter);
//			snd_azx_codec_amp_put(azx_codec, 0x1B, 0x80, 0x80, OUTPUT, 0);
			}
			// set pin widget 18h (mic1) for input
			// for mic you need to also enable the vref
			nid = 0x18;
			parameter = 0x24;
			snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_PIN_WIDGET_CONTROL, parameter);

			// need to mute the output
			// for Line In PW
			snd_azx_codec_amp_put(azx_codec, 0x1A, 0x80, 0x80, OUTPUT, 0);
			// for Mic1 PW
			snd_azx_codec_amp_put(azx_codec, 0x18, 0x80, 0x80, OUTPUT, 0);

			// set variable to let pcm engine know that it need to setup for multichannel
			mixer_values.channel_setting = 2;

			parameter = mixer_values.current_mux_select_val;
			break;
		case 1 :
			// need to set the codec line out and mic 1 pin widgets to outputs
			// set pin widget 1Ah (line in) for output
			parameter = 0x40;
			nid = 0x1A;
			snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_PIN_WIDGET_CONTROL, parameter);

			// set pin widget 18h (mic1) for output
			nid = 0x18;
			snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_PIN_WIDGET_CONTROL, parameter);

			// need to unmute the output
			// for Line In PW
			snd_azx_codec_amp_put(azx_codec, 0x1A, 0x00, 0x00, OUTPUT, 0x00);
			
			// for Mic1 PW
			snd_azx_codec_amp_put(azx_codec, 0x18, 0x00, 0x00, OUTPUT, 0x00);
			
			// set variable to let pcm engine know that it need to setup for multichannel
			mixer_values.channel_setting = 6;
			
			// set the selector for retasking inputs to outputs
			if (table_index == 5) {
				nid = 0x1B;
				snd_azx_codec_amp_put(azx_codec, 0x1B, 0x0, 0x0, OUTPUT, 0);
				snd_azx_codec_setup_multi_channel_selector(azx_codec, 0x01, 0x02);
			} else {
				snd_azx_codec_setup_multi_channel_selector(azx_codec, 0x03, 0x02);
			}
			
			parameter = mixer_values.current_mux_select_val;
			break;
	}

	changed = 1;

	return changed;
}

// 
static int snd_azx_mux1_info(snd_kcontrol_t* kcontrol, snd_ctl_elem_info_t* uinfo)
{
	static char *texts[2] = {
		"6ch", "8ch"
	};

	uinfo->type = SNDRV_CTL_ELEM_TYPE_ENUMERATED;
	uinfo->count = 1;
	uinfo->value.enumerated.items = 2;
	strcpy(uinfo->value.enumerated.name, texts[uinfo->value.enumerated.item]);
	return 0;
}

static int snd_azx_mux1_get(snd_kcontrol_t* kcontrol, snd_ctl_elem_value_t* ucontrol)
{
	ucontrol->value.enumerated.item[0] = mixer_values.current_mux_select_val;
	return 0;
}

static int snd_azx_mux1_put(snd_kcontrol_t* kcontrol, snd_ctl_elem_value_t* ucontrol)
{
	azx_codec_t* azx_codec = _snd_kcontrol_chip(kcontrol);
	u32 nid = 0x07;
	u32 direct = 0x0;
	u32 parameter = 0;

	int changed;
	mixer_values.current_mux_select_val = ucontrol->value.enumerated.item[0];
	switch (ucontrol->value.enumerated.item[0]) {
		case 0 :
			// set the path ways for 6 channel output
			// need to set the codec line out and mic 1 pin widgets to inputs
			// set pin widget 1Ah (line in) for input
			parameter = 0x20;
			nid = 0x1A;
			snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_PIN_WIDGET_CONTROL, parameter);


			// need to mute the output
			// for Line In PW
			snd_azx_codec_amp_put(azx_codec, 0x1A, 0x80, 0x80, OUTPUT, 0);

			// set variable to let pcm engine know that it need to setup for multichannel
			mixer_values.channel_setting = 6;

			parameter = mixer_values.current_mux_select_val;
			break;
		case 1 :
			// need to set the codec line out and mic 1 pin widgets to outputs
			// set pin widget 1Ah (line in) for output
			parameter = 0x40;
			nid = 0x1A;
			snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_PIN_WIDGET_CONTROL, parameter);

			// need to unmute the output
			// for Line In PW
			snd_azx_codec_amp_put(azx_codec, 0x1A, 0x00, 0x00, OUTPUT, 0x00);
			
			// set variable to let pcm engine know that it need to setup for multichannel
			mixer_values.channel_setting = 8;

			// set the selector for retasking inputs to outputs
//			snd_azx_codec_setup_multi_channel_selector(azx_codec, 0x03, 0x02);

	// output
	// for surround channel output using Line In 1
	// set select widget connection (nid = 0x12) - to summer node for front NID = 0x0C...offset 0 in connection list
	nid = 0x12;
	parameter = 1;
	snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_CONNECT_SEL, parameter);

	// Program PW for Line In pin NID = 0x1A;
	nid = 0x1A;
	// set select pin widget connection - to selector node NID = 0x10...offset 0 in connection list
	parameter = 0x0;
	snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_CONNECT_SEL, parameter);

	// Rear Pin Widget AMP SETUP (LEFT)

	// **** set the summer gain/mute to unmute **************************
	parameter = 0xB000; //0xB080
	snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_AMP_GAIN_MUTE, parameter);

	// program R_PW controls to output instead of default input
//	parameter = 0x40;
//	snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_PIN_WIDGET_CONTROL, parameter);

			parameter = mixer_values.current_mux_select_val;
			break;
	}

	changed = 1;

	return changed;
}

static int snd_azx_mux_digital_info(snd_kcontrol_t* kcontrol, snd_ctl_elem_info_t* uinfo)
{
	static char *texts[2] = {
		"Off", "On"
	};

	uinfo->type = SNDRV_CTL_ELEM_TYPE_ENUMERATED;
	uinfo->count = 1;
	uinfo->value.enumerated.items = 2;
	strcpy(uinfo->value.enumerated.name, texts[uinfo->value.enumerated.item]);
	return 0;

}

static int snd_azx_mux_digital_get(snd_kcontrol_t* kcontrol, snd_ctl_elem_value_t* ucontrol)
{
	ucontrol->value.enumerated.item[0] = mixer_values.current_mux_digital_select_val;

	return 0;
}

static int snd_azx_mux_digital_put(snd_kcontrol_t* kcontrol, snd_ctl_elem_value_t* ucontrol)
{
	azx_codec_t* azx_codec = _snd_kcontrol_chip(kcontrol);


	u32 nid = 0x06;
//	u32 cad = 0x02;
	u32 direct = 0x0;
	u32 parameter = 0;

	int changed;
	mixer_values.current_mux_digital_select_val = ucontrol->value.enumerated.item[0];

	switch (ucontrol->value.enumerated.item[0]) {
		case 0 :
			parameter = 0x0;
			snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_DIGI_CONVERT_1, parameter);
			break;
		case 1 :
			parameter = 0x01;
			snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_DIGI_CONVERT_1, parameter);
			break;
	}

	changed = 1;

	return changed;
}


// ALC260
// example of what nid_converter_t will contain
// example of what enum_list_t will contain
static enum_list_t alc260_base_capture_source = {
	.num_items = 3,
	.labels = {"Mic-1", "Line", "CD"},
};

// example of what enum_table_t will contain
static enum_table_t alc260_enum_config[] = {
	// format = {enum list string}, address of the enum list}
	{"ALC260 Implementation Capture Source\n", &alc260_base_capture_source},
	{"ALC260 Implementation Capture Source\n", &alc260_base_capture_source},
	{"", NULL}
};

// example of what nid_list_t will contain
static nid_list_t alc260_cap_select_table[] = {
	{{0x0, 0x2, 0x4}},  // Three Stack Implementation
	{{0x0, 0x2, 0x4}},  // Three Stack Implementation
};

static nid_converter_t alc260_converter_nids[] = {
	// format = {number of ADC, number of DAC, list of ADC nids {adc0, adc1, adc2, etc}, list of DAC nids {front, rear, clfe, rear_surr, etc.}
	{2, 1, {0x4, 0x5}, {0x2}},	// Three Stack Implementation
	{2, 1, {0x4, 0x5}, {0x2}},	// Three Stack Implementation
};

snd_kcontrol_new_t alc260_three_stack_rear_panel_mixer[] = {
//	AZX_CODEC_VOLUME("control name here", nid, index, direction, max_val, type)
	AZX_CODEC_VOLUME("PCM Playback Volume", 0x08, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("PCM Playback Switch", 0x0F, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("CD Playback Volume", 0x07, 0x04, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("CD Playback Switch", 0x07, 0x04, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Line Playback Volume", 0x07, 0x02, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Line Playback Switch", 0x07, 0x02, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Mic-1 Playback Volume", 0x07, 0x0, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Mic-1 Playback Switch", 0x07, 0x0, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Headphone Playback Volume", 0x09, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("Headphone Playback Switch", 0x10, 0x0, OUTPUT, 0x01, MUTE),
	// 10th entry
	AZX_CODEC_VOLUME("Capture Volume", 0x04, 0x0, INPUT, 0x23, VOLUME),
	AZX_CODEC_ENUM("Capture Source", 0x04, 0x0),
};

// HP 
snd_kcontrol_new_t alc260_hp_mixer[] = {
//	AZX_CODEC_VOLUME("control name here", nid, index, direction, max_val, type)
	AZX_CODEC_VOLUME("PCM Playback Volume", 0x08, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("PCM Playback Switch", 0x015, 0x0, OUTPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("CD Playback Volume", 0x07, 0x04, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("CD Playback Switch", 0x07, 0x04, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Line Playback Volume", 0x07, 0x02, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Line Playback Switch", 0x07, 0x02, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Mic-1 Playback Volume", 0x07, 0x0, INPUT, 0x41, VOLUME),
	AZX_CODEC_MUTE("Mic-1 Playback Switch", 0x07, 0x0, INPUT, 0x01, MUTE),
	AZX_CODEC_VOLUME("Headphone Playback Volume", 0x09, 0x0, OUTPUT, 0x40, VOLUME),
	AZX_CODEC_MUTE("Headphone Playback Switch", 0x10, 0x0, OUTPUT, 0x01, MUTE),
	// 10th entry
	AZX_CODEC_VOLUME("Capture Volume", 0x04, 0x0, INPUT, 0x23, VOLUME),
	AZX_CODEC_ENUM("Capture Source", 0x04, 0x0),
};

static verb_t alc260_default_volume[] = {
// format = {nid, direct, command, parameter}
// Line In pin widget(nid=0x14) for input
	{0x14, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x20},
// CD pin widget(nid=0x1C) for input
	{0x16, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x20},
// Mic1 (rear panel) pin widget(nid=0x18) for input and vref at 80%
	{0x12, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x24},
// Mic2 (front panel) pin widget(nid=0x1B) for input and vref at 80%
	{0x13, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x24},
// unmute amp left and right
	{0x04, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x7000},
// set connection select to line in (default select for this ADC)
	{0x04, 0x0, AC_VERB_SET_CONNECT_SEL, 0x02},
// unmute front mixer amp left and right and set to max volume
	{0x08, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
// unmute pin widget amp left and right (no gain on this amp)
	{0x0F, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
	{0x09, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
// unmute pin widget amp left and right (no gain on this amp)
	{0x10, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
// Amp Indexes: CD = 0x04, Line In 1 = 0x02, Mic 1 = 0x00 & Line In 2 = 0x03
// unmute CD
	{0x07, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x04 << 8))},
// unmute Line In
	{0x07, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x02 << 8))},
// unmute Mic 1
	{0x07, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
// Amp Indexes: DAC = 0x01 & mixer = 0x00
// Unmute Front out path
	{0x08, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x08, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},
// Unmute Headphone out path
	{0x09, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x09, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},
	{0x0, 0x0, 0x0, 0x0}
};

// HP 
static verb_t alc260_hp_volume[] = {
// format = {nid, direct, command, parameter}
// Line In pin widget(nid=0x14) for input
	{0x14, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x20},
// CD pin widget(nid=0x1C) for input
	{0x16, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x20},
// Mic1 (rear panel) pin widget(nid=0x18) for input and vref at 80%
	{0x12, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x24},
// Mic2 (front panel) pin widget(nid=0x1B) for input and vref at 80%
	{0x13, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x24},
	{0x15, 0x0, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x40},
// unmute amp left and right
	{0x04, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x7000},
// set connection select to line in (default select for this ADC)
	{0x04, 0x0, AC_VERB_SET_CONNECT_SEL, 0x02},
// unmute front mixer amp left and right and set to max volume
	{0x08, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
// unmute pin widget amp left and right (no gain on this amp)
//	{0x0F, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
	{0x15, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
	{0x09, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0xB040},
// unmute pin widget amp left and right (no gain on this amp)
	{0x10, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, 0x0B000},
// Amp Indexes: CD = 0x04, Line In 1 = 0x02, Mic 1 = 0x00 & Line In 2 = 0x03
// unmute CD
	{0x07, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x04 << 8))},
// unmute Line In
	{0x07, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x02 << 8))},
// unmute Mic 1
	{0x07, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
// Amp Indexes: DAC = 0x01 & mixer = 0x00
// Unmute Front out path
	{0x08, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x08, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},
// Unmute Headphone out path
	{0x09, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x00 << 8))},
	{0x09, 0x0, AC_VERB_SET_AMP_GAIN_MUTE, (0x7000 | (0x01 << 8))},
	{0x0, 0x0, 0x0, 0x0}
};

static codec_mixer_t alc260_mixer[] = {
	{"*-*- ALC260 Implementation Mixer\n", 12,  alc260_three_stack_rear_panel_mixer, 17, alc260_default_volume},
	{"*-*- ALC260 for HP Mixer\n", 12,  alc260_hp_mixer, 18, alc260_hp_volume},
	{"",0, NULL, 0, NULL}
};

int pcm_codec_setup_alc260(azx_t *chip, azx_dev_t *azx_dev, u32 num_channels, u32 board_config)
{
	u32 format = azx_dev->format_val;
	u32 stream_tag = azx_dev->stream_tag;
	u8 operation = chip->operation;		// 0 = capture, 1 = playback
	u8 nid = 0;
	u8 channel_id;
	u8 table_index = (board_config & 0xFF0) >> 4;

	if (operation) {
		// playback
		// stereo & headphone support (both uses the same DAC)
		nid = alc260_converter_nids[table_index].dac[FRONT];
		channel_id = 0x0;
		snd_azx_pcm_setup_codec(chip->azx_codec, nid, stream_tag, channel_id, format);
	} else {
		// capture
		nid = alc260_converter_nids[table_index].adc[CAP0];
		channel_id = 0x0;
		snd_azx_pcm_setup_codec(chip->azx_codec, nid, stream_tag, channel_id, format);
	}
	return 0;
}

int mixer_codec_setup_alc260(azx_codec_t* azx_codec, snd_card_t* card, u8 table_index)
{
	int err = 0;
	int temp;
	snd_kcontrol_new_t* mixer_controls = alc260_mixer[table_index].mixer_ctrls;
	
	// set codec default volume and connect
	verb_t *def_ctrl = alc260_mixer[table_index].mixer_default_vol;
	for (temp = 0; temp < alc260_mixer[table_index].num_default; temp++) {
		snd_azx_codec_write(azx_codec, def_ctrl[temp].nid, def_ctrl[temp].direct, def_ctrl[temp].command, def_ctrl[temp].parameter);
	}

	// set mixer controller
	for (temp = 0; temp < alc260_mixer[table_index].num_controls; temp++) {
		if ((err = snd_ctl_add(card, snd_ctl_new1(&(mixer_controls[temp]), azx_codec))) < 0) {
			printk("Error adding %s to mixer\n", three_stack_rear_panel_mixer[temp].name);
		}
	}
snd_azx_codec_write(azx_codec, 0x0, 0, 0x716, 8);
	return err;
}
int azx_alc260_codec_driver_create(u32 board_config)
{
//printk("alc260.\n");
	mixer_values.channel_setting = 2;
	return 0;
}

int azx_alc260_codec_driver_free(void)
{
	return 0;
}

void patch_alc260(azx_codec_t *azx_codec) 
{
	azx_codec->cap_select_table = &alc260_cap_select_table[0];
	codec_callback_info.pcm_codec_setup = pcm_codec_setup_alc260;
	codec_callback_info.mixer_codec_setup = mixer_codec_setup_alc260;
	codec_callback_info.codec_driver_free = azx_alc260_codec_driver_free;
	codec_callback_info.codec_driver_create = azx_alc260_codec_driver_create;
	codec_callback_info.pcm_setup_info = NULL;
	codec_callback_info.model_num = 0;
}

/*
 *
 */
// mixer control configurations

int snd_azx_codec_amp_put(azx_codec_t* azx_codec, u8 nid, u8 left_val, u8 right_val, u8 direction, u32 index)
{
	u32 direct = 0;
	u32 l_parameter = 0;
	u32 r_parameter = 0;
	u32 parameter = 0;
	int changed = 0;

	if (direction) {
		// output direction = 1
		l_parameter = 0xA000;
		r_parameter = 0x9000;
	} else {
		// input direction = 0
		l_parameter = 0x6000;
		r_parameter = 0x5000;
	}

	// left volume
	parameter = l_parameter | left_val | (index << 8);
//	printk("Setting left channel to new value...[%lx]\n", (unsigned long)parameter);
	snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_AMP_GAIN_MUTE, parameter);
	
	// right volume
	parameter = r_parameter | right_val | (index << 8);
//	printk("Setting right channel to new value...[%lx]\n", (unsigned long)parameter);
	snd_azx_codec_write(azx_codec, nid, direct, AC_VERB_SET_AMP_GAIN_MUTE, parameter);

	changed = 1;

	return changed;
}

int snd_azx_codec_amp_get(azx_codec_t* azx_codec, u8 nid, u8* left_val, u8* right_val, u8 direction, u32 index)
{
	u32 direct = 0;
	u32 l_parameter = 0;
	u32 r_parameter = 0;
	u32 parameter = 0;

	// need to call a function to get the val from the codec
	if (direction) {
		// output direction = 1
		l_parameter = 0xA000;
		r_parameter = 0x8000;
	} else {
		// input direction = 0
		l_parameter = 0x2000;
		r_parameter = 0x0000;
	}

//	printk("Getting the left channel value...\n");

	// get left value
	parameter = l_parameter | index;
	*left_val = snd_azx_codec_read(azx_codec, nid, direct, AC_VERB_GET_AMP_GAIN_MUTE, parameter) & 0xff;

//	printk("Getting the right channel value...\n");
	parameter = r_parameter | index;
	*right_val = snd_azx_codec_read(azx_codec, nid, direct, AC_VERB_GET_AMP_GAIN_MUTE, parameter) & 0xff;

	return 0;
}

static int snd_azx_codec_info_volume(snd_kcontrol_t * kcontrol, snd_ctl_elem_info_t * uinfo)
{
	int max_val = (kcontrol->private_value >> 13) & 0xFF;
	int type = (kcontrol->private_value >> 21) & 0x01;

	uinfo->type = type == 0 ? SNDRV_CTL_ELEM_TYPE_BOOLEAN : SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 2;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = max_val;
	return 0;
}

int snd_azx_codec_info_mute(snd_kcontrol_t* kcontrol, snd_ctl_elem_info_t* uinfo)
{
	int max_val = (kcontrol->private_value >> 13) & 0xFF;
	int type = (kcontrol->private_value >> 21) & 0x01;

	uinfo->type = type == 0 ? SNDRV_CTL_ELEM_TYPE_BOOLEAN : SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 1;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = max_val;
	return 0;
}

int snd_azx_info_enum(snd_kcontrol_t* kcontrol, snd_ctl_elem_info_t* uinfo)
{
	enum_list_t* capture_list = NULL;
	azx_codec_t* azx_codec = _snd_kcontrol_chip(kcontrol);
//	azx_t* chip = snd_kcontrol_chip(kcontrol);
	u8 index = (azx_codec->board_config & 0xFF0) >> 4;

	capture_list = alc880_enum_config[index].enum_list;

	if (capture_list) {
		int num_items = capture_list->num_items;

		uinfo->type = SNDRV_CTL_ELEM_TYPE_ENUMERATED;
		uinfo->count = 1;
		uinfo->value.enumerated.items = num_items;
		strcpy(uinfo->value.enumerated.name, capture_list->labels[uinfo->value.enumerated.item]);
//printk("info value = 0x%4x, %s.\n", uinfo->value.enumerated.item, capture_list->labels[uinfo->value.enumerated.item]);
	}

	return 0;
}
 
int snd_azx_enum_get(snd_kcontrol_t* kcontrol, snd_ctl_elem_value_t* ucontrol)
{
	azx_codec_t*  azx_codec = _snd_kcontrol_chip(kcontrol);

	ucontrol->value.integer.value[0] = azx_codec->current_mux_select_val;
//printk("get val = 0x%4x.\n", azx_codec->current_mux_select_val);
	return 0;
}

int snd_azx_enum_put(snd_kcontrol_t* kcontrol, snd_ctl_elem_value_t* ucontrol)
{
	azx_codec_t* azx_codec = _snd_kcontrol_chip(kcontrol);
//	u32 cad = chip->cad;
	u32 direct = 0;
	u32 nid = kcontrol->private_value & 0xFF;

	u8 index = ucontrol->value.integer.value[0];

	u32 parameter;

//	parameter = alc880_cap_select_table[table_index].cap_select_nid[index];
	parameter = azx_codec->cap_select_table->cap_select_nid[index];

	// set connection select for AIW to index 0x02 (nid 0x1A)
	snd_azx_codec_read(azx_codec, nid, direct, AC_VERB_SET_CONNECT_SEL, parameter);

	azx_codec->current_mux_select_val = index;
//printk("put index = 0x%4x.\n", index);
	return 1;
}

static int snd_azx_codec_get_volume(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	azx_codec_t *azx_codec = _snd_kcontrol_chip(kcontrol);
	u32 nid = kcontrol->private_value & 0xFF;
	u32 index = (kcontrol->private_value & 0x0F00) >> 8;
	u8 direction = (kcontrol->private_value & 0x1000) >> 12;

	u8 left_val;
	u8 right_val;;

	snd_azx_codec_amp_get(azx_codec, nid, &left_val, &right_val, direction, index);

	//strip the mute bit from the returned value
	left_val = left_val & 0x7F;
	right_val = right_val & 0x7F;

	ucontrol->value.integer.value[0] = left_val; // get left channel gain
	ucontrol->value.integer.value[1] = right_val; // get right channel gain
	return 0;
}                                                                                                                                                                                                                                                                                                            

static int snd_azx_codec_put_volume(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	azx_codec_t *azx_codec = _snd_kcontrol_chip(kcontrol);
	int changed = 0;
	u8 left_val;
	u8 right_val;

	u32 nid = kcontrol->private_value & 0xFF;
	u32 index = (kcontrol->private_value & 0x0F00) >> 8;
	u8 direction = (kcontrol->private_value & 0x1000) >> 12;


	snd_azx_codec_amp_get(azx_codec, nid, &left_val, &right_val, direction, index);
	if (left_val & 0x80) {
		// value muted
		// note: I am not implementing a mute for left and right
		left_val = (u8)ucontrol->value.integer.value[0] | 0x80;
		right_val = (u8)ucontrol->value.integer.value[1] | 0x80;
	} else {
		left_val = (u8)ucontrol->value.integer.value[0];
		right_val = (u8)ucontrol->value.integer.value[1];
	}
	changed = snd_azx_codec_amp_put(azx_codec, nid, left_val, right_val, direction, index);

	return changed;
}
                
int snd_azx_codec_mute_get(snd_kcontrol_t* kcontrol, snd_ctl_elem_value_t* ucontrol)
{

	azx_codec_t *azx_codec = _snd_kcontrol_chip(kcontrol);

	u32 nid = kcontrol->private_value & 0xFF;
	u32 index = (kcontrol->private_value & 0x0F00) >> 8;
	u8 direction = (kcontrol->private_value & 0x1000) >> 12;

	u8 left_val;
	u8 right_val;
	u8 mute;

	snd_azx_codec_amp_get(azx_codec, nid, &left_val, &right_val, direction, index);

	// strip the volume from the mute value
	mute = (left_val & 0x80) >> 7;

	// note: mute is an inverted value for ALSA
	mute = ~mute & 0x01;

	ucontrol->value.integer.value[0] = mute; // get left channel gain

	return 0;
}
                      
int snd_azx_codec_mute_put(snd_kcontrol_t* kcontrol, snd_ctl_elem_value_t* ucontrol)
{

	azx_codec_t *azx_codec = _snd_kcontrol_chip(kcontrol);
	int changed = 0;
	u8 left_val = (u8)ucontrol->value.integer.value[0];
	u8 right_val = (u8)ucontrol->value.integer.value[1];

	u32 nid = kcontrol->private_value & 0xFF;
	u32 index = (kcontrol->private_value & 0x0F00) >> 8;
	u8 direction = (kcontrol->private_value & 0x1000) >> 12;
	u8 mute = (u8)ucontrol->value.integer.value[0];

	snd_azx_codec_amp_get(azx_codec, nid, &left_val, &right_val, direction, index);

	left_val = left_val & 0x7F;
	right_val = right_val & 0x7F;

	if (mute) {
		left_val = left_val & 0x7F;
		right_val = right_val & 0x7F;
	} else {
		left_val = left_val | 0x80;
		right_val = right_val | 0x80;
	}

	changed = snd_azx_codec_amp_put(azx_codec, nid, left_val, right_val, direction, index);
//if ((nid == 0x17) | (nid == 0x16) |(nid==0x15) | (nid==0x14)) {
//	snd_azx_codec_amp_get(azx_codec, nid, &left_val, &right_val, direction, index);
//	printk("0x%4x 0x%4x, 0x%4x.\n", nid, left_val, right_val);  	
//}
	return changed;
}


static inline int printable(unsigned int x)
{
	x &= 0xff;
	if (x < ' ' || x >= 0x71) {
		if (x <= 0x89)
			return x - 0x71 + 'A';
		return '?';
	}
	return x;
}

int snd_azx_codec_get_name(azx_codec_t *azx_codec, unsigned int id, char *name)
{
	const azx_codec_id_t *pid;

	sprintf(name, "0x%x %c%c%c", id,
		printable(id >> 24),
		printable(id >> 16),
		printable(id >> 8));
	for (pid = snd_azx_codec_id_vendors; pid->id; pid++)
		if (pid->id == (id & pid->mask)) {
			strcpy(name, pid->name);
			goto __vendor_ok;
		}
	return -EINVAL;

      __vendor_ok:
	for (pid = snd_azx_codec_ids; pid->id; pid++) {
		if (pid->id == (id & pid->mask)) {
			strcat(name, " ");
			strcat(name, pid->name);
			if (pid->mask != 0xffffffff)
				sprintf(name + strlen(name), " rev %d", id & ~pid->mask);
			if (pid->patch)
				pid->patch(azx_codec);
			return 0;
		}
	}
	sprintf(name + strlen(name), " id %x", id & 0xff);
	return -EINVAL;
}

/*
 * proc interface
 */
#if 0
static void snd_azx_codec_proc_read_main(azx_codec_t *azx_codec, snd_info_buffer_t * buffer, int subidx)
{
}

static void snd_azx_codec_proc_read(snd_info_entry_t *entry, snd_info_buffer_t * buffer)
{
}

static void snd_azx_codec_proc_regs_read_main(azx_codec_t *azx_codec, snd_info_buffer_t * buffer, int subidx)
{
}
#endif
void snd_azx_codec_proc_done(azx_codec_t * azx_codec)
{
	if (azx_codec->proc_regs) {
		snd_info_unregister(azx_codec->proc_regs);
		azx_codec->proc_regs = NULL;
	}
	if (azx_codec->proc) {
		snd_info_unregister(azx_codec->proc);
		azx_codec->proc = NULL;
	}
}

#if 0
void snd_azx_codec_bus_proc_init(azx_codec_bus_t * bus)
{
	snd_info_entry_t *entry;
	char name[32];

	sprintf(name, "codec97#%d", bus->num);
	if ((entry = snd_info_create_card_entry(bus->card, name, bus->card->proc_root)) != NULL) {
		entry->mode = S_IFDIR | S_IRUGO | S_IXUGO;
		if (snd_info_register(entry) < 0) {
			snd_info_free_entry(entry);
			entry = NULL;
		}
	}
	bus->proc = entry;
}

void snd_azx_codec_bus_proc_done(azx_codec_bus_t * bus)
{
	if (bus->proc) {
		snd_info_unregister(bus->proc);
		bus->proc = NULL;
	}
}

// bus function
static int snd_azx_codec_bus_free(azx_codec_bus_t *bus)
{
	if (bus) {
		snd_azx_codec_bus_proc_done(bus);
		if (bus->pcms)
			kfree(bus->pcms);
		if (bus->private_free)
			bus->private_free(bus);
		snd_magic_kfree(bus);
	}
	return 0;
}

static int snd_azx_codec_bus_dev_free(snd_device_t *device)
{
	azx_codec_bus_t *bus = snd_magic_cast(azx_codec_bus_t, device->device_data, return -ENXIO);
	return snd_azx_codec_bus_free(bus);
}

int snd_azx_codec_bus(snd_card_t * card, azx_codec_bus_t * _bus, azx_codec_bus_t ** rbus)
{
	int err;
	azx_codec_bus_t *bus;
	static snd_device_ops_t ops = {
		.dev_free =	snd_azx_codec_bus_dev_free,
	};

	snd_assert(card != NULL, return -EINVAL);
	snd_assert(_bus != NULL && rbus != NULL, return -EINVAL);
	bus = snd_magic_kmalloc(azx_codec_bus_t, 0, GFP_KERNEL);
	if (bus == NULL)
		return -ENOMEM;
	*bus = *_bus;
	bus->card = card;
	if (bus->clock == 0)
		bus->clock = 48000;
	spin_lock_init(&bus->bus_lock);
	snd_azx_codec_bus_proc_init(bus);
	if ((err = snd_device_new(card, SNDRV_DEV_LOWLEVEL, bus, &ops)) < 0) 		{
		snd_azx_codec_bus_free(bus);
		return err;
	}
	*rbus = bus;
	return 0;
}
#endif

static int snd_azx_codec_free(azx_codec_t *azx_codec)
{
	if (azx_codec) {
		snd_azx_codec_proc_done(azx_codec);
		if (azx_codec->bus)
			azx_codec->bus->codec[azx_codec->num] = NULL;
		if (azx_codec->private_free)
			azx_codec->private_free(azx_codec);
		snd_magic_kfree(azx_codec);
printk("Release azx codec .\n");
	}
	return 0;
}

static int snd_azx_codec_dev_free(snd_device_t *device)
{
	azx_codec_t *azx_codec = snd_magic_cast(azx_codec_t, device->device_data, return -ENXIO);
	return snd_azx_codec_free(azx_codec);
}


/**
 * snd_azx_codec_mixer - create an Codec97 component
 */
int snd_azx_codec_mixer(azx_codec_t * _azx_codec, azx_codec_t ** razx_codec, int board_config)
{
	snd_card_t *card;
	azx_codec_t * azx_codec;
	char name[64];
	int temp, err;
	u8 table_index = (board_config & 0xFF0) >> 4;
	static snd_device_ops_t ops = {
		.dev_free =	snd_azx_codec_dev_free,
	};

	snd_assert(razx_codec != NULL, return -EINVAL);
	*razx_codec = NULL;
//	snd_assert(bus != NULL && _azx_codec != NULL, return -EINVAL);
//	snd_assert(_azx_codec->num < 4 && bus->codec[_azx_codec->num] == NULL, return -EINVAL);
	snd_assert(_azx_codec != NULL, return -EINVAL);
	snd_assert(_azx_codec->num < 4 , return -EINVAL);
	card = _azx_codec->card;
	azx_codec = snd_magic_kcalloc(azx_codec_t, 0, GFP_KERNEL);
	if (azx_codec == NULL)
		return -ENOMEM;
	*azx_codec = *_azx_codec;
//	azx_codec->bus = bus;

//	bus->codec[azx_codec->num] = azx_codec;
	spin_lock_init(&azx_codec->reg_lock);

	azx_codec->revision = snd_azx_codec_read(azx_codec, AC_NODE_ROOT, 0, AC_VERB_PARAMETERS, AC_PRAR_REV_ID);
		
	azx_codec->functiontype = snd_azx_codec_read(azx_codec, AC_NODE_AUDIO_FUNCTION, 0, AC_VERB_PARAMETERS, AC_PRAR_FUNCTION_TYPE);

	if (!(azx_codec->functiontype & 1)) { // is audio function
		printk(" no audio function.\n");
		err = -EINVAL;
		goto __error_azx_mixer;
	}
		
	temp = snd_azx_codec_read(azx_codec, AC_NODE_AUDIO_FUNCTION,0, AC_VERB_PARAMETERS, AC_PRAR_NODE_COUNT);

	azx_codec->node_start = (unsigned char)(temp >> 16) & 0x0ff;
	azx_codec->node_total = (unsigned char)(temp) & 0x0ff;
printk("node1 0x%4x. 0x%4x\n", azx_codec->node_start, azx_codec->node_total);
	azx_codec->rates[AZX_RATES_FRONT_DAC] = SNDRV_PCM_RATE_48000;
	azx_codec->rates[AZX_RATES_SURR_DAC] = SNDRV_PCM_RATE_48000;
	azx_codec->rates[AZX_RATES_LFE_DAC] = SNDRV_PCM_RATE_48000;
	azx_codec->rates[AZX_RATES_ADC] = SNDRV_PCM_RATE_48000;
	azx_codec->rates[AZX_RATES_MIC_ADC] = SNDRV_PCM_RATE_48000;

	if ((err = snd_azx_codec_get_name(azx_codec, azx_codec->id, name)) != 0) {
		printk(" not Realtek's Azalia codec.\n");
		goto __error_azx_mixer;
	}
//	snd_azx_codec_get_name(NULL, azx_codec->id, name);  // azx_codec->id might be changed in the special setup code

	if (card->mixername[0] == '\0') {
		strcpy(card->mixername, name);
	} else {
		if (strlen(card->mixername) + 1 + strlen(name) + 1 <= sizeof(card->mixername)) {
			strcat(card->mixername, ",");
			strcat(card->mixername, name);
		}
	}
	printk("%s.\n", card->mixername);
	if ((err = snd_component_add(card, "Azalia")) < 0) {
		goto __error_azx_mixer;
	}

	if ((err = codec_callback_info.codec_driver_create(board_config)) < 0) {
		printk("Error: codec_driver_create FAILED!\n");
		goto __error_azx_mixer;
	}

	if ((err = codec_callback_info.mixer_codec_setup(azx_codec, card, table_index)) < 0) {
		printk("Error returned from codec_function->mixer_codec_setup: %i\n", err);
		goto __error_azx_mixer;
	}	
//return -EINVAL;
//	snd_azx_codec_proc_init(azx_codec);
	if ((err = snd_device_new(card, SNDRV_DEV_LOWLEVEL, azx_codec, &ops)) < 0) {
		snd_printk("failed snd_device_new.\n");
		goto __error_azx_mixer;
	}
	*razx_codec = azx_codec;

	return 0;

__error_azx_mixer:
	snd_azx_codec_free(azx_codec);
	return err;
}

/*
 * ac97 mixer section
 */
static int __devinit snd_azx_mixer_new(azx_t *chip, int clock)
{
//	azx_codec_bus_t bus, *pbus;
	azx_codec_t azx_codec;
	int err;

//	if (snd_azx_codec_detect(chip) < 0)
//		return -ENXIO;
	
	memset(&azx_codec, 0, sizeof(azx_codec));
	azx_codec.private_data = chip;
	azx_codec.card = chip->card;
//	azx_codec.pci = chip->pci;
	if (clock == 0)		clock = 48000;
	azx_codec.clock = clock;
	azx_codec.num = 0; //i;
	
	// check codec address
	azx_codec.id = snd_azx_codec_read(&azx_codec, AC_NODE_ROOT, 0, AC_VERB_PARAMETERS, AC_PRAR_VENDOR_ID);
		
// salers issue
	if (chip->board_config & 0x1000) {
		azx_codec.id = 0x10ec0860;
	}
//================================================================
	if ((err = snd_azx_codec_mixer(&azx_codec, &chip->azx_codec, chip->board_config)) < 0) {
			return err;
	}
	azx_codec.board_config = chip->board_config;
	/* snd_azx_codec_tune_hardware(chip->azx_codec, azx_codec_quirks); */

	return 0;
}


/*
 * proc interface for register dump
 */

static void snd_azx_proc_read(snd_info_entry_t *entry, snd_info_buffer_t *buffer)
{
	azx_t *chip = snd_magic_cast(azx_t, entry->private_data, return);
	int i;

	for (i = 0; i < 256; i += 4)
		snd_iprintf(buffer, "%02x: %08x\n", i, readl(chip->remap_addr + i));
}

static void __devinit snd_azx_proc_init(azx_t *chip)
{
	snd_info_entry_t *entry;

	if (! snd_card_proc_new(chip->card, "azalia", &entry))
		snd_info_set_text_ops(entry, chip, 1024, snd_azx_proc_read);
}



/*
 * destructor
 */

static int snd_azx_free(azx_t *chip)
{
	// free codec driver
//	chip->codec_function->codec_driver_free();

	if (chip->remap_addr) {
		// stop playback stream
		int i;
		for (i = 0; i < ICH6_MAX_DEV; i++)
			azx_stream_stop(chip, &chip->azx_dev[i]);

		// disable interrupts
		snd_azx_int_disable(chip);
		snd_azx_int_clear(chip);

		/* disable CORB/RIRB */
		azx_free_cmd_io(chip);
		
		/* disable position buffer */
		azx_writel(chip, DPLBASE, 0);
		azx_writel(chip, DPUBASE, 0);

		// wait a little for interrupts to finish
		mdelay(10);

		iounmap((void *)chip->remap_addr);
	}

	if (chip->irq >= 0) {
		printk("Releasing chip->irq...\n");
		free_irq(chip->irq, (void*)chip);
	}

	if (chip->bdl.area) {
		printk("Releasing BDL memory...\n");
		snd_dma_free_pages(&dev_dma, &chip->bdl);
	}

#ifdef USE_POSBUF
	if (chip->posbuf.area)
		snd_dma_free_pages(&dev_dma, &chip->posbuf);
#endif

	if (chip) {
		printk("Releasing chip...\n");
		pci_release_regions(chip->pci);
		pci_disable_device(chip->pci);
//		kfree(chip);
		snd_magic_kfree(chip);
	}

	return 0;
}


//struct azx_sd_info {
//	u32	sd_offset;		// interrupt mask for register 0x24h (Int register)
//	u32	sd_int_sta_mask;	// stream descriptor offset relative to chip->remap_addr
//};

int snd_azx_init_chip(azx_t* chip)
{
	u16 index, output_index, output_count, i;
	azx_dev_t* azx_dev;

	unsigned char tcsel_reg;

	/* Clear bits 0-2 of PCI register TCSEL (at offset 0x44)
	 * TCSEL == Traffic Class Select Register, which sets PCI express QOS
	 * Ensuring these bits are 0 clears playback static on some HD Audio codecs
	 */
	pci_read_config_byte (chip->pci, ICH6_PCIREG_TCSEL, &tcsel_reg);
	pci_write_config_byte(chip->pci, ICH6_PCIREG_TCSEL, tcsel_reg & 0xf8);

	// reset controller
	if (snd_azx_reset(chip) < 0) {
		printk("failed Azalia reset.\n");
		return -EBUSY;
	}


	// initialize each stream (aka device)
	// assign the starting bdl address to each stream (device) and initialize
	// maxium capture and playback is 4 
	for (index = 0; index < ICH6_MAX_DEV; index++) {
		unsigned int off = sizeof(u32) * (index * AZX_MAX_FRAG * 4);
		azx_dev = &chip->azx_dev[index];
		azx_dev->device = index;
		azx_dev->bdl = (u32 *)(chip->bdl.area + off);
		azx_dev->bdl_addr = chip->bdl.addr + off;
#ifdef USE_POSBUF
		azx_dev->posbuf = (volatile u32 *)(chip->posbuf.area + index * 8);
#endif
		
/* offset: SDI0=0x80, SDI1=0xa0, ... SDO3=0x160 */
		azx_dev->sd_addr = chip->remap_addr + (0x20 * index + 0x80);
		/* int mask: SDI0=0x01, SDI1=0x02, ... SDO3=0x80 */
		azx_dev->sd_int_sta_mask = 1 << index;
		/* stream tag: must be non-zero and unique */
		azx_dev->index = index;
//		azx_dev->stream_tag = index + 1;
	}

	index = ICH6_MAX_CAPTURE_DEV;
	output_index = chip->input_streams;
	output_count = 0;
	chip->playback_offset = i = 0x80 + 0x20 * output_index;	//first output stream 
	while ((output_count < chip->output_streams) && (output_count < ICH6_MAX_PLAYBACK_DEV)) {
		azx_dev = &chip->azx_dev[index];
		azx_dev->sd_addr = chip->remap_addr +(0x80 + 0x20 * output_index);
		azx_dev->sd_int_sta_mask = 1 << (u32)output_index;
		index++;
		output_index++;
		output_count++;
	}

	(&chip->azx_dev[SDI0])->stream_tag = 0xA00000;	// ADC for Line In
	(&chip->azx_dev[SDI1])->stream_tag = 0xB00000;	// ADC for Mic1
	(&chip->azx_dev[SDI2])->stream_tag = 0xC00000;	// ADC for CD
	(&chip->azx_dev[SDI3])->stream_tag = 0xD00000;	// ADC not supported in codec
	(&chip->azx_dev[SDO0])->stream_tag = 0x100000;	// DAC for front
	(&chip->azx_dev[SDO1])->stream_tag = 0x100000;	// DAC for rear
	(&chip->azx_dev[SDO2])->stream_tag = 0x100000;	// DAC for c/lfe
	(&chip->azx_dev[SDO3])->stream_tag = 0x100000;	// DAC for rear center L/R (used for hp output)
//	chip->isr_count = 0;
	chip->operation = 0;			// default capture
	chip->current_channel_setting = 2;	// default channel setting to 2
						// will be changed to 6 when retasking jacks

	snd_azx_int_clear(chip);
	snd_azx_int_enable(chip);

	/* initialize the codec command I/O */
	azx_init_cmd_io(chip);

#ifdef USE_POSBUF
	/* program the position buffer */
	azx_writel(chip, DPLBASE, (u32)chip->posbuf.addr);
	azx_writel(chip, DPUBASE, upper_32bit(chip->posbuf.addr));
#endif
	return 0;
}

static int snd_azx_dev_free(snd_device_t *device)
{
	azx_t *chip = snd_magic_cast(azx_t, device->device_data, return -ENXIO);
	return snd_azx_free(chip);
}

/*
 * constructor for chip instance
 */
static int __devinit snd_azx_create(snd_card_t *card,
				      struct pci_dev *pci,
				      azx_t **r_chip)
{
	static snd_device_ops_t ops = {
		.dev_free =	snd_azx_dev_free,
	};
	azx_t *chip;
	int err;
	unsigned short index;
//	unsigned char	index;	// Codec Address

	if ((err = pci_enable_device(pci)) < 0)
		return err;

	chip = snd_magic_kcalloc(azx_t, 0, GFP_KERNEL);
//	chip = kcalloc(1, sizeof(*chip), GFP_KERNEL);
	if (chip == NULL)
		return -ENOMEM;

	spin_lock_init(&chip->reg_lock);
	spin_lock_init(&chip->azx_codec_lock);
	init_MUTEX(&chip->open_mutex);
	chip->card = card;
	chip->pci = pci;
	chip->irq = -1;

	// Get PCI resources
	if ((err = pci_request_regions(pci, "HD audio")) < 0) {
		snd_magic_kfree(chip);
		return err;
	}

	chip->addr = pci_resource_start(pci, 0);
	chip->remap_addr = ioremap_nocache(chip->addr, pci_resource_len(pci,0));
	if (chip->remap_addr == NULL) {
		snd_printk("Azalia space ioremap problem\n");
		goto __error_azx;
	}

	if (request_irq(pci->irq, snd_azx_interrupt, SA_INTERRUPT|SA_SHIRQ, card->shortname, (void *)chip)) {
		snd_printk("unable to grab IRQ %d\n", pci->irq);
		goto __error_azx;
	}
	chip->irq = pci->irq;
	pci_set_master(pci);
	synchronize_irq(chip->irq);

	pci_read_config_word(pci, PCI_SUBSYSTEM_VENDOR_ID, &chip->subsystem_vendor);
	pci_read_config_word(pci, PCI_SUBSYSTEM_ID, &chip->subsystem_device);
printk("sub_vendor = 0x%4x, sub_device = 0x%4x.\n", chip->subsystem_vendor, chip->subsystem_device);
	index = 0;
	while ((alc_mixer_index[index].sub_vendor != 0) ||
		(alc_mixer_index[index].sub_device !=0)) {
		if ((alc_mixer_index[index].sub_vendor == chip->subsystem_vendor) &&
		   (alc_mixer_index[index].sub_device == chip->subsystem_device))	{
			chip->board_config = alc_mixer_index[index].board_config;
			break;
		}
		index++;
	}

	if ((alc_mixer_index[index].sub_vendor == 0) && 
		(alc_mixer_index[index].sub_device == 0))
		chip->board_config = 0;
printk("board config 0x%8x.\n", chip->board_config);

	stat_board_config = chip->board_config;
	index = (unsigned short)(azx_readw(chip, GCAP));
	printk("output 0x%4x, input 0x%4x, bi 0x%4x.\n", index >> 12, (index&0x0f00) >> 8, (index & 0x00f8) >> 3);

	chip->output_streams = (unsigned short)(azx_readw(chip, GCAP) >> 12);
	chip->input_streams = (unsigned short)((azx_readw(chip, GCAP) & 0x0f00) >> 8);
	chip->biput_streams = (unsigned short)((azx_readw(chip, GCAP) & 0x00f8) >> 3);

	/* allocate memory for the BDL for each stream */
	dev_dma.type = SNDRV_DMA_TYPE_DEV;
	dev_dma.dev = snd_dma_pci_data(chip->pci);
	if ((err = snd_dma_alloc_pages(&dev_dma, PAGE_SIZE, &chip->bdl)) < 0) {
		snd_printk("cannot allocate BDL\n");
		goto __error_azx;
	}
#ifdef USE_POSBUF
	/* allocate memory for the position buffer */
	if ((err = snd_dma_alloc_pages(&dev_dma, ICH6_MAX_DEV * 8, &chip->posbuf)) < 0) {
		snd_printk("cannot allocate posbuf\n");
		goto __error_azx;
	}
#endif

	if ((err = azx_alloc_cmd_io(chip)) < 0) {
		snd_printk("cannot allocate CORB/RIRB\n");
		goto __error_azx;
	}

	if ((err = snd_azx_init_chip(chip)) < 0) {
		printk(" failed init chip.\n");
		goto __error_azx;
	}

	if ((err = snd_device_new(card, SNDRV_DEV_LOWLEVEL, chip, &ops)) < 0) {
		printk("failed added device new.\n");
		goto __error_azx;
	}

	snd_card_set_dev(card, &pci->dev);

	*r_chip = chip;


	return 0;

__error_azx:
	snd_azx_free(chip);
	return -EBUSY;
}


static int __devinit snd_azx_probe(struct pci_dev *pci,
				     const struct pci_device_id *pci_id)
{
	static int dev;
	snd_card_t *card;
	azx_t *chip;
	unsigned char revision;
	int err;

	if (dev >= SNDRV_CARDS)
		return -ENODEV;
	if (!enable[dev]) {
		dev++;
		return -ENOENT;
	}

//	int board_config = pci_id->driver_data;
//board_config = 0x20;
//printk("board_config = %x\n", board_config);

	card = snd_card_new(index[dev], id[dev], THIS_MODULE, 0);
	if (card == NULL)
		return -ENOMEM;

	pci_read_config_byte(pci, PCI_REVISION_ID, &revision);

	strcpy(card->driver, "Azalia");
	strcpy(card->shortname, "HDA");
	if ((err = snd_azx_create(card, pci, &chip)) < 0)
		goto __error;

	if ((err = snd_azx_mixer_new(chip, azx_clock[dev])) < 0)
		goto __error;

	if ((err = snd_azx_pcm_new(chip)) < 0)
		goto __error;
	
	snd_azx_proc_init(chip);

	sprintf(card->longname, "%s rev %x at 0x%lx, irq %i",
		card->shortname, revision, chip->addr, chip->irq);

	snd_card_set_dev(card, &pci->dev);

	if ((err = snd_card_register(card)) < 0)
		goto __error;

	pci_set_drvdata(pci, chip);
	dev++;
	return 0;

 __error:
	snd_card_free(card);
	return err;
}

static void __devexit snd_azx_remove(struct pci_dev *pci)
{
	azx_t *chip = snd_magic_cast(azx_t, pci_get_drvdata(pci), return);
	if (chip)
		snd_card_free(chip->card);
	pci_set_drvdata(pci, NULL);
}

static struct pci_driver driver = {
	.name = "AZALIA controller",
	.id_table = snd_azx_ids,
	.probe = snd_azx_probe,
	.remove = __devexit_p(snd_azx_remove),
};


static int __init alsa_card_azx_init(void)
{
	int err;

        if ((err = pci_module_init(&driver)) < 0) {
#ifdef MODULE
		printk(KERN_ERR "AZALIA controller not found or device busy\n");
#endif
                return err;
        }

        return 0;
}

static void __exit alsa_card_azx_exit(void)
{
	pci_unregister_driver(&driver);
}

module_init(alsa_card_azx_init)
module_exit(alsa_card_azx_exit)

#ifndef MODULE

/* format is: snd-azalia=enable,index,id,azx_clock,spdif_aclink */

static int __init alsa_card_azx_setup(char *str)
{
	static unsigned __initdata nr_dev = 0;

	if (nr_dev >= SNDRV_CARDS)
		return 0;
	(void)(get_option(&str,&enable[nr_dev]) == 2 &&
	       get_option(&str,&index[nr_dev]) == 2 &&
	       get_id(&str,&id[nr_dev]) == 2 &&
	       get_option(&str,&azx_clock[nr_dev]) == 2 
	       );
	nr_dev++;
	return 1;
}

__setup("snd-azalia=", alsa_card_azx_setup);

#endif /* ifndef MODULE */
