/*
 * Copyright 1999-2008  University of Chicago
 * Copyright 2008-2010  Members of the EGEE Collaboration
 * Copyright 2008-  Stichting Fundamenteel Onderzoek der Materie (FOM-Nikhef)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "config.h"
#include "xacml_soapH.h"

#include "xacml_server.h"
#include "xacml_i.h"

#include <limits.h>
#include <unistd.h>
#include <dlfcn.h>

#define NDEBUG
#include <cassert>
#include <sstream>
#include <iomanip>

extern SOAP_NMAC struct Namespace xacml_soap_namespaces[];

#ifndef _POSIX_HOST_NAME_MAX
#   define _POSIX_HOST_NAME_MAX 255
#endif

/* When undefined, use the threading version by default, actual threading is
 * enabled using the XACML_THREADING_MODEL environment variable */
#ifndef XACML_THREADING
#   define XACML_THREADING  1
#endif

extern "C"
int
xacml_i_accept(
    struct soap                        *soap,
    int                                 socket,
    struct sockaddr                    *addr,
    int                                *n);

/* Global flag to be set by the server_thread when the setup has failed. In that
 * case we want to do a pthread_join straight away */
static int thread_failed=0;

/* Global timeout values used as soap.send_timeout, soap.recv_timeout and
 * soap.accept_timeout, can be set/read using xacml_server_set_timeouts() and
 * xacml_server_get_timeouts() */
static int global_send_timeout = 10;
static int global_recv_timeout = 10;
static int global_accept_timeout = 2;

#ifndef DONT_DOCUMENT_INTERNAL
namespace xacml
{
#if XACML_THREADING==1 /* Adding multi-threading support to the libxacml.so - by Aram and Oscar*/
static void *
service_client_thread(void * arg)
{
    struct soap * client_soap = (struct soap *) arg;

    pthread_detach(pthread_self());

    /* Time out of X seconds on recv */
    client_soap -> recv_timeout = global_recv_timeout;
    /* Time out of  seconds on send */
    client_soap -> send_timeout = global_send_timeout;

    /* printf ("Thread: soap_serve().\n"); */
    xacml_soap_serve(client_soap);

    /* printf ("Thread: soap_destroy().\n"); */
    soap_destroy(client_soap);

    /* printf ("Thread: soap_end().\n"); */
    soap_end(client_soap);

    /* printf ("Thread: xacml_request_destroy().\n"); */
    if (client_soap -> user)
    {
        xacml_request_destroy((xacml_request_t)(client_soap -> user));
        client_soap -> user = NULL;
    }

    /* printf ("Thread: soap_free().\n"); */
    soap_free(client_soap);

    /* printf ("Thread: exit().\n"); */
    pthread_exit(NULL);
}
#endif

static void *
service_thread(void * arg)
{
    xacml_server_t server = (xacml_server_t) arg;
    struct sockaddr addr;
    socklen_t namelen;
    struct soap soap;
    char serv[6];
    int s = 0;
    int failed=0;   /* will become 1 in case the setup failed */
    int *threaderrno=NULL;
#if XACML_THREADING==1 /* Adding multi-threading support to the libxacml.so - by Aram and Oscar*/
    struct soap * dropped_soap = NULL;
    pthread_t threadid;
    xacml_request_t x_request=NULL;
    int xacml_threading=0;  /* Default no threading */
    char *thread_envstring=getenv("XACML_THREAD_MODEL");

    /* Set correct value to xacml_threading */
    if (thread_envstring)   {
	if (strcmp(thread_envstring,"pthread")==0)
	    xacml_threading=1;
	else if (strcmp(thread_envstring,"none")==0)
	    xacml_threading=0;
	else {		/* illegal value */
	    failed=1;
	    goto out;
	}
    }
#endif /* XACML_THREADING */


    pthread_mutex_lock(&server->lock);

    if (server->state != XACML_SERVER_STARTED)
    {
        /* stopped before this thread started */
	failed=1;
        goto out;
    }
    /* See http://www.cs.fsu.edu/~engelen/soapdoc2.html, 19.36 around
       soapcpp2 -c -n -pmyClient1 client1.h */
    soap_init(&soap);
    soap_set_namespaces(&soap, xacml_soap_namespaces);

    soap.send_timeout = global_send_timeout; 
    soap.recv_timeout = global_recv_timeout;
    soap.accept_timeout = global_accept_timeout; 
    soap.bind_flags = SO_REUSEADDR;

    /* create an empty request to couple server to into soap.user, do NOT use
     * soap.user with different *type* of data. This is also needed for
     * soap_bind which calls xacml_i_close(), which needs a valid request
     * structure, upon failure. */
    server->request = NULL;
    xacml_request_init(&server->request);
    server->request->server = server;
    soap.user = server->request;

    if (server->accept_func != NULL)
    {
        /* Use custom I/O handler wrappers */
        soap.faccept = xacml_i_accept;
        soap.fopen = xacml_i_connect;
        soap.fsend = xacml_i_send;
        soap.frecv = xacml_i_recv;
        soap.fclose = xacml_i_close;
    }

    /* Note: soap_bind will call soap.fclose == xacml_i_close() in case of
     * failure, xacml_i_close will use soap->user (a request_t) to get the
     * close_func, hence soap.user must be valid at this point */
    server->listener = soap_bind(&soap, NULL, server->port, 100);
    if (server->listener < 0)
    {
	failed=1;
        goto out;
    }
    if (server->fd != -1)
    {
        close(server->listener);
        server->listener = server->fd;
        soap.master = server->fd;
    }
    namelen = sizeof(&addr);
    getsockname(server->listener, &addr, &namelen);
    getnameinfo(&addr, namelen, NULL, 0, serv, sizeof(serv), NI_NUMERICSERV);

    sscanf(serv, "%hu", &server->port);
    server->state = XACML_SERVER_READY;

    pthread_cond_signal(&server->cond);

    while (server->state == XACML_SERVER_READY)
    {
        pthread_mutex_unlock(&server->lock);
   
	/* accept function will use and fill the request in soap.user */
        s = soap_accept(&soap);
        if (s < 0)
        {
            pthread_mutex_lock(&server->lock);
            continue;
        }

#if XACML_THREADING==1 /* Adding multi-threading support to the libxacml.so - by Aram and Oscar*/
	if (xacml_threading)	{
	    x_request = NULL;
	    xacml_request_init(&x_request);
	    x_request->server = server;
	    soap.user = x_request;

	    dropped_soap = soap_copy (&soap);

	    pthread_create (&threadid, NULL, (void*(*)(void*)) service_client_thread, (void *) dropped_soap);
	} else {
#endif
	    xacml_soap_serve(&soap);

	    soap_destroy(&soap);
	    soap_end(&soap);

	    /* Destroy old request */
	    xacml_request_destroy(server->request);
	    server->request = NULL;
	
	    /* Create new empty request, do this here and not just before
	     * soap_accept or we will leak memory when we shutdown (the loop
	     * typically ends between the while and the continue above) */
	    xacml_request_init(&server->request);
	    server->request->server = server;
	    soap.user = server->request;
#if XACML_THREADING==1 /* Adding multi-threading support to the libxacml.so - by Aram and Oscar*/
	}
#endif /* XACML_THREADING */

        pthread_mutex_lock(&server->lock);
    }
out:
    /* Cleanup soap memory: even needed when we succesfully ended the while
     * loop, since in case of shutdown, we usually end from within the
     * soap_accept() */
    soap_destroy(&soap);
    soap_end(&soap);

    /* Cleanup last request */
    xacml_request_destroy(server->request);
    server->request = NULL;

    soap.user = NULL;
    switch (server->state)
    {
        case XACML_SERVER_NEW:
            assert(server->state != XACML_SERVER_NEW);
            break;
        case XACML_SERVER_STARTED:
            server->state = XACML_SERVER_STOPPED;
            break;
        case XACML_SERVER_READY:
            assert(server->state != XACML_SERVER_READY);
            break;
        case XACML_SERVER_STOPPING:
            server->state = XACML_SERVER_STOPPED;
            break;
        case XACML_SERVER_STOPPED:
            assert(server->state != XACML_SERVER_STOPPED);
            break;
    }
    soap_done(&soap);
    /* Set failed/non-failed flag *before* sending the conditional signal */
    thread_failed=failed; 
    pthread_cond_signal(&server->cond);
    pthread_mutex_unlock(&server->lock);

    /* Save errno in memory to return to thread calling join */
    threaderrno=(int*)malloc(sizeof(int));
    *threaderrno=errno;

    return (void*)threaderrno;
}

static void
extract_attribute_value(
    class XACMLcontext__AttributeValueType * attribute,
    std::string &                       value)
{
#if GSOAP_MAJOR > 2 ||\
    (GSOAP_MAJOR == 2 && GSOAP_MINOR > 8) ||\
    (GSOAP_MAJOR == 2 && GSOAP_MINOR == 8 && GSOAP_PATCH > 10)
    value = attribute->__mixed;
#else
    value = attribute->__item;
#endif
}

/**
 * Parses query as input and stores in request struct
 */
static xacml_result_t
parse_xacml_query(
    const class SOAP_CMAC XACMLsamlp__XACMLAuthzDecisionQueryType *
                                        query,
    xacml_request_t                     request)
{
    XACMLcontext__RequestType *         req = query->XACMLcontext__Request;

    if (query->saml__Issuer != NULL)
    {
	xacml_request_set_subject(request, query->saml__Issuer->__item.c_str());
	xacml_request_set_issuer(request, query->saml__Issuer->__item.c_str());
    }
    
    /* Note: we change the gsoap input to always get ID, Version and
     * IssueInstant instead of samlp__ID, samlp__Version and samlp__IssueInstant
     * Otherwise, for gsoap >= 2.8.16 we get samlp:ID etc in the
     * XACMLsamlp:XACMLAuthzDecisionQueryType and we will then also get
     * query->samlp__ID etc. instead of query->ID */
    xacml_request_set_issue_instant(request, query->IssueInstant);
    /* Always replace the request ID */
    xacml_request_set_id(request, query->ID.c_str());

    if (query->ReturnContext)
    {
        xacml_request_set_return_context(request, true);
    }

    if(req){
    
      /* TODO Really needs an if(req->Subject) check here... */
      if(1){
        for (std::vector<class XACMLcontext__SubjectType * >::iterator i =
                    req->XACMLcontext__Subject.begin();
            i != req->XACMLcontext__Subject.end();
            i++)
        {
            if (*i == NULL)
            {
                continue;
            }

            for (std::vector<class XACMLcontext__AttributeType * >::iterator j =
                (*i)->XACMLcontext__Attribute.begin();
                j != (*i)->XACMLcontext__Attribute.end();
                j++)
            {
                for (std::vector<class XACMLcontext__AttributeValueType *>::iterator k =
                            (*j)->XACMLcontext__AttributeValue.begin();
                     k != (*j)->XACMLcontext__AttributeValue.end();
                     k++)
                {
                    std::string aval;

                    extract_attribute_value(*k, aval);

                    xacml_request_add_subject_attribute(
                        request,
                        (*i)->SubjectCategory.c_str(),
                        (*j)->AttributeId.c_str(),
                        (*j)->DataType.c_str(),
                        (*j)->Issuer ? (*j)->Issuer->c_str() : NULL,
                        aval.c_str());
                }
            }
        }
      }

      /* TODO Really needs an if(req->Resource) check here... */
      if(1){
        for (std::vector<class XACMLcontext__ResourceType * >::iterator i =
                    req->XACMLcontext__Resource.begin();
            i != req->XACMLcontext__Resource.end();
            i++)
        {
            size_t attribute_count = 0;

            for (std::vector<class XACMLcontext__AttributeType * >::iterator j =
                    (*i)->XACMLcontext__Attribute.begin();
                j != (*i)->XACMLcontext__Attribute.end();
                j++)
            {
                for (std::vector<class XACMLcontext__AttributeValueType *>::iterator k =
                            (*j)->XACMLcontext__AttributeValue.begin();
                     k != (*j)->XACMLcontext__AttributeValue.end();
                     k++)
                {
                    attribute_count++;
                }
            }

            xacml_resource_attribute_s ra;

            for (std::vector<class XACMLcontext__AttributeType * >::iterator j =
                    (*i)->XACMLcontext__Attribute.begin();
                j != (*i)->XACMLcontext__Attribute.end();
                j++)
            {
                for (std::vector<class XACMLcontext__AttributeValueType *>::iterator k =
                            (*j)->XACMLcontext__AttributeValue.begin();
                     k != (*j)->XACMLcontext__AttributeValue.end();
                     k++)
                {
                    std::string val;
                    
                    extract_attribute_value(*k, val);
                    xacml_resource_attribute_add(
                            &ra,
                            (*j)->AttributeId.c_str(),
                            (*j)->DataType.c_str(),
                            (*j)->Issuer ? (*j)->Issuer->c_str() : NULL,
                            val.c_str());
                }
            }
            xacml_request_add_resource_attribute(
                    request,
                    &ra);
        }
      
      }
      if(req->XACMLcontext__Action){
        for (std::vector<class XACMLcontext__AttributeType *>::iterator i =
                    req->XACMLcontext__Action->XACMLcontext__Attribute.begin();
             i != req->XACMLcontext__Action->XACMLcontext__Attribute.end();
             i++)
        {
            for (std::vector<class XACMLcontext__AttributeValueType *>::iterator j =
                        (*i)->XACMLcontext__AttributeValue.begin();
                 j != (*i)->XACMLcontext__AttributeValue.end();
                 j++)
            {
                std::string action;

                extract_attribute_value(*j, action);

                xacml_request_add_action_attribute(
                    request,
                    (*i)->AttributeId.c_str(),
                    (*i)->DataType.c_str(),
                    (*i)->Issuer ? (*i)->Issuer->c_str() : NULL,
                    action.c_str());
            }
        }
      
      }
      if (req->XACMLcontext__Environment)
      {
          for (std::vector<class XACMLcontext__AttributeType *>::iterator i =
                      req->XACMLcontext__Environment->XACMLcontext__Attribute.begin();
               i != req->XACMLcontext__Environment->XACMLcontext__Attribute.end();
               i++)
          {
              for (std::vector<class XACMLcontext__AttributeValueType *>::iterator j =
                          (*i)->XACMLcontext__AttributeValue.begin();
                   j != (*i)->XACMLcontext__AttributeValue.end();
                   j++)
              {
                  std::string envval;

                  extract_attribute_value(*j, envval);

                  xacml_request_add_environment_attribute(
                      request,
                      (*i)->AttributeId.c_str(),
                      (*i)->DataType.c_str(),
                      (*i)->Issuer ? (*i)->Issuer->c_str() : NULL,
                      envval.c_str());
              }
          }
      }
    }


    return XACML_RESULT_SUCCESS;
}

/**
 * Takes the response struct as input and uses it to fill the samlp__Response
 */
static int
prepare_response(
    struct soap *                       soap,
    xacml_response_t                    response,
    class SOAP_CMAC samlp__ResponseType *        samlp__Response)
{
    
    if (response == NULL || samlp__Response == NULL)
	return SOAP_SVR_FAULT;

    // Make sure response->id is set
    if (response->id == "")
	if (xacml_response_set_id(response, NULL)==NULL)
	    return SOAP_SVR_FAULT;

    // Set response to reasonable default if it's still empty
    if (response->issuer == "")
	response->issuer = "XACMLService";

    samlp__Response->saml__Issuer = soap_new_saml__NameIDType(soap, -1);
    samlp__Response->saml__Issuer->Format = soap_new_std__string(soap, -1);
    samlp__Response->saml__Issuer->Format->assign("urn:oasis:names:tc:SAML:2.0:nameid-format:entity");
    // Take over the response issuer from the XACML response, since it might
    // already have been set using xacml_response_set_issuer() in the handler
    // called in __XACMLService__Authorize() which is called before we are
    // called here. If it's unset, use "XACMLService". The question is if we
    // want to use the same response issuer for both the SAML-P response issuer
    // and the attribute containing the XACML decision as SAML statement. I
    // would say they are normally the same
    samlp__Response->saml__Issuer->__item = response->issuer;
    // Also take over response ID, replace by "" when empty (error)
    samlp__Response->ID = response->id;

    samlp__Response->samlp__Status = soap_new_samlp__StatusType(soap, -1);
    samlp__Response->samlp__Status->samlp__StatusCode = soap_new_samlp__StatusCodeType(soap, -1);
    samlp__Response->samlp__Status->samlp__StatusCode->Value =
	saml_status_code_strings[response->saml_status_code];

    // Set the 

    samlp__Response->InResponseTo = soap_new_std__string(soap, -1);
    samlp__Response->InResponseTo->assign(response->in_response_to);

    samlp__Response->Version = "2.0";

    samlp__Response->IssueInstant =
            response->issue_instant ? response->issue_instant : time(NULL);

    // We're now going to create a assertion inside the SAML-P response (for
    // which we also have set the issuer above.
    samlp__Response->__size_ResponseType = 1;
    samlp__Response->__union_ResponseType = soap_new___samlp__union_ResponseType(soap, 1);
    samlp__Response->__union_ResponseType[0].__union_ResponseType =
            SOAP_UNION__samlp__union_ResponseType_saml__Assertion;
    samlp__Response->__union_ResponseType[0].union_ResponseType.saml__Assertion =
            soap_new_saml__AssertionType(soap, -1);

    samlp__Response->__union_ResponseType[0].union_ResponseType.saml__Assertion->IssueInstant =
            time(NULL);

    samlp__Response->__union_ResponseType[0].union_ResponseType.saml__Assertion->saml__Issuer =
            soap_new_saml__NameIDType(soap, -1);
    samlp__Response->__union_ResponseType[0].union_ResponseType.saml__Assertion->saml__Issuer->Format
            =
            soap_new_std__string(soap, -1);
    samlp__Response->__union_ResponseType[0].union_ResponseType.saml__Assertion->saml__Issuer->Format->assign("urn:oasis:names:tc:SAML:2.0:nameid-format:entity");

    // Note: I believe we need to set the saml__Issuer->__item, not saml__Issuer
    // itself: that would result in an empty attribute issuer
    samlp__Response->__union_ResponseType->union_ResponseType.saml__Assertion->saml__Issuer->__item =
	    response->issuer;

    saml__AssertionType * response_assertion =
            samlp__Response->__union_ResponseType->union_ResponseType.saml__Assertion;

    response_assertion->__size_AssertionType = 1;
    response_assertion->__union_AssertionType = soap_new___saml__union_AssertionType(soap, 1);

    response_assertion->__union_AssertionType->__union_AssertionType =
	            SOAP_UNION__saml__union_AssertionType_saml__Statement;
    // Here the XACML decision is created which will be a SAML statement in the
    // attribute which is part of the SAML-P response
    response_assertion->__union_AssertionType->union_AssertionType.saml__Statement =
            soap_new_XACMLassertion__XACMLAuthzDecisionStatementType(soap, -1);

    XACMLassertion__XACMLAuthzDecisionStatementType * xacml_decision =
            dynamic_cast<XACMLassertion__XACMLAuthzDecisionStatementType *>
                (response_assertion->__union_AssertionType->union_AssertionType.saml__Statement);

    xacml_decision->XACMLcontext__Response =
            soap_new_XACMLcontext__ResponseType(soap, -1);

    // It seems we're doing it ok, had expected we might need to set
    // xacml_decision->__item here

    XACMLcontext__ResultType * result = soap_new_XACMLcontext__ResultType(soap, -1);
    xacml_decision->XACMLcontext__Response->XACMLcontext__Result.push_back(result);

    switch (response->decision)
    {
        case XACML_DECISION_Permit:
            result->XACMLcontext__Decision = XACMLcontext__DecisionType__Permit;
            break;
        case XACML_DECISION_Deny:
            result->XACMLcontext__Decision = XACMLcontext__DecisionType__Deny;
            break;
        case XACML_DECISION_Indeterminate:
        case XACML_DECISION_NotApplicable:
            return SOAP_SVR_FAULT;
    }

    result->XACMLcontext__Status = soap_new_XACMLcontext__StatusType(soap, -1);

    result->XACMLcontext__Status->XACMLcontext__StatusCode = 
            soap_new_XACMLcontext__StatusCodeType(soap, -1);

    result->XACMLcontext__Status->XACMLcontext__StatusCode->Value =
            xacml_status_code_strings[response->xacml_status_code];

    if (response->obligations.size() != 0)
    {
        result->XACMLpolicy__Obligations =
                soap_new_XACMLpolicy__ObligationsType(soap, -1);

        for (xacml::obligations::iterator i = response->obligations.begin();
             i != response->obligations.end();
             i++)
        {
            XACMLpolicy__ObligationType * obligation =
                    soap_new_XACMLpolicy__ObligationType(soap, -1);

            result->XACMLpolicy__Obligations->
                    XACMLpolicy__Obligation.push_back(obligation);

            obligation->ObligationId = i->obligation.obligation_id;

            switch (i->obligation.fulfill_on)
            {
                case XACML_EFFECT_Permit:
                    obligation->FulfillOn = XACMLpolicy__EffectType__Permit;
                    break;
                case XACML_EFFECT_Deny:
                    obligation->FulfillOn = XACMLpolicy__EffectType__Deny;
                    break;
                default:
                    return SOAP_SVR_FAULT;
            }

            for (xacml::attributes::iterator j = i->obligation.attributes.begin();
                 j != i->obligation.attributes.end();
                 j++)
            {
                XACMLpolicy__AttributeAssignmentType * attr =
                    soap_new_XACMLpolicy__AttributeAssignmentType(soap, -1);
                obligation->XACMLpolicy__AttributeAssignment.push_back(attr);

                attr->DataType = j->data_type;
                attr->AttributeId = j->attribute_id;
                attr->__mixed = (char *) soap_malloc(soap, j->value.length()+1);
                std::strcpy(attr->__mixed, j->value.c_str());
            }
        }
    }

    if (response->request != NULL)
    {
        xacml_decision->XACMLcontext__Request =
            xacml::create_xacml_request(soap, response->request);
    }

    return SOAP_OK;
}
} // namespace xacml
#endif /* DONT_DOCUMENT_INTERNAL */

/**
 * Initialize XACML server
 * @ingroup xacml_server
 *
 * Create a new XACML authorization server instance. This service, when
 * started with xacml_server_start(), will accept TCP/IP connections, parse an
 * XACML authorization query, and then call the @a handler callback function
 * with the request information. By default, the server will listen for
 * connections on TCP port 8080.
 * 
 * @param server 
 *     Pointer to the service handle to intialize.
 * @param handler
 *     Callback function to perform authorization and create obligations.
 * @param arg
 *     Application-specific argument to @a handler.
 *
 * @retval XACML_RESULT_SUCCESS
 *     Success.
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter.
 * @see xacml_server_start(), xacml_server_destroy()
 */
xacml_result_t
xacml_server_init(
    xacml_server_t *                    server,
    xacml_authorization_handler_t       handler,
    void *                              arg)
{
    if (server == NULL || handler == NULL)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }
    (*server) = new xacml_server_s;
    (*server)->port = 8080;
    (*server)->state = XACML_SERVER_NEW;
    (*server)->handler = handler;
    (*server)->handler_arg = arg;
    (*server)->io_module = NULL;
    (*server)->accept_func = NULL;
    (*server)->request = NULL;
    (*server)->fd = -1;
    pthread_mutex_init(&(*server)->lock, NULL);
    pthread_cond_init(&(*server)->cond, NULL);

    return XACML_RESULT_SUCCESS;
}
/* xacml_server_init() */

/**
 * Set the TCP port for a server
 * @ingroup xacml_server
 *
 * Change the TCP port to use for an XACML server. This must be called
 * before the server has been started, or an @a XACML_RESULT_INVALID_STATE
 * error will result.
 *
 * @param server
 *     Server to modify.
 * @param port
 *     TCP port number to use for connections.
 *
 * @retval XACML_RESULT_SUCCESS
 *     Success.
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter.
 * @retval XACML_RESULT_INVALID_STATE
 *     Invalid state.
 *
 * @see xacml_server_get_port()
 */
xacml_result_t
xacml_server_set_port(
    xacml_server_t                      server,
    unsigned short                      port)
{
    xacml_result_t rc = XACML_RESULT_SUCCESS;

    if (server == NULL)
    {
        rc = XACML_RESULT_INVALID_PARAMETER;
        goto out;
    }

    pthread_mutex_lock(&server->lock);
    switch (server->state)
    {
        case XACML_SERVER_NEW:
            server->port = port;
            break;
        case XACML_SERVER_STARTED:
        case XACML_SERVER_READY:
        case XACML_SERVER_STOPPING:
        case XACML_SERVER_STOPPED:
            rc = XACML_RESULT_INVALID_STATE;
            break;
    }
    pthread_mutex_unlock(&server->lock);

out:
    return rc;
}
/* xacml_server_set_port() */


/**
 * Get the TCP port for a server
 * @ingroup xacml_server
 *
 * Return the TCP port to use for an XACML server. 
 *
 * @param server
 *     Server to query.
 * @param port
 *     Pointer to be set to the TCP port number to use for connections.
 *
 * @retval XACML_RESULT_SUCCESS
 *     Success.
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter.
 *
 * @see xacml_server_set_port()
 */
xacml_result_t
xacml_server_get_port(
    const xacml_server_t                server,
    unsigned short *                    port)
{
    if (server == NULL || port == NULL)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }
    pthread_mutex_lock(&server->lock);
    while (server->state == XACML_SERVER_STARTED)
    {
        pthread_cond_wait(&server->cond, &server->lock);
    }
    *port = server->port;
    pthread_mutex_unlock(&server->lock);

    return XACML_RESULT_SUCCESS;
}
/* xacml_server_get_port() */

/**
 * Start processing XACML Authorization Queries
 * @ingroup xacml_server
 *
 * Creates a thread to procss XACML authorization queries. This thread
 * will continue to do so until xacml_server_destroy() is called. When
 * a legitimate XACML query has been parsed, the authorization handler
 * function passed to xacml_server_init() will be called with the parsed
 * XACML Authorization query and the application specific argument.
 *
 * @param server
 *     XACML server to start.
 *
 * @retval XACML_RESULT_SUCCESS
 *     Success
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter
 * @retval XACML_RESULT_INVALID_STATE;
 *     Invalid state
 *
 * @see xacml_server_init(), xacml_server_destroy()
 */
xacml_result_t
xacml_server_start(
    xacml_server_t                      server)
{
    int                                 r;
    int *retval;
    int rr;
    xacml_result_t                      rc = XACML_RESULT_SUCCESS;

    pthread_mutex_lock(&server->lock);
    switch (server->state)
    {
        case XACML_SERVER_NEW:
            r = pthread_create(&server->service_thread, NULL,
                                xacml::service_thread, server);
            if (r == 0)
            {
                server->state = XACML_SERVER_STARTED;
            }
            else
            {
                rc = XACML_RESULT_INVALID_STATE;
            }
            break;

        case XACML_SERVER_STARTED:
        case XACML_SERVER_READY:
        case XACML_SERVER_STOPPING:
            rc = XACML_RESULT_INVALID_STATE;
            break;
        case XACML_SERVER_STOPPED:
            /* This state only occurs when the server is already being
             * destroyed 
             */
            assert(server->state != XACML_SERVER_STOPPED);
            break;
    }

    /* Wait for the signal from server_thread: Need to do BEFORE unlocking */
    pthread_cond_wait(&server->cond,&server->lock);
    pthread_mutex_unlock(&server->lock);

    if (thread_failed)  {
	thread_failed=0; /* reset flag, probably not needed */
	rc=XACML_RESULT_INVALID_STATE; /* return value to indicate failure */
	/* Call pthread_join now, since the thread has failed and hence has
	 * already exited */
	rr=pthread_join(server->service_thread,(void**)&retval);
	if (rr==0) {  /* successfully joined, retval now contains errno */
	    errno=*retval;
	    free(retval); /* cleanup memory */
	    retval=NULL;
	}
	else	    /* pthread_join failed, set errno to its own error code */
	    errno=rr;
    }

    return rc;
}
/* xacml_server_start() */

/**
 * Destroy an XACML server
 * @ingroup xacml_server
 *
 * Stop servicing XACML authorization queries, and destroy the server
 * handle. The @a server should not be used after this function returns.
 *
 * @param server
 *     Server to destroy.
 *
 * @return void
 * @see xacml_server_init(), xacml_server_start()
 */
void
xacml_server_destroy(
    xacml_server_t                      server)
{
    int *retval=NULL;
    int rr;

    if (server == NULL)
    {
        return;
    }

    pthread_mutex_lock(&server->lock);
    switch (server->state)
    {
        case XACML_SERVER_NEW:
            server->state = XACML_SERVER_STOPPED;
            break;

        case XACML_SERVER_STARTED:
        case XACML_SERVER_READY:
            server->state = XACML_SERVER_STOPPING;

            /* FALLSTHROUGH */
        case XACML_SERVER_STOPPING:
            while (server->state == XACML_SERVER_STOPPING)
            {
                pthread_cond_wait(&server->cond, &server->lock);
            }
	    rr=pthread_join(server->service_thread,(void**)&retval);
	    if (rr==0) {  /* successfully joined, retval now contains errno */
		errno=*retval;
		free(retval); /* cleanup memory */
		retval=NULL;
	    } else    /* pthread_join failed, set errno to its own error code */
		errno=rr;
            break;
        case XACML_SERVER_STOPPED:
            assert(server->state != XACML_SERVER_STOPPED);
            break;
    }
    pthread_mutex_unlock(&server->lock);

    pthread_mutex_destroy(&server->lock);
    pthread_cond_destroy(&server->cond);

    if (server->io_module)
    {
        dlclose(server->io_module);
    }

    delete server;
}



/**
 * Load and use an I/O module from a shared object for a server's requests
 * @ingroup xacml_io
 * Open the module named by @a module and configures the @a server handle
 * to use the I/O descriptor named "xacml_io_descriptor" in that module to
 * handle the server's I/O.
 *
 * @param server
 *     XACML server handle.
 * @param module
 *     Name of a shared object containing the xacml_io_descriptor_t.
 *
 * @retval XACML_RESULT_SUCCESS
 *     Success.
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter.
 * 
 * @note
 *     If an error occurs loading the I/O module, an error message will be
 *     sent to stderr.
 * @see xacml_request_set_io_descriptor()
 */
xacml_result_t
xacml_server_set_io_module(
    xacml_server_t                      server,
    const char                         *module)
{
    const xacml_io_descriptor_t        *desc;
    xacml_result_t                      rc;

    server->io_module = dlopen(module, RTLD_NOW|RTLD_LOCAL);
    if (server->io_module == NULL)
    {
        std::cerr << "Error loading module " << module << " "
             << dlerror() << std::endl;
        return XACML_RESULT_INVALID_PARAMETER;
    }
    desc = reinterpret_cast<xacml_io_descriptor_t *>(
            dlsym(server->io_module, XACML_IO_DESCRIPTOR));

    rc = xacml_server_set_io_descriptor(server, desc);

    if (rc !=  XACML_RESULT_SUCCESS)
    {
        dlclose(server->io_module);
        server->io_module = NULL;
    }

    return rc;
}
/* xacml_server_set_io_module() */

/**
 * Set the socket file descriptor used to accept new connections
 * @ingroup xacml_io
 *
 * @param server
 *     XACML server handle.
 * @param fd
 *     Bound server file descriptor which has had listen() called on it.
 *
 * @retval XACML_RESULT_SUCCESS
 *     Success.
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter.
 * 
 * @note
 *     If an error occurs loading the I/O module, an error message will be
 *     sent to stderr.
 * @see xacml_request_set_io_descriptor()
 */
xacml_result_t
xacml_server_set_fd(
    xacml_server_t                      server,
    int                                 fd)
{
    xacml_result_t                      rc = XACML_RESULT_SUCCESS;

    if (server == NULL)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }
    server->fd = fd;
    return rc;
}
/* xacml_server_set_fd() */

/**
 * Use an I/O module for a server
 * @ingroup xacml_io
 * 
 * Configure a server handle to use the I/O callbacks contained in the
 * descriptor. 
 *
 * @param server
 *     XACML server handle.
 * @param descriptor
 *     Descriptor with the I/O callbacks to be used when processing 
 *     @a request.
 *
 * @retval XACML_RESULT_SUCCESS
 *     Success.
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter.
 * @see xacml_request_set_io_descriptor()
 */
xacml_result_t
xacml_server_set_io_descriptor(
    xacml_server_t                      server,
    const xacml_io_descriptor_t        *descriptor)
{
    if (server == NULL || descriptor == NULL)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }
    server->accept_func = descriptor->accept_func;
    server->connect_func = descriptor->connect_func;
    server->send_func = descriptor->send_func;
    server->recv_func = descriptor->recv_func;
    server->close_func = descriptor->close_func;

    return XACML_RESULT_SUCCESS;
}
/* xacml_server_set_io_descriptor() */


/**
 * Set the global send, receive and accept timeouts for the gsoap structure
 *
 * @param send_timeout
 *     send timeout, -1 to keep default
 * @param recv_timeout
 *     receive timeout, -1 to keep default
 * @param accept_timeout
 *     accept timeout, -1 to keep default
 *
 */
void
xacml_server_set_timeouts(
    int send_timeout,
    int recv_timeout,
    int accept_timeout
    )
{
    if (send_timeout!=-1)
	global_send_timeout = send_timeout;
    if (recv_timeout!=-1)
	global_recv_timeout = recv_timeout;
    if (accept_timeout!=-1)
	global_accept_timeout = accept_timeout;
}
/* xacml_server_set_timeouts() */

/**
 * Get the global send, receive and accept timeouts for the gsoap structure
 *
 * @param send_timeout
 *     returns current send timeout when non-NULL
 * @param recv_timeout
 *     returns current receive timeout when non-NULL
 * @param accept_timeout
 *     returns current accept timeout when non-NULL
 *
 */
void
xacml_server_get_timeouts(
    int *send_timeout,
    int *recv_timeout,
    int *accept_timeout
    )
{
    if (send_timeout)
	*send_timeout=  global_send_timeout;
    if (recv_timeout)
	*recv_timeout=  global_recv_timeout;
    if (accept_timeout)
	*accept_timeout=global_accept_timeout;
}
/* xacml_server_get_timeouts() */

#ifndef DONT_DOCUMENT_INTERNAL
int
__XACMLService__Authorize(
    struct soap *                       soap,
    class SOAP_CMAC XACMLsamlp__XACMLAuthzDecisionQueryType *
                                        XACMLsamlp__XACMLAuthzDecisionQuery,
    class SOAP_CMAC samlp__ResponseType *        samlp__Response)
{
    int                                 rc;
    xacml_server_t                      server;
    xacml_request_t                     request;
    xacml_response_t                    response;

    request = reinterpret_cast<xacml_request_t>(soap->user);
    server = request->server;

    rc = xacml::parse_xacml_query(XACMLsamlp__XACMLAuthzDecisionQuery, request);

    if (rc != 0)
    {
        return SOAP_CLI_FAULT;
    }

    rc = xacml_response_init(&response);
    if (rc != 0)
    {
        return SOAP_SVR_FAULT;
    }
    rc = server->handler(server->handler_arg, request, response);
    if (rc != 0)
    {
        return SOAP_SVR_FAULT;
    }

    if (response->request == NULL && request->return_context)
    {
        /* Client requested the service return a context in the reply to
         * indicate what attribtues were used by the service to reach the 
         * decision. Fake this with the entire input context if the service
         * implementation doesn't provide one.
         */
        xacml_response_set_request_context(response, request);
    }
   
    // Already put in the ID of the request in the in_response_to field
    /* Note: we change the gsoap input to always get ID, Version and
     * IssueInstant instead of samlp__ID, samlp__Version and samlp__IssueInstant
     * Otherwise, for gsoap >= 2.8.16 we get samlp:ID etc in the
     * XACMLsamlp:XACMLAuthzDecisionQueryType and we will then also get
     * query->samlp__ID etc. instead of query->ID */
    response->in_response_to = XACMLsamlp__XACMLAuthzDecisionQuery->ID.c_str();

    rc = xacml::prepare_response(soap, response, samlp__Response);
    if (rc != 0)
    {
        return SOAP_SVR_FAULT;
    }
    xacml_response_destroy(response);

    return SOAP_OK;
}
/* __XACMLService__Authorize() */

extern "C"
int
xacml_i_accept(
    struct soap                        *soap,
    int                                 socket,
    struct sockaddr                    *addr,
    int                                *n)
{
    xacml_server_t                      server;
    xacml_request_t                     request;
    socklen_t                           len;
    int                                 sock_out = 0;
    void *                              io_arg;

    /* *n is int, while socklen_t is typically an unsigned */
    if (*n<0)
    {
	soap->error = SOAP_ERR;
	return -1;
    }
    len=(socklen_t)(*n);

    /* Input soap->user is probably a dummy but initialized request structure.
     * We will reuse it, to prevent memory loss, with the proper one */
    request = (xacml_request_t)(soap->user);
    server = request->server;

    io_arg = server->accept_func(socket, addr, &len, &sock_out);

    /* *n is int, while socklen_t is typically an unsigned */
    if (io_arg == NULL || len>INT_MAX)
    {
        soap->error = SOAP_ERR;
        return -1;
    }
    *n = (int)len;

    request->connect_func = server->connect_func;
    request->send_func = server->send_func;
    request->recv_func = server->recv_func;
    request->close_func = server->close_func;
    request->io_arg = io_arg;
    request->server = server;
    server->request = request;

    return sock_out;
}
/* xacml_i_accept() */

#endif /* DONT_DOCUMENT_INTERNAL */
