/*
 * Copyright 1999-2008  University of Chicago
 * Copyright 2008-2010  Members of the EGEE Collaboration
 * Copyright 2008-  Stichting Fundamenteel Onderzoek der Materie (FOM-Nikhef)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <sstream>
#include <stdlib.h>
#include <sys/time.h>

#include "xacml_i.h"

/**
 * Note: this function might initialize twice if we run it multi-threaded, but
 * will not call lrand48() uninitialized.
 */
static long xacml_rand(long max)	{
    static int initialized=0;
    struct timeval tv;

    /* seed only once, use a fast changing variable, such as milliseconds */
    if (!initialized)	{
	gettimeofday(&tv,NULL);
	/* Create number of microseconds from tv_usec and tv_sec together, make
	 * sure to stay <= 2.147e9 < 2^31 */
	srand48((long)tv.tv_usec+1000000L*(long)(tv.tv_sec%2148));
	initialized=1;
    }

    /* return lrand48 modulo a max */
    return (lrand48() % max);
}

/**
 * @defgroup xacml_response Responses
 * @ingroup xacml_common
 */

/**
 * Initialize an XACML query response
 * @ ingroup xacml_response
 *
 * Creates an XACML response structure which can be used to contain a
 * response with obligations to send to a SAML / XACML client. After the
 * response is no longer needed, the caller must destroy it by calling
 * xacml_response_destroy().
 *
 * @param response
 *     Response to initialize. The value pointed to by this can be passed
 *     to other xacml_response_* functions.
 *
 * @retval XACML_RESULT_SUCCESS
 *     Success
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter
 *
 * @see xacml_response_destroy().
 */
xacml_result_t
xacml_response_init(
    xacml_response_t *                  response)
{
    // Note that using
    if (response == NULL)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }
    *response = new xacml_response_s;
    (*response)->issue_instant = 0;
    (*response)->saml_status_code = SAML_STATUS_Success;
    (*response)->decision = XACML_DECISION_Permit;
    (*response)->xacml_status_code = XACML_STATUS_ok;
    (*response)->request = NULL;

    // Will typically be set using xacml_response_set_issuer()
    (*response)->issuer = "";
    
    // Will typically be set using xacml_response_set_id()
    (*response)->id = "";
    
    return XACML_RESULT_SUCCESS;
}
/* xacml_response_init() */

/**
 * Destroy an XACML query response
 * @ ingroup xacml_response
 *
 * Frees resources associated with an XACML response structure. After
 * calling this, the @a response value must not be used by the caller.
 *
 * @param response
 *     Response to destroy.
 *
 * @return void
 *
 * @see xacml_response_init().
 */
void
xacml_response_destroy(
    xacml_response_t                    response)
{
    if (response != NULL)
    {
        if (response->request != NULL)
        {
            xacml_request_destroy(response->request);
        }
        delete response;
    }
}
/* xacml_response_destroy() */

/**
 * Set the issue instant of a response
 * @ ingroup xacml_response
 * Sets the response IssueInstance to the time specified in @a issue_instant.
 * On the server, if this is not set, the XACML library will automatically add
 * one when the response is sent.
 * 
 * @param response
 *     The XACML response to inspect.
 * @param issue_instant
 *     New value of the IssueInstant.
 * 
 * @retval XACML_RESULT_SUCCESS
 *     Success
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter
 */
xacml_result_t
xacml_response_set_issue_instant(
    xacml_response_t                    response,
    time_t                              issue_instant)
{
    if (response == NULL)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }
    response->issue_instant = issue_instant;

    return XACML_RESULT_SUCCESS;
}
/* xacml_response_set_issue_instant() */

/**
 * Determine when a response was issued
 * @ ingroup xacml_response
 * Checks an SAML protocol response and copies the timestamp when it was issued
 * to the value pointed to by @a issue_instant.
 * 
 * @param response
 *     The XACML response to inspect.
 * @param issue_instant
 *     Pointer to be set to the timestamp when the response was issued.
 * 
 * @retval XACML_RESULT_SUCCESS
 *     Success
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter
 */
xacml_result_t
xacml_response_get_issue_instant(
    const xacml_response_t              response,
    time_t *                            issue_instant)
{
    if (response == NULL || issue_instant == NULL)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }
    *issue_instant = response->issue_instant;

    return XACML_RESULT_SUCCESS;
}
/* xacml_response_get_issue_instant() */

/**
 * Set the issuer of the response
 * @ ingroup xacml_response
 *
 * @param response
 *     Response to modify.
 * @param issuer
 *     Value of the issuer
 *
 * @retval XACML_RESULT_SUCCESS
 * @retval XACML_RESULT_INVALID_PARAMETER
 */
xacml_result_t
xacml_response_set_issuer(
    xacml_response_t                    response,
    const char *                        issuer)
{
    if (response == NULL || issuer == NULL)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }
    response->issuer = issuer;

    return XACML_RESULT_SUCCESS;
}
/* xacml_response_set_issuer() */

/**
 * Determine the name of the issuer of a response
 * @ ingroup xacml_response
 * 
 * @param response
 *     The XACML response to inspect.
 * @param issuer
 *     Pointer to be set to the name of the issuer. The caller must not free
 *     this value or access it after the response has been destroyed.
 * 
 * @retval XACML_RESULT_SUCCESS
 *     Success
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter
 */
xacml_result_t
xacml_response_get_issuer(
    const xacml_response_t              response,
    const char **                       issuer)
{
    if (response == NULL || issuer == NULL)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }
    *issuer = response->issuer == "" ? NULL : response->issuer.c_str();

    return XACML_RESULT_SUCCESS;
}
/* xacml_response_get_issuer() */

/**
 * Set the ID of the response
 * @ ingroup xacml_response
 *
 * @param response
 *     Response to modify.
 * @param issuer
 *     Value of the ID
 *
 * @retval id or NULL on error
 */
const char *
xacml_response_set_id(
    xacml_response_t                    response,
    const char *                        id)
{
    std::ostringstream ostr;

    if (response == NULL)
        return NULL;

    if (id == NULL) { // Initialize the ID
	ostr << "ID-";
	ostr.fill('0');
	ostr.width(9);
	ostr << xacml_rand((long)1e9);
	response->id = ostr.str();
    } else
	response->id = id;

    return response->id.c_str();
}
/* xacml_response_set_id() */

/**
 * Determine the ID of a response
 * @ ingroup xacml_response
 * 
 * @param response
 *     The XACML response to inspect.
 * @param ID
 *     Pointer to be set to the ID. The caller must not free this value or
 *     access it after the response has been destroyed.
 * 
 * @retval XACML_RESULT_SUCCESS
 *     Success
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter
 */
xacml_result_t
xacml_response_get_id(
    const xacml_response_t              response,
    const char **                       id)
{
    if (response == NULL || id == NULL)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }
    *id = response->id == "" ? NULL : response->id.c_str();

    return XACML_RESULT_SUCCESS;
}
/* xacml_response_get_id() */

/**
 * Set the in_response_to field of the response
 * @ ingroup xacml_response
 *
 * @param response
 *     Response to modify.
 * @param in_response_to
 *     Value of the in_response_to field
 *
 * @retval XACML_RESULT_SUCCESS
 * @retval XACML_RESULT_INVALID_PARAMETER
 */
xacml_result_t
xacml_response_set_in_response_to(
    xacml_response_t                    response,
    const char *                        in_response_to)
{
    if (response == NULL || in_response_to == NULL)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }
    response->in_response_to = in_response_to;

    return XACML_RESULT_SUCCESS;
}
/* xacml_response_set_in_response_to() */

/**
 * Determine the ID int the in_response_to field of a response
 * @ ingroup xacml_response
 * 
 * @param response
 *     The XACML response to inspect.
 * @param in_response_to
 *     Pointer to be set to the ID in the in_response_to field. The caller must
 *     not free this value or access it after the response has been destroyed.
 * 
 * @retval XACML_RESULT_SUCCESS
 *     Success
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter
 */
xacml_result_t
xacml_response_get_in_response_to(
    const xacml_response_t              response,
    const char **                       in_response_to)
{
    if (response == NULL || in_response_to == NULL)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }
    *in_response_to = response->in_response_to == "" 
			    ? NULL
			    : response->in_response_to.c_str();

    return XACML_RESULT_SUCCESS;
}
/* xacml_response_get_in_response_to() */

/**
 * Set the status code of a SAML response
 * @ ingroup xacml_response
 *
 * Set the status code describing the status of a query. The
 * possible values of the status code are defined in @ref saml_status_code_t
 *
 * @param response
 *     Response to modify.
 * @param status_code
 *     New status code value.
 * 
 * @retval XACML_RESULT_SUCCESS
 *     Success
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter
 *
 * @see xacml_response_get_saml_status_code()
 */
xacml_result_t
xacml_response_set_saml_status_code(
    xacml_response_t                    response,
    saml_status_code_t                  status_code)
{
    if (response == NULL ||
        status_code < SAML_STATUS_Success ||
        status_code > SAML_STATUS_UnsupportedBinding)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }
    response->saml_status_code = status_code;

    return XACML_RESULT_SUCCESS;
}
/* xacml_response_set_saml_status_code() */

/**
 * Get the status code of a SAML response
 * @ ingroup xacml_response
 *
 * Retrieves the status code describing the status of the query. The possible
 * values of the status code are defined in @ref saml_status_code_t
 *
 * @param response
 *     Response to inspect.
 * @param status_code
 *     Pointer to be set to the status code.
 * 
 * @retval XACML_RESULT_SUCCESS
 *     Success
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter
 *
 * @see xacml_response_set_saml_status_code()
 */
xacml_result_t
xacml_response_get_saml_status_code(
    const xacml_response_t              response,
    saml_status_code_t *                status_code)
{
    if (response == NULL || status_code == NULL)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }
    *status_code = response->saml_status_code;

    return XACML_RESULT_SUCCESS;
}
/* xacml_response_get_saml_status_code() */

/**
 * Set the decision of an XACML response
 * @ ingroup xacml_response
 *
 * Set the decision value describing the result of a query. The
 * possible values of the status code are defined in @ref xacml_decision_t
 *
 * @param response
 *     Response to modify.
 * @param decision
 *     New decision value.
 * 
 * @retval XACML_RESULT_SUCCESS
 *     Success
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter
 *
 * @see xacml_response_get_xacml_decision()
 */
xacml_result_t
xacml_response_set_xacml_decision(
    xacml_response_t                    response,
    xacml_decision_t                    decision)
{
    if (response == NULL ||
        decision < XACML_DECISION_Permit ||
        decision > XACML_DECISION_NotApplicable)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }
    response->decision = decision;

    return XACML_RESULT_SUCCESS;
}
/* xacml_response_set_xacml_decision() */

/**
 * Get the XACML decision from a response
 * @ ingroup xacml_response
 *
 * Retrieves the decision of the response to a query. The possible
 * values of the status code are defined in @ref xacml_decision_t
 *
 * @param response
 *     Response to inspect.
 * @param decision
 *     Pointer to be set to the decision.
 * 
 * @retval XACML_RESULT_SUCCESS
 *     Success
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter
 *
 * @see xacml_response_set_xacml_decision()
 */
xacml_result_t
xacml_response_get_xacml_decision(
    const xacml_response_t              response,
    xacml_decision_t *                  decision)
{
    if (response == NULL || decision == NULL)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }
    *decision = response->decision;

    return XACML_RESULT_SUCCESS;
}
/* xacml_response_get_xacml_decision() */


/**
 * Set the XACML status code of a response
 * @ ingroup xacml_response
 *
 * Set the status code describing the status of a query. The
 * possible values of the status code are defined in @ref xacml_status_code_t
 *
 * @param response
 *     Response to modify.
 * @param status_code
 *     New status code value.
 * 
 * @retval XACML_RESULT_SUCCESS
 *     Success
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter
 *
 * @see xacml_response_get_xacml_status_code()
 */
xacml_result_t
xacml_response_set_xacml_status_code(
    xacml_response_t                    response,
    xacml_status_code_t                 status_code)
{
    if (response == NULL ||
        status_code < XACML_STATUS_ok ||
        status_code > XACML_STATUS_processing_error)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }
    response->xacml_status_code = status_code;

    return XACML_RESULT_SUCCESS;
}
/* xacml_response_set_xacml_status_code() */

/**
 * Get the XACML status code of a response
 * @ ingroup xacml_response
 *
 * Retrieves the status code describing the status of the XACML query. The
 * possible values of the status code are defined in @ref xacml_status_code_t
 *
 * @param response
 *     Response to inspect.
 * @param status_code
 *     Pointer to be set to the status code.
 * 
 * @retval XACML_RESULT_SUCCESS
 *     Success
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter
 *
 * @see xacml_response_set_xacml_status_code()
 */
xacml_result_t
xacml_response_get_xacml_status_code(
    const xacml_response_t              response,
    xacml_status_code_t *               status_code)
{
    if (response == NULL || status_code == NULL)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }
    *status_code = response->xacml_status_code;

    return XACML_RESULT_SUCCESS;
}
/* xacml_response_get_xacml_status_code() */

/**
 * Add an obligation to a response
 * @ingroup xacml_request
 *
 * @param response
 *     Response to add the obligation to.
 * @param obligation
 *     Value of the obligation, including any attributes associated with it.
 *     The caller may free this obligation after this function returns.
 *
 * @retval XACML_RESULT_SUCCESS
 *     Success.
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter.
 */
xacml_result_t
xacml_response_add_obligation(
    xacml_response_t                    response,
    const xacml_obligation_t            obligation)
{

    if (response == NULL || obligation == NULL)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }

    response->obligations.push_back(*obligation);

    return XACML_RESULT_SUCCESS;
}
/* xacml_response_add_obligation() */

/**
 * Count the number of obligations in a response
 * @ingroup xacml_request
 * Modifies the value pointed to by @a count to contain the number of
 * obligations in @a response. Values from 0 to the value returned
 * in @a count can be passed to xacml_response_get_obligation() to
 * iterate through the set of obligations.
 *
 * @param response
 *     Response to inspect.
 * @param count
 *     Pointer to be set to the number of subject attributes.
 *
 * @retval XACML_RESULT_SUCCESS
 *     Success.
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter.
 *
 * @see xacml_response_get_obligation()
 */
xacml_result_t
xacml_response_get_obligation_count(
    const xacml_response_t              response,
    size_t *                            count)
{
    if (response == NULL || count == NULL)
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }

    *count = response->obligations.size();

    return XACML_RESULT_SUCCESS;
}
/* xacml_response_get_obligation_count() */

/**
 * Get the value of an obligation
 * @ingroup xacml_response
 *
 * Retrieves the obligation based on the its index. The total number of subject
 * obligations can be determined by calling
 * xacml_response_get_obligation_count().
 * 
 * @param response
 *     The response to inspect.
 * @param num
 *     Obligation index.
 * @param obligation
 *     Pointer to be set to the obligation. The caller
 *     must not modify or free this value.
 *
 * @retval XACML_RESULT_SUCCESS
 *     Success.
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter.
 *
 * @see xacml_response_add_obligation(),
 *      xacml_response_get_obligation_count()
 */
xacml_result_t
xacml_response_get_obligation(
    const xacml_response_t              response,
    size_t                              num,
    xacml_obligation_t *                obligation)
{
    if (response == NULL || 
        obligation == NULL ||
        num > response->obligations.size())
    {
        return XACML_RESULT_INVALID_PARAMETER;
    }

    *obligation = &response->obligations[num];

    return XACML_RESULT_SUCCESS;
}
/* xacml_response_get_obligation() */

/**
 * Set the value of the request attributes used to reach a decision
 * @ingroup xacml_response
 *
 * @param response
 *     XACML response to update.
 * @param request
 *     Request containing the set of attributes that where used by the
 *     policy implementation to reach the decision.
 *
 * @retval XACML_RESULT_SUCCESS
 *     Success.
 * @retval XACML_RESULT_INVALID_PARAMETER
 *     Invalid parameter.
 */
xacml_result_t
xacml_response_set_request_context(
    xacml_response_t                    response,
    xacml_request_t                     request)
{
    xacml_result_t                      result;

    if (response == NULL || request == NULL)
    {
        result = XACML_RESULT_INVALID_PARAMETER;

        goto out;
    }

    if (response->request != NULL)
    {
        xacml_request_destroy(response->request);
    }

    result = xacml_request_init(&response->request);
    if (result != XACML_RESULT_SUCCESS)
    {
        goto out;
    }

    response->request->subjects = request->subjects;
    response->request->resource_attributes = request->resource_attributes;
    response->request->action_attributes = request->action_attributes;
    response->request->environment_attributes = request->environment_attributes;
    response->request->subject = request->subject;

out:
    return result;
}
/* xacml_response_set_request_context() */

