/*
 * Copyright 1999-2008  University of Chicago
 * Copyright 2008-2010  Members of the EGEE Collaboration
 * Copyright 2008-  Stichting Fundamenteel Onderzoek der Materie (FOM-Nikhef)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef XACML_H
#define XACML_H

#include "xacml_datatypes.h"

#ifdef __cplusplus
#include <ctime> /* Need <ctime> instead of <time.h> for C++, otherwise error:
                    'time_t' in namespace 'std' does not name a type */
#else
#include <time.h>
#endif

#ifndef DONT_DOCUMENT_INTERNAL
EXTERN_C_BEGIN
#endif

/**
 * @mainpage SAML 2.0 profile of XACML v2.0 Implementation
 *
 * This API provides a basic implementation of the SAML 2.0 profile of XACML
 * v2.0, including support for obligations in XACML response messages. It aids
 * in writing XACML clients and servers.
 *
 * - @link xacml_common Datatypes @endlink
 * - @link xacml_client Client Functions @endlink
 * - @link xacml_server Server functions @endlink
 * - @link xacml_io I/O Handlers @endlink
 */

/** 
 * @defgroup xacml_common Common Library Functions and Constants
 */

/**
 * @defgroup xacml_server Server Library Functions
 */

xacml_result_t
xacml_init(void);

xacml_result_t
xacml_request_init(
    xacml_request_t *                   request);

void
xacml_request_destroy(
    xacml_request_t                     request);


xacml_result_t
xacml_request_set_io_module(
    xacml_request_t                     request,
    const char *                        module);

xacml_result_t
xacml_request_set_io_descriptor(
    xacml_request_t                     request,
    const xacml_io_descriptor_t        *descriptor);

xacml_result_t
xacml_request_add_subject_attribute(
    xacml_request_t                     request,
    const char *                        subject_category,
    const char *                        attribute_id,
    const char *                        data_type,
    const char *                        issuer,
    const char *                        value);

xacml_result_t
xacml_request_get_subject_attribute_count(
    const xacml_request_t               request,
    size_t *                            count);

xacml_result_t
xacml_request_get_subject_attribute(
    const xacml_request_t               request,
    size_t                              num,
    const char **                       subject_category,
    const char **                       attribute_id,
    const char **                       data_type,
    const char **                       issuer,
    const char **                       value);

xacml_result_t
xacml_request_set_return_context(
    const xacml_request_t               request,
    int                                 return_context);

xacml_result_t
xacml_request_get_return_context(
    const xacml_request_t               request,
    int *                               return_context);

xacml_result_t
xacml_resource_attribute_init(
    xacml_resource_attribute_t *        attribute);

xacml_result_t
xacml_resource_attribute_add(
    xacml_resource_attribute_t          attribute,
    const char *                        attribute_id,
    const char *                        data_type,
    const char *                        issuer,
    const char *                        value);

void
xacml_resource_attribute_destroy(
    xacml_resource_attribute_t          attribute);

xacml_result_t
xacml_resource_attribute_get_count(
    xacml_resource_attribute_t          attribute,
    size_t *                            count);

xacml_result_t
xacml_resource_attribute_get_attribute(
    const xacml_resource_attribute_t    attribute,
    size_t                              num,
    const char **                       attribute_id,
    const char **                       data_type,
    const char **                       issuer,
    const char **                       value);

xacml_result_t
xacml_request_add_resource_attribute(
    xacml_request_t                     request,
    const xacml_resource_attribute_t    attribute);

xacml_result_t
xacml_request_get_resource_attribute_count(
    const xacml_request_t               request,
    size_t *                            count);

xacml_result_t
xacml_request_get_resource_attribute(
    const xacml_request_t               request,
    size_t                              num,
    xacml_resource_attribute_t *        attribute);

xacml_result_t
xacml_request_add_action_attribute(
    xacml_request_t                     request,
    const char *                        attribute_id,
    const char *                        data_type,
    const char *                        issuer,
    const char *                        value);

xacml_result_t
xacml_request_get_action_attribute_count(
    const xacml_request_t               request,
    size_t *                            count);

xacml_result_t
xacml_request_get_action_attribute(
    const xacml_request_t               request,
    size_t                              num,
    const char **                       attribute_id,
    const char **                       data_type,
    const char **                       issuer,
    const char **                       value);

xacml_result_t
xacml_request_add_environment_attribute(
    xacml_request_t                     request,
    const char *                        attribute_id,
    const char *                        data_type,
    const char *                        issuer,
    const char *                        value);

xacml_result_t
xacml_request_get_environment_attribute_count(
    const xacml_request_t               request,
    size_t *                            count);

xacml_result_t
xacml_request_get_environment_attribute(
    const xacml_request_t               request,
    size_t                              num,
    const char **                       attribute_id,
    const char **                       data_type,
    const char **                       issuer,
    const char **                       value);

xacml_result_t
xacml_request_set_subject(
    xacml_request_t                     request,
    const char *                        subject);

xacml_result_t
xacml_request_get_subject(
    const xacml_request_t               request,
    const char **                       subject);

xacml_result_t
xacml_request_set_issue_instant(
    xacml_request_t                    request,
    time_t                              issue_instant);

xacml_result_t
xacml_request_get_issue_instant(
    xacml_request_t                    request,
    time_t *                            issue_instant);

const char *
xacml_request_set_id(
    xacml_request_t                    request,
    const char *                        id);

xacml_result_t
xacml_request_get_id(
    const xacml_request_t              request,
    const char **                       id);

xacml_result_t
xacml_request_set_issuer(
    xacml_request_t                    request,
    const char *                        issuer);

xacml_result_t
xacml_request_get_issuer(
    xacml_request_t                    request,
    const char **                       issuer);


/* XACML Response */
xacml_result_t
xacml_response_init(
    xacml_response_t *                  response);

void
xacml_response_destroy(
    xacml_response_t                    response);

xacml_result_t
xacml_response_set_issue_instant(
    xacml_response_t                    response,
    time_t                              issue_instant);

xacml_result_t
xacml_response_get_issue_instant(
    xacml_response_t                    response,
    time_t *                            issue_instant);

const char *
xacml_response_set_id(
    xacml_response_t                    response,
    const char *                        id);

xacml_result_t
xacml_response_get_id(
    const xacml_response_t              response,
    const char **                       id);

xacml_result_t
xacml_response_set_issuer(
    xacml_response_t                    response,
    const char *                        issuer);

xacml_result_t
xacml_response_get_issuer(
    xacml_response_t                    response,
    const char **                       issuer);

xacml_result_t
xacml_response_set_in_response_to(
    xacml_response_t                    response,
    const char *                        in_response_to);

xacml_result_t
xacml_response_get_in_response_to(
    xacml_response_t                    response,
    const char **                       in_response_to);

xacml_result_t
xacml_response_set_saml_status_code(
    xacml_response_t                    response,
    saml_status_code_t                  status_code);

xacml_result_t
xacml_response_get_saml_status_code(
    const xacml_response_t              response,
    saml_status_code_t *                status_code);

xacml_result_t
xacml_response_set_xacml_decision(
    xacml_response_t                    response,
    xacml_decision_t                    decision);

xacml_result_t
xacml_response_get_xacml_decision(
    const xacml_response_t              response,
    xacml_decision_t *                  decision);

xacml_result_t
xacml_response_set_xacml_status_code(
    xacml_response_t                    response,
    xacml_status_code_t                 status_code);

xacml_result_t
xacml_response_get_xacml_status_code(
    const xacml_response_t              response,
    xacml_status_code_t *               status_code);

xacml_result_t
xacml_obligation_init(
    xacml_obligation_t *                obligation,
    const char *                        obligation_id,
    xacml_effect_t                      fulfill_on);

void
xacml_obligation_destroy(
    xacml_obligation_t                  obligation);

xacml_result_t
xacml_obligation_add_attribute(
    xacml_obligation_t                  obligation,
    const char *                        attribute_id,
    const char *                        data_type,
    const char *                        value);

xacml_result_t
xacml_obligation_get_id(
    const xacml_obligation_t            obligation,
    const char **                       obligation_id);

xacml_result_t
xacml_obligation_get_effect(
    const xacml_obligation_t            obligation,
    xacml_effect_t *                    fulfill_on);

xacml_result_t
xacml_obligation_get_attribute_count(
    const xacml_obligation_t            obligation,
    size_t *                            count);

xacml_result_t
xacml_obligation_get_attribute(
    const xacml_obligation_t            obligation,
    size_t                              num,
    const char **                       attribute_id,
    const char **                       data_type,
    const char **                       value);

xacml_result_t
xacml_response_add_obligation(
    xacml_response_t                    response,
    const xacml_obligation_t            obligation);

xacml_result_t
xacml_response_get_obligation_count(
    const xacml_response_t              response,
    size_t *                            count);

xacml_result_t
xacml_response_get_obligation(
    const xacml_response_t              response,
    size_t                              num,
    xacml_obligation_t *                obligation);

xacml_result_t
xacml_response_set_request_context(
    xacml_response_t                    response,
    xacml_request_t                     request);

#ifndef DONT_DOCUMENT_INTERNAL
EXTERN_C_END
#endif

#endif /* XACML_H */
