#------------------------------------------------------------------------------
# Copyright 2008-2012 Istituto Nazionale di Fisica Nucleare (INFN)
#
# Licensed under the EUPL, Version 1.1 only (the "Licence").
# You may not use this work except in compliance with the Licence.
# You may obtain a copy of the Licence at:
#
# http://joinup.ec.europa.eu/system/files/EN/EUPL%20v.1.1%20-%20Licence.pdf
#
# Unless required by applicable law or agreed to in
# writing, software distributed under the Licence is
# distributed on an "AS IS" basis,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
# either express or implied.
# See the Licence for the specific language governing
# permissions and limitations under the Licence.
#------------------------------------------------------------------------------
"""
This file is part of WNoDeS.
"""

import sys
import os
import smtplib
import gzip
#import logging
import logging.handlers
import inspect
import subprocess
import time
import ConfigParser

__version__ = (1, 1, 4)

LOG_LEVELS = {'debug': logging.DEBUG,
          'info': logging.INFO,
          'warning': logging.WARNING,
          'error': logging.ERROR,
          'critical': logging.CRITICAL}


class arbitrary_dict(dict):
    """
    A dictionary which applies an arbitrary key-altering function before
    accessing the keys
    """

    def __keytransform__(self, key):
        return key

    # Overrided methods. List from a
    #http://stackoverflow.com/questions/2390827/how-to-properly-subclass-dict
    def __init__(self, *args, **kwargs):
        self.update(*args, **kwargs)

    def __getitem__(self, key):
        return super(arbitrary_dict, self).__getitem__(
            self.__keytransform__(key))

    def __setitem__(self, key, value):
        return super(arbitrary_dict, self).__setitem__(
            self.__keytransform__(key), value)

    def __delitem__(self, key):
        return super(arbitrary_dict, self).__delitem__(
            self.__keytransform__(key))

    def __contains__(self, key):
        return super(arbitrary_dict, self).__contains__(
            self.__keytransform__(key))

    def update(self, *args, **kwargs):
        trans = self.__keytransform__
        super(arbitrary_dict, self).update(
            *[(trans(k), v) for k, v in args],
            **dict((trans(k), kwargs[k]) for k in kwargs))


class lcdict(arbitrary_dict):

    def __keytransform__(self, key):
        return str(key).lower()


def daemonize(stdin="/dev/null", stdout="/tmp/out", stderr="/tmp/err"):
    """ A function to daemonize the current process.
        Taken from the Python Cookbook."""

    # Perform first fork.
    try:
        pid = os.fork()
        if pid > 0:
            sys.exit(0)  # Exit first parent.
    except OSError, e:
        sys.stderr.write("fork #1 failed: (%d) %sn" % (e.errno, e.strerror))
        sys.exit(1)
    # Decouple from parent environment.
    os.chdir("/")
    os.umask(0)
    os.setsid()
    # Perform second fork.
    try:
        pid = os.fork()
        if pid > 0:
            sys.exit(0)  # Exit second parent.
    except OSError, e:
        sys.stderr.write("fork #2 failed: (%d) %sn" % (e.errno, e.strerror))
        sys.exit(1)
    # The process is now daemonized, redirect standard file descriptors.
    sys.stdout.flush()
    sys.stderr.flush()
    si = file(stdin, 'r')
    so = file(stdout, 'a+')
    se = file(stderr, 'a+', 0)
    os.dup2(si.fileno(), sys.stdin.fileno())
    os.dup2(so.fileno(), sys.stdout.fileno())
    os.dup2(se.fileno(), sys.stderr.fileno())


def sendmail(m_server, m_from, m_to, m_subject, msg):
    """Send an email using smtplib."""

    msg = "From: %s\r\nTo: %s\r\nSubject: %s\r\n\r\n" % \
        (m_from, m_to, m_subject) + msg

    srv = smtplib.SMTP(m_server)
    srv.sendmail(m_from, m_to, msg)
    srv.quit()


def synchronized(lock):
    """ Synchronization decorator. """
    def wrap(func):
        def new_function(*args, **kw):
            lock.acquire()
            try:
                return func(*args, **kw)
            finally:
                lock.release()
        new_function.__name__ = func.__name__
        new_function.__doc__ = func.__doc__
        return new_function
    return wrap


def whoami():
    """ Return the name of the calling function. """
    return inspect.stack()[1][3]


def getmyargs():
    """ Return name and value of the args of the calling function. """
    frame = inspect.stack()[1][0]
    args, _, _, values = inspect.getargvalues(frame)
    return [(i, values[i]) for i in args]


class RingBuffer(object):
    """ A not-yet-full buffer, implemented as an iterable object. """
    def __init__(self, size_max):
        self.max = size_max
        self.data = []

    def append(self, x):
        """ append an element at the end of the buffer. """
        self.data.append(x)
        if self.max and (len(self.data) == self.max):
            self.cur = 0
            # Permanently change access methods from non-full to full
            self.append = self._full_append
            self.get = self._full_get

    def get(self):
        """ Return a list of elements from the oldest to the newest. """
        return self.data

    def _full_append(self, x):
        """ Append an element overwriting the oldest one. """
        self.data[self.cur] = x
        self.cur = (self.cur + 1) % self.max

    def _full_get(self):
        """ return list of elements in correct order. """
        return self.data[self.cur:] + self.data[:self.cur]

    def __iter__(self):
        self.index = 0
        return self

    def next(self):
        if self.index == len(self.data):
            raise StopIteration
        else:
            self.index = self.index + 1
            return self.data[self.index - 1]

    def __repr__(self):
        return str(self.get())


class CompressedRotatingFileHandler(logging.handlers.RotatingFileHandler):
    """ A class extending RotatingFileHandler to compress files. """
    def doRollover(self):
        """ Do a rollover and compress the files. """
        self.stream.close()
        if self.backupCount > 0:
            for i in range(self.backupCount - 1, 0, -1):
                sfn = "%s.%d.gz" % (self.baseFilename, i)
                dfn = "%s.%d.gz" % (self.baseFilename, i + 1)
                if os.path.exists(sfn):
                    if os.path.exists(dfn):
                        os.remove(dfn)
                    os.rename(sfn, dfn)
            dfn = self.baseFilename + ".1.gz"
            if os.path.exists(dfn):
                os.remove(dfn)
            # gzip here
            f_in = open(self.baseFilename, "r")
            f_out = gzip.open(dfn, "w")
            f_out.writelines(f_in)
            f_out.close()
            f_in.close()
            os.remove(self.baseFilename)
        self.mode = "w"
        self.stream = open(self.baseFilename, "w")

# try to avoid using the deprecated md5 module
try:
    import hashlib
    _MD5_CALL = hashlib.md5
except ImportError:
    # for Python < 2.5
    import md5
    _MD5_CALL = md5.new


def md5sum(name):
    """ Return an md5 hash for a file name, for stdin if name is "-",
        or for a file-like object. Return False in case of error. """

    if isinstance(name, file):
        # parameter is a file descriptor
        m = _MD5_CALL()

        while True:
            d = name.read(8096)
            if not d:
                break
            m.update(d)
        return m.hexdigest()

    if name == "-":
        # parameter is stdin
        return md5sum(sys.stdin)
    else:
        # parameter is a file name
        try:
            fdesc = open(name, "rb")
        except IOError:
            return False
        else:
            return md5sum(fdesc)


def runCommand(cmd, timeout=20, use_timeout='yes'):
    """ Convenience routine to execute commands. """

    wait_time = 0
    exec_cmd = subprocess.Popen(cmd, bufsize=0, shell=True,
                                stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    while 1:
        if exec_cmd.poll() == 0 and exec_cmd.poll() is not None:
                (output, error) = exec_cmd.communicate()
                pid = exec_cmd.pid
                returncode = 0
                break
        elif  exec_cmd.poll() != 0 and exec_cmd.poll() is not None:
                time.sleep(5)
                if exec_cmd.poll() != 0 and exec_cmd.poll() is not None:
                    (output, error) = exec_cmd.communicate()
                    returncode = 1
                    pid = exec_cmd.pid
                    break
        elif use_timeout == 'yes':
                if not wait_time >= timeout:
                    wait_time = wait_time + 1
                    time.sleep(1)
                else:
                    subprocess.Popen(['kill', '-9', str(exec_cmd.pid)],
                                     bufsize=0)
                    output = ''
                    error = ''
                    pid = ''
                    returncode = 2
                    break
        elif use_timeout == 'no':
                time.sleep(5)
    returnstuff = [returncode, output, error, pid]
    return returnstuff


def getValueFromFileSectionOption(my_file, section, option):
    """ get the values for a specific options
        in a [section] of a ConfigParse file"""
    try:
        conf = ConfigParser.RawConfigParser()
        conf.read(my_file)
        return_list = conf.get(section, option)
        return [0, return_list]
    except ConfigParser.NoSectionError:
        err_msg = "Cannot find a section called: %s, \
        error %s" % (section, str(sys.exc_info()[0]))
        return [1, err_msg]
    except ConfigParser.NoOptionError:
        err_msg = "Cannot find an option called: %s, \
        error %s" % (option, str(sys.exc_info()[0]))
        return [1, err_msg]
    except Exception:
        err_msg = "Exception in section: %s, \
        error %s, %s, %s" % (section, sys.exc_info()[:])
        return [1, err_msg]

try:
    # the uuid module was introduced with Python 2.5
    import uuid

    def guid():
        uid = str(uuid.uuid1()).upper()
        return uid
except ImportError:
    import random
    import time
    # the hashlib has a specific python component for python version less than
    # 2.6
    import hashlib
    import socket

    def guid():
        t = long(time.time() * 1000)
        r = long(random.random() * 100000000000000000L)
        try:
            a = socket.gethostbyname(socket.gethostname())
        except socket.error:
            # if we can't get a network address, just imagine one
            a = random.random() * 100000000000000000L
        data = str(t) + ' ' + str(r) + ' ' + str(a)
        data = hashlib.md5(data).hexdigest().lower()
        uid = (data[:8] + '-' + data[8:12] + '-' + data[12:16] + '-'
               + data[16:20] + '-' + data[20:])
        return uid
