#------------------------------------------------------------------------------
# Copyright 2008-2012 Istituto Nazionale di Fisica Nucleare (INFN)
# 
# Licensed under the EUPL, Version 1.1 only (the "Licence").
# You may not use this work except in compliance with the Licence.
# You may obtain a copy of the Licence at:
# 
# http://joinup.ec.europa.eu/system/files/EN/EUPL%20v.1.1%20-%20Licence.pdf
# 
# Unless required by applicable law or agreed to in
# writing, software distributed under the Licence is
# distributed on an "AS IS" basis,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
# either express or implied.
# See the Licence for the specific language governing
# permissions and limitations under the Licence.
#------------------------------------------------------------------------------
#!/bin/env python
"""
A tail module written in Python.

It can be called either directly, or imported by
other Python programs in the following way:

import tail
t = tail.Tailer("filename")

See the docs for the class Tailer for details.
"""

__version__ = "1.4.0"
__author__ = "Davide Salomoni"

import os
import sys
import time
import getopt
import threading
import Queue
import xmlrpclib
import urlparse

try:
    True, False
except NameError:
    setattr(__builtins__, "True", 1)
    setattr(__builtins__, "False", not True)


class Tailer(object):
    """
    A class to tail files.
    """
    NLINES = 10
    STDIN_NAME = "standard input"

    def __init__(self, *files, **kw):
        """
        Initialize the Tailer.

        files can be specified by file name or by file handler. Multiple files
        should be passed as separate arguments, e.g. Tailer(file1, file2).

        Keyword arguments:
            - buffered (False)
                Use buffered=True, which reads the entire input into memory,
                when using streams (e.g. sys.stdin) or string buffers
                (e.g. StringIO.StringIO).
            - lines (10)
                Number of lines to tail.
            - printing (False)
                Print lines to stdout.
            - handler (None)
                Call handler for every line. Handler must be a function taking
                two arguments, the name of the file and the line being
                tailed/followed.

                If handler is a string beginning with "http://" or "https://",
                it is assumed to be a URI referring to an XML-RPC call.
                Remember that an XML-RPC handler should return something, maybe
                just an empty string, i.e. "".

                An XML-RPC handler could be implemented in Python on the
                server side for example with:
                    def remoteHandler(name, line):
                        do_something_with(name, line)
                        return ""

                    from SimpleXMLRPCServer import SimpleXMLRPCServer
                    server = SimpleXMLRPCServer(("localhost", 8888))
                    server.register_function(remoteHandler)
                    server.serve_forever()

                In this example the handler name for the Tailer class would be
                    http://localhost:8888/remoteHandler
        """

        self.buffered = kw.get('buffered', False)
        self.lines = kw.get('lines', self.NLINES)
        self.printing = kw.get('printing', False)
        self.handler = kw.get('handler', None)
        self.remoteHandler = False

        if self.handler:
            if isinstance(self.handler, str):
                h = self.handler.lower()
                if h.startswith("http://") or h.startswith("https://"):
                    url = urlparse.urlsplit(h)
                    self.remoteHandler = True
                    self.remoteServer = xmlrpclib.ServerProxy(url[0]+"://"+url[1])
                    self.handler = url[2][1:]  # remove leading '/'
                else:
                    raise TypeError, "unrecognized XML-RPC handler '%s'" % \
                          self.handler
            else:
                if not callable(self.handler):
                    raise TypeError, "handler (%s) is not callable" % \
                          type(self.handler)

        self.detached = False  # used to signal if the threads are detached
        self.multiple = False
        self.table = []  # list of (file name, file handler)
        self.files = files
        self.stdin = False  # are we processin stdin?
        self.handlerLock = threading.Lock()

        if self.buffered:
            # handle the case of streams (e.g. sys.stdin) or string buffers.
            self._getLines = self._getLinesBuffered
            self.stdin = True
            for stream in self.files:
                self.table.append((self.STDIN_NAME, stream))
        else:
            self._getLines = self._getLinesUnbuffered
            for fn in self.files:
                if isinstance(fn, file):
                    # argument is a file handler
                    self.table.append((fn.name, fn))
                else:
                    # argument is a file name
                    self.table.append((fn, open(fn, 'r')))

        if len(self.files) > 1:
            self.multiple = True

    def tail(self):
        """
        Tail the current files(s).
        Return a list of the last lines.
        """

        result = []
        if self.lines <= 0:
            for (filename, fh) in self.table:
                fh.seek(0, 2)  # move to EOF
            return result

        # how to call the handler (XML-RPC vs. local)
        if self.remoteHandler:
            handlerCall = "self.remoteServer.%s(filename, line)" % \
                          self.handler
        else:
            handlerCall = "self.handler(filename, line)"

        for (filename, fh) in self.table:
            if self.multiple:
                header = "==> %s <==" % filename
                result.append(header)
                if self.printing:
                    print header

            for line in self._getLines(fh, self.lines):
                result.append(line)
                if self.printing:
                    print line
                if self.handler:
                    # note that the handler only receives lines belonging
                    # to the file(s), not lines like "==> name <==" ; these
                    # only appear in the list returned by tail (or follow).
                    try:
                        eval(handlerCall)
                    finally:
                        pass

            if self.multiple:
                result.append("")
                if self.printing:
                    print

        return result

    def follow(self, detached=False, until=None, store=False, interval=1.0):
        """
        Follow the current file(s).
        Return None or a list of the last lines in the current files.

        Keyword arguments:
            - detached (False)
                Run detached.
            - until (None)
                Run until the specified time. The format is "YYYYMMDDhhmm",
                e.g. until="200406252035".
            - store (False)
                Store output in a list and return that list.
            - interval (1.0)
                Set how often (in seconds) the program should check for new
                lines in the file(s) being followed.
        """

        if self.stdin:
            # following stdin is not supported
            return self.tail()

        if self.detached:
            raise Exception, "detached instance already running"

        self.until = until
        self.store = store
        self.interval = interval
        self.qLines = Queue.Queue(0)  # used by the threads to store tail output

        # First of all, tail the last lines.
        # Call tail() in a separate thread so that if we are deferred we can
        # return as soon as possible (even if there is a handler that takes a
        # long time to complete), and make sure that the Follower threads do
        # not start until that thread has completed.
        self._tailDone = threading.Event()
        self._tailThread = threading.Thread(target=self._tailWrapper)
        self._tailThread.start()

        self.threads = []
        # Now follow file output via threads.
        for (filename, fh) in self.table:
            f = _Follower(filename, fh, self)
            f.start()
            self.threads.append(f)

        if detached:
            self.detached = True
            return None

        try:
            # infinite loop terminating on KeyboardInterrupt
            while True:
                if self._isTimeToGo():
                    raise KeyboardInterrupt
                else:
                    # the sleep here should not be confused with the sleep
                    # in the threads (controled by the 'interval' parameter).
                    time.sleep(1)

        except KeyboardInterrupt:
            # terminate threads and get queue output
            self._stopThreads()
            lines = self._getQueue()
            if lines:
                self.result.extend(lines)

            if store:
                return self.result
            else:
                return None

    def _tailWrapper(self):
        """
        Wrapper method used to execute self.tail() in a separated thread.
        When done, notify all waiting Follower threads.
        """
        self.result = self.tail()
        self._tailDone.set()

    def end(self):
        """
        Stop a detached tail and return the tailed lines.
        """
        if not self.detached:
            return

        self._stopThreads()
        lines = self._getQueue()
        if lines:
            self.result.extend(lines)

        self.detached = False
        return self.result

    def _isTimeToGo(self):
        """
        Check if it is time to exit the follow() loop.
        """
        if self.until and (time.strftime("%Y%m%d%H%M") > self.until):
            return True
        else:
            return False

    def _stopThreads(self):
        """
        Stop all running Follower threads.
        """
        for thread in self.threads:
            thread.stop()

        if self._tailThread.isAlive():
            # the Follower threads are all waiting for tail() to finish,
            # so it is safe to return; they will end as soon as tail()
            # completes.
            return

        # now wait until all threads are really gone.
        while 1:
            n = [thread for thread in self.threads if thread.isAlive()]
            if not n:
                break
            else:
                time.sleep(0.1)

    def _getQueue(self):
        """
        Return the content of the queue where tailed lines are stored.
        """
        content = []
        while True:
            try:
                item = self.qLines.get(False)
            except Queue.Empty:
                # the queue is empty, we are done
                break
            content.append(item)
    
        return content
    
    def _getLinesBuffered(self, fd, linesback):
        # A version of the tailer that reads all lines into memory.
        # Not recommended for very large files.
        lines = fd.read().splitlines()
        return lines[-linesback:]

    def _getLinesUnbuffered(self, fd, linesback):
        # Contributed to the Python Cookbook by Ed Pascoe (2003).
        # Note that it does not work for piped streams or for string buffers.
        avgcharsperline = 75
    
        while True:
            try:
                fd.seek(-1 * avgcharsperline * linesback, 2)
            except IOError:
                fd.seek(0)
    
            if fd.tell() == 0:
                atstart = 1
            else:
                atstart = 0
    
            lines = fd.read().split("\n")
            if (len(lines) > (linesback+1)) or atstart:
                break
    
            avgcharsperline=avgcharsperline * 1.3
    
        if len(lines) > linesback:
            start = len(lines) - linesback - 1
        else:
            start = 0

        return lines[start:len(lines)-1]

class _Follower(threading.Thread):
    def __init__(self, filename, fh, tailer):
        """
        The constructor takes the following parameters:
            - the file name being followed
            - the open file handler
            - an instance of the Tailer class.
        """

        threading.Thread.__init__(self)

        self.exit = False
        self.filename = filename
        self.fh = fh
        self.tailer = tailer
        self.setName(self.filename)
    
    def run(self):
        tailer = self.tailer

        # how to call the handler (XML-RPC vs. local)
        if tailer.remoteHandler:
            handlerCall = "tailer.remoteServer.%s(self.filename, line)" % \
                           tailer.handler
        else:
            handlerCall = "tailer.handler(self.filename, line)"

        # do not start until the threaded tail() call has done its job
        tailer._tailDone.wait()
        
        while not self.exit:

            if tailer.detached and tailer._isTimeToGo():
                break
    
            # process line
            line = self.fh.readline()
            if not line:
                where = self.fh.tell()
                fh_results = os.fstat(self.fh.fileno())
                try:
                    st_results = os.stat(self.filename)
                except OSError:
                    st_results = fh_results
    
                if st_results[1] == fh_results[1]:
                    time.sleep(tailer.interval)
                    self.fh.seek(where)
                else:
                    msg = "%s changed inode numbers from %d to %d" % \
                          (self.filename, fh_results[1], st_results[1])
                    if tailer.store: tailer.qLines.put(msg)
                    if tailer.printing: print msg
                    self.fh = open(self.filename, 'r')
            else:
                # process the line just read
                if line.endswith('\n'): line = line[:-1]
                if tailer.multiple:
                    header = "\n==> %s <==\n%s" % (self.filename, line)
                    if tailer.store: tailer.qLines.put(header)
                    if tailer.printing: print header
                else:
                    if tailer.store: tailer.qLines.put(line)
                    if tailer.printing: print line

                if tailer.handler:
                    # synchronize the handler across threads
                    tailer.handlerLock.acquire()
                    try:
                        eval(handlerCall)
                    finally:
                        tailer.handlerLock.release()

    def stop(self):
        """Stop this thread as soon as possible."""
        self.exit = True

def main():
        
    USAGE = """Usage: %s [OPTION] <FILE>...
  -n, --number    print the last n lines
  -f, --follow    continuous tail
  -h, --help      print this help message
  """ % os.path.basename(sys.argv[0])

    NLINES = Tailer.NLINES
    FOLLOW = False
    
    try:
        opts, args = getopt.getopt(sys.argv[1:], "n:fh", 
            ["number=", "follow", "help"])
    except getopt.error, msg:
        print msg
        print USAGE
        sys.exit(1)

    for o, v in opts:
        if o in ("-h", "--help"):
            print USAGE
            sys.exit(0)
        if o in ("-n", "--number"):
            NLINES = int(v)
        if o in ("-f", "--follow"):
            FOLLOW = True
    
    if len(args)==0:
        # process STDIN
        tailer = Tailer(sys.stdin, buffered=True)
    else:
        tailer = Tailer(*args)

    tailer.printing = True
    tailer.lines = NLINES

    if FOLLOW:
        tailer.follow(store=False)
    else:
        tailer.tail()

if __name__ == "__main__":
    try:
        main()
    except KeyboardInterrupt:
        pass
