#------------------------------------------------------------------------------
# Copyright 2008-2012 Istituto Nazionale di Fisica Nucleare (INFN)
#
# Licensed under the EUPL, Version 1.1 only (the "Licence").
# You may not use this work except in compliance with the Licence.
# You may obtain a copy of the Licence at:
#
# http://joinup.ec.europa.eu/system/files/EN/EUPL%20v.1.1%20-%20Licence.pdf
#
# Unless required by applicable law or agreed to in
# writing, software distributed under the Licence is
# distributed on an "AS IS" basis,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
# either express or implied.
# See the Licence for the specific language governing
# permissions and limitations under the Licence.
#------------------------------------------------------------------------------
"""
This is a plugin file for WNoD.
"""

import tempfile
import commands
import os
import sys
try:
    from wnodes.utils import utils
except ImportError:
    sys.exit("python module 'utils' not found.")


def _write_cpu_info(image, dev):
    """ Write info about the dom0 CPU into a file in the VM. """

    CPU_INFO_FILE = "/etc/wnodes/hypervisor/wnodes-realcpu.cfg"

    cmdline = """
#!/bin/sh

cpu=`grep 'model name' /proc/cpuinfo | uniq | awk -F':' '{print $2}' | sed -e 's/^[ \t]*//;s/[ \t]*$//;s/  */ /g'`

guestfish << EOF
       add %s
       run
       mount %s /
       write_file %s "$cpu" 0
EOF
    """ % (image, dev, CPU_INFO_FILE)
    fd, name = tempfile.mkstemp(text=True)
    os.write(fd, cmdline)
    os.close(fd)
    os.chmod(name, 0777)
    commands.getoutput(name)
    os.unlink(name)


def _modify_syslog_conf(image, hv_hostname, dev):
    """ Direct the VM syslogd messages to the current hypervisor. """

    cmdline = """
#!/bin/sh


guestfish << EOF
       add %s
       run
       mount %s /
       sh 'echo "*.*    @%s" >> /etc/syslog.conf'
EOF
    """ % (image, dev, hv_hostname)
    fd, name = tempfile.mkstemp(text=True)
    os.write(fd, cmdline)
    os.close(fd)
    os.chmod(name, 0777)
    out = commands.getoutput(name)
    print out
    os.unlink(name)


def _enable_virtio(image, dev):
    """ Modify modprobe.conf to enable the virtio driver. """

    # TODO: does this work for all O/S?
    return
    # change"alias eth0 e1000" in "alias eth0 virtio"
    cmdline = """
#!/bin/sh

guestfish << EOF
    add %s
    run
    mount %s /
    sh "perl -ni -e 'print unless /alias eth/' /etc/modprobe.conf"
    sh "rm -f /etc/sysconfig/network-scripts/ifcfg-eth0"
EOF
    """ % (image, dev)
    fd, name = tempfile.mkstemp(text=True)
    os.write(fd, cmdline)
    os.close(fd)
    os.chmod(name, 0777)
    commands.getoutput(name)
    os.unlink(name)


def _copy_ssh_key(image, ssh_key_file, dev):
    """
        Copy ssh key to each VM istance
    """
    pub_key_filename = '%s.pub' % ssh_key_file
    pub_key_file = open(pub_key_filename)
    SSH_PUB_KEY = pub_key_file.readline().strip()
    pub_key_file.close()

    cmdline = """
#!/bin/sh

guestfish << EOF
    add %s
    run
    mount %s /
    sh 'mkdir -p /root/.ssh'
    sh 'echo %s >> /root/.ssh/authorized_keys'
EOF
    """ % (image, dev, SSH_PUB_KEY)
    fd, name = tempfile.mkstemp(text=True)
    os.write(fd, cmdline)
    os.close(fd)
    os.chmod(name, 0777)
    out = commands.getoutput(name)
    print out
    os.unlink(name)


def _docet_status_set(status, vm_name):
    """
    Set VM status in the DOCET database.
    """
    try:
        import docet
    except ImportError:
        raise Exception, "%s: cannot import the DOCET module" % (utils.whoami())

    DOCET_CONF_FILE = "/etc/docet.conf"
    if not os.path.isfile(DOCET_CONF_FILE):
        raise Exception, "%s: DOCET configuration file not found: %s" % (utils.whoami(), DOCET_CONF_FILE)

    q = docet.docet(DOCET_CONF_FILE)

    s = status.upper()
    if s is "START":
        pass
    elif s is "STOP":
        pass
    else:
        raise Exception, "%s: unknown status received: %s" % (utils.whoami(), status)


def post_copy_image(*args):
    """
    Called after an image copy has been performed. Parameters:
    - image : the KVM image which has been copied.
    - logger : a function which can be used to log messages,
               to the WNoDeS log file.
               This function takes a message as a mandatory argument,
               and an optional logging level,
               as defined by the Python logging module.
    - ssh key file : a public SSH key for the VM
    - hv hostname : the hostname of the HV
    """
    image = args[0]
    logger = args[1]
    ssh_key_file = args[2]
    hv_hostname = args[3]
    dev = args[4]

    if not os.path.isfile(image):
        logger(" -- VM image not found: %s" % image, "error")
        return

    _write_cpu_info(image, dev)
    _modify_syslog_conf(image, hv_hostname, dev)
    _enable_virtio(image, dev)
    _copy_ssh_key(image, ssh_key_file, dev)


def post_vm_create(vm_name, logger):
    """
    Called after a VM has been created. Parameters:
    - vm_name: the name of the VM that has been created.
    - logger : a function which can be used to log messages
               to the WNoDeS log file.
               This function takes a message as a mandatory argument
               and an optional logging level,
               as defined by the Python logging module.

    """
    #_docet_status_set("START", vm_name)


def post_vm_destroy(vm_name, logger):
    """
    Called after a VM has been destroyed. Parameters:
    - vm_name: the name of the VM that has been created.
    - logger : a function which can be used to log messages
               to the WNoDeS log file.
               This function takes a message as a mandatory argument
               and an optional logging level,
               as defined by the Python logging module.

    """
    #_docet_status_set("STOP", vm_name)
