#------------------------------------------------------------------------------
# Copyright 2008-2012 Istituto Nazionale di Fisica Nucleare (INFN)
#
# Licensed under the EUPL, Version 1.1 only (the "Licence").
# You may not use this work except in compliance with the Licence.
# You may obtain a copy of the Licence at:
#
# http://joinup.ec.europa.eu/system/files/EN/EUPL%20v.1.1%20-%20Licence.pdf
#
# Unless required by applicable law or agreed to in
# writing, software distributed under the Licence is
# distributed on an "AS IS" basis,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
# either express or implied.
# See the Licence for the specific language governing
# permissions and limitations under the Licence.
#------------------------------------------------------------------------------
"""
Part of WNoDeS framework.
Interface between WNoDeS processes an the batch system PBS
"""

from wnodes.utils import command
from wnodes.utils import batch_system

import sys
import threading
import commands

try:
    from xml.etree import ElementTree
except ImportError:
    try:
        from elementtree import ElementTree  # pylint: disable-msg=F0401
    except ImportError:
        sys.exit("package ElementTree is missing, exiting.")

class Pbsnodes(command.Command):
    def __init__(self, option, hostname):
        self.__hostname__ = hostname
        self.__option__ = option
        self.type_node = {'hopen':'-c',
            'hclose':'-o',
            '':'-x'}

    def _get_command(self):
        cmd = 'pbsnodes %s %s' % (self.type_node[self.__option__],
            self.__hostname__)
        return cmd

class Qalter(command.Command):
    def __init__(self, action, id, hostname = ''):
        self.__hostname__ = hostname
        self.__jobid__ = id
        self.__action__ = action

    def _get_command(self):
        cmd = 'qalter -l nodes=1:'
        if self.__action__ == 'add':
            cmd += '%s %s' % (self.__hostname__, self.__jobid__)
        else:
            cmd += 'bait %s' % self.__jobid__
        return cmd

class Qdel(command.Command):
    def __init__(self, jobid, user = ''):
        self.__jobid__ = jobid
        self.__user__ = user

    def _get_command(self):
        if self.__user__ == '':
            cmd = 'qdel %s' % self.__jobid__
        else:
            cmd = 'su -l %s -c "qdel %s"' % (self.__user__, self.__jobid__)
        return cmd

class Qrun(command.Command):
    def __init__(self, hostname, jobid):
        self.__hostname__ = hostname
        self.__jobid__ = jobid

    def _get_command(self):
        cmd = 'qrun -H %s %s' % (self.__hostname__, self.__jobid__)
        return cmd

class Qstat(command.Command):
    def __init__(self, jobid = '', user = ''):
        self.__user__ = user
        self.__jobid__ = jobid

    def _get_command(self):
        cmd = 'qstat -x %s' % self.__jobid__
        return cmd

    def _map_qstat_in_bjobs(self):
        map = {'Job_Id':'JOBID',
            'euser':'USER',
            'job_state':'STAT',
            'queue':'QUEUE',
            'submit_host':'FROM_HOST',
            'exec_host':'EXEC_HOST',
            'Job_Name':'JOB_NAME',
            'start_time':'SUBMIT_TIME'}
        return map

    def parse_output(self, output):
        qstat_mask = {}
        for x in self._map_qstat_in_bjobs().keys():
            qstat_mask[x] = 'Empty'

        if output[0] == 0:
            if len(output[1]) != 0:
                job_info = ElementTree.fromstring(output[1])
                for i in job_info.getiterator():
                    for x in qstat_mask.keys():
                        if x == i.tag:
                            if i.tag == 'Job_Id':
                                qstat_mask[x] = i.text.split('.')[0]
                            else:
                                qstat_mask[x] = i.text

                if qstat_mask['job_state'] == 'Q':
                    qstat_mask['job_state'] = 'PEND'
                elif qstat_mask['job_state'] == 'R':
                    qstat_mask['job_state'] = 'RUN'
                elif 'UNKNOWN' in output[1].upper():
                    qstat_mask['job_state'] = 'DONE'
                else:
                    qstat_mask['job_state'] = 'WAITING'
            else:
                if self.__jobid__ != '':
                    qstat_mask['Job_Id'] = self.__jobid__
                if self.__user__ != '':
                    qstat_mask['euser'] = self.__user__
                qstat_mask['job_state'] = 'WAITING'
        else:
            # handle the case where the job is finished
            # or the batch system does not respond to the query because busy
            if 'UNKNOWN' in output[1].upper():
                qstat_mask['job_state'] = 'DONE'
            else:
                qstat_mask['job_state'] = 'WAITING'
            if self.__jobid__ != '':
                qstat_mask['Job_Id'] = self.__jobid__
            if self.__user__ != '':
                qstat_mask['euser'] = self.__user__

        sr = '%s %s %s %s %s %s %s %s' % (qstat_mask['Job_Id'],
            qstat_mask['euser'], qstat_mask['job_state'],
            qstat_mask['queue'], qstat_mask['submit_host'],
            qstat_mask['exec_host'], qstat_mask['Job_Name'],
            qstat_mask['start_time'])

        new_output = [output[0], sr]

        return new_output

class Qsub(command.Command):
    def __init__(self, option, user = ''):
        self.__option__ = option
        self.__user__ = user

    def _get_command(self):
        cmd = 'su -l %s -c "qsub %s"' % (self.__user__, self.__option__)
        return cmd

class PbsCommands(batch_system.BatchSystem):
    """ Set of lsf batch system commands"""
    def __init__(self, expiration = 15):
        self.expiration = expiration

    def badmin(self, option, hostname):
        """badmin"""
        b_admin = Pbsnodes(option, hostname)
        b_admin_output = b_admin.get_output()
        return b_admin_output

    def bhost(self, hostname):
        """bhost""" 
        b_host = Pbsnodes('', hostname)
        b_host_output = b_host.get_output()
        return b_host_output

    def bjobs(self, jobid = '', user = ''):
        """bjobs"""
        try:
            job_id = jobid.split('.')[0]
        except:
            job_id = jobid

        b_jobs = Qstat(jobid = job_id, user = user)
        b_jobs_output = b_jobs.get_output()
        b_jobs_new_output = b_jobs.parse_output(b_jobs_output)
        return b_jobs_new_output

    def bkill(self, jobid, user = ''):
        """bkill"""
        b_kill = Qdel(jobid, user = user)
        b_kill_output = b_kill.get_output()
        return b_kill_output

    def breserve(self, action, id, hostname = ''):
        """breserve"""
        b_reserve = Qalter(action, id, hostname = hostname)
        b_reserve_output = b_reserve.get_output()
        return b_reserve_output

    def brun(self, hostname, jobid):
        """brun"""
        b_run = Qrun(hostname, jobid)
        b_run_output = b_run.get_output()
        return b_run_output

    def bsub(self, option, user = ''):
        """bsub""" 
        b_sub = Qsub(option, user = user)
        b_sub_output = b_sub.get_output()
        return b_sub_output
