#------------------------------------------------------------------------------
# Copyright 2008-2012 Istituto Nazionale di Fisica Nucleare (INFN)
#
# Licensed under the EUPL, Version 1.1 only (the "Licence").
# You may not use this work except in compliance with the Licence.
# You may obtain a copy of the Licence at:
#
# http://joinup.ec.europa.eu/system/files/EN/EUPL%20v.1.1%20-%20Licence.pdf
#
# Unless required by applicable law or agreed to in
# writing, software distributed under the Licence is
# distributed on an "AS IS" basis,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
# either express or implied.
# See the Licence for the specific language governing
# permissions and limitations under the Licence.
#------------------------------------------------------------------------------
"""
Part of WNoDeS framework.
Interface between WNoDeS processes an the batch system LSF
"""

from wnodes.utils import command
from wnodes.utils import batch_system
#import sys
#import commands
#import threading


#try:
#    from xml.etree import ElementTree
#except ImportError:
#    try:
#        from elementtree import ElementTree  # pylint: disable-msg=F0401
#    except ImportError:
#        sys.exit("package ElementTree is missing, exiting.")


class Badmin(command.Command):
    def __init__(self, profile, option, hostname):
        self.__profile__ = profile
        self.__hostname__ = hostname
        self.__option__ = option

    def _get_command(self):
        cmd = 'source %s;' % self.__profile__
        cmd += ' badmin %s' % self.__option__
        cmd += ' -C "WNoDeS admin action" %s' % self.__hostname__
        return cmd


class Bjobs(command.Command):
    def __init__(self, profile, jobid='', user=''):
        self.__profile__ = profile
        self.__jobid__ = jobid
        self.__user__ = user

    def _get_command(self):
        cmd = 'source %s; bjobs -w' % self.__profile__
        if self.__user__ != '':
            cmd += ' -u %s' % self.__user__
        if self.__jobid__ != '':
            cmd += ' %s | grep %s' % (self.__jobid__, self.__jobid__)
        return cmd

class Bhost(command.Command):
    def __init__(self, profile, hostname = ''):
        
        self.__profile__ = profile
        self.__hostname__ = hostname
        

    def _get_command(self):
        cmd = 'source %s; bhosts -w %s | grep %s' % (
                                 self.__profile__,
                                 self.__hostname__,
                                 self.__hostname__,
                                 )
        return cmd

class Bkill(command.Command):
    def __init__(self, profile, jobid, user=''):
        self.__profile__ = profile
        self.__jobid__ = jobid
        self.__user__ = user

    def _get_command(self):
        cmd = 'source %s;' % self.__profile__
        if self.__user__ != '':
            cmd += ' su -s /bin/bash -l %s -c' % self.__user__
        if self.__user__ != '':
            cmd += ' "bkill %s"' % self.__jobid__
        else:
            cmd += ' bkill %s' % self.__jobid__
        return cmd


class Bmod(command.Command):
    def __init__(self, profile, option, jobid):
        self.__profile__ = profile
        self.__option__ = option
        self.__jobid__ = jobid

    def _get_command(self):
        cmd = 'source %s;' % self.__profile__
        cmd += ' bmod %s %s' % (self.__option__, self.__jobid__)
        return cmd


class Breserve(command.Command):
    def __init__(self, profile, action, ID, hostname='', option='',
        user=''):
        self.__profile__ = profile
        self.__hostname__ = hostname
        self.__id_reservation__ = ID
        self.__option__ = option
        self.__user__ = user
        self.__action__ = action
        self.type_reservation = {'add': 'brsvadd', 'del': 'brsvdel'}

    def _get_command(self):
        cmd = 'source %s;' % self.__profile__
        cmd += ' %s' % self.type_reservation[self.__action__]
        if 'add' == self.__action__:
            cmd += ' -m %s %s' % (self.__hostname__, self.__option__)
            cmd += ' -N %s -u %s' % (self.__id_reservation__, self.__user__)
        else:
            cmd += ' %s' % self.__id_reservation__
        return cmd


class Brun(command.Command):
    def __init__(self, profile, hostname, jobid):
        self.__hostname__ = hostname
        self.__profile__ = profile
        self.__jobid__ = jobid

    def _get_command(self):
        cmd = 'source %s;' % self.__profile__
        cmd += ' brun -m %s %s' % (self.__hostname__, self.__jobid__)
        return cmd


class Bsub(command.Command):
    def __init__(self, profile, option, user=''):
        self.__profile__ = profile
        self.__option__ = option
        self.__user__ = user

    def _get_command(self):
        cmd = 'source %s;' % self.__profile__
        if self.__user__ != '':
            cmd += ' su -s /bin/bash -l %s -c' % self.__user__
        if self.__user__ != '':
            cmd += ' "bsub < %s"' % self.__option__
        else:
            cmd += ' bsub < %s' % self.__option__
        return cmd


class LsfCommands(batch_system.BatchSystem):
    """ Set of lsf batch system commands"""
    def __init__(self, lsf_profile, expiration=15):
        self.lsf_profile = lsf_profile
        self.expiration = expiration
        self.expiration_period = [0, 0]
        self.b_jobs_output = []
        self.jobs_info = {}
        self.host_list = {}

    def parse_cmd_output(self, my_input):
        return my_input

    def badmin(self, option, hostname):
        """badmin"""
        b_admin = Badmin(self.lsf_profile, option, hostname)
        b_admin_output = b_admin.get_output()
        return b_admin_output

    def bjobs(self, jobid='', user=''):
        """bjobs"""
        b_jobs = Bjobs(self.lsf_profile, jobid=jobid, user=user)
        b_jobs_output = b_jobs.get_output()
        return b_jobs_output
    
    def bhost(self, hostname = '', fake = False, fake_status = 1):
        """bhosts"""
        
        b_host = Bhost(self.lsf_profile, hostname = hostname)
        b_host_output = b_host.get_output()
        return b_host_output

    def bkill(self, jobid, user=''):
        """bkill"""
        b_kill = Bkill(self.lsf_profile, jobid, user=user)
        b_kill_output = b_kill.get_output()
        return b_kill_output

    def bmod(self, option, jobid):
        """bmod"""
        b_mod = Bmod(self.lsf_profile, option, jobid)
        b_mod_output = b_mod.get_output()
        return b_mod_output

    def breserve(self, action, ID, hostname='', option='', user=''):
        """breserve"""
        b_reserve = Breserve(self.lsf_profile, action, ID, hostname=hostname,
            option=option, user=user)
        b_reserve_output = b_reserve.get_output()
        return b_reserve_output

    def brun(self, hostname, jobid):
        """brun"""
        b_run = Brun(self.lsf_profile, hostname, jobid)
        b_run_output = b_run.get_output()
        return b_run_output

    def bsub(self, option, user=''):
        """bsub"""
        b_sub = Bsub(self.lsf_profile, option, user=user)
        b_sub_output = b_sub.get_output()
        return b_sub_output
