#------------------------------------------------------------------------------
# Copyright 2008-2012 Istituto Nazionale di Fisica Nucleare (INFN)
#
# Licensed under the EUPL, Version 1.1 only (the "Licence").
# You may not use this work except in compliance with the Licence.
# You may obtain a copy of the Licence at:
#
# http://joinup.ec.europa.eu/system/files/EN/EUPL%20v.1.1%20-%20Licence.pdf
#
# Unless required by applicable law or agreed to in
# writing, software distributed under the Licence is
# distributed on an "AS IS" basis,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
# either express or implied.
# See the Licence for the specific language governing
# permissions and limitations under the Licence.
#------------------------------------------------------------------------------
"""
Part of WNoDeS framework.
Interface between WNoDeS processes an the batch system PBS
"""

import unittest
from wnodes.utils import command
from wnodes.utils import batch_pbs

import sys
import threading
import commands

try:
    from xml.etree import ElementTree
except ImportError:
    try:
        from elementtree import ElementTree  # pylint: disable-msg=F0401
    except ImportError:
        sys.exit("package ElementTree is missing, exiting.")

class PbsnodesFake(batch_pbs.Pbsnodes):
    def get_output(self, option='', status=0):
        fake_output = {0:{'':''},
            1:{'Unknown':'pbsnodes'}
            }
        return [status, fake_output[status][option]]

class QalterFake(batch_pbs.Qalter):
    def get_output(self):
        fake_output = ''
        return [0, fake_output]

class QdelFake(batch_pbs.Qdel):
    def get_output(self):
        return ['0', '']

class QrunFake(batch_pbs.Qrun):
    def get_output(self):
        fake_output = ''
        return [0, fake_output]

class QstatFake(batch_pbs.Qstat):
    def _map_option_file(self):
        return {'Running':'qstat_running.xml',
        'Queued':'qstat_queued.xml'}

    def get_output(self, option='Running', status=0):
        fake_output = {0:
            {'Running':'',
            'Queued':''},
            1:{'Unknown':'qstat: Unknown Job Id'}
            }
        if status == 0:
            try:
                path = '/etc/wnodes/utils/'
                fh = open(path + self._map_option_file()[option])
            except IOError:
                print 'Test file does not exist'
                v = ''
            else:
                v = fh.read().split('\n')[0]
                fh.close()
            fake_output[status][option] = v
        return [status, fake_output[status][option]]

class QsubFake(batch_pbs.Qsub):
    def get_output(self):
        fake_output = '11924801.gridba3.ba.infn.it'
        return ['0', fake_output]

class BatchPbsTestCase(unittest.TestCase):
    def __init__(self, method_name):
        unittest.TestCase.__init__(self, method_name)

    def test_badmin_hopen(self):
        """test badmin hopen"""
        b_admin = PbsnodesFake('hopen', 'a.c.d.it')
        b_admin_cmd = b_admin.get_command()
        print b_admin_cmd
        b_admin_output = b_admin.get_output()

    def test_badmin_hclose(self):
        """test badmin hclose"""
        b_admin = PbsnodesFake('hclose', 'a.c.d.it')
        b_admin_cmd = b_admin.get_command()
        print b_admin_cmd
        b_admin_output = b_admin.get_output()

    def test_bhost(self):
        """test bhost""" 
        b_host = PbsnodesFake('', 'a.c.d.it')
        b_host_cmd = b_host.get_command()
        print b_host_cmd
        b_host_output = b_host.get_output(option='Unknown', status=1)

    def test_bjobs_jobid_running(self):
        """test bjobs with jobid and running"""
        b_jobs = QstatFake(jobid = '11924718', user = '')
        b_jobs_cmd = b_jobs.get_command()
        print b_jobs_cmd
        b_jobs_output = b_jobs.get_output(option='Running')
        self.assert_(b_jobs_output[1] != '')
        b_jobs_new_output = b_jobs.parse_output(b_jobs_output)
        print b_jobs_new_output

    def test_bjobs_jobid_queued(self):
        """test bjobs with jobid and queued"""
        b_jobs = QstatFake(jobid = '11910242', user = '')
        b_jobs_cmd = b_jobs.get_command()
        print b_jobs_cmd
        b_jobs_output = b_jobs.get_output(option='Queued')
        self.assert_(b_jobs_output[1] != '')
        b_jobs_new_output = b_jobs.parse_output(b_jobs_output)
        print b_jobs_new_output

    def test_bjobs_jobid_unknown(self):
        """test bjobs with jobid and unknown"""
        b_jobs = QstatFake(jobid = '11910242', user = '')
        b_jobs_cmd = b_jobs.get_command()
        print b_jobs_cmd
        b_jobs_output = b_jobs.get_output(option='Unknown', status=1)
        b_jobs_new_output = b_jobs.parse_output(b_jobs_output)
        print b_jobs_new_output

    def test_bkill(self):
        """bkill"""
        b_kill = QdelFake('54534534', user = '')
        b_kill_cmd = b_kill.get_command()
        print b_kill_cmd
        b_kill_output = b_kill.get_output()

    def test_breserve_add(self):
        """test breserve add"""
        b_reserve = QalterFake('add', '54534534', hostname = 'a.c.d.it')
        b_reserve_cmd = b_reserve.get_command()
        print b_reserve_cmd
        b_reserve_output = b_reserve.get_output()

    def test_breserve_del(self):
        """test breserve del"""
        b_reserve = QalterFake('del', '54534534')
        b_reserve_cmd = b_reserve.get_command()
        print b_reserve_cmd
        b_reserve_output = b_reserve.get_output()

    def test_brun(self):
        """test brun"""
        b_run = QrunFake('a.c.d.it', '54534534')
        b_run_cmd = b_run.get_command()
        print b_run_cmd
        b_run_output = b_run.get_output()

    def test_bsub(self):
        """test bsub""" 
        b_sub = QsubFake('cippa', user = 'joda001')
        b_sub_cmd = b_sub.get_command()
        print b_sub_cmd
        b_sub_output = b_sub.get_output()

def main():
    suite = unittest.TestLoader().loadTestsFromTestCase(BatchPbsTestCase)
    unittest.TextTestRunner(verbosity=2).run(suite)

if __name__ == '__main__':
    main()
