#------------------------------------------------------------------------------
# Copyright 2008-2012 Istituto Nazionale di Fisica Nucleare (INFN)
#
# Licensed under the EUPL, Version 1.1 only (the "Licence").
# You may not use this work except in compliance with the Licence.
# You may obtain a copy of the Licence at:
#
# http://joinup.ec.europa.eu/system/files/EN/EUPL%20v.1.1%20-%20Licence.pdf
#
# Unless required by applicable law or agreed to in
# writing, software distributed under the Licence is
# distributed on an "AS IS" basis,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
# either express or implied.
# See the Licence for the specific language governing
# permissions and limitations under the Licence.
#------------------------------------------------------------------------------
"""
Part of WNoDeS framework.
Interface between WNoDeS processes an the batch system LSF
"""

import unittest
import time
from wnodes.utils import command
from wnodes.utils import batch_lsf

import sys
import threading
import commands

try:
    from xml.etree import ElementTree
except ImportError:
    try:
        from elementtree import ElementTree  # pylint: disable-msg=F0401
    except ImportError:
        sys.exit("package ElementTree is missing, exiting.")

class BadminFake(batch_lsf.Badmin):
    def get_output(self, option, status = 0):
        fake_output = {0:{'hclose':'Close ...... done',
            'hopen':'Open ...... done'}
            }
        return [status, fake_output[status][option]]

class BjobsFake(batch_lsf.Bjobs):
    def get_output(self, status = 0):
        ll = 'JOBID   USER    STAT  QUEUE      FROM_HOST   EXEC_HOST'
        ll += '   JOB_NAME   SUBMIT_TIME\n4432916 superb036 RUN   superb'
        ll += '     ce04-lcg    wn-205-01-15-02-b cr004_624025965 Mar  1 16:16'
        fake_output = {0:ll,
            1:'No'}
        return [status, fake_output[status]]

class BkillFake(batch_lsf.Bkill):
    def get_output(self):
        fake_output = 'Job <%s>: Job has already finished' % self.__jobid__
        return [0, fake_output]

class BmodFake(batch_lsf.Bmod):
    def get_output(self):
        fake_output = 'Parameter of <%s> are being changed' % self.__jobid__
        return [0, fake_output]

class BreserveFake(batch_lsf.Breserve):
    def get_output(self):
        fake_output = ''
        return [0, fake_output]

class BrunFake(batch_lsf.Brun):
    def get_output(self):
        fake_output = 'Job <%s> is being forced to run.' % self.__jobid__
        return [0, fake_output]

class BsubFake(batch_lsf.Bsub):
    def get_output(self):
        fake_output = 'Job <54534534> is submitted to default queue <argo>.'
        return [0, fake_output]

class BatchLsfTestCase(unittest.TestCase):
    def __init__(self, method_name):
        unittest.TestCase.__init__(self, method_name)        
        self.profile = 'pap'

    def test_badmin_hclose(self):
        """test badmin hclose"""
        b_admin = BadminFake(self.profile, 'hclose', 'a.b.c.d')
        b_admin_cmd = b_admin.get_command()
        print b_admin_cmd
        b_admin_output = b_admin.get_output('hclose')
        #print b_admin_output
        self.assert_('Close' in b_admin_output[1])

    def test_badmin_hopen(self):
        """test badmin hopen"""
        b_admin = BadminFake(self.profile, 'hopen', 'a.b.c.d')
        b_admin_cmd = b_admin.get_command()
        print b_admin_cmd
        b_admin_output = b_admin.get_output('hopen')
        #print b_admin_output
        self.assert_('Open' in b_admin_output[1])

    def test_bjobs(self):
        """test bjobs"""
        b_jobs = BjobsFake(self.profile, jobid = '', user = '')
        b_jobs_cmd = b_jobs.get_command()
        print b_jobs_cmd
        b_jobs_output = b_jobs.get_output(status=1)
        self.assertEqual('No', b_jobs_output[1])

    def test_bjobs_jobid(self):
        """test bjobs with jobid"""
        b_jobs = BjobsFake(self.profile, jobid = '54534534', user = '')
        b_jobs_cmd = b_jobs.get_command()
        print b_jobs_cmd
        b_jobs_output = b_jobs.get_output(status=1)
        self.assertEqual('No', b_jobs_output[1])

    def test_bjobs_user(self):
        """test bjobs with user"""
        b_jobs = BjobsFake(self.profile, jobid = '', user = 'joda001')
        b_jobs_cmd = b_jobs.get_command()
        print b_jobs_cmd
        b_jobs_output = b_jobs.get_output(status=1)
        self.assertEqual('No', b_jobs_output[1])

    def test_bjobs_jobid_success(self):
        """test bjobs with jobid and success"""
        b_jobs = BjobsFake(self.profile, jobid = '54534534', user = '')
        b_jobs_cmd = b_jobs.get_command()
        print b_jobs_cmd
        b_jobs_output = b_jobs.get_output()
        self.assert_('JOBID' in b_jobs_output[1])

    def test_bkill(self):
        """test bkill"""
        b_kill = BkillFake(self.profile, '54534534', user = '')
        b_kill_cmd = b_kill.get_command()
        print b_kill_cmd
        b_kill_output = b_kill.get_output()
        self.assert_('54534534' in b_kill_output[1])

    def test_bkill_user(self):
        """test bkill with user"""
        b_kill = BkillFake(self.profile, '54534534', user = 'joda001')
        b_kill_cmd = b_kill.get_command()
        print b_kill_cmd
        b_kill_output = b_kill.get_output()
        self.assert_('54534534' in b_kill_output[1])

    def test_bmod_jobid_option(self):
        """test bmod with jobid and option"""
        b_mod = BmodFake(self.profile, '-Un', '54534534')
        b_mod_cmd = b_mod.get_command()
        print b_mod_cmd
        b_mod_output = b_mod.get_output()
        self.assert_('54534534' in b_mod_output[1])

    def test_breserve_add_hostname_option_user(self):
        """test breserve add with hotname, option and user"""
        start_reservation = time.strftime("%m:%d:%H:%M",
            time.localtime(time.time() + 70))
        end_reservation = time.strftime("%m:%d:%H:%M",
            time.localtime(time.time() + 200))

        b_reserve = BreserveFake(self.profile, 'add', '97698789',
            hostname = 'a.c.d.it', option = '-n 1 -o -b %s -e %s' % 
            (start_reservation, end_reservation), user = 'joda001')
        b_reserve_cmd = b_reserve.get_command()
        print b_reserve_cmd
        b_reserve_output = b_reserve.get_output()

    def test_breserve_del(self):
        """test breserve del"""
        b_reserve = BreserveFake(self.profile, 'del', '97698789')
        b_reserve_cmd = b_reserve.get_command()
        print b_reserve_cmd
        b_reserve_output = b_reserve.get_output()

    def test_brun(self):
        """test brun"""
        b_run = BrunFake(self.profile, 'a.c.d.it', '54534534')
        b_run_cmd = b_run.get_command()
        print b_run_cmd 
        b_run_output = b_run.get_output()

    def test_bsub(self):
        """test bsub"""
        b_sub = BsubFake(self.profile, 'cippa', user = '')
        b_sub_cmd = b_sub.get_command()
        print b_sub_cmd
        b_sub_output = b_sub.get_output()

    def test_bsub_user(self):
        """test bsub with user"""
        b_sub = BsubFake(self.profile, 'cippa', user = 'joda001')
        b_sub_cmd = b_sub.get_command()
        print b_sub_cmd
        b_sub_output = b_sub.get_output()

def main():
    suite = unittest.TestLoader().loadTestsFromTestCase(BatchLsfTestCase)
    unittest.TextTestRunner(verbosity=2).run(suite)

if __name__ == '__main__':
    main()
