#-------------------------------------------------------------------------------
# Copyright 2008-2012 Istituto Nazionale di Fisica Nucleare (INFN)
# 
# Licensed under the EUPL, Version 1.1 only (the "Licence").
# You may not use this work except in compliance with the Licence. 
# You may obtain a copy of the Licence at:
# 
# http://joinup.ec.europa.eu/system/files/EN/EUPL%20v.1.1%20-%20Licence.pdf
# 
# Unless required by applicable law or agreed to in 
# writing, software distributed under the Licence is 
# distributed on an "AS IS" basis, 
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
# either express or implied. 
# See the Licence for the specific language governing 
# permissions and limitations under the Licence.
#-------------------------------------------------------------------------------
"""
Part of WNoD framework. Configure the network in order to support several VLANs on the same Hypervisor
"""

import commands 
import os
import threading



def synchronized(lock):
    """ Synchronization decorator. """
    def wrap(func):
        def new_function(*args, **kw):
            lock.acquire()
            try:
                return func(*args, **kw)
            finally:
                lock.release()
        new_function.__name__ = func.__name__
        new_function.__doc__ = func.__doc__
        return new_function
    return wrap


class NetworkCfg:
    """ A Class to configure and support multiple VLAN on the same dom0 """
    
    LOCK_NS = threading.Lock()
    
    def __init__(self, fileNetworkStatus):
        
        LOCK_NS = threading.Lock()
        self.net_conf = commands.getoutput('route -n| grep UG').split('\n')
        self.default_nic = self.net_conf[0].split()[7]
        self.default_gw = self.net_conf[0].split()[1]
        self.ip = commands.getoutput('ifconfig ' + self.default_nic + ' | grep Mask').split('\n')
        self.ipaddress = self.ip[0].split()[1]
        self.ipaddress = self.ipaddress.split(':')[1]
        self.ipmask = self.ip[0].split()[3]
        self.ipmask = self.ipmask.split(':')[1]
        self.localhost = commands.getoutput('hostname -s')
        self.fileNetworkStatus = fileNetworkStatus
        # local_net_status structure [network element name][type, status, vlanID, bridgeName, host]
        self.networkStatus = {}
        
        
        if os.path.isfile(self.fileNetworkStatus):
            self.last_networkstatus = open(self.fileNetworkStatus, 'r')

            for line in self.last_networkstatus.readlines():
                INTERFACE = line.strip().split(':')
                if INTERFACE[1] == 'nic':  self.default_nic = INTERFACE[0]
            self.last_networkstatus.close()

            self.last_networkstatus = open(self.fileNetworkStatus, 'r')
            for line in self.last_networkstatus.readlines():
                INTERFACE = line.strip().split(':')
                if INTERFACE[0] == self.default_nic +'.default':
                    self.networkStatus.setdefault(INTERFACE[0], INTERFACE[1:])
                else:
                    IFCONFIG = commands.getstatusoutput('ifconfig %s' % INTERFACE[0])
                    if IFCONFIG[0] == 0:
                        self.networkStatus.setdefault(INTERFACE[0], INTERFACE[1:])
            self.last_networkstatus.close()
        else:
            self.networkStatus.setdefault(self.default_nic, ['nic', 'UP', 'default', 'NoBridge', self.localhost])


    def listItem(self, keysearch):
        
        self.keysearch = keysearch
        self.if_list = []
        self.if_is_in = 'no'
        
        if '&&' in self.keysearch:
            self.keysearch = self.keysearch.split('&&')
            for self.interface in self.networkStatus.keys():
                for key in self.keysearch:
                    if key in self.networkStatus[self.interface]:
                        self.if_is_in = 'yes'
                    else:
                        self.if_is_in = 'no'
                        break
                if self.if_is_in == 'yes':
                    self.if_list.append(self.interface)
            return self.if_list
        if '||' in self.keysearch:
            self.keysearch = self.keysearch.split('||')
            for self.interface in self.networkStatus.keys():
                for key in self.keysearch:
                    if key in self.networkStatus[self.interface]:
                        self.if_list.append(self.interface)
                        break
            return self.if_list
        else:
            self.keysearch = [self.keysearch]
            return [ interface for interface in self.networkStatus.keys() for key in self.keysearch \
                                                    if key in self.networkStatus[interface]]
                
    @synchronized(LOCK_NS)
    def writeNetworkStatus(self):
        """
        Update local data container and the log file for the currently network status.
        Thread safe.
        """
        
        # Create local_image_dir if not exist
        NetworkStatusDir = os.path.dirname(self.fileNetworkStatus)
        if not os.path.exists(NetworkStatusDir):
            os.makedirs(NetworkStatusDir)
            # uodate log non importato...
            #self.updateLog('Successfully created directory %s', NetworkStatusDir)
        
        event = open(self.fileNetworkStatus, 'w')
        for i in self.networkStatus.keys():
            event.write(str(i) +':'+ ':'.join(self.networkStatus[i]) +'\n')
        event.close()
    
            
        
        
class Vlan:
    """ Class to configure VLAN """
    
    def __init__(self, netCfgIstance):
        self.netCfgIstance = netCfgIstance
        
        
    def createVlan(self, vlanID):
        """ Create the 802.1Q tag on the default NIC"""
        
        self.nic = self.netCfgIstance.default_nic
        self.vlanID = vlanID

        self.cmd = "vconfig add %s %s" % (self.nic, self.vlanID)
        self.vlanInterface = "%s.%s" % (self.nic, self.vlanID)
        self.status, self.output = commands.getstatusoutput(self.cmd)
        if self.status == 0:
            self.netCfgIstance.networkStatus.setdefault(self.vlanInterface, ['vlan', 'UP', self.vlanID, \
                                                                        'NoBridge', self.netCfgIstance.localhost])
            self.netCfgIstance.writeNetworkStatus()
            return self.vlanInterface
        else:
            return 1
        
    def deleteVlan(self, vlanID):
        """ Remove the 802.1Q tag from the default NIC"""
        
        self.vlanID = vlanID
        self.vlanIf = self.netCfgIstance.listItem('vlan&&'+str(self.vlanID))[0]
        
        self.cmd = "vconfig rem %s" % (self.vlanIf)
        self.status, self.output = commands.getstatusoutput(self.cmd)
        if self.status == 0:
            self.netCfgIstance.networkStatus.pop(self.vlanIf)
            self.netCfgIstance.writeNetworkStatus()
            return 0
        else:
            return 1
        
    def getListVlan(self):
        """ Get the list of the 802.1Q tags configured """
        
        return self.netCfgIstance.listItem('vlan')

    def vlanExist(self, vlanID):
        """ Method to check whether a VLAN is already configured """
        
        self.vlanID = vlanID
        
        if len(self.netCfgIstance.listItem('vlan&&' + self.vlanID)) == 1:
            return 0
        else:
            return 1

        
        
class Brctl:
    """ Class to configure network bridge """
    
    def __init__(self, netCfgIstance):
        self.netCfgIstance = netCfgIstance

    def bridgeExist(self, vlanID):
        """ Method to check whether a VLAN is already configured """
        
        self.vlanID = vlanID
        
        if len(self.netCfgIstance.listItem('bridge&&' + self.vlanID)) == 1:
            return 0
        else:
            return 1

    def createBridge(self, vlanID):
        """ Create new bridge """
        
        self.vlanID = vlanID
        self.bridgeName = "br.%s" % (self.vlanID)
        
        self.cmd = "brctl addbr %s" % (self.bridgeName)
        self.status, self.output = commands.getstatusoutput(self.cmd)
        if self.status == 0:
            self.netCfgIstance.networkStatus.setdefault(self.bridgeName, ['bridge', 'DOWN', self.vlanID, \
                                                                          self.bridgeName,  self.netCfgIstance.localhost])
            self.netCfgIstance.writeNetworkStatus()
            return self.bridgeName
        else:
            return 1
    
    def deleteBridge(self, bridgeName):
        """ delete an existing empty bridge """
        
        self.bridgeName = bridgeName
        
        ifOnTheBridge = self.netCfgIstance.listItem(self.bridgeName)
        if len(ifOnTheBridge) == 1 and self.netCfgIstance.networkStatus[self.bridgeName][1] == 'DOWN':
            self.cmd = "brctl delbr %s" % (self.bridgeName)
            self.status, self.output = commands.getstatusoutput(self.cmd)
            if self.status == 0:
                self.netCfgIstance.networkStatus.pop(self.bridgeName)
                self.netCfgIstance.writeNetworkStatus()
                return 0
            else:
                return 1
        else:
            return 1
        
    def listBridge(self):
        """ List the configured bridges """
        
        return self.netCfgIstance.listItem('bridge')
    
    def listIfOnTheBridge(self, bridgeName):
        """ List all the Interface connected to the bridge """
        
        self.bridgeName = bridgeName
        
        list = self.netCfgIstance.listItem(self.bridgeName)
        list.remove(self.bridgeName)
        return list
    
    def addIfToBridge(self, interface):
        """ Add an interface to a bridge """
        
        self.interface = interface
        self.vlanID = self.netCfgIstance.networkStatus[self.interface][2]
        self.bridgeName = self.netCfgIstance.listItem('bridge&&' + self.vlanID)[0]
        
        self.cmd = "brctl addif %s %s" % (self.bridgeName, self.interface)
        self.status, self.output = commands.getstatusoutput(self.cmd)
        if self.status == 0:
            self.netCfgIstance.networkStatus[self.interface][3] = self.bridgeName
            self.netCfgIstance.writeNetworkStatus()
            return 0
        else:
            return 1
        
    def delIfFromBridge(self, interface):
        """ Delete an already DOWN interface from the bridge """
        
        self.interface = interface
        self.ifStatus =  self.netCfgIstance.networkStatus[self.interface][1]
        self.bridgeName = self.netCfgIstance.networkStatus[self.interface][3]
        
        if self.ifStatus == 'DOWN':
            self.cmd = "brctl delif %s %s" % (self.bridgeName, self.interface)
            self.status, self.output = commands.getstatusoutput(self.cmd)
            if self.status == 0:
                self.netCfgIstance.networkStatus[self.interface][3] = 'NoBridge'
                self.netCfgIstance.writeNetworkStatus()
                return 0
            else:
                return 1
        else:
            return 1
        
class Tap:
    """ Class to configure taps """
    
    LOCK_TAP = threading.Lock()
        
    def __init__(self, netCfgIstance):
        self.netCfgIstance = netCfgIstance
    
    @synchronized(LOCK_TAP)
    def createTap(self, vlanID, hostname):
        """ Create a new tap """
        
        self.vlanID = vlanID
        self.hostName = hostname
        self.taps = self.netCfgIstance.listItem('tap')
        self.id = []
        for self.element in self.taps:
            self.id.append(int(self.element[3:]))
        self.id.sort()
        if not len(self.taps) == 0: self.previous = self.id[0]
        self.oldtap = 'no'
        for self.element in self.id:
            self.ids = self.element
            if not self.ids == self.previous and not self.ids == self.previous + 1:
                # there is a tap between two taps still UP
                self.nextTapID = self.previous + 1
                self.oldtap = 'yes'
                break
            self.previous = self.ids
        if self.oldtap == 'no': 
            self.nextTapID = len(self.taps)

        self.cmd = "tunctl -t tap%d" % (self.nextTapID)
        self.status, self.output = commands.getstatusoutput(self.cmd)
        if self.status == 0:
            self.netCfgIstance.networkStatus.setdefault('tap'+str(self.nextTapID), \
                                                        ['tap', 'DOWN', self.vlanID, 'NoBridge', self.hostName])
            self.netCfgIstance.writeNetworkStatus()
            return 'tap'+str(self.nextTapID)
        else:
            return 1
    
    @synchronized(LOCK_TAP)
    def deleteTap(self, tap):
        """ Delete an already DOWN tap interface """
        
        self.tap = tap
        self.ifStatus =  self.netCfgIstance.networkStatus[self.tap][1]
        
        if self.ifStatus == 'DOWN':
            self.cmd = "tunctl -d %s" % (self.tap)
            self.status, self.output = commands.getstatusoutput(self.cmd)
            if self.status == 0:
                self.netCfgIstance.networkStatus.pop(self.tap)
                self.netCfgIstance.writeNetworkStatus()
                return 0
            else:
                return 1
        else:
            return 1
        
class Ifconfig:
    """ Class to configure all network interfaces using ifconfig"""
    
    def __init__(self, netCfgIstance):
        self.netCfgIstance = netCfgIstance

    def ifUp(self, interface):
        """ Startup network interface """
        
        self.interface = interface
        
        self.cmd = "ifconfig %s 0.0.0.0 up" % (self.interface)
        self.status, self.output = commands.getstatusoutput(self.cmd)
        if self.status == 0:
            self.netCfgIstance.networkStatus[self.interface][1] = 'UP'
            self.netCfgIstance.writeNetworkStatus()
            return 0
        else:
            return 1
        
    def ifDown(self, interface):
        """ Shutdown network interface """
        
        self.interface = interface
        
        self.cmd = "ifconfig %s down" % (self.interface)
        self.status, self.output = commands.getstatusoutput(self.cmd)
        if self.status == 0:
            self.netCfgIstance.networkStatus[self.interface][1] = 'DOWN'
            self.netCfgIstance.writeNetworkStatus()
            return 0
        else:
            return 1
    
    def configAddress(self, interface, address, netmask='255.255.255.0'):
        """ Configure ip address for the interface """
        
        self.interface = interface
        self.address = address
        self.netmask = netmask
        
        self.cmd = "ifconfig %s %s netmask %s" % (self.interface, self.address, self.netmask)
        self.status, self.output = commands.getstatusoutput(self.cmd)
        if self.status == 0:
            return 0
        else:
            return 1
