#!/usr/bin/env python
#------------------------------------------------------------------------------
# Copyright 2008-2011 Istituto Nazionale di Fisica Nucleare (INFN)
#
# Licensed under the EUPL, Version 1.1 only (the "Licence").
# You may not use this work except in compliance with the Licence.
# You may obtain a copy of the Licence at:
#
# http://www.osor.eu/eupl/european-union-public-licence-eupl-v.1.1
#
# Unless required by applicable law or agreed to in
# writing, software distributed under the Licence is
# distributed on an "AS IS" basis,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
# either express or implied.
# See the Licence for the specific language governing
# permissions and limitations under the Licence.
#------------------------------------------------------------------------------

import commands
import logging
#import optparse
import os
#import signal
import sys
import threading
import time
import traceback

__short_name__ = os.path.basename(os.path.splitext(sys.argv[0])[0])

try:
    from wnodes.utils import utils
except ImportError:
    sys.exit("%s: python module 'wnodes.utils' not found."
        % __short_name__)
try:
    from wnodes.utils import wsocket
except ImportError:
    sys.exit("%s: python module 'wnodes.utils' not found."
        % __short_name__)
try:
    from wnodes.cachemanager.database import database
except ImportError:
    sys.exit("%s: python module 'wnodes.cachemanager.database' not found."
        % __short_name__)

class CacheManager(wsocket.ClientRequestHandler):

    # lock to protect the logging system
    LOCK_LOG = threading.Lock()
    # lock to protect the database management system
    LOCK_DB = threading.Lock()

    def __init__(self, config):
        # insert here other constructor options
        """NOTE:
        keypair is the keypair used to login via ssh to configure VM
        ('<prv_filename>:<pub_filename>')
        client_key is the filename of the .pem private key
            used for socket encryption
        client_cert is the filename of the .pem certificate
            corresponding to client_key used for socket encryption
        TODO:
        consider using the same key for each task (VM configuration/socket)
        """
        self._cm_conf__max_log_size = int(config._cm_conf__max_log_size)
        self._cm_conf__log_file = config._cm_conf__log_file
        self._cm_conf__cache_refresh_interval = float(config._cm_conf__cache_refresh_interval)
        self._cm_conf__cache_size = int(config._cm_conf__cache_size)
        self._cm_conf__timeout = float(config._cm_conf__timeout)
        self._batch_sys__queue = config._batch_sys__queue
        self._batch_sys__type = config._batch_sys__type
        self._batch_sys__user = config._batch_sys__user
        self._bait_conf__port = int(config._bait_conf__port)
        self._cm_conf__port = int(config._cm_conf__port)
        self._cm_conf__host = config._cm_conf__host
        self._ns_conf__host = config._ns_conf__host
        self._ns_conf__port = int(config._ns_conf__port)
        self._webapp_conf__host = config._webapp_conf__host
        self._key_cert__ca_cert = config._key_cert__ca_cert
        self._key_cert__server_cert = config._key_cert__server_cert
        self._key_cert__server_key = config._key_cert__server_key
        #define private and public keys of cloud
        self._key_cert__cm_private_key = config._key_cert__cm_private_key
        self._key_cert__cm_public_key = config._key_cert__cm_public_key
        self._default_vm_option = {}
        self._default_vm_option['IMG'] = config._default_vm_option__img
        self._default_vm_option['MEM'] = int(config._default_vm_option__mem)
        self._default_vm_option['STORAGE'] = \
            int(config._default_vm_option__storage)
        self._default_vm_option['CPU'] = int(config._default_vm_option__cpu)
        self._default_vm_option['BANDWIDTH'] = \
            int(config._default_vm_option__bandwidth)

        wsocket.ClientRequestHandler.__init__(self, None, None, \
            key=None, cert=None, ca=None)

        # flag used to check if the cache is being refreshed
        self.refreshing_cache = False
        #flag used to kill the active threads on exit
        self.exit = False

        if (len(self._key_cert__cm_private_key) > 0
            and not os.path.isfile(self._key_cert__cm_private_key)):
            sys.exit('Specified private key file does not exist: %s'
                     % self._key_cert__cm_private_key)

        if (len(self._key_cert__cm_public_key) > 0
            and not os.path.isfile(self._key_cert__cm_public_key)):
            sys.exit('Specified public key file does not exist: %s'
                     % self._key_cert__cm_public_key)

        if (len(self._key_cert__server_key) > 0
            and not os.path.isfile(self._key_cert__server_key)):
            sys.exit('Specified private key file does not exist: %s'
                     % self._key_cert__server_key)

        if (len(self._key_cert__server_key) > 0
            and not os.path.isfile(self._key_cert__server_key)):
            sys.exit('Specified public key file does not exist: %s'
                     % self._key_cert__server_key)

        if (len(self._key_cert__ca_cert) > 0
            and not os.path.isfile(self._key_cert__ca_cert)):
            sys.exit('Specified public key file does not exist: %s'
                     % self._key_cert__ca_cert)

        # define which parameters are MANDATORY,
        # i.e. can't be configured run-time
        self.mandatoryList = ['IMG', 'MEM', 'STORAGE', 'BANDWIDTH', 'CPU']
        # define which parameters are CONFIGURABLE run-time
        self.configurableList = ['MOUNT', 'PUB_KEY']
        # define the list of possible statuses
        self.status_cache_pending = "CACHE_P"
        self.status_cache_ready = "CACHE_R"
        self.status_cache_available = "CACHE_A"
        self.status_cache_destroyed = "CACHE_D"
        self.status_pending = "PENDING"
        self.status_allocated = "ALLOC"
        self.status_destroyed = "DESTR"
        self.status_aborted = "ABORTED"
        self.status_unknown = "UNKNOWN"

        # set up logging
        max_log_count = 5
        self.logging_default = logging.DEBUG
        self.cm_logger = logging.getLogger('CacheManagerLogger')
        self.cm_logger.setLevel(self.logging_default)
        fmt = logging.Formatter("%(asctime)s - %(levelname)s - %(message)s")
        handler = utils.CompressedRotatingFileHandler(\
            self._cm_conf__log_file,
            maxBytes=self._cm_conf__max_log_size,
            backupCount=max_log_count)
        handler.setFormatter(fmt)
        self.cm_logger.addHandler(handler)

        # import batch_system module
        try:
            if (self._batch_sys__type.lower() == 'pbs' or
                self._batch_sys__type.lower() == 'torque'):
                try:
                    #from wnodes.utils.batch import pbsCmds
                    from wnodes.utils.batch_pbs import PbsCommands
                except:
                    sys.exit("%s: python module 'batch' not found."
                             % __short_name__)
                #self.batch_cmd = pbsCmds()
                self.batch_cmd = PbsCommands()
            elif self._batch_sys__type.lower() == 'lsf':
                try:
                    #from wnodes.utils.batch import lsfCmds
                    from wnodes.utils.batch_lsf import LsfCommands
                except ImportError:
                    sys.exit("%s: python module 'batch' not found."
                             % __short_name__)
                #self.batch_cmd = lsfCmds('/etc/profile.d/lsf.sh')
                self.batch_cmd = LsfCommands('/etc/profile.d/lsf.sh')
            else:
                sys.exit('Batch system not supported')
        except KeyError:
            sys.exit('Variable of Batch System Type, ' +
                     'is not defined in the configuration array')

    def init_database(self, db_session, compute_table, get_next_id):
        # Initialize database
        # TODO: This is a quick and dirty development, it can be improved
        self.database = database.Database(db_session, \
             compute_table, \
             get_next_id)

    @utils.synchronized(LOCK_LOG)
    def updateLog(self, msg, level=None):
        """
        Emit a string to the WNoDeS log file.
        Thread safe.
        """
        print str(msg)
        if level is None:
            level = self.logging_default
        else:
            level = utils.LOG_LEVELS.get(level, logging.NOTSET)

        self.cm_logger.log(level, msg)

    @utils.synchronized(LOCK_DB)
    def add_element_to_db(self, obj):
        """
        Add an element to database.
        Thread safe.
        """
        self.database.session.add(obj)
        return obj

    @utils.synchronized(LOCK_DB)
    def db_commit(self):
        """
        Commit changes to database.
        Thread safe.
        """
        self.database.session.commit()

    def get_available_hosts(self):
        """
        Returns the list of hosts in status HOST_AVAILABLE.
        NOTE that an host AVAILABLE is not necessarily FREE!!
        """
        """
        NOTE: At the moment this function
        does NOT WORK as expected by WNoDeS architecture:
        CURRENT: ask to batch system for clouduser RUNNING jobs
        and ask to the corresponding bait for each job the status of each VM
        FUTURE: ask a list of cloud machines and their status to the Nameserver
        """

        #NOTE: each host is an array like: host = [ bait_name, jobid, vm_name ]
        hostslist = []

        output = self.batch_cmd.bjobs(user=self._batch_sys__user)[1]
        lines = output.split('\n')
        for line in lines:
            try:
                words = line.split()
                jobStatus = words[2]

                # the following variable will represent
                # the bait list to ask for available VM's
                # this is a trick to avoid multiple requests
                # to batch system for performance issues
                # baits_to_analyze = {
                #                 '<bait1_name>':
                #                 ['jobID1', 'jobID2', ...],
                #                 '<bait2_name>':
                #                 ['jobID1', 'jobID2', ...],
                #                 ...}
                baits_to_analyze = {}
                if jobStatus == 'RUN':
                    if '-bait' in words[5]:
                        jobid = words[0]
                        bait = words[5]
                        try:
                            baits_to_analyze[bait].append(jobid)
                        except KeyError:
                            baits_to_analyze[bait] = [jobid]

                for bait in baits_to_analyze.keys():
                    msg = {'getStatus': []}
                    baitStatus = self.sendRequest(bait,
                                                  self._bait_conf__port,
                                                  msg)
                    if not baitStatus[0]:
                        baitStatus = baitStatus[1]
                        if not baitStatus[0]:
                            baitStatus = baitStatus[1]
                        else:
                            self.updateLog("Method error on request " +
                                           "%s to bait %s"
                                           % (str(msg), bait))
                    else:
                        self.updateLog("Communication error requesting " +
                                       "the status to bait %s" % bait)

                    # now check for hosts available
                    VMStatus = baitStatus[1]
                    for jobid in baits_to_analyze[bait]:
                        try:
                            if VMStatus[jobid][0] == 'HOST_AVAILABLE':
                                hostslist.append([bait, jobid,
                                                  VMStatus[jobid][2]])
                        except:
                            self.updateLog("Exception caught while " +
                                            "retrieving VM status on " +
                                            "bait %s for job %s"
                                            % (bait, jobid))
            except:
                #this exception may happen for example
                # when bjobs doesn't find any job or lfs tells to wait
                pass

        return hostslist

    def is_host_available(self, host):
        """
        Check if an host is AVAILABLE (i.e. is in status HOST_AVAILABLE)
        This function asks to the bait for the status of the given host
        return True if status is HOST_AVAILABLE or False otherwise
        """
        try:
            bait = host[0]
            jobid = host[1]
            #hostname = host[2]

            msg = {'getStatus': []}
            baitStatus = self.sendRequest(bait,
                                          self._bait_conf__port,
                                          msg)
            if not baitStatus[0]:
                baitStatus = baitStatus[1]
                if not baitStatus[0]:
                    baitStatus = baitStatus[1]
                    VMsInfo = baitStatus[1]
                    VMStatus = VMsInfo[jobid][0]
                    if VMStatus == 'HOST_AVAILABLE':
                        return True
                    else:
                        return False
                else:
                    err_msg = ("Method error on request %s to bait %s"
                               % (str(msg), bait))
                    self.updateLog(err_msg, "error")
                    return False
            else:
                err_msg = ("Communication error on request %s to bait %s"
                           % (str(msg), bait))
                self.updateLog(err_msg, "error")
                return False
        except:
            a, b, c = sys.exc_info()
            traceback.print_exception(a, b, c)
            err_msg = ("Exception while checking if host %s"
                       % host +
                       "is available, see stderr for details")
            self.updateLog(err_msg, "error")
            return False

    def is_host_free(self, host):
        """
        Check if an host is FREE (i.e. the machine is not allocated to a user)
        """
        """
        NOTE: At the moment this function
        does not work as expected by WNoDeS architecture:
        CURRENT: login as root into VM
        and check for user connected using command 'users'
        FUTURE: check the VM status as given by Nameserver
        """
        output = []
        try:
            self.updateLog("IS HOST FREE")
            command_input = ('ssh -i %s '
                      % self._key_cert__cm_private_key +
                      '-o UserKnownHostsFile=/dev/null ' +
                      '-o StrictHostKeyChecking=no ' +
                      '-q root@%s users' % host[2])
            self.updateLog("SSH command " + command_input)
            output = commands.getoutput(command_input)
            output = output.split()
        except:
            pass
        #print output
        if not len(output):
            return True
        else:
            return False

    def get_vm_bait(self, vm_hostname):
        """
        Contacts Nameserver to retrieve the bait name for the given host
        """
        msg = {'whoIs_TheBait': [vm_hostname]}
        bait = self.sendRequest(self._ns_conf__host,
                                self._ns_conf__port,
                                msg)
        if not bait[0]:
            bait = bait[1]
            if not bait[0]:
                bait = bait[1]
            else:
                self.updateLog("Method error on request %s to nameserver %s"
                               % (str(msg), self._ns_conf__host))
                bait = "unknown_bait"
        else:
            self.updateLog("Communication error on request %s to nameserver %s"
                            % (str(msg), self._ns_conf__host))
            bait = "unknown_bait"
        return bait

    def get_free_hosts(self):
        """
        Returns the list of hosts in status which should be in cache hosts.
        This function contacts the DB and looks for hosts in status
        self.status_cache_ready
        (SEE __init__ FUNCTION FOR ACTUAL VARIABLE VALUE)
        NOTE that an host AVAILABLE is not necessarily FREE!!
        """
        free_hosts = []
        vm_db_instances = (self.database.session.query
                           (self.database.tables['Compute']).filter_by
                           (STATUS=self.status_cache_ready).all())
        for vm_db_instance in vm_db_instances:
            jobid = str(vm_db_instance.JOBID)
            vm_hostname = vm_db_instance.HOSTNAME
            bait = self.get_vm_bait(vm_hostname)
            free_hosts.append([bait, jobid, vm_hostname])

        return free_hosts

    def sync_cache(self):
        """
        This function is run at CM startup.
        It tries to deduce from batch system an WNoDeS the actual status
        of the VMs and synchronize the information with the DB
        """
        self.updateLog("Synchronizing cache", "info")
        jobids = []

        # Control that VM in status CACHE_P on database
        # really correspond to an existing request
        vm_db_instances = (self.database.session.query
                           (self.database.tables['Compute']).filter_by
                           (STATUS=self.status_cache_pending).all())
        for vm_db_instance in vm_db_instances:
            try:
                jobid = str(vm_db_instance.JOBID)
                self.updateLog('Found a VM in status CACHE_P corresponding ' +
                               'to job %s in database' % jobid, 'info')
                lines = self.batch_cmd.bjobs(jobid=jobid)[1]
                lines = lines.split()
                if lines[0] != str(jobid) or lines[1] != self._batch_sys__user:
                    raise Exception
                jobids.append(jobid)
                self.updateLog('VM with jobid %s is synchronized with database'
                               % jobid, 'info')
            except:
                self.updateLog('Could not find a cloud VM corresponding ' +
                               'to jobid %s as indicated by db. ' % jobid +
                               'The status on db will be set to UNKNOWN',
                    'info')
                vm_db_instance.STATUS = self.status_unknown

        # Control that VM in status CACHE_R on database
        # really correspond to an existing VM
        vm_db_instances = (self.database.session.query
                           (self.database.tables['Compute']).filter_by
                           (STATUS=self.status_cache_ready).all())
        for vm_db_instance in vm_db_instances:
            jobid = str(vm_db_instance.JOBID)
            vm_hostname = vm_db_instance.HOSTNAME
            self.updateLog("Found a VM in status CACHE_R corresponding " +
                           "to job %s in database" % jobid, "info")
            try:
                lines = self.batch_cmd.bjobs(jobid=jobid)[1]
                lines = lines.split()
                bait = self.get_vm_bait(vm_hostname)
                free = self.is_host_free([bait, jobid, vm_hostname])
                if (lines[0] != str(jobid).split('.')[0]
                    or lines[1] != self._batch_sys__user
                    or lines[2] != "RUN"
                    or not free):
                    raise Exception
                jobids.append(jobid)
                self.updateLog("VM with jobid %s is synchronized with database"
                               % jobid, "info")
            except:
                self.updateLog("Could not find an available cloud VM " +
                               "corresponding to jobid %s " % jobid +
                               "as indicated by db. " +
                               "The status on db will be set to UNKNOWN",
                               "info")
                vm_db_instance.STATUS = self.status_unknown

        # Control that VM in status ALLOC on database
        # really correspond to an existing VM
        vm_db_instances = (self.database.session.query
                           (self.database.tables['Compute']).filter_by
                           (STATUS=self.status_allocated).all())
        for vm_db_instance in vm_db_instances:
            jobid = str(vm_db_instance.JOBID)
            vm_hostname = vm_db_instance.HOSTNAME
            self.updateLog("Found a VM in status ALLOC" +
                           "corresponding to job %s in database"
                           % jobid, "info")
            try:
                lines = self.batch_cmd.bjobs(jobid=jobid)[1]
                lines = lines.split()
                bait = self.get_vm_bait(vm_hostname)
                available = self.is_host_available([bait, jobid, vm_hostname])
                if (lines[0] != str(jobid)
                    or lines[1] != self._batch_sys__user
                    or lines[2] != "RUN"
                    or not available):
                    raise Exception
                jobids.append(jobid)
                self.updateLog("VM with jobid %s is synchronized with database"
                               % jobid, "info")
            except:
                self.updateLog("Could not find an available VM " +
                               "corresponding to jobid %s " % jobid +
                               "as indicated by db. " +
                               "The status on db will be set to DESTR and " +
                               "the corresponding job (if any) will be killed",
                               "info")
                self.batch_cmd.bkill(jobid, user=self._batch_sys__user)
                vm_db_instance.STATUS = self.status_destroyed

        # Manage PENDING request (at the moment the only way
        # is to destroy those requests!)
        self.updateLog("Looking for requests still pending", "info")
        vm_db_instances = (self.database.session.query
                           (self.database.tables['Compute']).filter_by
                           (STATUS=self.status_pending).all())
        for vm_db_instance in vm_db_instances:
            jobid = "NULL"
            try:
                jobid = str(vm_db_instance.JOBID)
            except:
                jobid = None
            self.updateLog("Found a pending request with jobid %s: destroying"
                           % jobid, "info")
            self.batch_cmd.bkill(jobid, user=self._batch_sys__user)
            vm_db_instance.STATUS = self.status_aborted

        self.db_commit()

        # Control that every job submitted by the cloud user
        # corresponds to a possible state of [CACHE_P, CACHE_R, ALLOC]
        self.updateLog("Looking for orphan jobs", "info")
        try:
            lines = self.batch_cmd.bjobs(user=self._batch_sys__user)[1]

            lines = lines.split('\n')
            for line in lines:
                line = line.split()
                jobid = line[0]
                if (jobid == 'JOBID') | (jobid == 'No'):
                    continue
                if jobid not in jobids:
                    self.updateLog("Found orphan job with jobid %s, killing it"
                                   % jobid, "info")
                    self.batch_cmd.bkill(jobid, user=self._batch_sys__user)
        except:
            a, b, c = sys.exc_info()
            traceback.print_exception(a, b, c)
            self.updateLog("Error while checking for orphans jobs, " +
                           "see stderr for details. Continuing anyway", "info")

        self.updateLog("Cache synchronized", "info")

    def refresh_cache(self):
        """
        This function is spawn in a separate thread inside main()
        it checks every self._cm_conf__cache_refresh_interval the cache status
        if the cache is too small:
            it submit the right number of parallel requests to WNoDeS
        if the cache is too big:
            it destroys the right number of VMs
        The status of the cache is controlled through the DB:

        Possible VM statuses on DB
        (SEE __init__ FUNCTION FOR ACTUAL STATUS VARIABLES VALUE):
        - self.status_cache_pending (cache_pending):
            The VM is being created
            and is supposed to be a cached VM once created
        - self.status_cache_ready (cache_ready):
            The VM is ready and is part of the cache.
            Actually, the CM looks for VM in this status
            to look for VMs which can be allocated to users
        - self.status_cache_available (cache_allocated):
            The VM WAS part of the cache and now it is allocated to an user
            (i.e. is no more part of the cache).
            For each DB record in this status, there should be another record
            with the same host infos (jobid, hostname, ecc) corresponding
            to the actual allocation to the user, in status
            "ALLOC/DESTR/ABORTED, ecc"
            NOTE that for this record the status of the VM remains
            self.status_cache_available even ater VM destruction
        - self.status_cache_destroyed (cache_destroyed):
            The VM is put in this status if it is destroyed
            because the cache was found too big
        - self.status_allocated:
            The VM is allocated to a user
        - self.status_pending:
            The VM is under creation
        - self.status_destroyed:
            The VM is destroyed
        - self.status_aborted:
            The VM creation was aborted for some reason
        - self.status_unknown:
            The VM status is unknown
            because the actual information fetched from BatchSystem and WNoDeS
            does not match the informations fetched from DB

        This functions is able to manage the pending requests through the DB:
        it looks for VM in status self.status_cache_pending on DB
        and assumes that these VM are being created.
        """
        start = 0
        # Main loop to check if the thread must exit
        while not self.exit:
            # Loop to manage the time elapsed
            if (time.time() - start) > self._cm_conf__cache_refresh_interval:
                start = time.time()
                self.refreshing_cache = True
                fhosts = self.get_free_hosts()

                db_instances = (self.database.session.query
                                (self.database.tables['Compute']).filter_by
                                (STATUS=self.status_cache_pending).all())
                pending = len(db_instances)

                # Total cache size is vm_available + vm_requested
                cacheSize = len(fhosts) + pending
                self.updateLog("found %s VM's in cache, they should be %s"
                                % (cacheSize, self._cm_conf__cache_size))

                # Manage cache size
                if cacheSize < self._cm_conf__cache_size:
                    VmNumber = self._cm_conf__cache_size - cacheSize
                    self.updateLog("too few VM in cache, " +
                                   "submitting %s standard request(s)"
                                   % VmNumber)
                    for i in range(0, VmNumber):
                        KWARGS = {'cache': True}
                        create_vm_thread = (threading.Thread
                                            (target=self.create_vm,
                                             kwargs=KWARGS))
                        create_vm_thread.start()

                if cacheSize > self._cm_conf__cache_size:
                    self.updateLog("too much VM in cache, " +
                                   "destroying exceeding VMs")
                    for i in range(0, cacheSize - self._cm_conf__cache_size):
                        self.destroy_vm(fhosts[i], True)

                if cacheSize == self._cm_conf__cache_size:
                    self.updateLog("Chache refreshed (%s pending requests)"
                                   % pending)

            else:
                time.sleep(1)

    def create_vm(self, requirements={}, cache=False, vm_db_instance=None):
        """
        Function used to create a VM with specified requirements.
        For mandatory parameters not specified in requirements,
        default ones will be used (see configfile)
        This function updates the variable self._requests
        to manage pending requests, through the thread safe functions
        It acts as follow:
        1) submit job
        2) use bjobs to periodically check if job is running
        3) when job is running, get the bait_name from bjobs output
        4) periodically query the bait to get vm status until is HOST_AVAILABLE
        5) when host is available return a structure like
        ['<bait_hostname>', '<jobid>', '<vm_hostname>']

        If the VM is not available in a time < timeout,
        the creation will be aborted

        If variable cache = True it means that:
        the VM to be created is supposed to be part of the cache.
        In this case the status of this VM will be updated in the DB during
        the creation, to make the function able to manage this pending request:
        self.refresh_cache
        (SEE __init__ FUNCTION FOR ACTUAL STATUS VARIABLES VALUE)
        The status on DB is updated as follow:
        1) creation of db_instance
        2) db_instance.STATUS = self.status_cache_pending (pending request)
        3) Once the job is submitted:
            db_instance.JOBID = <jobid>
        4) Once the VM is in status HOST_AVAILABLE:
            db_instance.STATUS = self.status_cache_ready;
            db_instance.HOSTNAME = <vm_hostname>
        5) If something goes wrong:
            db_instance.STATUS = self.status_aborted

        NOTE: Do NOT use
        cache = True
        for requests that are not supposed to go in the cache once ready
        (ex. requests submitted after a cache-miss request),
        OR the CM will think that the VM is in the cache!!
        """
        newhost = []
        # options = {}

        options = self._default_vm_option

        # Override the default parameters
        # with the ones specified in requirements
        for key in requirements.keys():
            # insert here a control on keys,
            # something like if key not in ['IMAGE', 'RAM', ecc]
            options[key] = requirements[key]

        self.updateLog("---- Creating VM with options: %s"
                       % str(options), 'info')

        if cache:
            vm_db_instance = self.add_element_to_db(
                  self.database.tables['Compute'](
                      # ID=self.database.get_next_id
                      #     (self.database.tables['Compute']),
                      ARCH="x86_64",
                      MEMORY=str(options['MEM']),
                      NAME="NO_NAME",
                      THROUGHPUT=str(options['BANDWIDTH']),
                      STATUS=self.status_pending,
                      UUID=str(utils.guid()),
                      CORES=str(options['CPU']),

                      #DB_ADDED
                      IMG_TAG="NULL",
                      #BASENAME="NULL",
                      DATE="NULL",
                      OS="NULL",
                      PUBLICKEY="NULL",
                      PUID="NULL",
                      #QUANT="NULL",
                      STORAGE="NULL",
                      TIMESTAMP="NULL",
                      SSH="NULL"

                      # WARNING the following columns can be useful
                      #DB_MOD_TAG
                      #TYPE="CLOUD",
                      #SESSION_START_TIME=str(time.time())
                  )
              )

        # Update the status on db to PENDING
        # (only for requests which must create a cached VM)
        if cache:
            vm_db_instance.STATUS = self.status_cache_pending
            self.db_commit()

        # Create the wrapper for LSF or PBS
        wrapperFile = '/tmp/bsubwrapper_%s' % str(time.time())
        output = open(wrapperFile, 'a')

        if self._batch_sys__type == 'lsf':
            output.write("#!/bin/bash \n")
            output.write("# LSF directives: \n")
            output.write("#BSUB -L /bin/bash \n")
            output.write("#BSUB -q %s \n" % self._batch_sys__queue)

            output.write("while [[ 1 ]]; do \n")
            output.write("    /bin/sleep 10d \n")
            output.write("done \n")

            output.write("##### WNoDeS variable definition ##### \n")
            output.write("# WNoDeS_VM_TYPE:CLOUD\n")
            output.write("# WNoDeS_VM_NETWORK_TYPE:OPEN\n")
            output.write("# WNoDeS_VM_IMG:" + options['IMG'] + "\n")
            output.write("# WNoDeS_VM_MEM:" + str(options['MEM']) + "\n")
            output.write("# WNoDeS_VM_STORAGE:" +
                         str(options['STORAGE']) + "\n")
            output.write("# WNoDeS_VM_BANDWIDTH:" +
                         str(options['BANDWIDTH']) + "\n")
            output.write("# WNoDeS_VM_CPU:" + str(options['CPU']) + "\n")

#            if len(options['MOUNT']) > 1:
#                i = 0
#                for MOUNT in options['MOUNT'].split(','):
#                    MPARAM = MOUNT.split(':')
#                    output.write("# WNoDeS_VM_CONFIG_MOUNT_%s" % str(i) +
#                                 "-MOUNT:%s:%s %s\n"
#                                 % (MPARAM[0], MPARAM[1], MPARAM[2]))
#                    i = i + 1
#            else:
#                output.write("# WNoDeS_VM_CONFIG_MOUNT_111-MOUNT:" +
#                             "/dev/vdb /mnt\n")
            pub_file = open(self._key_cert__cm_public_key, 'r')
            PUB_KEY_STRING = pub_file.readline().strip()
            output.write("# WNoDeS_VM_CONFIG_SSH_PUBKEY:%s\n" % PUB_KEY_STRING)
            output.write("##### END WNoD variable definition ##### \n")
            output.write("##### WNoDeS end variable definition ##### \n")

        elif (self._batch_sys__type == 'pbs'
              or self._batch_sys__type == 'torque'):
            output.write("#!/bin/bash \n")
            output.write("# PBS directives: \n")
            output.write("#PBS -q %s \n" % self._batch_sys__queue)
            output.write("while [[ 1 ]]; do \n")
            output.write("    /bin/sleep 10d \n")
            output.write("done \n")
            pub_file = open(self._key_cert__cm_public_key, 'r')
            PUB_KEY_STRING = pub_file.readline().strip()

        output.close()

        bait = ""
        timeout = self._cm_conf__timeout
        elapsed = 0.0  # min
        start = time.time()
        ready = False
        while ((not ready)
               and (elapsed < timeout)
               and (not self.exit)):
            elapsed = float(time.time() - start) / 60.0
            if self._batch_sys__type == 'lsf':
#                self.updateLog('batch type %s; option %s; suer %s' %
#                               (self.batch_sys, wrapperFile,
#                                self._batch_sys__user),"debug")
                bsub_out = self.batch_cmd.bsub(wrapperFile,
                                               user=self._batch_sys__user)
            elif (self._batch_sys__type == 'pbs'
                  or self._batch_sys__type == 'torque'):
                command_submit = (wrapperFile + " -v WNoDeS_VM_TYPE=\"CLOUD" +
                                  "\",WNoDeS_VM_NETWORK_TYPE=\"OPEN\"," +
                                  "WNoDeS_VM_IMG=\"" +
                                  options['IMG'] +
                                  "\",WNoDeS_VM_MEM=\"" +
                                  str(options['MEM']) +
                                  "\",WNoDeS_VM_STORAGE=\"" +
                                  str(options['STORAGE']) +
                                  "\",WNoDeS_VM_BANDWIDTH=\"" +
                                  str(options['BANDWIDTH']) +
                                  "\",WNoDeS_VM_CPU=\"" +
                                  str(options['CPU']) +
                                  "\",WNoDeS_VM_CONFIG_SSH_PUBKEY=\"" +
                                  PUB_KEY_STRING +
                                  "\"")
                bsub_out = self.batch_cmd.bsub(command_submit,
                                               user=self._batch_sys__user)
            self.updateLog(bsub_out, 'debug')
            if bsub_out[0]:
                self.updateLog('bsub failed! Aborting VM creation', "error")
                self.updateLog('bsub_out: %s' % bsub_out[1], "debug")
                if vm_db_instance:
                    vm_db_instance.STATUS = self.status_aborted
                    self.db_commit()
                return newhost
            #print bsub_out
            jobid = ""
            """
            WARNING!! the following code uses the string:
            "is submitted to queue"
            to identify the line specifing the jobid,
            so it may not work for batch systems different from LSF!
            """
            #self.updateLog(bsub_out, "debug")
            if self._batch_sys__type == 'lsf':
                for line in bsub_out[1].split("\n"):
                    if "is submitted to queue" in line:
                        jobid = line.split()[1]
                        jobid = jobid[1:len(jobid) - 1]
            elif (self._batch_sys__type == 'pbs'
                  or self._batch_sys__type == 'torque'):
                for line in bsub_out[1].split("\."):
                    jobid = line.split()[0]
            if vm_db_instance:
                vm_db_instance.JOBID = str(jobid)
                time.sleep(1)
                self.db_commit()
                time.sleep(1)
            self.updateLog('My jobid is: %s' % jobid)

            self.updateLog("Waiting for job running")
            jobRunning = False

            # Loop to wait for job running
            while ((elapsed < timeout)
                   and (not jobRunning)
                   and (not self.exit)):
                elapsed = float(time.time() - start) / 60.0
                lines = self.batch_cmd.bjobs(jobid=jobid)[1]
                lines = lines.split('\n')

                for line in lines:
                    line = line.split()
                try:
                    if line[0] in jobid:
                        hostname_new = line[5].split('/')[0].split('.')[0]
                        if hostname_new == 'Empty':
                            bait = 'unknown_bait'
                        else:
                            bait = self.get_vm_bait(hostname_new)
                        #self.updateLog("jobid %s for the bait is ---- for line %s : %s" % (jobid, hostname_new, bait))
                        if ((line[2] == 'RUN')
                            and (hostname_new == bait)):
                            jobRunning = True

                            self.updateLog("job running on bait: %s" % bait)
                            self.updateLog("Waiting for VM available")
                            VMReady = False

                            # Loop to wait for HOST_AVAILABLE
                            max_request_retries = 24
                            retries = 1
                            lastStatus = "NULL"
                            while ((not VMReady)
                                   and (elapsed < timeout)
                                   and (not self.exit)):
                                elapsed = float(time.time() - start) / 60.0
                                msg = {'getStatus': []}
                                baitStatus = (self.sendRequest
                                              (bait,
                                               self._bait_conf__port,
                                               msg))
                                if not baitStatus[0]:
                                    baitStatus = baitStatus[1]
                                    if not baitStatus[0]:
                                        baitStatus = baitStatus[1]
                                    else:
                                        self.updateLog("Method error " +
                                                       "on request " +
                                                       "%s to bait %s"
                                                       % (str(msg), bait))
                                        time.sleep(5)
                                        continue
                                else:
                                    self.updateLog("Communication error" +
                                                   "requesting " +
                                                   "the status to bait %s"
                                                   % bait)
                                    time.sleep(5)
                                    continue

                                # Now check if host is in status HOST_AVAILABLE
                                VMStatus = None
                                try:
                                    VMStatus = baitStatus[1][jobid]
                                except:
                                    self.updateLog("Bait %s does not know "
                                                   % bait +
                                                   "the status of VM with " +
                                                   "jobid %s. (%s/%s)"
                                                   % (jobid, retries,
                                                      max_request_retries))
                                    retries += 1
                                    if retries >= max_request_retries:
                                        self.updateLog("Max limit of status " +
                                                       "requests reached, " +
                                                       "checking again the " +
                                                       "status of job %s"
                                                       % jobid, "info")
                                        jobRunning = False
                                        time.sleep(5)
                                        break
                                    else:
                                        time.sleep(5)
                                        continue
                                try:
                                    #lastStatus = "NULL"
                                    self.updateLog("START CHECKING Status %s " % VMStatus[0])
                                    if VMStatus[0] == 'HOST_AVAILABLE':
                                        newhost = [bait, jobid, VMStatus[2]]
                                        if cache:
                                            vm_db_instance.HOSTNAME = \
                                                newhost[2]
                                            vm_db_instance.STATUS = \
                                                self.status_cache_ready
                                            self.db_commit()
                                        self.updateLog("The new host is " +
                                                       "available: %s"
                                                       % newhost)
                                        VMReady = True
                                        ready = True
                                        break
                                    elif VMStatus[0] == 'UNKNOWN':
                                        self.updateLog("Unable to retrieve " +
                                            "the status of VM with jobid %s, "
                                            % jobid +
                                            "retrying in 5 seconds",
                                            "info")
                                        time.sleep(5)
                                    else:
                                        currentStatus = VMStatus[0]
                                        if currentStatus != lastStatus:
                                            lastStatus = currentStatus
                                            self.updateLog("The VM with jobid "
                                                           + "%s is in status:"
                                                           % jobid
                                                           + " %s."
                                                           % currentStatus,
                                                           "info")
                                        time.sleep(5)
                                except:
                                    a, b, c = sys.exc_info()
                                    traceback.print_exception(a, b, c)
                                    self.updateLog("Exception caught while " +
                                                   "retrieving VM status on " +
                                                   "bait %s for job %s, "
                                                   % (bait, jobid) +
                                                   "retrying in 5 seconds",
                                                   "info")
                                    time.sleep(5)
                    else:
                        time.sleep(15)
                except:
                    a, b, c = sys.exc_info()
                    traceback.print_exception(a, b, c)
                    self.updateLog("Unable to get the job status," +
                                   "retrying in 5 seconds")
                    jobRunning = False
                    time.sleep(5)
            else:
                if self.exit:
                    self.batch_cmd.bkill(jobid, user=self._batch_sys__user)

                    if vm_db_instance:
                        vm_db_instance.STATUS = self.status_aborted
                        self.db_commit()
                    self.updateLog("VM creation aborted! No VM created")
                if elapsed > timeout:
                    self.batch_cmd.bkill(jobid, user=self._batch_sys__user)

                    if vm_db_instance:
                        vm_db_instance.STATUS = self.status_aborted
                        self.db_commit()
                    self.updateLog("Request Timed Out!! No VM created")
        # Return the newhost
        return newhost

    def get_host_info(self, host):
        """
        Function used to get the hardware informations for a specific host
        host must be in the format of: ['<bait_name>', '<jobid>', '<vm_name>']
        it queries the bait to get the informations about this VM
        """
        hw_info = {}

        bait = host[0]
        jobid = host[1]
        vm_host = host[2]

        msg = {'getStatus': []}
        baitStatus = self.sendRequest(bait, self._bait_conf__port, msg)
        if not baitStatus[0]:
            baitStatus = baitStatus[1]
            if not baitStatus[0]:
                baitStatus = baitStatus[1]
            else:
                self.updateLog("Method error on request %s to bait %s"
                               % (str(msg), bait))
        else:
            self.updateLog("Communication error " +
                           "requesting the status to bait %s"
                            % bait)

        vm_list = baitStatus[2]
        try:
            # output[2] is a dictionary containing
            # all the VM managed by the specified bait
            for key in vm_list.keys():
                #vm_hw[1] is the hostname of that virtual machine
                if vm_list[key][1] == vm_host:
                    # vm_hw[3] is a dictionary containing
                    # the hardware information that virtual machine
                    hw_info = vm_list[key][3]
        except:
            self.updateLog("Unable to get info for host %s on bait %s"
                            % (vm_host, bait))

        for key in hw_info.keys():
            hw_info[key] = str(hw_info[key])
        return hw_info

    def extractMandatoryParameters(self, requirements):
        """
        This function return a dictionary containing only
        the MANDATORY parameters contained in 'requirements'
        the list of mandatory parameters is defined
        in self.mandatoryList (see class constructor)
        """
        mandatoryParameters = {}
        for key in self.mandatoryList:
            try:
                mandatoryParameters[key] = requirements[key]
            except KeyError:
                return {}
        return mandatoryParameters

    def extractConfigurableParameters(self, requirements):
        """
        This function return a dictionary containing only
        the CONFIGURABLE parameters contained in 'requirements'
        the list of configurables parameters is defined
        in self.configurableList (see class constructor)
        """
        configurableParameters = {}
        for key in self.configurableList:
            try:
                configurableParameters[key] = requirements[key]
            except:
                pass
        return configurableParameters

    def host_matches_requirements(self, requirements, hw_info):
        """
        This functions takes as argments two requirements dictionaries
        and checks if mandatory parameters of 'hw_info'
        satisfies mandatory parameters of 'requirements'
        """
        mandatoryRequested = self.extractMandatoryParameters(requirements)
        mandatoryAvailable = self.extractMandatoryParameters(hw_info)
        if len(mandatoryRequested) == 0 or len(mandatoryAvailable) == 0:
            return False
        if mandatoryRequested == mandatoryAvailable:
            return True
        else:
            return False

    def configure_vm(self, hostname, configurableParameters):
        """
        This function acts the same way of the one in hypervisor code.
        Consider to configure the VM recalling the same function on hypervisor
        """
        # NOTE: we should configure VM by a request to HyperVisor??
        CONFIG_COMPONENTS = {}
        for key in configurableParameters.keys():
            if 'VM_CONFIG_' in key:
                COMPONENT = key.split('_')[2]
                CONFIG_COMPONENTS[COMPONENT] = configurableParameters[key]

        PLUGINS_OUTPUT = ""
        for COMPONENT in CONFIG_COMPONENTS.keys():
            print "----- Configuring component: %s" % COMPONENT
            PLUGIN = ""
            try:
                cmd = ('from wnodes.utils.plugins import plugin_%s as PLUGIN'
                       % COMPONENT.lower())
                exec cmd
                PLUGIN_RETURN_OUTPUT = PLUGIN.handlePlugin(
                                           self._key_cert__cm_private_key,
                                           hostname,
                                           CONFIG_COMPONENTS[COMPONENT])
                PLUGINS_OUTPUT += str(PLUGIN_RETURN_OUTPUT)
                if PLUGIN_RETURN_OUTPUT[0] == 1:
                    return [1, PLUGINS_OUTPUT]
            except:
                err_msg = ('Plugin missing to configure the component: %s'
                           % COMPONENT)
                a, b, c = sys.exc_info()
                traceback.print_exception(a, b, c)
                self.updateLog(err_msg, "error")
                return [1, COMPONENT]

        return [0, PLUGINS_OUTPUT]

    def _get_first_requested_host(self, *args):
        """
        This function controls the cache to get
        the first free host matching requested req
        if a host is found, it directly return it.
        if no host is found it submit a request (and the user must wait...)
        NOTE that since this function
        uses above functions to manage and check the cache,
        the architecture is still not the right one
        (see for example:
        self.get_available_hosts or self.is_host_free)
        """

        parameters = args[0]
        req = {}
        first_host = []
        # parameters[0] must be the dictionary with the req
        try:
            req = parameters[0]
        except:
            req = self._default_vm_option

        mandatoryParams = self.extractMandatoryParameters(req)
        if not len(mandatoryParams):
            err_msg = ('Cannot extract mandatory parameters from ' +
                       '%s!!' % str(req))
            self.updateLog(err_msg, "error")
            return [1, err_msg]
        configurableParameters = self.extractConfigurableParameters(req)
        """
        Now change configurableParameters in the form used by utils plugins
        """

        # PUBKEY:
        try:
            if configurableParameters['PUB_KEY'] != "":
                SSH_CONFIG = {}
                SSH_CONFIG['PUBKEY'] = configurableParameters['PUB_KEY']
                configurableParameters['VM_CONFIG_SSH'] = SSH_CONFIG
            del configurableParameters['PUB_KEY']
        except KeyError:
            pass

        # MOUNT:
        try:
            MOUNT_CONFIG = {}
            i = 0
            for MOUNT in configurableParameters['MOUNT'].split(','):
                MPARAM = MOUNT.split(':')
                MOUNT_CONFIG["MOUNT_%s-MOUNT" % str(i)] = \
                "%s:%s %s" % (MPARAM[0], MPARAM[1], MPARAM[2])
                i += 1
            configurableParameters['VM_CONFIG_MOUNT'] = MOUNT_CONFIG
            del configurableParameters['MOUNT']
        except KeyError:
            pass
        print ("----- POST-Adapted configurable parameters are: %s"
               % configurableParameters)
        """
        now mandatoryParams contains ONLY the list of hardware req
        and configurableParameters contains ONLY the list of configurable
        parameters in a form directly usable by the plugin modules
        """
        # create the corresponding DB record
        vm_db_instance = None
        """
        now manage the web-app specific options
        """

        webrequestParameters = {}
        if 'VM_TOKEN' in req.keys():
            try:
                webrequestParameters['VM_TOKEN'] = req['VM_TOKEN']
                webrequestParameters['VM_NAME'] = req['VM_NAME']

                # For webapp request, the record on the DB
                # is created by the webapp and uniquely identified by the token
                vm_db_instance = (
                    self.database.session.query
                    (self.database.tables['Compute']).filter_by
                    (TOKEN=webrequestParameters['VM_TOKEN']).first())
                self.db_commit()
                if not vm_db_instance:
                    self.updateLog("Could find record on Database for VM " +
                                   "with token %s"
                                   % webrequestParameters['VM_TOKEN'],
                                   "error")
                    newhost = ['unexistent_bait', '-1', 'unexistent_vm']
                    return [1, newhost]
                #vm_db_instance.TOKEN = webrequestParameters['VM_TOKEN']
                #vm_db_instance.NAME = webrequestParameters['VM_NAME']
                ##vm_db_instance.TYPE = "CLOUD_WEB" #DB_MOD_TAG
            except KeyError:
                self.updateLog("One or more needed key " +
                               "for WEB request missing!!",
                               "error")
                newhost = ['unexistent_bait', '-1', 'unexistent_vm']
                return [1, newhost]
        else:
            # For VIP request, we need to create the DB record
            vm_db_instance = self.add_element_to_db(
                self.database.tables['Compute'](
                    ARCH="x86_64",
                    MEMORY=str(req['MEM']),
                    NAME="NO_NAME",
                    THROUGHPUT=str(req['BANDWIDTH']),
                    STATUS=self.status_pending,
                    UUID=str(commands.getoutput('uuidgen')),
                    CORES=str(req['CPU']),

                    #DB_ADDED
                    IMG_TAG="NULL",
                    #BASENAME="NULL",
                    DATE="NULL",
                    OS="NULL",
                    PUBLICKEY="NULL",
                    PUID="NULL",
                    #QUANT = "NULL",
                    STORAGE="NULL",
                    TIMESTAMP="NULL",
                    SSH="NULL"

                    # WARNING the following columns can be useful
                    #DB_MOD_TAG
                    #TYPE="CLOUD",
                    #SESSION_START_TIME = str(time.time())
                )
            )
            #vm_db_instance.TYPE = "CLOUD_VIP" #DB_MOD_TAG
        vm_db_instance.STATUS = self.status_pending
        self.db_commit()

        """
        now look for compatible VM in cache
        """

        #ahosts = self.get_available_hosts()
        #for host in ahosts:
        #    if (self.is_host_free(host)):
        fhosts = self.get_free_hosts()
        for host in fhosts:
                hw_info = self.get_host_info(host)
                # NOTE that one can also use full 'req'
                # istead of 'mandatoryParams',
                # because 'host_matches_requirements'
                # automatically extracts mandatory parameters
                if self.host_matches_requirements(mandatoryParams, hw_info):
                    self.updateLog("Found a free host in cache: %s"
                                   % str(host))
                    cached_vm_db_instance = (self.database.session.query
                        (self.database.tables['Compute']).filter_by
                        (JOBID=host[1]).filter_by
                        (HOSTNAME=host[2]).first())
                    cached_vm_db_instance.STATUS = self.status_cache_available
                    self.db_commit()
                    create_vm_thread = threading.Thread(target=self.create_vm,
                                                        args=(mandatoryParams,
                                                              True))
                    create_vm_thread.start()
                    first_host = host
                    break

        # if no matching VM is found, submit a creation request
        if not len(first_host):
            self.updateLog("No free host found in cache, " +
                           "need to create a new one (this may take a while)")
            newhost = self.create_vm(mandatoryParams)
            if not len(newhost):
                self.updateLog("Unable to create the requested host!")
                vm_db_instance.STATUS = self.status_aborted
                self.db_commit()
                newhost = ['unexistent_bait', '-1', 'unexistent_vm']
                return [1, newhost]
            else:
                self.updateLog("This is your new host: %s" % str(newhost[1]))
            first_host = newhost

        """
        Will now configure the VM using plugins
        """
        print ("Will now configure VM with parameters: %s"
               % configurableParameters)
        setupResult = self.configure_vm(first_host[2], configurableParameters)
        if setupResult[0]:
            err_msg = ("Unable to configure components: %s"
                       % str(setupResult[1]))
            self.updateLog(err_msg, "error")
            return [1, err_msg]

        if len(webrequestParameters) > 0:
            # execute PUT to web-app
            curl_command_string = ('curl' +
                                   ' -X PUT' +
                                   ' -d "token=%s&hostname=%s"'
                                   % (webrequestParameters['VM_TOKEN'],
                                      first_host[2]) +
                                   ' -H "Content-Type: ' +
                                   'application/x-www-form-urlencoded" ' +
                                   self._webapp_conf__host
                                   + ':8080/grid-cloud-0.0.1/jaxrs/compute/%s'
                                   % webrequestParameters['VM_NAME'])
            print curl_command_string
            commands.getstatusoutput(curl_command_string)

        #TODO: should now update database!
        vm_db_instance.HOSTNAME = first_host[2]
        vm_db_instance.JOBID = first_host[1]
        vm_db_instance.STATUS = self.status_allocated
        self.db_commit()

        return [0, first_host]

    def get_first_requested_host(self, *ARGS):
        """
        This is a dummy function to manage the requests,
        coming from webapp and VIP in a diffrent way.
        To see the actual code for first requested host,
        see _get_first_requested_host
        """

        requirements = {}
        try:
            requirements = ARGS[0][0]
        except:
            error_msg = "Error fetching parameters from args: %s" % ARGS
            self.updateLog(error_msg, "error")
            return [1, error_msg]

        """
        Distinguish the requests coming from webapp or from VIP
        """
        if 'VM_TOKEN' in requirements.keys():
            """
            Request coming from webapp, the actual request will be managed
            in a separate thread to avoid blocking socket on webapp
            (ask Daniele Andreotti fo details)
            a generic and immediate response will be given
            """
            get_first_requested_host_thread = \
                (threading.Thread
                (target=self._get_first_requested_host,
                args=ARGS))
            get_first_requested_host_thread.start()
            dummy_host = ["dummy_bait", "-1", "dummy_hostname"]
            return [0, dummy_host]
        else:
            """
            Request coming from VIP
            the socket will remain opened
            until the actual request is resolved
            """
            response = self._get_first_requested_host(*ARGS)
            return response

    def destroy_vm(self, host, cache=False):
        """
        This function destroys the give host killing the corresponding job
        """
        self.updateLog("will now destroy the VM %s" % host)
        kill_output = self.batch_cmd.bkill(host[1], user=self._batch_sys__user)
        time.sleep(5)
        if kill_output[0] != 0:
            self.updateLog("Unable to kill the job %s with error: %s"
                           % (host[1], kill_output[1]))
        if cache:
            try:
                vm_db_instance = (self.database.session.query
                                  (self.database.tables['Compute']).filter_by
                                  (JOBID=host[1]).filter_by
                                  (HOSTNAME=host[2]).first())
                vm_db_instance.STATUS = self.status_cache_destroyed
                self.db_commit()
            except:
                self.updateLog("Could not find a record corresponding " +
                               "to host %s on db!" % str(host),
                               "error")
        return kill_output[0]

    def request_vm_destroy(self, *args):
        """
        This function can be recalled via socket to destroy a VM
        using the above function and updating the database
        """

        dictio = args[0][0]

        try:
            bait = dictio['BAIT']
            jobid = dictio['JOBID']
            vm_hostname = dictio['VM_HOSTNAME']

        except:
            self.updateLog("Dictionary for vm destruction " +
                           "is not formatted as expected! needed: " +
                           "{'BAIT': <bait_hostname>, " +
                           "'JOBID': <jobid>, " +
                           "'VM_HOSTNAME': <vm_hostname>,  " +
                           " got: %s" % dictio)
            return [1, "Unable to destroy VM"]

        if (('UUID' in dictio) &
            (bait == 'unknown') &
            (jobid == 'UNKNOWN')):

            uuid = dictio['UUID']

            # case with (BAIT == 'unknown') & (JOBID== 'UNKNOWN')
            # used to remove ABORTED VMs
            vm_db_instance = (self.database.session.query
                              (self.database.tables['Compute']).filter_by
                              (UUID=uuid).all()[-1])
            vm_db_instance.STATUS = self.status_destroyed
            exit_status = 0
            self.updateLog("VM_HOSTNAME %s set as destroyed," % vm_hostname
                           + " because BAIT or JOBID are 'unknown'", "info")
            self.database.session.commit()
            return [0, "VM %s successfully destroyed" % vm_hostname]
        else:
            host = [bait,
                    jobid,
                    vm_hostname]

            destroy_vm_thread = threading.Thread(target=self.destroy_vm(host))
            destroy_vm_thread.start()

            vm_db_instance = (self.database.session.query
                  (self.database.tables['Compute']).filter_by
                  (JOBID=jobid).all()[-1])
            if vm_db_instance:
                vm_db_instance.STATUS = self.status_destroyed
            else:
                self.updateLog("could not find a database record with JOBID " +
                               " = %s" % jobid, "error")

            return [0, "VM %s successfully destroyed" % host]

        if exit_status:
            return [1, "Unable to destroy VM %s" % host]
        else:
            return [0, "VM %s successfully destroyed" % host]

    def exit_function(self, sig=None, data=None):
        """
        This function is called at the end of the process
        it is called when ^C is pressed or when SIGTERM is catched
        SIGTERM is often spawned by the init scipt with command 'stop'
        (i.e. service wnodes_cachemanager stop)
        it waits until all threads are finished and then exit
        if timeout is reached the exit is forced
        """
        self.updateLog("shutting down cache manager (this may take some time)")
        self.exit = True

        # Will now wait until all thread exit (with timeout)
        start = time.time()
        timeout = 600  # seconds
        elapsed = 0
        canExit = False
        thread_list = threading.enumerate()

        # Remove the MainThread (this one!!) from list
        del thread_list[0]
        while elapsed < timeout and not canExit:
            elapsed = time.time() - start
            canExit = True
            for th in thread_list:
                try:
                    if th.isAlive():
                        #cm.updateLog("Thread %s is still alive!" % th.getName)
                        canExit = False
                except:
                    self.updateLog("Exception for thread %s!" % th.getName)
                    pass
            time.sleep(1)

        if not canExit:
            self.updateLog("Timeout reached before every thread finished!")
            sys.exit(1)

        self.updateLog("Successfully exit!")
        sys.exit(0)
