package GetStatusListing; 

###############################################################################
# SUPER-UX
#
#    based on a version for UNICOS
#
###############################################################################

require Exporter;
@ISA = qw(Exporter);

@EXPORT_OK = qw(get_status_listing);

use Reporting qw(debug_report failed_report ok_report start_report command_report report_and_die);

use strict;

# SUPER-UX

# Queries the BSS for the state of (at least) all Unicore jobs
# and returns a list of their identifier together with the
# stats.
#
# Called in response to a #TSI_JOBQUERY in the script

# uses functions from Reporting

# No args
#
# Returns void to TSI
#         fail/success to NJS and a list of jobs
#
#        First line is "QSTAT",
#        followed by a line per found job, first word is BSS job identifier
#        and the second word is one of RUNNING, QUEUED or SUSPENDED or FROZEN
#
#        List must include all Unicore jobs (but can contain  extra jobs)
#
sub get_status_listing {

    my($command, $output, $result, $bssid, $state, $ustate, $result);

    debug_report("Finding all jobs on BSS");

    # Form the request command. 
    $command = "$main::qstat_cmd";

    command_report($command);

    $output = `($command) 2>&1`;

    # Parse output
    if($? != 0) {
        failed_report($output);
    }
    else {
        # Command succeeded. Parse the output and return
        # a line for each job found with two words, the first
        # is the job id the second its _Unicore_ state.

        # The output of "qstat -a" is:
        #para-2a@prometeo;  type=BATCH;  [ENABLED, RUNNING];  pri=10
        #  0 exit;  1 run;  0 suspend;  0 stage;  2 queued;  0 wait;  0 hold;  0 arrive;
        #
        #         REQUEST NAME         REQUEST ID            USER  PRI    STATE       JID
        #    1:      fixi3.4.2     58229.prometeo           ccost  31    RUNNING      997
        #    2:     lm_f90_2.1     58232.prometeo        ballabio  31     QUEUED    
        #                          ^^^^^                                  ^
        #                          bssid                               state
        #
        # There are also some uninteresting lines. 
        #
        # Do not differentiate between pipe and batch queues
        #
        # Ignore the minor states

        $result = "QSTAT\n";

        $_ = $output;
        #
        #Corresponding regular expression:
        #
        #^s*\d+\: -> lines starting with digit "   nnn:"
        #       \s+  -> 1 or more spaces
        #          \S+ -> 1 or more non-spaces (REQUEST NAME)
        #             \s+ -> 1 or more spaces
        #                (\d+)\. -> digit, put it in $1 (REQUEST ID)
        #                       .* -> any amount of junk (skip USER, PRI)
        #                        " " -> marks start of second last word on a line
        #                           (\S) -> first char of STATE, put it in $2
        #                               \S* -> 0 or more characters
        #                                  \s+  -> 1 or more spaces
        #                                     \d* -> 0 or more digits (JID can be absent)
        #                                        \s*$ -> 0 or more characters, end of line
        #
        # Note g to resume from end of last match
        while(/^\s*\d+\:\s+\S+\s+(\d+)\..* (\S)\S*\s+\d*\s*$/gm) {

            $bssid = $1;
            $state = $2;

            # Catches all known values.
            # Absent statuses are treated as unknown as status will contain a number
            $ustate = "UNKNOWN";

            # A - arriving, queue has not processed, so not running
            # D - departing, leaving pipe so not running
            # Q - queued
            # W - waiting (for date or pipe q available)
            $ustate = "QUEUED"  if $state =~ /A|D|Q|W/;

            # E - exiting, finished execution but implicit output written?
            # R - running (or ROUTING in pipe queue AAAARRRGGGGHH :-( )
            $ustate = "RUNNING" if $state =~ /R|E/ ;

            # C - checkpointed ?
            # H - held
            # P - preempted, assume that it will be automatically restarted
            # S - suspended
            $ustate = "SUSPENDED" if $state =~ /C|H|P|S/ ;

            # U - unknown

            # can also report FROZEN if this is supported by BSS (held and resources released)

            # Add to the returned lines
            $result = "$result $bssid $ustate\n";

        }

        print main::CMD_SOCK $result;
        debug_report("qstat executed OK");
    }

}
#
#                   Copyright (c) Fujitsu Ltd 2000 - 2004
#
#                Use and distribution is subject a License.
# A copy was supplied with the distribution (see documentation or the jar file).
#
# This product includes software developed by Fujitsu Limited (http://www.fujitsu.com).
