package Submit;

###############################################################################
# PBSpro 10.4
# based on the Torque version plus
# contributions from Michele Carpené (CINECA)
###############################################################################

require Exporter;
@ISA = qw(Exporter);

@EXPORT_OK = qw(submit);

use File::Path qw(mkpath);

use Reporting
  qw(debug_report failed_report ok_report start_report command_report report_and_die);

use strict;

#
# Submit the script to the BSS
#
# arg 1 = The script (this is called because the script contains the string "#TSI_SUBMIT");
#
# Returns void to TSI
#         on success returns the BSS identifier assigned to the job
#         on fail return a message
#

BEGIN {

    # --------------------------------------------------------------------
    # Create a file name to which scripts are written for the submits to
    # the batch sub-system. This needs to be unique as there may be
    # more than one TSI running, so base on the process id.

    $Submit::tsi_unique_file_name = "TSI_temp_file_$$";

    # Always cd to some neutral place when not doing something
    $Submit::neutral_dir = $ENV{PWD};
    if ( $Submit::neutral_dir =~ m/(.*)/s ) {
        $Submit::neutral_dir = $1;
    }

}

sub submit {

    my $from_njs = shift;

    # Correction for $HOME or $USER in root directories SvdB 11 Feb 2002
    # Substitute for these values as the executed script's
    # environment will not (and if it did the values will
    # be wrong). This is safe as thes script seen here is
    # completely generated by the NJS and so we will not be
    # messing with any user created code.

    $from_njs =~ s/\$USER/$ENV{USER}/g;
    $from_njs =~ s/\$HOME/$ENV{HOME}/g;

    # Get the information from the NJS (embedded as comments)
    #
    # Not picking up defnitions of Software Resources sent as:
    # #TSI_SWR<name>

    # First clear all command line sections from previous iterations
    my $jobname       = "";
    my $outcome_dir   = "";
    my $uspace_dir    = "";
    my $time          = "";
    my $memory        = "";
    my $nodes         = "";
    my $processors    = "";
    my $total_processors    = "";
    my $processors_per_node = "";
    my $fast_fs       = "";
    my $large_fs      = "";
    my $home_fast_fs  = "";
    my $home_large_fs = "";
    my $queue         = "";
    my $email         = "";
    my $interactive   = "";
    my $stdout        = "stdout";
    my $stderr        = "stderr";
    my $umask         = "";
    my $project       = "";
    $_ = $from_njs;

    while (/#TSI_(\S+) (.*)\n/g) {
        $jobname       = $2 if $1 eq "JOBNAME";
        $outcome_dir   = $2 if $1 eq "OUTCOME_DIR";
        $uspace_dir    = $2 if $1 eq "USPACE_DIR";
        $time          = $2 if $1 eq "TIME";
        $memory        = $2 if $1 eq "MEMORY";
        $nodes         = $2 if $1 eq "NODES";
        $processors    = $2 if $1 eq "PROCESSORS";
        $total_processors    = $2 if $1 eq "TOTAL_PROCESSORS";
        $processors_per_node = $2 if $1 eq "PROCESSORS_PER_NODE";
        $fast_fs       = $2 if $1 eq "FASTFS";
        $large_fs      = $2 if $1 eq "LARGEFS";
        $home_fast_fs  = $2 if $1 eq "HOMEFASTFS";
        $home_large_fs = $2 if $1 eq "HOMELARGEFS";
        $queue         = $2 if $1 eq "QUEUE";
        $email         = $2 if $1 eq "EMAIL";
        $interactive   = $2 if $1 eq "PREFER_INTERACTIVE";
        $stdout        = $2 if $1 eq "STDOUT";
        $stderr        = $2 if $1 eq "STDERR";
        $umask         = $2 if $1 eq "UMASK";
        $project       = $2 if $1 eq "PROJECT";
    }

    if ( $interactive eq "true" ) {
        start_report("Interactively executing a job");
    }
    else {
        start_report("Submitting a job to the BSS");
    }

    # will aggregate all PBS settings in this variable
    my $resource_list = "";

    # Jobname
    if ( $jobname eq "NONE" ) {
        $jobname = "#PBS -N $main::default_job_name";
    }
    else {

        # Valid names start with a character and
        # are 15 or less characters in total
        if ( $jobname =~ /([a-zA-Z]\S{0,14})/ ) {
            $jobname = "#PBS -N $1";
        }
        else {
            $jobname = "#PBS -N $main::default_job_name";
        }
    }

    $resource_list = "$jobname\n";

    # Queue
    if ( $queue eq "NONE" ) {
        # just use the default queue
    }
    else {
        $resource_list = "$resource_list" . "#PBS -q $queue \n";
    }

    # Project
    if ( !$project || $project eq "NONE" ) {
        # default project shall be used
    }
    else {
        $resource_list = "$resource_list" . "#PBS -A $project\n";
    }

    # Job time requirement. Wallclock time in seconds.
    my $hours = ( $time / ( 60 * 60 ) ) % 24;
    my $mins  = ( $time / 60 ) % 60;
    my $secs  = $time % 60;

    $resource_list =
      "$resource_list" . "#PBS -l walltime=$hours:$mins:$secs \n";

    #---------------------------------------------

    # Job memory requirements in megabytes
    $memory = $memory / 1024;
    $memory = $memory . "gb";

    $resource_list = "$resource_list" . "#PBS -l mem=$memory \n";

    # Nodes count
    #
    if ( $nodes eq "NONE" ) {
        # Single node job
        $nodes = "1";    #need this for vmem calculation later
        if ( $total_processors ne "" ) {
            $processors = $total_processors;
        }
        $resource_list = "$resource_list" . "#PBS -l nodes=1:ppn=$processors\n";
    }
    else {
        # Multiple node and/or processors
        $resource_list =
          "$resource_list" . "#PBS -l nodes=$nodes:ppn=$processors_per_node\n";
    }

    # Email
    if ( $email eq "NONE" ) {
        $email = "";
    }
    else {
        $resource_list = "$resource_list" . "#PBS -mb -me -mu $email \n";
    }

    # Tell the BSS to put the batch job's stdout and stderr
    # into these files
    $resource_list = "$resource_list" . "#PBS -o $outcome_dir/$stdout \n";
    $resource_list = "$resource_list" . "#PBS -e $outcome_dir/$stderr \n";

    # job default umask
    if ( $umask ne "" ) {
        $resource_list = $resource_list . "#PBS -W umask=$umask\n";
    }

    # Set the working dir (even if we're doing a cd later on...)
    # Without this, jobs will fail if users have no home
    $resource_list = $resource_list . "#PBS -d $uspace_dir \n";

    # cd to the Uspace directory, will write a file soon and
    # this means that there does not need to be a TSI working
    # directory
    if ( chdir($uspace_dir) == 0 ) {
        failed_report("Could not cd to Uspace $uspace_dir because $!");
        return 0;
    }

 # Make sure that the Outcome directory is there for the stdout and stderr files
    mkpath $outcome_dir, 0, 0700 unless -e $outcome_dir;
    chmod 0700, $outcome_dir;

 # add PBS settings to "from_njs" in order to add them to job description
    $from_njs = "#!/bin/sh\n$resource_list\n$from_njs";

    debug_report($from_njs);    #printing job description

    open( EMSCRIPT, ">$Submit::tsi_unique_file_name" );
    print EMSCRIPT $from_njs;
    close(EMSCRIPT);

    if ( $interactive eq "true" ) {
        my $command = "$Submit::tsi_unique_file_name";
        chmod 0700, $command;
        command_report($command);
        `./$command > $outcome_dir/stdout 2> $outcome_dir/stderr`;
        ok_report();
    }
    else {
        my $command = "$main::submit_cmd $Submit::tsi_unique_file_name";
        command_report($command);

        # and execute the command
        my $output = `($command) 2>&1`;

        # Parse output
        if ( $? != 0 ) {
            failed_report($output);
        }
        else {

            # Succeeded, return Job Identifier
            debug_report("output report: $output");
            if ( $output =~ /[0-9]*/ ) {
                my $jobid = $output;

                # qsub does not report a failure code so we need to
                # check here that we got an integer.
                #my $res = $jobid;
                #$res =~ s/[0-9]//g;
                if ( length($jobid) > 0 ) {
                    debug_report("Job submitted OK. Identifier: $jobid");
                    print main::CMD_SOCK "$jobid\n";
                }
                else {
                    failed_report("Job submit failed?: $jobid $output");
                }
            }
            else {
                failed_report("Request id not found in: $output");
            }

        }
    }

    chdir $Submit::neutral_dir;

}

#
#                   Copyright (c) Fujitsu Ltd 2000 - 2004
#
#                Use and distribution is subject a License.
# A copy was supplied with the distribution (see documentation or the jar file).
#
# This product includes software developed by Fujitsu Limited (http://www.fujitsu.com).
