package GetStatusListing;

###############################################################################
# PBS
#
#    Based on 3.0 PBS Linux port from Philipp Wieder at FZ Juelich
#                     and PBS by Achim Streit at Paderborn
#
###############################################################################

require Exporter;
@ISA = qw(Exporter);

@EXPORT_OK = qw(get_status_listing);

use Reporting qw(debug_report failed_report ok_report start_report command_report report_and_die);

use strict;

# Queries the BSS for the state of (at least) all Unicore jobs
# and returns a list of their identifier together with the
# stats.
#
# Called in response to a #TSI_JOBQUERY in the script

# uses functions from Reporting

# No args
#
# Returns void to TSI
#         fail/success to NJS and a list of jobs
#
#        First line is "QSTAT",
#        followed by a line per found job, first word is BSS job identifier
#        and the second word is one of RUNNING, QUEUED or SUSPENDED or FROZEN
#
#        List must include all Unicore jobs (but can contain  extra jobs)
#
sub get_status_listing {

    my($command, $output, $result, $bssid, $state, $ustate, $result);

    debug_report("Finding all jobs on BSS");

    # Form the request command. 
    $command = "$main::qstat_cmd";

    command_report($command);

    $output = `($command) 2>&1`;

    # Parse output
    if($? != 0) {
        failed_report($output);
    }
    else {

        # Command succeeded. Parse the output and return
        # a line for each job found with two words, the first
        # is the job id the second its Unicore state.

        $result = "QSTAT\n";

        # An exemplary output of the current PBS $qstat_cmd 
        # is presented below: 
        # Job id      Name             User      Time Use S Queue
        # ----------- ---------------- --------- -------- - -----
        # 285.zam008  batch_mpi_1node  zdv190    00:00:00 R default
        # 286.zam008  batch_mpi_1node  zdv190            0 Q default
        # 287.zam008  batch_mpi_1node  zdv190            0 Q default    

        # The interesting fields are the "job id" and 
        # the "S" (status) fields.
        # The job ID is the numerical part before the dot, 
        # so we are intersted in
        # the xxx from xxx.zam008 (see example above).
    
        # The number of junk fields is variable and we interpret 
        # the first letter
        # of status (status may be absent!)
        #
        # There are also some uninteresting lines. 
        #
        # Do not differentiate between pipe and batch queues.
        #
        # Ignore the minor states.

        $_ = $output;
        #
        # Terms are: start of a line
        #            digit (integer) <- put into $1
        #            "." marks end of integer
        #            any amount of junk
        #            " " marks start of last word on a line
        #            Character       <- put into $2
        #            (possibly) more characters in the word
        #            some white space
        #            last word
        #            (possibly) space
        #            ties match to one line
        #
        # Note g to resume from end of last match
        while(/^(\d+?)\..* (\S)\S*\s+\S+\s*$/gm) {

            $bssid = $1;
            $state = $2;

            # Catches all known values.
            # Absent statuses are treated as unknown as status will contain a number
            $ustate = "UNKNOWN";

            # Q - Job is queued, eligable to run or routed.
            # T - Job is being moved to new location.
            # W - Job is waiting for its execution time
            #     (submitted with the -a option (via qsub)).
            $ustate = "QUEUED"  if $state =~ /Q|T|W/;

            # E - exiting, finished execution but implicit output written?
            # R - running
            $ustate = "RUNNING" if $state =~ /R|E/ ;

            # H - held
            # S - suspended
            $ustate = "SUSPENDED" if $state =~ /H|S/ ;

            # can also report FROZEN if this is supported by BSS (held and resources released)

            # U - unknown

            # Add to the returned lines
            $result = "$result $bssid $ustate\n";

        }

        print main::CMD_SOCK $result;
        debug_report("qstat executed OK");
    }

}
#
#                   Copyright (c) Fujitsu Ltd 2000 - 2004
#
#                Use and distribution is subject a License.
# A copy was supplied with the distribution (see documentation or the jar file).
#
# This product includes software developed by Fujitsu Limited (http://www.fujitsu.com).
