package Submit;

###############################################################################
# TORQUE / Cray XT
#
#    Based on linux_torque TSI module with some XT specific changes
#
###############################################################################
use IPC::Open3;
require Exporter;
@ISA = qw(Exporter);

@EXPORT_OK = qw(submit);

use File::Path qw(mkpath);

use Reporting qw(debug_report failed_report ok_report start_report command_report report_and_die);

use strict;

# Submit the script to the BSS

# arg 1 = The script (this is called because the script contains the string "#TSI_SUBMIT");
#
# Returns void to TSI
#         on success returns the BSS identifier assigned to the job
#         on fail return a message 
#

BEGIN {
    # --------------------------------------------------------------------
    # Create a file name to which scripts are written for the submits to
    # the batch sub-system. This needs to be unique as there may be
    # more than one TSI running, so base on the process id.

    $Submit::tsi_unique_file_name = "TSI_temp_file_$$";

    $Submit::pbs_submit_script = "tsi_submit_$$";


	# Always cd to some neutral place when not doing something
	$Submit::neutral_dir = $ENV{PWD};
	if($Submit::neutral_dir =~ m/(.*)/s) {
		$Submit::neutral_dir = $1;
	}

}

sub submit {

    my $from_njs = shift;

	# Correction for $HOME or $USER in root directories
	# Substitute for these values as the executed script's
	# environment will not (and if it did the values will
	# be wrong). This is safe as the script seen here is
	# completely generated by the NJS and so we will not be
	# messing with any user created code.

	$from_njs =~ s/\$USER/$ENV{USER}/g;
	$from_njs =~ s/\$HOME/$ENV{HOME}/g;

    # Get the information from the NJS (embedded as comments)

    # First clear all command line sections from previous iterations
    my $jobname =       "";
    my $outcome_dir =   "";
    my $uspace_dir =    "";
    my $time =          "";
    my $memory =        "";
    my $nodes =         "";
    my $processors =    "";
    my $total_processors = "";
    my $processors_per_node = "";
    my $fast_fs =       "";
    my $large_fs =      "";
    my $home_fast_fs =  "";
    my $home_large_fs = "";
    my $queue =         "";
    my $email =         "";
    my $interactive =   "";
    my $stdout = "stdout";
    my $stderr = "stderr";

    $_ = $from_njs;
    while(/#TSI_(\S+) (.*)\n/g) {
        $jobname =       $2 if $1 eq "JOBNAME";
        $outcome_dir =   $2 if $1 eq "OUTCOME_DIR";
        $uspace_dir =    $2 if $1 eq "USPACE_DIR";
        $time =          $2 if $1 eq "TIME"; 
        $memory =        $2 if $1 eq "MEMORY";
        $nodes =         $2 if $1 eq "NODES";
        $processors =    $2 if $1 eq "PROCESSORS";
        $total_processors =    $2 if $1 eq "TOTAL_PROCESSORS";
        $processors_per_node = $2 if $1 eq "PROCESSORS_PER_NODE";
        $fast_fs =       $2 if $1 eq "FASTFS";
        $large_fs =      $2 if $1 eq "LARGEFS";
        $home_fast_fs =  $2 if $1 eq "HOMEFASTFS";
        $home_large_fs = $2 if $1 eq "HOMELARGEFS";
        $queue =         $2 if $1 eq "QUEUE";
        $email =         $2 if $1 eq "EMAIL";
        $interactive =   $2 if $1 eq "PREFER_INTERACTIVE";
	$stdout = $2 if $1 eq "STDOUT";
	$stderr = $2 if $1 eq "STDERR";

    }

   if($interactive eq "true") {
	start_report("Interactively executing a job");
   }
   else {
	start_report("Submitting a job to the BSS");
   }


    # Jobname
    if($jobname eq "NONE" ) {
        $jobname = "#PBS -N $main::default_job_name\n";
    }
    else {
        # Valid names start with a character and
        # are 15 or less characters in total
        if($jobname =~ /([a-zA-Z]\S{0,14})/) {
            $jobname = "#PBS -N $1";
        }
        else {
            $jobname = "#PBS -N $main::default_job_name";
        }
    }
    my $resource_list="";
    $resource_list = "$jobname\n";


   # Queue
   $resource_list = "$resource_list"."#PBS -q batch\n";

	# Job memory requirements in megabytes, this can be
	# either for the whole job, for each processor or
	# for each node depending on how the IDB is set up
	# (see PER_xxxx_LIMITS)
	#
	# Comment (SvdB) vmem = "max virtual memory used by all .. processes"
	#                The IDB must bet set up to send the correct value
	#                to the TSI (PER_JOB_LIMITS ?)
	#                Alternative: pvmem and PER_PROCESSOR_LIMITS?
	#                
	# Paderborn alternative: $resource_list = "-l ";  # no memory value passed on
    #my $resource_list = "-l vmem=$memory"."mb";
    #$resource_list = "$resource_list"."#PBS -l mem=$memory"."mb\n";

	# $nodes is number of nodes required
	#
	# $processors (processors per node) is 1 if the system
	# is not an SMP
	#       
    # Nodes count, NONE implies serial, otherwise a valid number 
    #
    if($nodes eq "NONE") {
        # Single node job
        if($total_processors ne ""){
	  $processors=$total_processors;
        }
        # ORNL style
        $resource_list = "$resource_list"."#PBS -l size=$processors\n";
        # NERSC style
        #$resource_list = "$resource_list"."#PBS -l mppwidth=1,mppnppn=$processors\n";
    }
    else {
        # Multiple node and/or processors
        # ORNL style
        $resource_list = "$resource_list"."#PBS -l size=".($nodes*$processors_per_node)."\n" ;
        # NERSC style
        #$resource_list = "$resource_list"."#PBS -l mppwidth=$nodes,mppnppn=$processors_per_node\n";
    }
        
    # Job time requirement. Wallclock time in seconds.
    $resource_list = "$resource_list"."#PBS -l walltime=$time\n";

    #send email to user?
    if($email eq "NONE") {
        $email = "#PBS -m n";
    }
    else {
        $email = "#PBS -m abe -M $email" ;
    }
    $resource_list = "$resource_list"."$email\n";

	# Tell the BSS to put the batch job's stdout and stderr
	# into these files

	# >>>>>>>>>>>>>>>>>>>>> NEEDS CHECKING

	$resource_list = "$resource_list"."#PBS -o $outcome_dir/$stdout\n";
        $resource_list = "$resource_list"."#PBS -e $outcome_dir/$stderr\n";

	# cd to the Uspace directory, will write a file soon and
	# this means that there does not need to be a TSI working
	# directory
	if(chdir($uspace_dir) == 0) {
		failed_report("Could not cd to Uspace $uspace_dir because $!");
		return 0;
	}

	# Make sure that the Outcome directory is there for the stdout and stderr files
	mkpath $outcome_dir, 0, 0700 unless -e $outcome_dir;
	chmod 0700, $outcome_dir;
        


    # Ignoring all other fields
    
    # Torque requires a full qualified path for the script which has to be executed
    $resource_list = "$resource_list"."$uspace_dir"."/"."$Submit::tsi_unique_file_name";   


    open(EMSCRIPT,">$Submit::tsi_unique_file_name");
    print EMSCRIPT $from_njs;
    close(EMSCRIPT);

  #  print "\n\n$resource_list\n\n";

    open(PBSSCRIPT,">$Submit::pbs_submit_script");
    print PBSSCRIPT $resource_list;
    print PBSSCRIPT "\n";
    close (PBSSCRIPT);

	if ($interactive eq "true"){
		my $command = "$Submit::tsi_unique_file_name";
		chmod 0700, $command;
		command_report($command);
		`./$command > $outcome_dir/$stdout 2> $outcome_dir/$stderr`;
		ok_report();
	}
	else {


                chmod 0700, $Submit::tsi_unique_file_name;
                my $command = "$main::submit_cmd $Submit::pbs_submit_script";

		

		command_report($command);

		# and execute the command
                my $pid = open3(\*QSUBIN,\*QSUBOUT,\*QSUBERR,$command);
                close(QSUBIN);
                waitpid($pid, 0);
                my $output = "";
                while ( <QSUBOUT> ) {
                   $output .= $_;
                }
                close(QSUBOUT);
                my $err = "";
                while ( <QSUBERR> ) {
                    $err .= $_;
                }
                close(QSUBERR);

		# Parse output
		if($? != 0) {
			failed_report($output."\n".$err);
		}
		else {

			# Succeeded, return Job Identifier
			# looking for line "xxxx.blah" (want the xxxx)
			# note: ? => minimal match in case other "." in string

			if($output =~ /(.+?)\..*/) {
				my $jobid = $1;

				# qsub does not report a failure code so we need to
				# check here that we got an integer. If we did assume
				# OK, otherwise report a failure
				my $res = $jobid;
				$res =~ s/[0-9]//g;
				if (length($res) == 0 ) {
					debug_report("Job submitted OK. Identifier: $jobid");
                                        if ( $err ne "" ) {
                                           debug_report($err);
                                        }
					print main::CMD_SOCK "$jobid\n";
				}
				else {
					failed_report("Job submit failed?: $jobid $output\n$err");
				}
			}
			else {
				failed_report("Request id not found in: $output");
			}

		}

	}


	chdir $Submit::neutral_dir;

    # print pbs_submit_script;

}

#
#                   Copyright (c) Fujitsu Ltd 2000 - 2004
#
#                Use and distribution is subject a License.
# A copy was supplied with the distribution (see documentation or the jar file).
#
# This product includes software developed by Fujitsu Limited (http://www.fujitsu.com).
