package Submit;    #SUN GRID ENGINE

require Exporter;
@ISA = qw(Exporter);

@EXPORT_OK = qw(submit);

use IPC::Open3;

use File::Path qw(mkpath);

use Reporting
  qw(debug_report failed_report ok_report start_report command_report report_and_die);
use LocalInfo qw(add_local_info);
use CommonUtils qw(addperms);
use strict;

# Submit the script to the BSS

# arg 1 = The script (this is called because the script contains the string "#TSI_SUBMIT");
#
# Returns void to TSI
#         on success returns the BSS identifier assigned to the job
#         on fail return a message
#

BEGIN {

    # --------------------------------------------------------------------
    # Create a file name to which scripts are written for the submits to
    # the batch sub-system. This needs to be unique as there may be
    # more than one TSI running, so base on the process id.

    $Submit::tsi_unique_file_name = "TSI_temp_file_$$";
    $Submit::sge_submit_script    = "tsi_submit_$$";

    # Always cd to some neutral place when not doing something
    $Submit::neutral_dir = $ENV{PWD};
    if ( $Submit::neutral_dir =~ m/(.*)/s ) {
        $Submit::neutral_dir = $1;
    }

}

sub submit {

    my $from_njs = shift;

    # change the beginning of the script to add local SGE info
    $from_njs = add_local_info($from_njs);

    # Correction for $HOME or $USER in root directories SvdB 11 Feb 2002
    # Substitute for these values as the executed script's
    # environment will not (and if it did the values will
    # be wrong). This is safe as thes script seen here is
    # completely generated by the NJS and so we will not be
    # messing with any user created code.

    $from_njs =~ s/\$USER/$ENV{USER}/g;
    $from_njs =~ s/\$HOME/$ENV{HOME}/g;

    # Get the information from the NJS (embedded as comments)
    # Not picking up defnitions of Software Resources sent as:
    # TSI_SWR<name>

    # First clear all command line sections from previous iterations
    my $jobname             = "";
    my $outcome_dir         = "";
    my $uspace_dir          = "";
    my $time                = "";
    my $memory              = "";
    my $nodes               = "";
    my $processors          = "";
    my $total_processors    = "";
    my $processors_per_node = "";
    my $fast_fs             = "";
    my $large_fs            = "";
    my $home_fast_fs        = "";
    my $home_large_fs       = "";
    my $queue               = "";
    my $email               = "";
    my $interactive         = "";
    my $stdout              = "stdout";
    my $stderr              = "stderr";
    my $project             = "";

    $_ = $from_njs;

    while (/#TSI_(\S+) (.*)\n/g) {
        $jobname             = $2 if $1 eq "JOBNAME";
        $outcome_dir         = $2 if $1 eq "OUTCOME_DIR";
        $uspace_dir          = $2 if $1 eq "USPACE_DIR";
        $time                = $2 if $1 eq "TIME";
        $memory              = $2 if $1 eq "MEMORY";
        $nodes               = $2 if $1 eq "NODES";
        $processors          = $2 if $1 eq "PROCESSORS";
        $total_processors    = $2 if $1 eq "TOTAL_PROCESSORS";
        $processors_per_node = $2 if $1 eq "PROCESSORS_PER_NODE";
        $fast_fs             = $2 if $1 eq "FASTFS";
        $large_fs            = $2 if $1 eq "LARGEFS";
        $home_fast_fs        = $2 if $1 eq "HOMEFASTFS";
        $home_large_fs       = $2 if $1 eq "HOMELARGEFS";
        $queue               = $2 if $1 eq "QUEUE";
        $email               = $2 if $1 eq "EMAIL";
        $interactive         = $2 if $1 eq "PREFER_INTERACTIVE";
        $stdout              = $2 if $1 eq "STDOUT";
        $stderr              = $2 if $1 eq "STDERR";
        $project             = $2 if $1 eq "PROJECT";
    }

    if ( $interactive eq "true" ) {
        start_report("Interactively executing a job");
    }
    else {
        start_report("Submitting a job to the BSS");
    }

    my $resource_list = "#! /bin/sh\n";

    $resource_list = $resource_list . "#\$ -S /bin/sh\n";

    # Jobname (source man page)
    if ( $jobname eq "NONE" ) {
        $jobname = "$main::default_job_name";
    }
    else {
        $jobname = "$jobname";
    }

    $resource_list = $resource_list . "#\$ -N $jobname\n";

    # Queue (source man page)
    #
    if ( $queue ne "NONE" ) {
        $resource_list = "$resource_list" . "#\$ -q $queue\n";
    }

    # Project
    #
    if ( $project && $project ne "NONE" ) {
        $resource_list = "$resource_list" . "#\$ -A $project\n";
    }

    # Job memory requirements in megabytes, this can be
    # either for the whole job, for each processor or
    # for each node depending on how the IDB is set up
    # (see PER_xxxx_LIMITS)

    # memory must be defined
    $resource_list = "$resource_list" . "#\$ -l h_vmem=$memory" . "M\n";

    # $time = Job time requirement. Wallclock time in seconds.
    my ( $sek, $min, $stu );
    $stu           = int( $time / 3600 );
    $min           = int( $time / 60 - 60 * $stu );
    $sek           = $time % 60;
    $time          = sprintf "%02d:%02d:%02d", $stu, $min, $sek;
    $resource_list = "$resource_list" . "#\$ -l h_rt=$time\n";

    # Note that this makes no allowance for
    # the time spent on the command processor, it may be necessary to add
    # a field for this e.g. -Txxxx where xxxxx is large enough to prevent
    # jobs being killed for exceeding this limit.

    # Nodes count, NONE implies serial, otherwise a valid number
    # $nodes is number of nodes required
    # $processors (processors per node) is 1 if the system
    # is not an SMP
    # new for nqs on origin
    # Syntax of time limits change depending on if executed on application or
    # command processors.
    if ( $nodes eq "NONE" || $main::is_pvp ) {

 # PVP or Single processor job, nothing parallel requested, run on command procs
        $nodes = "";
    }
    else {

        # Some sort of parallel resource asked for (including single proc jobs)
        # so run on application procs.
        $resource_list =
          "$resource_list" . "#\$ -pe $main::pe_name $total_processors\n";
    }

    # Email source: HS
    #if($email eq "NONE") {
    #    $email = "";
    #}
    #else {
    #    $email = "-mb -me -mu $email " ;
    #}

    # Tell the BSS to put the batch job's stdout and stderr
    # into these files
    $resource_list = $resource_list . "#\$ -o $outcome_dir/$stdout\n";
    $resource_list = $resource_list . "#\$ -e $outcome_dir/$stderr\n";

    # cd to the Uspace directory, will write a file soon and
    # this means that there does not need to be a TSI working
    # directory
    if ( chdir($uspace_dir) == 0 ) {
        failed_report("Could not cd to Uspace $uspace_dir because $!");
        return 0;
    }

 # Make sure that the Outcome directory is there for the stdout and stderr files
    mkpath $outcome_dir unless -e $outcome_dir;
    addperms 0700, $outcome_dir;

    unlink $Submit::tsi_unique_file_name;

    open( EMSCRIPT, ">$Submit::tsi_unique_file_name" );
    print EMSCRIPT $from_njs;
    close(EMSCRIPT);

    # actual script (obtained from NJS)
    $resource_list =
      "$resource_list" . "$uspace_dir" . "/" . "$Submit::tsi_unique_file_name";

    open( PBSSCRIPT, ">$Submit::sge_submit_script" );
    print PBSSCRIPT $resource_list;
    print PBSSCRIPT "\n";
    close(PBSSCRIPT);

    if ( $interactive eq "true" ) {
        my $command = "$Submit::tsi_unique_file_name";
        chmod 0700, $command;
        command_report($command);
        `./$command > $outcome_dir/$stdout 2> $outcome_dir/$stderr`;
        ok_report();
    }
    else {

        chmod 0700, $Submit::tsi_unique_file_name;
        my $command = "$main::submit_cmd $Submit::sge_submit_script";

        command_report($command);

        # and execute the command
        my $pid = open3( \*QSUBIN, \*QSUBOUT, \*QSUBERR, $command );
        close(QSUBIN);
        waitpid( $pid, 0 );

        my $output = "";
        while (<QSUBOUT>) {
            $output .= $_;
        }
        close(QSUBOUT);
        my $err = "";
        while (<QSUBERR>) {
            $err .= $_;
        }
        close(QSUBERR);

        # Parse output
        if ( $? != 0 ) {
            failed_report($err);
        }
        else {

            # Succeeded, return Job Identifier
            # looking for line "xxxx.blah" (want the xxxx)
            # note: ? => minimal match in case other "." in string

            if ( $output =~ /Your job (\d+) \(\".+?\"\) has been submitted/ ) {
                my $jobid = $1;

                # qsub does not report a failure code so we need to
                # check here that we got an integer. If we did assume
                # OK, otherwise report a failure
                my $res = $jobid;
                $res =~ s/[0-9]//g;
                if ( length($res) == 0 ) {
                    debug_report("Job submitted OK. Identifier: $jobid");
                    print main::CMD_SOCK "$jobid\n";
                }
                else {
                    failed_report("Job submit failed?: $jobid $output");
                }
            }
            else {
                failed_report("Request id not found in: $output");
            }

        }

    }

    chdir $Submit::neutral_dir;

}

# 11-11-11 Bastian Demuth : write SGE options to a file and use parallel environments (-pe) for requesting slots
# 16-09-08 Bernd Schuller : updated using the patches from Nadya Williams UZH
# 04-05-01 Sven van den Berghe fecit Created from 3.0 TSI for Unicore 3.5
# 24-07-01 SvdB Put stdout and stderr in Outcome dir, use Uspace for script file
# 21-08-01 Sven van den Berghe fecit Fixed neutral dir (cleaned for root too)
#
#                  Copyright (c) Fujitsu Ltd 2000 - 2002
#
#     Use and distribution is subject to the Community Source Licence.
#
# Further information is available from www.unicore.org or arcon@fecit.co.uk.
#

