package Submit;

###############################################################################
# TORQUE / Linux
#
#    Based on 3.0 PBS Linux port from Philipp Wieder at FZ Juelich
#                     and PBS by Achim Streit at Paderborn
#
###############################################################################

use IPC::Open3;
require Exporter;
@ISA = qw(Exporter);

@EXPORT_OK = qw(submit);

use File::Path qw(mkpath);

use Reporting
  qw(debug_report failed_report ok_report start_report command_report report_and_die);
use CommonUtils qw(addperms);

use strict;

# Submit the script to the BSS

# arg 1 = The script (this is called because the script contains the string "#TSI_SUBMIT");
#
# Returns void to TSI
#         on success returns the BSS identifier assigned to the job
#         on fail return a message
#

BEGIN {

    # --------------------------------------------------------------------
    # Create a file name to which scripts are written for the submits to
    # the batch sub-system. This needs to be unique as there may be
    # more than one TSI running, so base on the process id.

    $Submit::tsi_unique_file_name = "TSI_temp_file_$$";

    $Submit::pbs_submit_script = "tsi_submit_$$";

    # Always cd to some neutral place when not doing something
    $Submit::neutral_dir = $ENV{PWD};
    if ( $Submit::neutral_dir =~ m/(.*)/s ) {
        $Submit::neutral_dir = $1;
    }

}

sub submit {

    my $from_njs = shift;

    # Correction for $HOME or $USER in root directories
    # Substitute for these values as the executed script's
    # environment will not (and if it did the values will
    # be wrong). This is safe as the script seen here is
    # completely generated by the NJS and so we will not be
    # messing with any user created code.

    $from_njs =~ s/\$USER/$ENV{USER}/g;
    $from_njs =~ s/\$HOME/$ENV{HOME}/g;

    # Get the information from the NJS (embedded as comments)

    # First clear all command line sections from previous iterations
    my $jobname             = "";
    my $outcome_dir         = "";
    my $uspace_dir          = "";
    my $time                = "";
    my $memory              = "";
    my $nodes               = "";
    my $processors          = "";
    my $total_processors    = "";
    my $processors_per_node = "";
    my $fast_fs             = "";
    my $large_fs            = "";
    my $home_fast_fs        = "";
    my $home_large_fs       = "";
    my $queue               = "";
    my $email               = "";
    my $interactive         = "";
    my $reservation_id      = "";
    my $stdout              = "stdout";
    my $stderr              = "stderr";
    my $umask               = "";
    my $project             = "";
    my $nodes_filter        = $main::nodes_filter;

    if ($nodes_filter ne "") {
        $nodes_filter = ":$nodes_filter";
    }

    $_ = $from_njs;
    while (/#TSI_(\S+) (.*)\n/g) {
        $jobname             = $2 if $1 eq "JOBNAME";
        $outcome_dir         = $2 if $1 eq "OUTCOME_DIR";
        $uspace_dir          = $2 if $1 eq "USPACE_DIR";
        $time                = $2 if $1 eq "TIME";
        $memory              = $2 if $1 eq "MEMORY";
        $nodes               = $2 if $1 eq "NODES";
        $processors          = $2 if $1 eq "PROCESSORS";
        $total_processors    = $2 if $1 eq "TOTAL_PROCESSORS";
        $processors_per_node = $2 if $1 eq "PROCESSORS_PER_NODE";
        $fast_fs             = $2 if $1 eq "FASTFS";
        $large_fs            = $2 if $1 eq "LARGEFS";
        $home_fast_fs        = $2 if $1 eq "HOMEFASTFS";
        $home_large_fs       = $2 if $1 eq "HOMELARGEFS";
        $queue               = $2 if $1 eq "QUEUE";
        $email               = $2 if $1 eq "EMAIL";
        $interactive         = $2 if $1 eq "PREFER_INTERACTIVE";
        $reservation_id      = $2 if $1 eq "RESERVATION_REFERENCE";
        $stdout              = $2 if $1 eq "STDOUT";
        $stderr              = $2 if $1 eq "STDERR";
        $umask               = $2 if $1 eq "UMASK";
        $project             = $2 if $1 eq "PROJECT";
	$nodes_filter        = $nodes_filter . ':' . $2  
				  if $1 eq "BSS_NODES_FILTER";
    }

    if ( $interactive eq "true" ) {
        start_report("Interactively executing a job");
    }
    else {
        start_report("Submitting a job to the BSS");
    }

    # Jobname
    if ( $jobname eq "NONE" ) {
        $jobname = "#PBS -N $main::default_job_name\n";
    }
    else {
        # Valid names start with a character and
        # are 15 or less characters in total
        if ( $jobname =~ /([a-zA-Z]\S{0,14})/ ) {
            $jobname = "#PBS -N $1";
        }
        else {
            $jobname = "#PBS -N $main::default_job_name";
        }
    }
    my $resource_list = "";
    $resource_list = "$jobname\n";

    # Queue
    if ( !$queue || $queue eq "NONE" ) {
        # Torque's default queue is used when not specified otherwise.
    }
    else {
        $resource_list = "$resource_list" . "#PBS -q $queue\n";
    }

    # Project
    if ( !$project || $project eq "NONE" ) {
        # default project shall be used
    }
    else {
        $resource_list = "$resource_list" . "#PBS -A $project\n";
    }

    # Nodes count, NONE implies serial, otherwise a valid number
    #
    if ( $nodes eq "NONE" ) {
        # Single node job
        $nodes = "1";    #need this for vmem calculation later
        if ( $total_processors ne "" ) {
            $processors = $total_processors;
        }
        $resource_list = "$resource_list" . "#PBS -l nodes=1:ppn=$processors$nodes_filter\n";
    }
    else {
        # Multiple node and/or processors
        $resource_list =
          "$resource_list" . "#PBS -l nodes=$nodes:ppn=$processors_per_node$nodes_filter\n";
    }

    # Job memory requirements in megabytes
    # PBS vmem = "max virtual memory used by all .. processes"
    # XNJS sends "memory per node", so need to calculate correct vmem
    my $vmem = $memory * $nodes;
    $resource_list = "$resource_list" . "#PBS -l vmem=$vmem" . "mb\n";

    # Job time requirement. Wallclock time in seconds.
    $resource_list = "$resource_list" . "#PBS -l walltime=$time\n";

    #send email to user?
    if ( $email eq "NONE" ) {
        $email = "#PBS -m n";
    }
    else {
        $email = "#PBS -m abe -M $email";
    }
    $resource_list = "$resource_list" . "$email\n";

    # if a job is to be run in a reservation
    if ( !( $reservation_id eq "" ) ) {
        $resource_list =
          "$resource_list" . "#PBS -W x=FLAGS:ADVRES:$reservation_id\n";
    }

# Set the working dir (even if we're doing a cd later on...)
# Without this, jobs will fail if users have no home, cause whithout setting this,
# Torque falls back to the home directory - which doesn't exist
    $resource_list = $resource_list . "#PBS -d $uspace_dir \n";

    # Tell the BSS to put the batch job's stdout and stderr
    # into these files
    $resource_list = "$resource_list" . "#PBS -o $outcome_dir/$stdout\n";
    $resource_list = "$resource_list" . "#PBS -e $outcome_dir/$stderr\n";

    if ( $umask ne "" ) {
        $resource_list = $resource_list . "#PBS -W umask=$umask\n";
    }

    if ( chdir($uspace_dir) == 0 ) {
        failed_report("Could not cd to Uspace $uspace_dir because $!");
        return 0;
    }

 # Make sure that the Outcome directory is there for the stdout and stderr files
    mkpath $outcome_dir unless -e $outcome_dir;
    addperms 0700, $outcome_dir;

 # Torque requires a full qualified path for the script which has to be executed
    $resource_list =
      "$resource_list" . "$uspace_dir" . "/" . "$Submit::tsi_unique_file_name";

    open( EMSCRIPT, ">$Submit::tsi_unique_file_name" );
    print EMSCRIPT $from_njs;
    close(EMSCRIPT);

    open( PBSSCRIPT, ">$Submit::pbs_submit_script" );
    print PBSSCRIPT $resource_list;
    print PBSSCRIPT "\n";
    close(PBSSCRIPT);

    if ( $interactive eq "true" ) {
        my $command = "$Submit::tsi_unique_file_name";
        addperms 0700, $command;
        command_report($command);
        `./$command > $outcome_dir/$stdout 2> $outcome_dir/$stderr`;
        ok_report();
    }
    else {
        addperms 0700, $Submit::tsi_unique_file_name;
        my $command = "$main::submit_cmd $Submit::pbs_submit_script";

        command_report($command);

        # execute the submit command and capture its output and error
        my $pid = open3( \*QSUBIN, \*QSUBOUT, \*QSUBERR, $command );
        close(QSUBIN);
        waitpid( $pid, 0 );

        my $output = "";
        while (<QSUBOUT>) {
            $output .= $_;
        }
        close(QSUBOUT);
        my $err = "";
        while (<QSUBERR>) {
            $err .= $_;
        }
        close(QSUBERR);

        # Parse output
        if ( $? != 0 ) {
            failed_report($output."\n".$err);
        }
        else {

            # Succeeded, return Job Identifier
            # looking for line "xxxx.blah" (want the xxxx)
            # note: ? => minimal match in case other "." in string

            if ( $output =~ /(.+?)\..*/ ) {
                my $jobid = $1;

                # qsub does not report a failure code so we need to
                # check here that we got an integer. If we did assume
                # OK, otherwise report a failure
                my $res = $jobid;
                $res =~ s/[0-9]//g;
                if ( length($res) == 0 ) {
                    debug_report("Job submitted OK. Identifier: $jobid");
                    print main::CMD_SOCK "$jobid\n";
                }
                else {
                    failed_report("Job submit failed?: $jobid $output");
                }
            }
            else {
                failed_report("Request id not found in: $output");
            }

        }

    }

    chdir $Submit::neutral_dir;

}

#
#                   Copyright (c) Fujitsu Ltd 2000 - 2004
#
#                Use and distribution is subject a License.
# A copy was supplied with the distribution (see documentation or the jar file).
#
# This product includes software developed by Fujitsu Limited (http://www.fujitsu.com).
