#
# Maui implementation of resource reservation
# 
# Björn Hagemeier <b.hagemeier@fz-juelich.de>
# - created 2011-01-19
#
# Rafal Kluszczynski <klusi@icm.edu.pl>
# Grzegorz Marczak <lielow@icm.edu.pl>
# - updated 2012-08-29
#
package ResourceReservation;

require Exporter;
@ISA = qw(Exporter);

@EXPORT_OK = qw(execute_reservation make_reservation cancel_reservation query_reservation);

use Reporting qw(initial_report failed_report debug_report ok_report command_report);
use Submit qw(submit);
use POSIX qw(strftime);

use strict;


BEGIN {
        initial_report("Resource reservation Maui implementation");

	# Change this to your actual MAUI bin directory
	$main::reservation_bin_dir = "/usr/local/maui/bin";
}


#
# Make a reservation
#
# arg1: NJS command
#
# The NJS command includes the resources as used by the Submit command,
# and the requested starttime in ISO 8601 format
#
# Returns: 
#
#  TSI_OK
#  reservationID
#
#
# Example:
#
#   #TSI_MAKE_RESERVATION
#   #TSI_STARTTIME 2007-11-08T09:46:58+0100
#   #TSI_TIME 3600
#   #TSI_RESERVATION_OWNER user
#    # etc
#
sub make_reservation {
	my $njs_command = shift;
	#print $njs_command;

	#
	# STEP 1: parse NJS parameters
	#
	my $start_time;
	# extract start time (see above for format)
	if ($njs_command =~ m/#TSI_STARTTIME (\S+)/) {
		$start_time = format_starttime_for_maui($1);
	}

	#DURATION [[[DD:]HH:]MM:]SS  INFINITY 	duration of the reservation (not needed if ENDTIME is specified)
	# #TSI_TIME can be used directly, as it is formatted as seconds only
	my $duration;
	if ($njs_command =~ m/#TSI_TIME (\S+)/) {
		$duration = $1;
	}

	# #TSI_MEMORY already describes MB, which is expected by Maui
	my $memory;
	if ($njs_command =~ m/#TSI_MEMORY (\S+)/) {
		$memory = $1;
	}

	# #TSI_NODES describes number of nodes or equals NONE when 
	# only total processors value is used
	my $nodes;
	if ($njs_command =~ m/#TSI_NODES (\S+)/) {
		$nodes = $1;
		if ($nodes == "NONE") {
			undef $nodes;
		}
	}

	# #TSI_PROCESSORS_PER_NODE
	my $procs;
	if ($njs_command =~ m/#TSI_PROCESSORS_PER_NODE (\S+)/) {
		$procs = $1;
	}

	# #TSI_TOTAL_PROCESSORS
	my $total_procs;
	if ($njs_command =~ m/#TSI_TOTAL_PROCESSORS (\S+)/) {
		$total_procs = $1;
	}

	# #TSI_PROCESSORS
	# exists only for backward compatibility, is always the same as #TSI_PROCESSORS_PER_NODE
	if (!defined($procs) && ($njs_command =~ m/#TSI_PROCESSORS (\S+)/)) {
		$procs = $1;
	}

	# #TSI_HOST_NAME
	# this may not be desirable
	my $host;
	if ($njs_command =~ m/#TSI_HOST_NAME (\S+)/) {	  
		$host = $1;
		if ($host == "none") {
			undef $host;
		}
	}

	# #TSI_QUEUE
	my $queue;
	if ($njs_command =~ m/#TSI_QUEUE (\S+)/) {
		$queue = $1;
		if ($queue == "NONE") {
			undef $queue;
		}
	}

	my $reservation_owner;
	if ($njs_command =~ m/#TSI_RESERVATION_OWNER (\S+)/) {
		$reservation_owner = $1;
	}

	# getting user and group from #TSI_IDENTITY
	my $user;
	my $project;
	if ($njs_command =~ m/#TSI_IDENTITY (\S+) (\S+)/) {
		$user = $1;
		$project = $2;
	}

	#
	# STEP 2: evaluating parameters for reservation command
	# 
	my %resources; # for resources specification
	my $tasks = 1;

	if (!defined($nodes) || ($nodes eq "NONE")) {
		# defined only total processors: allocating slots
		# with resources specification below tasks are cpus
		$resources{"MEM"} = $memory;
		$resources{"PROCS"} = 1;
		
		$tasks = $total_procs;
	}
	else {
		# reserving whole nodes
		# whitout resources specifications tasks are nodes
		$tasks = $nodes;
	}
	# setting resource specification string
	my @resource_strings;
	while (my ($resource, $value) = each %resources) {
		push(@resource_strings, "$resource=$value");
	}
	my $resource_specification = join(":", @resource_strings);

	#
	# STEP 3: creating reservation command
	#
	my $reservation_command = "$main::reservation_bin_dir/setres ";
	$reservation_command .= "-s $start_time ";
	$reservation_command .= "-d $duration ";
	if (defined($reservation_owner)) {
		$reservation_command .= "-u $reservation_owner ";
	}
	if (defined($project) && ($project ne "NONE")) {
		$reservation_command .= "-g $project ";
	}

	# setting name of reservation based on xlogin
	if (defined($reservation_owner)) {
		$reservation_command .= "-n $reservation_owner ";
	}
	else {
		$reservation_command .= "-n $user ";
	}

	# queue is Maui's partition
	if (defined($queue) && ($queue ne "") && ($queue ne "NONE")) {
		$reservation_command .= "-p $queue ";
	}
	
	if (defined($resource_specification) && ($resource_specification ne "")) {
		$reservation_command .= "-r $resource_specification ";
	}
	$reservation_command .= "TASKS==$tasks";

	#
	# STEP 4: executing reservation command
	#
	debug_report("Reservation command: '$reservation_command'");
	#print "Reservation command: '$reservation_command'\n";
	my $output = `($reservation_command) 2>&1`;
	my $retval = $?;
	command_report($reservation_command);

	# parse output
	if ($retval != 0) {
		my $msg = "Could not create reservation";
		failed_report("$msg\n");
		print main::CMD_SOCK "$output\n";
	}
	else {
		#find the reservation id in the output
		my @id_array = split("'", $output);
		my $id_length = @id_array;
		my $reservation_id;
		if($id_length == 3) {
			$reservation_id = $id_array[1];
			#print "Reservation id: <$reservation_id>\n";

			ok_report("Created reservation with id: <$reservation_id>");
			print main::CMD_SOCK "$reservation_id\n";
		}
		else {
			failed_report("Unknown error");
			print main::CMD_SOCK "$output\n";
		}
	}
}


#
# Cancel a reservation
#
# arg1: NJS command
#
# The NJS command includes the reservation ID
#
#
# Example: 
#
# #TSI_CANCEL_RESERVATION
# #TSI_RESERVATION_REFERENCE job.1234
#
sub cancel_reservation {
	my $njs_command = shift;
	my $reservation_id;
	if ($njs_command =~ m/#TSI_RESERVATION_REFERENCE (\S+)/) {
		$reservation_id = $1;
	}

	my $output;
	my $retval;
	if (defined($reservation_id) && ($reservation_id ne "")) {
		my $release_command = "$main::reservation_bin_dir/releaseres $reservation_id";
		#print "Release command: $release_command\n";
		$output = `($release_command) 2>&1`;
		$retval = $?;

      		command_report($release_command);
	}
	else {
		$output = "Missing reservation reference from UNICORE/X";
		$retval = -1;
	}

	if ($retval != 0) {
		my $msg = "Reservation could not be cancelled";
		failed_report("$msg:\n$output\n");
	}
	else {
		ok_report("Cancelled reservation: <$reservation_id>");
	}
}


#
# Query a reservation
#
# arg1: NJS command
#
# The NJS command includes the reservation ID
#
# Result:
# The TSI replies with TSI_OK and a line containing the reservation status and the start time.
#
# Status is one of UNKNOWN, INVALID, WAITING, READY, ACTIVE, FINISHED or OTHER
# Start time is in the ISO 8601 format used in make_reservation() (see above)
#
#
# Example: 
#
# #TSI_QUERY_RESERVATION
# #TSI_RESERVATION_REFERENCE job.1234
#
sub query_reservation {
	my $njs_command = shift;

	my $reservation_id;
	if ($njs_command =~ m/#TSI_RESERVATION_REFERENCE (\S+)/) {
		$reservation_id = $1;
	}
	debug_report("Query reservation: <$reservation_id>");      

#Reservations
#
#ReservationID       Type S       Start         End    Duration    N/P    StartTime
#
#B7B36478-F10B-11E1-953B-8DCD092F1E54.0  User -    00:02:16     1:02:16     1:00:00    1/4    Tue Aug 28 14:29:30
#
#1 reservation located

	my $query_command = "$main::reservation_bin_dir/showres -v $reservation_id";
	#print "Query command: $query_command\n";

	my $output = `($query_command) 2>&1`;
	my $retval = $?;
        command_report($query_command);

	# parse output
	if ($retval != 0) {
		failed_report("$output\n");
	}
	else {
		$_ = $output;
		my $found = 0;

		my $reservationState;
		my $reservationStartTime;
	        my $reservationNP;
		while ( /^\s*(\S+\.\d+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+\s+\S+\s+\d+\s+\d+:\d+:\d+.+)/mg )
		{
		        my $reservationId = $1;
		        #my $reservationType = $2;
		        $reservationState = $3;
		        #my $reservationStart = $4;
		        #my $reservationEnd = $5;
		        #my $reservationDuration = $6;
		        $reservationNP = $7;
		        $reservationStartTime = $8;
			
			if ($reservationId eq $reservation_id) {
				$found = 1;
				last;
			}
		}

		my $status = "UNKNOWN";
		my $starttime = "";
		my $description = "";
		if ($found == 1) {
			if ($reservationState eq "S") {  #starting
				$status = "READY";
			}
			elsif ($reservationState eq "R") {  #running
				$status = "ACTIVE";
			}
			elsif ($reservationState eq "I") {  #idle
				$status = "READY";
			}
			elsif ($reservationState eq "-") {
				$status = "WAITING";
			}
			$starttime = starttime_to_iso8601($reservationStartTime);

			if ($status eq "UNKNOWN" || $status eq "WAITING") {
				my $now = strftime("%Y-%m-%dT%H:%M:%S%z", localtime(time()));
				if ($now gt $starttime) {
					$status = "READY";
				}
				else {
					$status = "WAITING";
				}
				# TODO: handle timezones
			}
			$starttime = " $starttime";
			$description = "Allocated nodes/CPUs: $reservationNP\n";
		}
		else {
			$status = "FINISHED";
		}

		ok_report("Reservation status = $status ($starttime)");
		print main::CMD_SOCK "$status$starttime\n$description";
	}
}


#
# Executes (claims) a reservation request
#
# arg1: reservation id
# arg2: NJS command
#
sub execute_reservation {
	my $reservation_id = shift;
	my $njs_command = shift;

	debug_report("Claim reservation with id <$reservation_id>");

	# simply submit it...
	submit($njs_command);
}


#
# Utils
#
sub format_starttime_for_maui {
	# src fmt: #TSI_STARTTIME 2007-11-08T09:46:58+0100
	# dst fmt: [HH[:MM[:SS]]][_MO[/DD[/YY]]] or +[[[DD:]HH:]MM:]SS
	my $orig_starttime = shift;
    
	my @date_time = split("T", $orig_starttime);
	my $date = $date_time[0];
	my $time = $date_time[1];

	my @date_components = split("-", $date);
	# this ignores the time zone
	my ($plain_time, $time_zone) = split('\+', $time);
	my @time_components = split(":", $plain_time);
	my %date_time = ( "year"   => $date_components[0],
			  "month"  => $date_components[1],
			  "day"    => $date_components[2]);
	my $date_time_str = join("_", $plain_time, join("/",$date_time{"month"},$date_time{"day"},$date_time{"year"}));
	return $date_time_str;
}

sub starttime_to_iso8601 {
        # src fmt: Tue Aug 28 14:29:30
        # dst fmt: 2007-11-08T09:46:58+0100
        my %months2num = ( Jan => "01", Feb => "02", Mar => "03",
                Apr => "04", May => "05", Jun => "06", Jul => "07",
                Aug => "08", Sep => "09", Oct => "10", Nov => "11",
                Dec => "12" );
        my $starttime = shift;

        my @tokens = split(/\s+/, $starttime);
        if ($#tokens >= 3) {
                my $month = $months2num{$tokens[1]};
                my $day = $tokens[2];
                my $time = $tokens[3];

                my $tmp = strftime("%Y;%z", localtime(time()));
                my ($year, $tz) = split(';', $tmp);
                if ($#tokens >= 4) {
                        $year = $tokens[4];
                }

                $day = "0$day" if ($day < 10);
                my $isotime = "$year-$month-$day" . 'T' . $time . $tz;
                return $isotime;
        }
        else {
                return "";
        }
}

