package Submit;

###############################################################################
#
# SLURM / Linux
#
#   Adapted from BSC's TSI by Xavier Delaruelle at CEA
#
###############################################################################

require Exporter;
@ISA = qw(Exporter);

@EXPORT_OK = qw(submit);

use File::Path qw(mkpath);
use POSIX qw(floor);

use Reporting
  qw(debug_report failed_report ok_report start_report command_report );
use CommonUtils qw(addperms);

use strict;

# Submit the script to the BSS

# arg 1 = The script (this is called because the script contains the string "#TSI_SUBMIT");
#
# Returns void to TSI
#         on success returns the BSS identifier assigned to the job
#         on fail return a message
#

BEGIN {

    # --------------------------------------------------------------------
    # Create a file name to which scripts are written for the submits to
    # the batch sub-system. This needs to be unique as there may be
    # more than one TSI running, so base on the process id.

    $Submit::tsi_unique_file_name = "TSI_temp_file_$$";

    # Always cd to some neutral place when not doing something
    $Submit::neutral_dir = $ENV{PWD};
    if ( $Submit::neutral_dir =~ m/(.*)/s ) {
        $Submit::neutral_dir = $1;
    }

}

sub submit {

    my $from_njs = shift;

    # Correction for $HOME or $USER in root directories SvdB 11 Feb 2002
    # Substitute for these values as the executed script's
    # environment will not (and if it did the values will
    # be wrong). This is safe as thes script seen here is
    # completely generated by the NJS and so we will not be
    # messing with any user created code.

    $from_njs =~ s/\$USER/$ENV{USER}/g;
    $from_njs =~ s/\$HOME/$ENV{HOME}/g;

    # Get the information from the NJS (embedded as comments)

    # First clear all command line sections from previous iterations
    my $jobname             = "";
    my $outcome_dir         = "";
    my $uspace_dir          = "";
    my $time                = "";
    my $memory              = "";
    my $nodes               = "";
    my $processors          = "";
    my $total_processors    = "";
    my $processors_per_node = "";
    my $fast_fs             = "";
    my $large_fs            = "";
    my $home_fast_fs        = "";
    my $home_large_fs       = "";
    my $queue               = "";
    my $email               = "";
    my $interactive         = "";
    my $reservation_id      = "";
    my $stdout              = "stdout";
    my $stderr              = "stderr";
    my $project             = "";
    my $nodes_filter        = $main::nodes_filter;

    $_ = $from_njs;
    while (/#TSI_(\S+) (.*)\n/g) {
        $jobname             = $2 if $1 eq "JOBNAME";
        $outcome_dir         = $2 if $1 eq "OUTCOME_DIR";
        $uspace_dir          = $2 if $1 eq "USPACE_DIR";
        $time                = $2 if $1 eq "TIME";
        $memory              = $2 if $1 eq "MEMORY";
        $nodes               = $2 if $1 eq "NODES";
        $processors          = $2 if $1 eq "PROCESSORS";
        $total_processors    = $2 if $1 eq "TOTAL_PROCESSORS";
        $processors_per_node = $2 if $1 eq "PROCESSORS_PER_NODE";
        $fast_fs             = $2 if $1 eq "FASTFS";
        $large_fs            = $2 if $1 eq "LARGEFS";
        $home_fast_fs        = $2 if $1 eq "HOMEFASTFS";
        $home_large_fs       = $2 if $1 eq "HOMELARGEFS";
        $queue               = $2 if $1 eq "QUEUE";
        $email               = $2 if $1 eq "EMAIL";
        $interactive         = $2 if $1 eq "PREFER_INTERACTIVE";
        $reservation_id      = $2 if $1 eq "RESERVATION_REFERENCE";
        $stdout              = $2 if $1 eq "STDOUT";
        $stderr              = $2 if $1 eq "STDERR";
        $project             = $2 if $1 eq "PROJECT";
	$nodes_filter        = ($nodes_filter eq "") ? $2 : $nodes_filter . '&' . $2  
				  if $1 eq "BSS_NODES_FILTER";
    }

    if ( $interactive eq "true" ) {
        start_report("Interactively executing a job");
    }
    else {
        start_report("Submitting a job to the BSS");
    }

    # Jobname (source man page)
    if ( !$jobname || $jobname eq "NONE" ) {
        $jobname = "--job-name=\"$main::default_job_name\"";
    }
    else {
        $jobname = "--job-name=\"$jobname\"";
    }

    # Queue (source man page)
    #
    # The IDB will send the queue name of NONE if it is
    # set up with a single queue whose name is "noname"

    if ( !$queue || $queue eq "NONE" ) {
        $queue = "";
    }
    else {
        $queue = "--partition=$queue";
    }

    # Nodes constraints (filter)
    #
    if ( $nodes_filter ne "" ) {
        $nodes_filter = "--constraint=\"$nodes_filter\"";
    }

    # Project (account to charge the job to)
    #
    if ( !$project || $project eq "NONE" ) {

        # default project shall be used
        $project = "";
    }
    else {
        $project = "--account=$project\n";
    }

    # Job memory requirements in megabytes, this can be
    # either for the whole job, for each processor or
    # for each node depending on how the IDB is set up
    # (see PER_xxxx_LIMITS)

    if ( !$memory || $memory eq "NONE" ) {
        $memory = "";
    }
    else {
        $memory = "--mem=$memory";
    }

    # Convert time in the SLURM understandable syntax
    my $minutes = floor( $time / 60 );
    my $seconds = $time - ( $minutes * 60 );
    $time = "--time=$minutes:$seconds";

    # Nodes count
    if ( $nodes eq "NONE" || $nodes eq "0" ) {
        $nodes = "";
    }
    else {
        $nodes = "--nodes=$nodes";
    }

    # Processors count
    if ( $total_processors eq "NONE" || $total_processors eq "0" ) {
        $processors = "";
    }
    else {
        $processors = "--ntasks=$total_processors";
    }

    # Processors per node count
    if ( $processors_per_node eq "NONE" || $processors_per_node eq "0" ) {
        $processors_per_node = "";
    }
    else {
        $processors_per_node = "--ntasks-per-node=$processors_per_node";
    }

    # Email (email address of recepient)
    if ( !$email || $email eq "NONE" ) {
        $email = "";
    }
    else {

        # No mail will be sent if no --mail-type argument specified
        $email = "--mail-user=$email";
    }

    # Resource reservation specification
    my $reservation_specification = "";
    if ($reservation_id ne "") {
        $reservation_specification = "--reservation=$reservation_id";
    }

    # Tell the BSS to put the batch job's stdout and stderr
    # into these files
    my $stdout_loc = "-o $outcome_dir/$stdout";
    my $stderr_loc = "-e $outcome_dir/$stderr";

    # cd to the Uspace directory, will write a file soon and
    # this means that there does not need to be a TSI working
    # directory
    if ( chdir($uspace_dir) == 0 ) {
        failed_report("Could not cd to Uspace $uspace_dir because $!");
        return 0;
    }

 # Make sure that the Outcome directory is there for the stdout and stderr files
    mkpath $outcome_dir unless -e $outcome_dir;
    addperms 0700, $outcome_dir;

    # Ignoring all other fields

    open( EMSCRIPT, ">$Submit::tsi_unique_file_name" );
    print EMSCRIPT $from_njs;
    close(EMSCRIPT);

    if ( $interactive eq "true" ) {
        my $command = "$Submit::tsi_unique_file_name";
        addperms 0700, $command;
        command_report($command);
        `./$command > $outcome_dir/$stdout 2> $outcome_dir/$stderr`;
        ok_report();
    }
    else {

        my $command =
"$main::submit_cmd $queue $reservation_specification $nodes_filter $nodes $processors $processors_per_node $email $memory $time $jobname $project $stdout_loc $stderr_loc <$uspace_dir/$Submit::tsi_unique_file_name";
        command_report($command);

        addperms 0700, $Submit::tsi_unique_file_name;

        # and execute the command
        my $output = `($command) 2>&1`;

        # Parse output
        if ( $? != 0 ) {
            failed_report($output);
        }
        else {

            # Succeeded, return Job Identifier
            # looking for "Submitted batch job xxxx" in line (we want the xxxx)

            if ( $output =~ /Submitted batch job (\d+)/ ) {
                my $jobid = $1;

                debug_report("Job submitted OK. Identifier: $jobid");
                print main::CMD_SOCK "$jobid\n";
            }
            else {
                failed_report("Request id not found in: $output");
            }

        }
    }

    unlink $Submit::tsi_unique_file_name;

    chdir $Submit::neutral_dir;

}

