#
# Slurm implementation of resource reservation
# 
# Rafal Kluszczynski <klusi@icm.edu.pl>
# Grzegorz Marczak <lielow@icm.edu.pl>
# - created 2012-08-29 (based on Maui implementation by Bjoern Hagemeier)
#
package ResourceReservation;

require Exporter;
@ISA = qw(Exporter);

@EXPORT_OK = qw(execute_reservation make_reservation cancel_reservation query_reservation);

use Reporting qw(initial_report failed_report debug_report ok_report command_report);
use Submit qw(submit);

#use Data::UUID;
#use OSSP::uuid;
use POSIX qw(strftime);

use strict;


BEGIN {
        initial_report("Resource reservation Slurm implementation");

	# Change this to your actual Slurm bin directory
	$main::reservation_bin_dir   = "/usr/bin";
}


#
# Make a reservation
#
# arg1: NJS command
#
# The NJS command includes the resources as used by the Submit command,
# and the requested starttime in ISO 8601 format
#
# Returns: 
#
#  TSI_OK
#  reservationID
#
#
# Example:
#
#   #TSI_MAKE_RESERVATION
#   #TSI_STARTTIME 2007-11-08T09:46:58+0100
#   #TSI_TIME 3600
#   #TSI_RESERVATION_OWNER user
#    # etc
#
sub make_reservation {
	my $njs_command = shift;
	#print $njs_command;

	#
	# STEP 1: parse NJS parameters
	#
	my $start_time;
	# extract start time
	if ($njs_command =~ m/#TSI_STARTTIME (\S+)/) {
		$start_time = format_starttime_for_slurm($1);
	}

	# #TSI_TIME is formatted as seconds
	my $duration;
	if ($njs_command =~ m/#TSI_TIME (\S+)/) {
		$duration = format_duration_for_slurm($1);
	}

	# #TSI_MEMORY describes memory in MB
	my $memory;
	if ($njs_command =~ m/#TSI_MEMORY (\S+)/) {
		$memory = $1;
	}

	# #TSI_NODES describes number of nodes or equals NONE when 
	# only total processors value is used
	my $nodes;
	if ($njs_command =~ m/#TSI_NODES (\S+)/) {
		$nodes = $1;
		if ($nodes == "NONE") {
			undef $nodes;
		}
	}

	# #TSI_PROCESSORS_PER_NODE
	my $procs;
	if ($njs_command =~ m/#TSI_PROCESSORS_PER_NODE (\S+)/) {
		$procs = $1;
	}

	# #TSI_TOTAL_PROCESSORS
	my $total_procs;
	if ($njs_command =~ m/#TSI_TOTAL_PROCESSORS (\S+)/) {
		$total_procs = $1;
	}

	# #TSI_PROCESSORS
	# exists only for backward compatibility, is always the same as #TSI_PROCESSORS_PER_NODE
	if (!defined($procs) && ($njs_command =~ m/#TSI_PROCESSORS (\S+)/)) {
		$procs = $1;
	}

	# #TSI_HOST_NAME
	# this may not be desirable
	my $host;
	if ($njs_command =~ m/#TSI_HOST_NAME (\S+)/) {	  
		$host = $1;
		if ($host == "none") {
			undef $host;
		}
	}

	# #TSI_QUEUE
	my $queue;
	if ($njs_command =~ m/#TSI_QUEUE (\S+)/) {
		$queue = $1;
		if ($queue == "NONE") {
			undef $queue;
		}
	}

	my $reservation_owner;
	if ($njs_command =~ m/#TSI_RESERVATION_OWNER (\S+)/) {
		$reservation_owner = $1;
	}

	# getting user and group from #TSI_IDENTITY
	my $user;
	my $project;
	if ($njs_command =~ m/#TSI_IDENTITY (\S+) (\S+)/) {
		$user = $1;
		$project = $2;
	}

	#
	# STEP 2: evaluating parameters for reservation command
	# 
	if (!defined($nodes) || ($nodes eq "NONE")) {
		my $msg = "Only nodes reservation is supported by Slurm!";
		failed_report("$msg\n");
		return;
	}

	#
	# STEP 3: creating reservation command
	#
	my $reservation_command = "$main::reservation_bin_dir/scontrol create reservation ";
	$reservation_command .= "starttime=$start_time ";
	$reservation_command .= "duration=$duration ";
	if (defined($reservation_owner)) {
		$reservation_command .= "user=$reservation_owner ";
	}

	# TODO: handle groups reservations (unsupported by slurm?)
	#if (defined($project) && ($project ne "NONE")) {
	#}

	# TODO: setting name (Slurm by default uses xlogin as prefix so skip it)
	# $reservation_command .= "reservation=$user ";

	# setting queue as Slurm partition
	if (defined($queue) && ($queue ne "") && ($queue ne "NONE")) {
		$reservation_command .= "partitionname=$queue ";
	}
	$reservation_command .= "nodecnt=$nodes ";

	#
	# STEP 4: executing reservation command
	#
	debug_report("Reservation command: '$reservation_command'");
	#print "Reservation command: '$reservation_command'\n";
	my $output = `($reservation_command) 2>&1`;
	my $retval = $?;
	command_report($reservation_command);

	# parse output
	if ($retval != 0) {
		my $msg = "Could not create reservation";
		failed_report("$msg\n");
		print main::CMD_SOCK "$output\n";
	}
	else {
		#find the reservation id in the output
		my $reservation_id;
		if ($output =~ m/Reservation created: (\S+)/) {
			$reservation_id = $1;
			#print "Reservation id: <$reservation_id>\n";

	                ok_report("Created reservation with id: <$reservation_id>");
        	        print main::CMD_SOCK "$reservation_id\n";
		}
		else {
			failed_report("Unknown error");
			print main::CMD_SOCK "$output\n";
		}
	}
}


#
# Cancel a reservation
#
# arg1: NJS command
#
# The NJS command includes the reservation ID
#
#
# Example: 
#
# #TSI_CANCEL_RESERVATION
# #TSI_RESERVATION_REFERENCE job.1234
#
sub cancel_reservation {
	my $njs_command = shift;
	my $reservation_id;
	if ($njs_command =~ m/#TSI_RESERVATION_REFERENCE (\S+)/) {
		$reservation_id = $1;
	}

	my $output;
	my $retval;
	if (defined($reservation_id) && ($reservation_id ne "")) {
		my $release_command = "$main::reservation_bin_dir/scontrol delete reservation=$reservation_id";
		#print "Release command: $release_command\n";
		$output = `($release_command) 2>&1`;
		$retval = $?;

      		command_report($release_command);
	}
	else {
		$output = "Missing reservation reference from UNICORE/X";
		$retval = -1;
	}

	if ($retval != 0) {
		my $msg = "Reservation could not be cancelled";
		failed_report("$msg:\n$output\n");
	}
	else {
		ok_report("Cancelled reservation: <$reservation_id>");
	}
}


#
# Query a reservation
#
# arg1: NJS command
#
# The NJS command includes the reservation ID
#
# Result:
# The TSI replies with TSI_OK and a line containing the reservation status and the start time.
#
# Status is one of UNKNOWN, INVALID, WAITING, READY, ACTIVE, FINISHED or OTHER
# Start time is in the ISO 8601 format used in make_reservation() (see above)
#
#
# Example: 
#
# #TSI_QUERY_RESERVATION
# #TSI_RESERVATION_REFERENCE job.1234
#
sub query_reservation {
	my $njs_command = shift;

	my $reservation_id;
	if ($njs_command =~ m/#TSI_RESERVATION_REFERENCE (\S+)/) {
		$reservation_id = $1;
	}
	debug_report("Query reservation: <$reservation_id>");      

#ReservationName=abcd StartTime=2012-09-06T16:00:00 EndTime=2012-09-06T18:00:00 Duration=02:00:00
#   Nodes=wn[1-2] NodeCnt=2 Features=(null) PartitionName=infinity.q Flags=IGNORE_JOBS
#   Users=lielow Accounts=(null) Licenses=cpu*4 State=INACTIVE

	my $query_command = "$main::reservation_bin_dir/scontrol show reservation=$reservation_id";
	#print "Query command: $query_command\n";

	my $output = `($query_command) 2>&1`;
	my $retval = $?;
        command_report($query_command);

	# parse output
	if ($retval != 0) {
		failed_report("$output\n");
	}
	else {
		$_ = $output;
		my $status = "UNKNOWN";
		my $reservationStartTime = "";
		my $description = "";

		if ($output =~ m/Reservation $reservation_id not found/) {
			$status = "FINISHED";
      		}
		else {
			# TODO: handle correctly timezones
			my $reservationState;
			my $reservationEndTime;
			my $reservationNodeCnt;
			my $reservationNodesNames;

			# getting state and start time info from output
			if ($output =~ m/ State=(\S+)/) {
                        	$reservationState = $1;
                	}
			if ($output =~ m/ StartTime=(\S+) /) {
                                $reservationStartTime = $1."+0100";
			}
			if ($output =~ m/ EndTime=(\S+) /) {
                                $reservationEndTime = $1."+0100";
			}
			if ($output =~ m/ NodeCnt=(\S+) /) {
				$reservationNodeCnt = $1;
			}
			if ($output =~ m/ Nodes=(\S+) /) {
				$reservationNodesNames = $1;
			}

			# setting correct reservation state
			if ($reservationState eq "ACTIVE") {
				$status = "READY";
			}
			elsif ($reservationState eq "INACTIVE") {
				$status = "WAITING";
			}

			if ($status eq "UNKNOWN" || $status eq "WAITING") {
               			my $now = strftime("%Y-%m-%dT%H:%M:%S%z", localtime(time()));
				if ($now gt $reservationEndTime) {
					$status = "FINISHED";
				}
				elsif ($now gt $reservationStartTime) {
					$status = "READY";
				}
				else {
					$status = "WAITING";
				}
			}
			$description = "Allocated $reservationNodeCnt nodes ($reservationNodesNames).\n";
		}

		ok_report("Reservation status = $status ($reservationStartTime)");
		print main::CMD_SOCK "$status $reservationStartTime\n$description";
	}
}


#
# Executes (claims) a reservation request
#
# arg1: reservation id
# arg2: NJS command
#
sub execute_reservation {
	my $reservation_id = shift;
	my $njs_command = shift;

	debug_report("Claim reservation with id <$reservation_id>");

	# simply submit it...
	submit($njs_command);
}


#
# Utils
#
sub format_starttime_for_slurm {
	# src fmt: #TSI_STARTTIME 2007-11-08T09:46:58+0100
	# dst fmt: 2007-11-08T09:46:58
	my $starttime = shift;
	my $sep = '+';    
    
	# TODO: handle timezones
	my @date_time = split /\Q$sep\E/, $starttime;
	return $date_time[0];
}

sub format_duration_for_slurm {
        # src fmt: #TSI_TIME 7200 (seconds)
	# dst fmt: [days-]hours:minutes:seconds
        my $duration = shift;

	my $seconds = $duration % 60;
	$duration = int($duration / 60);
	my $minutes = $duration % 60;
	$duration = int($duration / 60);
	my $hours = $duration % 24;
	my $days = int($duration / 24);

	my $result = "$hours:$minutes:$seconds";
	$result = "$days-$result" if ($days > 0);
	return $result;
}

