package GetStatusListing;

###############################################################################
#
# SLURM / Linux
#
#   Adapted from BSC's TSI by Xavier Delaruelle at CEA
#
###############################################################################

require Exporter;
@ISA = qw(Exporter);

@EXPORT_OK = qw(get_status_listing);

use Reporting
  qw(debug_report failed_report ok_report start_report command_report report_and_die);

use strict;

# Queries the BSS for the state of (at least) all Unicore jobs
# and returns a list of their identifier together with the
# stats.
#
# Called in response to a #TSI_JOBQUERY in the script

# uses functions from Reporting

# No args
#
# Returns void to TSI
#         fail/success to NJS and a list of jobs
#
#        First line is "QSTAT",
#        followed by a line per found job, first word is BSS job identifier
#        and the second word is one of RUNNING, QUEUED or SUSPENDED or FROZEN
#        Optionally, a third word gives the queue name
#        The list must include all UNICORE jobs (but can contain  extra jobs)
#
sub get_status_listing {

    my ( $command, $output, $result, $bssid, $state, $ustate );

    debug_report("Finding all jobs on BSS");

    # Form the request command.
    $command = "$main::qstat_cmd";

    command_report($command);

    $output = `($command) 2>&1`;

    # Parse output
    if ( $? != 0 ) {
        failed_report($output);
    }
    else {
        $result = "QSTAT\n";

        # Test for messages/errors from squeue
        if ( $output =~ /slurm.* error:/m ) {
            failed_report($output);
            return;
        }

        # The slurm command has been set up to return just the jobid
        # and the status, no header lines, so just process them
        #
        # The format of each line is: <bssid> <state>
        # where we want <bssid> and <state>.

        my %decoder = {};
        $decoder{'CANCELLED'}  = 'UNKNOWN';
        $decoder{'COMPLETED'}  = 'RUNNING';
        $decoder{'CONFIGURING'} = 'QUEUED';
        $decoder{'COMPLETING'} = 'RUNNING';
        $decoder{'FAILED'}     = 'UNKNOWN';
        $decoder{'NODE_FAIL'}  = 'UNKNOWN';
        $decoder{'PENDING'}    = 'QUEUED';
        $decoder{'PREEMPTED'} = 'SUSPENDED';
        $decoder{'RUNNING'}    = 'RUNNING';
        $decoder{'SUSPENDED'}  = 'SUSPENDED';
        $decoder{'TIMEOUT'}    = 'UNKNOWN';

        my @arr_output = split /\n/, $output;
        debug_report("qstat init");

        foreach (@arr_output) {
            if (/^(.+)\s(.+)$/) {
                $bssid = $1;
                $state = $2;

                $ustate = $decoder{$state};

                # Add to the returned lines
                $result = "$result $bssid $ustate\n";
            }
        }

        print main::CMD_SOCK $result;
        debug_report("qstat executed OK");
    }

}

