package Submit;

###############################################################################
#
# Loadleveler on AIX
#
###############################################################################

require Exporter;
@ISA = qw(Exporter);

@EXPORT_OK = qw(submit);

use File::Path qw(mkpath);

use Reporting
  qw(debug_report failed_report ok_report start_report command_report );
use CommonUtils qw(addperms);

use strict;

# UXP/V specialisation

# Submit the script to the BSS

# arg 1 = The script (this is called because the script contains the string "#TSI_SUBMIT");
#
# Returns void to TSI
#         on success returns the BSS identifier assigned to the job
#         on fail return a message
#

BEGIN {

    # --------------------------------------------------------------------
    # Create a file name to which scripts are written for the submits to
    # the batch sub-system. This needs to be unique as there may be
    # more than one TSI running, so base on the process id.

    $Submit::tsi_unique_file_name = "TSI_temp_file_$$";

    # Always cd to some neutral place when not doing something
    $Submit::neutral_dir = $ENV{PWD};
    if ( $Submit::neutral_dir =~ m/(.*)/s ) {
        $Submit::neutral_dir = $1;
    }

}

sub submit {

    my $from_njs = shift;

    $from_njs =~ s/\$USER/$ENV{USER}/g;
    $from_njs =~ s/\$HOME/$ENV{HOME}/g;

    my $jobname        = "";
    my $outcome_dir    = "";
    my $uspace_dir     = "";
    my $time           = "";
    my $memory         = "";
    my $nodes          = "";
    my $processors     = "";
    my $fast_fs        = "";
    my $large_fs       = "";
    my $home_fast_fs   = "";
    my $home_large_fs  = "";
    my $queue          = "";
    my $email          = "";
    my $interactive    = "";
    my $reservation_id = "";
    my $stdout         = "stdout";
    my $stderr         = "stderr";
    my $project        = "";

    $_ = $from_njs;
    while (/#TSI_(\S+) (.*)\n/g) {
        $jobname        = $2 if $1 eq "JOBNAME";
        $outcome_dir    = $2 if $1 eq "OUTCOME_DIR";
        $uspace_dir     = $2 if $1 eq "USPACE_DIR";
        $time           = $2 if $1 eq "TIME";
        $memory         = $2 if $1 eq "MEMORY";
        $nodes          = $2 if $1 eq "NODES";
        $processors     = $2 if $1 eq "PROCESSORS";
        $fast_fs        = $2 if $1 eq "FASTFS";
        $large_fs       = $2 if $1 eq "LARGEFS";
        $home_fast_fs   = $2 if $1 eq "HOMEFASTFS";
        $home_large_fs  = $2 if $1 eq "HOMELARGEFS";
        $queue          = $2 if $1 eq "QUEUE";
        $email          = $2 if $1 eq "EMAIL";
        $interactive    = $2 if $1 eq "PREFER_INTERACTIVE";
        $reservation_id = $2 if $1 eq "RESERVATION_REFERENCE";
        $stdout         = $2 if $1 eq "STDOUT";
        $stderr         = $2 if $1 eq "STDERR";
        $project        = $2 if $1 eq "PROJECT";

    }

    if ( $interactive eq "true" ) {
        start_report("Interactively executing a job");
    }
    else {
        start_report("Submitting a job to the BSS");
    }

    # Jobname
    if ( $jobname eq "NONE" ) {
        $jobname = "# @ job_name = $main::default_job_name\n";
    }
    else {
        $jobname = "# @ job_name = $jobname\n";
    }

    # reservation ID
    if ( !( $reservation_id eq "" ) ) {
        $reservation_id = "# @ ll_res_id = $reservation_id\n";
    }

    # Queue
    #
    # The IDB will send the queue name of NONE if it is
    # set up with a single queue whose name is "noname"

    if ( $queue eq "NONE" ) {
        $queue = "";
    }
    else {
        $queue = "# @ class = $queue\n";
    }

    # Project
    if ( !$project || $project eq "NONE" ) {

        # default project shall be used
        $project = "";
    }
    else {
        $project = "# @ account_no = $project\n";
    }

    # Job memory requirements in megabytes, this can be
    # either for the whole job, for each processor or
    # for each node depending on how the IDB is set up
    # (see PER_xxxx_LIMITS)
    $memory = "# @ requirements =  (Memory >= $memory)\n";

    # Job time requirement. Wallclock time in seconds.
    $time = "# @ cpu_limit = $time\n";

    # NOTE: With V3.5 the NJS support SMPs by always sending
    #       a node request and a processors per node request.
    #
    #       $nodes is number of nodes required
    #
    #       $processors (processors per node) is 1 if the system
    #       is not an SMP

    # Job type default is serial
    #
    my $jobtype = "# @ job_type = serial\n";

    # Nodes count, NONE keeps job type serial,
    # otherwise a valid number implies job type parallel
    #
    if ( $nodes eq "NONE" || $nodes eq "0" ) {
        $nodes = "";
    }
    else {
        $jobtype = "# @ job_type = parallel\n";
        $nodes   = "# @ node = $nodes\n";
    }

    # Processors count (CPUs/per_node on SMP system), NONE keeps job type,
    # otherwise a valid number implies job_type parallel
    #
    if ( $processors eq "NONE" || $processors eq "0" || $nodes eq "" ) {
        $processors = "";
    }
    else {
        $processors = "# @ tasks_per_node = $processors\n";
    }

    # Email (email address of recepient)
    if ( $email eq "NONE" ) {
        $email = "";
    }
    else {
        $email = "# @ notification = always\n# @ notify_user = $email\n";
    }

    # Tell the BSS to put the batch job's stdout and stderr
    # into these files
    my $stdout_loc = "# @ output = $outcome_dir/$stdout \n";
    my $stderr_loc = "# @ error = $outcome_dir/$stderr \n";

    # cd to the Uspace directory, will write a file soon and
    # this means that there does not need to be a TSI working
    # directory
    if ( chdir($uspace_dir) == 0 ) {
        failed_report("Could not cd to Uspace $uspace_dir because $!");
        return 0;
    }

 # Make sure that the Outcome directory is there for the stdout and stderr files
    mkpath $outcome_dir unless -e $outcome_dir;
    addperms 0700, $outcome_dir;

    # Put these requests into the script that is
    # submitted to Loadleveler
    $from_njs =
"#!/bin/sh\n$queue$project$jobtype$reservation_id$nodes$processors$memory$email$time$jobname$stdout_loc$stderr_loc# @ queue\n$from_njs";

    open( EMSCRIPT, ">$Submit::tsi_unique_file_name" );
    print EMSCRIPT $from_njs;
    close(EMSCRIPT);

    if ( $interactive eq "true" ) {
        my $command = "$Submit::tsi_unique_file_name";
        addperms 0700, $command;
        command_report($command);
        `./$command > $outcome_dir/$stdout 2> $outcome_dir/$stderr`;
        ok_report();
    }
    else {

        my $command = "$main::submit_cmd $Submit::tsi_unique_file_name";

        command_report($command);

        # and execute the command
        my $output = `($command) 2>&1`;

        # Parse output
        if ( $? != 0 ) {
            failed_report($output);
        }
        else {

            # Succeeded, return Job Identifier
            # parsing a line of format - The job "jobid.junk" ..... junk

            # Previous, wrong anyway? e.g. sp5cws.rz.uni-karlsruhe.de.xxx
            #if($output =~ /The job "(.+?)\..*/) {
            # note the minimal match - for first "."

            # Match the complete jobid not just numeric - assumes that failed
            # submits are caught by return code of by the else block
            if ( $output =~ /The job "(\S+)" has been submitted.*/ ) {

                my $jobid = $1;

                debug_report("Job submitted OK. Identifier: $jobid");
                print main::CMD_SOCK "$jobid\n";

            }
            else {
                failed_report("Request id not found in: $output");
            }

        }
    }

    unlink $Submit::tsi_unique_file_name;
    chdir $Submit::neutral_dir;

}

#
#                   Copyright (c) Fujitsu Ltd 2000 - 2004
#
#                Use and distribution is subject a License.
# A copy was supplied with the distribution (see documentation or the jar file).
#
# This product includes software developed by Fujitsu Limited (http://www.fujitsu.com).
