package UserCache;

require Exporter;
@ISA = qw(Exporter);

@EXPORT_OK =
  qw(get_gids_4user get_gid_4group get_members_4group get_gid_4user get_uid_4user get_home_4user);

use Reporting qw(initial_report failed_report debug_report);

use strict;

# This module provides a memory only cache with timed expiry for the data used by BecomeUser
# to change process identity.

#Caching
# groups - maps group names to gids
# members - maps group names to lists of group members
# all_groups - maps username to list of all its gids. List is a string concatenated with spaces.
# gids - maps user names to primary gids
# uids - maps user names to uids
# homes - maps user names to their home directories
#
# 2 cache namespaces
# users_cache - maps user names to cache timestamps
# groups_cache - maps group names to chache timestamps

#retrieves all gids the username is member of
sub get_gids_4user {
    my $user = shift;

    prepare_users($user);
    if ( !defined( $UserCache::all_groups{$user} ) ) {
        return "";
    }
    return $UserCache::all_groups{$user};
}

#resolves the group name
sub get_gid_4group {
    my $group = shift;
    prepare_groups($group);

    if ( !defined( $UserCache::groups{$group} ) ) {
        return -1;
    }
    return $UserCache::groups{$group};
}

#returns all members of a given group name
sub get_members_4group {
    my $group = shift;
    prepare_groups($group);

    if ( !defined( $UserCache::members{$group} ) ) {
        return -1;
    }
    return $UserCache::members{$group};
}

#returns primary gid for a username
sub get_gid_4user {
    my $user = shift;
    prepare_users($user);
    if ( !defined( $UserCache::gids{$user} ) ) {
        return -1;
    }
    return $UserCache::gids{$user};
}

#returns uid for a username
sub get_uid_4user {
    my $user = shift;
    prepare_users($user);
    if ( !defined( $UserCache::uids{$user} ) ) {
        return -1;
    }
    return $UserCache::uids{$user};
}

#returns home for a username
sub get_home_4user {
    my $user = shift;
    prepare_users($user);
    if ( !defined( $UserCache::homes{$user} ) ) {
        return "";
    }
    return $UserCache::homes{$user};
}

sub prepare_users {
    my $user = shift;
    if ( !defined( $UserCache::users_cache{$user} )
        || expired( $UserCache::users_cache{$user} ) )
    {
        update_user_info($user);
    }
    if ( !defined( $UserCache::users_cache{$user} ) ) {
        debug_report("Unknown user name requested: '$user'.");
    }
}

sub prepare_groups {
    my $group = shift;
    if ( !defined( $UserCache::groups_cache{$group} )
        || expired( $UserCache::groups_cache{$group} ) )
    {
        update_group_info($group);
    }
    if ( !defined( $UserCache::groups_cache{$group} ) ) {
        debug_report("Unknown group name requested: '$group'.");
    }
}

#checks if cache TTL is expired
sub expired {
    my $ts  = shift;
    my $cur = time();
    if ( $ts + $main::usersCacheTtl < $cur ) {
        return 1;
    }
    return 0;
}

# Establish the list of all (including supplementary) groups the user is member of
# It is done by getgrent(), results are cached.
# Arguments are username and primary group id.
sub get_gids_4user_nc {
    my $user = shift;
    my ( $group, $g_p, $gid, $members );
    my $sup_groups;
    $sup_groups = shift;

    while (1) {
        ( $group, $g_p, $gid, $members ) = getgrent();
        if ( !defined($group) ) {
            last;
        }
        if ( $members =~ /( |^)$user( |$)/ ) {
            $sup_groups = $sup_groups . " " . $gid;
        }
    }
    endgrent();
    debug_report(
        "Established supplementary groups list for the user $user: $sup_groups"
    );
    return $sup_groups;
}

# Fills up all per group caches with a freshly obtained information
sub update_group_info {
    my $req_group = shift;
    undef $UserCache::groups{$req_group};
    undef $UserCache::members{$req_group};
    undef $UserCache::groups_cache{$req_group};

    my ( $g_name, $g_passwd, $g_gid, $g_members ) = getgrnam($req_group);

    if ( !defined($g_gid) ) {
        return;
    }

    debug_report(
        "New group information obtained for $req_group ($g_name $g_gid)");

    $UserCache::groups{$req_group}       = $g_gid;
    $UserCache::members{$req_group}      = $g_members;
    $UserCache::groups_cache{$req_group} = time();
}

# Fills up all per user caches with a freshly obtained information
sub update_user_info {
    my $user = shift;

    undef $UserCache::uids{$user};
    undef $UserCache::gids{$user};
    undef $UserCache::homes{$user};
    undef $UserCache::all_groups{$user};
    undef $UserCache::users_cache{$user};

    my (
        $l_name,    $l_passwd, $l_uid, $l_gid, $l_quota,
        $l_comment, $l_gcos,   $l_dir, $l_shell
    ) = getpwnam($user);

    if ( !defined($l_uid) ) {
        return;
    }

    debug_report(
        "New user information obtained for $user ($l_name $l_uid $l_gid) ");
    $UserCache::all_groups{$user} = get_gids_4user_nc( $user, $l_gid );

    $UserCache::uids{$user}        = $l_uid;
    $UserCache::gids{$user}        = $l_gid;
    $UserCache::homes{$user}       = $l_dir;
    $UserCache::users_cache{$user} = time();
}

