package SharedConfiguration;

require Exporter;
@ISA       = qw(Exporter);
@EXPORT_OK = qw(configure_common startup read_configuration_file);
use strict;
use Initialisation qw(initialisation);
use MainLoop qw(infinite_loop);
use Reporting qw(initial_report);

# This file contains configuration of the TSI which is
# independent from the operating system and batch subsystem.
# You should at least review it (however modifications are
# rarely required here).

# TSI CONFIGURATION <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
sub configure_common() {

    # This section contains values that need to be set for each site
    #
    # Review each entry and confirm that the value set here conforms
    # to site policies and produces the behaviour that you want from the TSI

    # Values read by the functions in the Reporting package
    $main::debug = 1;    # Produce debug output?  0 = no, 1 = yes

    # For some debugging it is valuable to see the messages from
    # the NJS to the TSI. Set the variable below to 1 to do this.
    # Each message is written to a different file in the initial
    # working directory of the TSI process. If the TSI is run as
    # root the files will be written with the ownership of each Xlogin.
    $main::dump_messages = 0;
    $main::dump_dir      = "./DUMP";

    # Scripts executed on behalf of the user inherit their environment
    # from the environment of the shell executing this Perl script.
    #
    # If the TSI is run in setuid mode, then this environment might
    # not be correct for a normal user environment and general security
    # recommendations are that setuid scripts should be careful about
    # the environment supplied to child processes.

    # Set the child process SHELL variable to use the shell always used
    # by the NJS (user tasks with different shells are grandchildren)
    # Do not change from sh, but its location might change
    $ENV{SHELL} = "/bin/sh";

    # You may want to set this too
    #$ENV{IFS}=" \t\n"; # or something like this

    # Add any other site required environment variables here or, if they
    # are xlogin specific values in the becomeuser sub.

    # A name to be given to batch jobs if the user does not supply one
    # or if the given one is invalid
    $main::default_job_name = "UnicoreJob";

    # A filter property of working nodes (by default use all)
    $main::nodes_filter = "";

    # The Syslog facility to report to (relevant only if syslog is enabled)
    $main::logfacility = "LOG_USER";

    # The Syslog host where log output should go
    # By default logs are sent to the local Syslog daemon
    $main::loghost = "";

    # Cache ttl of users and groups info
    $main::usersCacheTtl = 600;
}

# END OF CONFIGURATION <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

sub read_configuration_file() {
    open FILE, $main::tsi_props
      or die "Can't read configuration file $main::tsi_props: $!";
    while (<FILE>) {
        if (/^\s*tsi.njs_machine\s*=\s*(\S+)/)  { $main::njs_machine = $1; }
        if (/^\s*tsi.njs_port\s*=\s*(\S+)/)     { $main::njs_port    = $1 }
        if (/^\s*tsi.my_port\s*=\s*(\S+)/)      { $main::my_port     = $1 }
        if (/^\s*tsi.keystore\s*=\s*(\S+)/)     { $main::keystore    = $1 }
        if (/^\s*tsi.keypass\s*=\s*(\S+)/)      { $main::keypass     = $1 }
        if (/^\s*tsi.truststore\s*=\s*(\S+)/)   { $main::truststore  = $1 }
        if (/^\s*tsi.logdir\s*=\s*(\S+)/)       { $main::logdir      = $1 }
        if (/^\s*tsi.logfacility\s*=\s*(\S+)/)  { $main::logfacility = $1 }
        if (/^\s*tsi.loghost\s*=\s*(\S+)/)      { $main::loghost     = $1 }
        if (/^\s*tsi.logformat\s*=\s*(.+)\s*$/) { $main::logformat   = $1 }
        if (/^\s*tsi.debug\s*=\s*(\S+)/)        { $main::debug       = $1 }
        if (/^\s*tsi.usersCacheTtl\s*=\s*(\S+)/){ $main::usersCacheTtl = $1 }
        if (/^\s*tsi.default_job_name\s*=\s*(\S+)/) {
            $main::default_job_name = $1;
        }
	if (/^\s*tsi.nodes_filter\s*=\s*(\S+)/) { 
            $main::nodes_filter = $1 
        }
        if (/^\s*tsi.enforce_gids_consistency\s*=\s*(\S+)/) {
            $main::enforce_os_gids_str = $1;
        }
        if (/^\s*tsi.fail_on_invalid_gids\s*=\s*(\S+)/) {
            $main::fail_on_invalid_gids_str = $1;
        }
        if ( $main::enforce_os_gids_str eq q{} ) {
            $main::enforce_os_gids = 1;
        }
        elsif ( $main::enforce_os_gids_str =~ /false/i ) {
            $main::enforce_os_gids = 0;
        }
        elsif ( $main::enforce_os_gids_str =~ /true/i ) {
            $main::enforce_os_gids = 1;
        }
        else {
            die
"Invalid value of the tsi.enforce_gids_consistency configuration parameter, should be 'true' or 'false'.";
        }
        if ( $main::fail_on_invalid_gids_str eq q{} ) {
            $main::fail_on_invalid_gids = 0;
        }
        elsif ( $main::fail_on_invalid_gids_str =~ /false/i ) {
            $main::fail_on_invalid_gids = 0;
        }
        elsif ( $main::fail_on_invalid_gids_str =~ /true/i ) {
            $main::fail_on_invalid_gids = 1;
        }
        else {
            die
"Invalid value of the tsi.fail_on_invalid_gids configuration parameter, should be 'true' or 'false'.";
        }

        if (/^\s*tsi.setfacl\s*=\s*(\S+)/)      { $main::setfacl_cmd      = $1 }
        if (/^\s*tsi.getfacl\s*=\s*(\S+)/)      { $main::getfacl_cmd      = $1 }
        if (/^\s*tsi.nfs4_setfacl\s*=\s*(\S+)/) { $main::nfs4_setfacl_cmd = $1 }
        if (/^\s*tsi.nfs4_getfacl\s*=\s*(\S+)/) { $main::nfs4_getfacl_cmd = $1 }

        if (/^\s*tsi.acl.(\S+)\s*=\s*(\S+)/) {
            if ( $2 ne "NONE" && $2 ne "NFS" && $2 ne "POSIX" ) {
                initial_report
"ERROR: ignoring wrong value of ACL specification for directory $1: $2 (must be 'NONE', 'POSIX' or 'NFS')";
            }
            else {
                $main::acl{$1} = $2;
            }
        }
    }
    close(FILE);
    initial_report "Evaluating configuration file: $main::tsi_props";
    initial_report "Found XNJS host $main::njs_machine";
    initial_report "Found XNJS port $main::njs_port";
    initial_report "Found shepherd port $main::my_port";

    if ( $main::keystore ne q{} ) {
        $main::ssl = 1;
        die "Invalid Keystore filename: $main::keystore."
          unless -f $main::keystore;
        if ( $main::truststore ne q{} ) {
            if ( !-f $main::truststore ) {
                initial_report
"Invalid Truststore filename: $main::truststore, using Keystore file.";
                $main::truststore = $main::keystore;
            }
        }
        else {
            initial_report
              "Truststore filename not specified, using Keystore file.";
            $main::truststore = $main::keystore;
        }
        initial_report "Found keystore file $main::keystore";
        initial_report "Found truststore file $main::truststore";
    }

    my $k;
    my $v;
    while ( ( $k, $v ) = each %main::acl ) {
        initial_report "ACL support for directory $k: $v\n";
    }

    if ( defined $main::setfacl_cmd && defined $main::getfacl_cmd ) {
        $main::posixacl_enabled = 1;
        initial_report "setfacl command: $main::setfacl_cmd";
        initial_report "getfacl command: $main::getfacl_cmd";
    }
    else {
        $main::posixacl_enabled = 0;
        initial_report "POSIX ACL support disabled (commands not configured)";
    }
    if ( defined $main::nfs4_setfacl_cmd && defined $main::nfs4_getfacl_cmd ) {
        $main::nfsacl_enabled = 1;
        initial_report "nfs4_setfacl command: $main::nfs4_setfacl_cmd";
        initial_report "nfs4_getfacl command: $main::nfs4_getfacl_cmd";
    }
    else {
        $main::nfsacl_enabled = 0;
        initial_report "NFS ACL support disabled  (commands not configured)";
    }

}

sub startup() {
    read_configuration_file();
    BecomeUser::initialize();

    # The size of the buffer to use when reading files from the NJS and
    # writing files to the NJS, used by GetDirectory and PutFiles
    $main::BUFFER_SIZE = 4096;

    # TSI Identification information
    $main::my_version = "6.5.1";

    initialisation();    # contact the NJS

    infinite_loop();     # ... and process the requests
}

