package MainLoop;

require Exporter;
@ISA = qw(Exporter);

@EXPORT_OK = qw(infinite_loop);

use BecomeUser qw(become_user restore_id);
use Reporting qw(failed_report report_and_die);
use GetStatusListing qw(get_status_listing);
use Submit qw(submit);
use ExecuteScript qw(execute_script);
use PutFiles qw(put_files);
use GetFileChunk qw(get_file_chunk);
use JobControl qw(abort_job cancel_job hold_job resume_job get_job_details);
use Dump2File qw(dump2file);
use ACL qw(process_acl);
use ResourceReservation
  qw(make_reservation execute_reservation cancel_reservation query_reservation);
use NoBatchOnly qw(reap_zombies);
use strict;

# Loop listening for input from NJS and processing it

# Writes directly to main::CMD_SOCK, reads directly from main::CMD_SOCK

sub infinite_loop() {

    $/ = "ENDOFMESSAGE\n";

    my $message;

    my $process_umask = umask();

    while ( $message = <main::CMD_SOCK> ) {

        # Strip message terminator _and_ untaint the data
        # I trust the source! and the setuid/setguid is downgrading!
        if ( $message =~ m/(.*)(ENDOFMESSAGE)/s ) {
            $message = $1;
        }

        # Got the message, do something about it.

        # Now find which command to execute

      SWITCH: {

            # Reap all zombies which have been created by the fork in
            # Submit.pm of NOBATCH TSI (for others won't be triggered)
            reap_zombies() if $NoBatchOnly::number_of_zombies > 0;

            # --- Commands that do not require a specific user

            # Are you still there? from NJS. No execution of commands
            # just reply with version number to NJS, no need to setuid
            $message =~ /#TSI_PING\n/ && do {
                print main::CMD_SOCK $main::my_version . "\n";
                last SWITCH;
            };

            # --- Now switch user for those that should be run as a user

            # Never execute any command as root, always setuid to a user
            # and if this fails abort execution of this command

            # Find which user we should be setting UID to
            # look for the line: #TSI_IDENTITY user_name project name
            my $user;
            my $project;

            if ( $message =~ m/#TSI_IDENTITY (\S+) (\S+)/ ) {
                $user    = $1;
                $project = $2;
            }
            else {
                failed_report("No user identity found in: $message ---");
                last SWITCH;
            }

            last SWITCH unless become_user( $user, $project ) == 1;

            dump2file($message);

            $message =~ /#TSI_GETSTATUSLISTING\n/ && do {
                get_status_listing($message);
                last SWITCH;
            };

            $message =~ /#TSI_SUBMIT\n/ && do {

                #set umask if was provided
                if ( $message =~ m/#TSI_UMASK (\S+)/ ) {
                    my $umask = oct($1);
                    umask($umask);
                }

                # check for presence of a reservation id
                if ( $message =~ m/#TSI_RESERVATION_REFERENCE (\S+)/ ) {
                    my $reservation_id = $1;
                    execute_reservation( $reservation_id, $message );
                }
                else {
                    submit($message);
                }

                #reset umask to the original one
                umask($process_umask);
                last SWITCH;
            };
            $message =~ /#TSI_MAKE_RESERVATION\n/ && do {
                make_reservation($message);
                last SWITCH;
            };

            $message =~ /#TSI_CANCEL_RESERVATION\n/ && do {
                cancel_reservation($message);
                last SWITCH;
            };

            $message =~ /#TSI_QUERY_RESERVATION\n/ && do {
                query_reservation($message);
                last SWITCH;
            };

            $message =~ /#TSI_EXECUTESCRIPT\n/ && do {
                execute_script($message);
                last SWITCH;
            };

            $message =~ /#TSI_ENDPROCESSING\n/ && do {
                end_processing($message);
                last SWITCH;
            };

            $message =~ /#TSI_PUTFILES\n/ && do {
                put_files($message);
                last SWITCH;
            };
            $message =~ /#TSI_GETFILECHUNK\n/ && do {
                get_file_chunk($message);
                last SWITCH;
            };

            $message =~ /#TSI_ABORTJOB\n/ && do {
                abort_job($message);
                last SWITCH;
            };

            $message =~ /#TSI_CANCELJOB\n/ && do {
                cancel_job($message);
                last SWITCH;
            };

            $message =~ /#TSI_HOLDJOB\n/ && do {
                hold_job($message);
                last SWITCH;
            };

            $message =~ /#TSI_RESUMEJOB\n/ && do {
                resume_job($message);
                last SWITCH;
            };

            $message =~ /#TSI_GETJOBDETAILS\n/ && do {
                get_job_details($message);
                last SWITCH;
            };

            $message =~ /#TSI_FILE_ACL\n/ && do {
                process_acl($message);
                last SWITCH;
            };

            failed_report("Unrecognised NJS command: $message ---");

        }

        restore_id();

        $message = "";

        print main::CMD_SOCK "ENDOFMESSAGE\n";

        $/ = "ENDOFMESSAGE\n";
    }

    report_and_die("End of inputs from NJS, finishing execution");

    exit;

}

#
#                   Copyright (c) Fujitsu Ltd 2000 - 2004
#
#                Use and distribution is subject a License.
# A copy was supplied with the distribution (see documentation or the jar file).
#
# This product includes software developed by Fujitsu Limited (http://www.fujitsu.com).
