package JobControl;

require Exporter;
@ISA = qw(Exporter);

@EXPORT_OK = qw(abort_job cancel_job hold_job resume_job get_job_details);

use Reporting qw(start_report failed_report ok_report command_report);

use strict;

# Generic BSS job control functions

#
# Assumes that the following variables have been set up
# before a call:
#
# $main::site_sez_no_holds: true if jobs on BSS cannot be held
#
# $main::abort_cmd:  command to execute to abort a BSS job (with process id as a argument)
# $main::cancel_cmd:  command to execute to cancel a BSS job (with process id as a argument)
# $main::hold_cmd:  command to execute to hold a BSS job (with process id as a argument)
# $main::freeze_cmd:  (optional) command to execute to freeze a BSS job (with process id as a argument)
# $main::resume_cmd:  command to execute to resume a BSS job (with process id as a argument)
# $main::details_cmd:  command to get details about a BSS job (with process id as a argument)

# Abort a job on the BSS.
#
# arg1: text from NJS with TSI_BSSID <request id>
#
sub abort_job {

    my $from_njs = shift;

    my ( $requestid, $output, $command );

    ($requestid) = ( $from_njs =~ /.*#TSI_BSSID (.+)\n/ );

    start_report("Aborting request $requestid");

    $command = "$main::abort_cmd $requestid\n";

    command_report($command);

    $output = `($command) 2>&1`;

    # Parse output
    if ( $? != 0 ) {
        failed_report($output);
    }
    else {
        ok_report($output);
    }
}

# Cancel a job on the BSS (abort plus dispose of output files)
#
# arg1: text from NJS with TSI_BSSID <request id>
#
sub cancel_job {

    my $from_njs = shift;

    my ( $requestid, $output, $command );

    ($requestid) = ( $from_njs =~ /.*#TSI_BSSID (.+)\n/ );

    start_report("Cancelling request $requestid");

    $command = "$main::cancel_cmd $requestid\n";

    command_report($command);

    $output = `($command) 2>&1`;

    # Parse output
    if ( $? != 0 ) {
        failed_report($output);
    }
    else {
        ok_report($output);
    }
}

# Hold a job on the BSS.
#
# arg1: text from NJS with TSI_BSSID <request id>
#
sub hold_job {

    if ($main::site_sez_no_holds) {
        ok_report("No holding of jobs on BSS");
    }
    else {

        # Freeze rather than hold? Yes, if the site supports it
        # (guess by having $main::freeze_cmd set) and it is requested

        my $from_njs = shift;

        my ( $requestid, $output, $command );

        ($requestid) = ( $from_njs =~ /.*#TSI_BSSID (.+)\n/ );

        if ( $from_njs =~ /#TSI_FREEZE/ && defined($main::freeze_cmd) ) {
            start_report("Freezing request $requestid");
            $command = "$main::freeze_cmd $requestid\n";
        }
        else {
            start_report("Holding request $requestid");
            $command = "$main::hold_cmd $requestid\n";
        }

        command_report($command);

        $output = `($command) 2>&1`;

        if ( $? != 0 ) {
            failed_report($output);
        }
        else {
            ok_report($output);
        }
    }

}

# Resume a job on the BSS.
#
# arg1: text from NJS with TSI_BSSID <request id>
#
sub resume_job {

    if ($main::site_sez_no_holds) {
        ok_report("No holding of jobs on BSS");
    }
    else {

        my $from_njs = shift;

        my ( $requestid, $output, $command );

        ($requestid) = ( $from_njs =~ /.*#TSI_BSSID (.+)\n/ );

        start_report("Resuming request $requestid");

        $command = "$main::resume_cmd $requestid\n";

        command_report($command);

        $output = `($command) 2>&1`;

        if ( $? != 0 ) {
            failed_report($output);
        }
        else {
            ok_report($output);
        }
    }
}

# Get detailed information about a job on the BSS.
#
# arg1: text from NJS with TSI_BSSID <request id>
# 
# writes result or error output to XNJS
#
sub get_job_details {

    my $from_njs = shift;

    my ( $requestid, $output, $command );

    ($requestid) = ( $from_njs =~ /.*#TSI_BSSID (.+)\n/ );

    start_report("Getting details for job $requestid");

    if( defined $main::details_cmd ) {
        $command = "$main::details_cmd $requestid\n";
        command_report($command);
        $output = `($command) 2>&1`;
	
        # check if command execution was OK
        if ( $? != 0 ) {
            failed_report($output."\n");
	    return;
        }

    }
    else{
	$output="No further information is available.";
    }
    ok_report();
    print main::CMD_SOCK $output."\n";
}

#
#                   Copyright (c) Fujitsu Ltd 2000 - 2004
#
#                Use and distribution is subject a License.
# A copy was supplied with the distribution (see documentation or the jar file).
#
# This product includes software developed by Fujitsu Limited (http://www.fujitsu.com).
