package BecomeUser;

require Exporter;
require UserCache;
@ISA = qw(Exporter);

@EXPORT_OK = qw(initialize become_user restore_id);

use Reporting qw(initial_report failed_report debug_report);

use strict;

# This module changes the process identity

sub initialize() {

    # Do we reset our UID to that of users?

    if ( $> == 0 ) {

        # Do setting if our effective UID is root

        # Remember the incoming effective ids, we restore
        # to these after every action. Both real and effective
        # are set to the original effective so that we look
        # look like a root process even from setuid scripts.

        # CODE < = Real UID, > = Eff UID, ( = Real GID, ) = Eff GID

        $BecomeUser::resting_uid = $>;
        $BecomeUser::resting_gid = $);

        # Test for support for setre*id, if unavailable
        # fall back on set*id

        eval {
            ( $(, $) ) = ( $BecomeUser::resting_gid, -1 );
            ( $<, $> ) = ( $BecomeUser::resting_uid, -1 );
        };

        if ($@) {

            # setre*id does not seem to be available, try set*id
            $( = $BecomeUser::resting_gid;
            $< = $BecomeUser::resting_uid;
            initial_report("Using set*id.");
            $BecomeUser::got_setre = 0;
        }
        else {
            initial_report("Using setre*id.");
            $BecomeUser::got_setre = 1;
        }

        initial_report(
            "Running privileged [$< : $(], will execute commands as the Xlogin"
        );

        $BecomeUser::setting_uids = 1;

        if ($main::enforce_os_gids) {
            initial_report(
"Groups of the user will be limited to those available for the Xlogin in the operating system."
            );
        }
        else {
            initial_report(
"XNJS will be free to assign any groups for the Xlogin regardless of the operating system settings."
            );
        }
    }
    else {
        initial_report(
            "Running unprivileged, will execute all commands as [$< : $(]");
        $BecomeUser::setting_uids = 0;
    }
}

sub check_membership {
    my $group     = shift;
    my $group_gid = shift;
    my $user      = shift;

    # if requested is the primary group or if checking is disabled return OK
    if ( $main::enforce_os_gids && $group_gid != UserCache::get_gid_4user($user) ) {

        # Check that this user is a member of the requested group
        my $mem_list = UserCache::get_members_4group($group);
        if ( !( $mem_list =~ m/$user/ ) ) {
            return 0;
        }
    }
    return 1;
}

# Change the process' identity (real and effective) to a user's (if process was started
# with sufficient privileges to allow this, does nothing otherwise)
#
# arg 1 = Name of the user
# arg 2 = Name of the project (group)
#
# Returns 1 if successful, 0 otherwise
#
# Side effects: modifies the ENV array, setting values for USER, LOGNAME and HOME
#
sub become_user {

    # make sure to prevent running things as root
    if ( !$BecomeUser::setting_uids && $> == 0 ) {
        failed_report(
"Running as root and not setting uids --- this is not allowed. Please check your TSI installation!"
        );
        return 0;
    }

    if ($BecomeUser::setting_uids) {

        my $new_name  = shift;
        my $new_group = shift;

        my $new_uid = UserCache::get_uid_4user($new_name);

        if ( $new_uid == -1 ) {
            failed_report("Attempt to run a task for an unknown user $new_name");
            return 0;
        }

        # Do not allow changes to root
        if ( $new_uid == 0 ) {
            failed_report("Attempt to run a command as root $new_name");
            return 0;
        }

        # Do project (group) mapping, $new_gid stores a new primary gid,
        #  $new_gids stores the $new_gid and all supplementary gids (numbers).
        my $new_gid;
        my $new_gids;

        my @req_groups = split( /:/, $new_group );
        if ( $req_groups[0] eq "NONE" ) {

            # None selected by user, set default from password file
            $new_gid  = UserCache::get_gid_4user($new_name);
            $new_gids = UserCache::get_gids_4user($new_name);
        }
        else {

            # XNJS sent something, check if OK and set

            # Handle the primary group
            if ( $req_groups[0] eq "DEFAULT_GID" ) {
                $new_gid = UserCache::get_gid_4user($new_name);
            }
            else {
                $new_gid = UserCache::get_gid_4group( $req_groups[0] );    # map name to gid
                if ( $new_gid == -1 ) {
                    if ($main::fail_on_invalid_gids) {
                        failed_report(
"Attempt to run a task with an unknown primary group: $req_groups[0]"
                        );
                        return 0;
                    }
                    else {
                        debug_report(
"WARNING: XNJS requested primary group $req_groups[0], but it is not available on the OS. Using default for the user $new_name."
                        );
                        $new_gid = UserCache::get_gid_4user($new_name);
                    }
                }
                if ( !check_membership( $req_groups[0], $new_gid, $new_name ) )
                {
                    if ($main::fail_on_invalid_gids) {
                        failed_report(
"The user $new_name is not a member of the group $req_groups[0]"
                        );
                        return 0;
                    }
                    else {
                        debug_report(
"WARNING: The user $new_name is not a member of the group $req_groups[0], default group will be used."
                        );
                        $new_gid = UserCache::get_gid_4user($new_name);
                    }
                }
            }

            # and supplementary groups
            my %sup_gids;
            my $added_def = 0;
            $sup_gids{$new_gid} = 1;

            for ( my $i = 1 ; $i <= $#req_groups ; $i++ ) {
                if ( ( $req_groups[$i] eq "DEFAULT_GID" ) && !$added_def ) {
                    $added_def = 1;
                    my $def_gids = UserCache::get_gids_4user($new_name);
                    my @def_gids_a = split( / /, $def_gids );
                    foreach my $tmp_gid (@def_gids_a) {
                        $sup_gids{$tmp_gid} = 1;
                    }
                }
                elsif ( ( $req_groups[$i] ne "DEFAULT_GID" ) ) {
                    my $tmp_gid = UserCache::get_gid_4group( $req_groups[$i] );
                    if ( $tmp_gid == -1 ) {
                        if ($main::fail_on_invalid_gids) {
                            failed_report(
"Attempt to run a task with an unknown supplementary group $req_groups[$i]."
                            );
                            return 0;
                        }
                        else {
                            initial_report(
"WARNING: XNJS requested supplementary group $req_groups[$i], but it is not available on the OS. Ignoring."
                            );
                            next;
                        }
                    }
                    if (
                        !check_membership(
                            $req_groups[$i], $tmp_gid, $new_name
                        )
                      )
                    {
                        if ($main::fail_on_invalid_gids) {
                            failed_report(
"The user $new_name is not a member of the group $req_groups[$i]"
                            );
                            return 0;
                        }
                        else {
                            debug_report(
"WARNING: The user $new_name is not a member of the group $req_groups[$i], skipping it."
                            );
                            next;
                        }
                    }
                    $sup_gids{$tmp_gid} = 1;
                }
            }

            foreach my $key ( keys %sup_gids ) {
                $new_gids = $new_gids . $key . " ";
            }
            chop($new_gids);
        }

        # Change identity
        #
        # Do this in a pair to try to force a call to setreuid
        # rather than calls to setruid and seteuid - not supported by AIX
        # The -1s force a single value to be changed, preserving the saved UID

        # CODE < = Real UID, > = Eff UID, ( = Real GID, ) = Eff GID

        # Go carefully, do groups while effective is root, then real
        # and last effective (permission to set uids is based on effective)

# Impl note: yes, the primary gid will appear twice in the list, however
# when there is no supplementary groups and only one gid (the primary gid) was given
# then the function would result in leaving the current process supplementary groups
# (i.e. root's). So don't change it!

        if ($BecomeUser::got_setre) {
            ( $(, $) ) = ( $new_gid, -1 );
            ( $(, $) ) = ( -1, "$new_gid $new_gids" );
            ( $<, $> ) = ( $new_uid, -1 );
            ( $<, $> ) = ( -1, $new_uid );
        }
        else {
            $( = $new_gid;
            $) = "$new_gid $new_gids";
            $< = $new_uid;
            $> = $new_uid;
        }

        if ( $< != $new_uid ) {
            failed_report(
                "Could not set TSI identity (real) to $new_name $new_uid.");
            return 0;
        }

        if ( $> != $new_uid ) {
            failed_report(
                "Could not set TSI identity (effective) to $new_name $new_uid."
            );
            return 0;
        }

        if ( $( != $new_gid ) {
            failed_report(
"Could not set TSI identity (group real) to $new_group $new_gid."
            );
            return 0;
        }

        if ( $) != $new_gid ) {
            failed_report(
"Could not set TSI identity (group effective) to $new_group $new_gid."
            );
            return 0;
        }

        debug_report("Executing command as $new_name [$< : $(] [$>: $)]");

        # Set some sort of user environment
        $ENV{USER}    = $new_name;
        $ENV{LOGNAME} = $new_name;
        $ENV{HOME}    = UserCache::get_home_4user($new_name);

    }

    return 1;

}

# Change the process identity back to the initial (probably root).
#
# No return
#
# Side effects: modifies the ENV array, setting dummy values for USER, LOGNAME and HOME

sub restore_id {

    if ($BecomeUser::setting_uids) {

        # Go carefully back to resting effective first -
        # this goes because of saved? - then real - so that
        # ps gives resting, then groups OK coz effective is root

        if ($BecomeUser::got_setre) {
            ( $<, $> ) = ( -1, $BecomeUser::resting_uid );
            ( $<, $> ) = ( $BecomeUser::resting_uid, -1 );
            ( $(, $) ) = ( -1, $BecomeUser::resting_gid );
            ( $(, $) ) = ( $BecomeUser::resting_gid, -1 );
        }
        else {
            $> = $BecomeUser::resting_uid;
            $< = $BecomeUser::resting_uid;
            $) = $BecomeUser::resting_gid;
            $( = $BecomeUser::resting_gid;
        }

        $ENV{USER}    = "nobody";
        $ENV{LOGNAME} = "nobody";
        $ENV{HOME}    = "/tmp";

        debug_report("Restored IDs to [$< : $(] [$> : $)]");
    }
}

#
#                   Copyright (c) Fujitsu Ltd 2000 - 2004
#
#                Use and distribution is subject a License.
# A copy was supplied with the distribution (see documentation or the jar file).
#
# This product includes software developed by Fujitsu Limited (http://www.fujitsu.com).
