###########################################################################
#
# This module allows for performing file ACL operations: getting
# file ACL, setting them and checking if ACL are supported for the FS.
# The supported ACL implementation is:
#  - POSIX 1003.2c draft 17 (via popular getfacl and setfacl UNIX commands)
#    in the Linux version. Currently few features of the Linux implementation are used:
#    it is assumed that getfacl returns also the default ACL entries, it is assumed that setfacl
#    automatically creates the mandatory default ACL entries when needed, -R option is used
#    to achieve recursive behavior.
#
# process_acl function expects the following argument from NJS:
# #TSI_ACL_OPERATION <CHECK_SUPPORT|GETFACL|SETFACL>
# #TSI_ACL_PATH /some/path
# #TSI_ACL_COMMAND <RM_ALL|MODIFY|RM> [RECURSIVE]
# #TSI_ACL_COMMAND_SPEC [D]<U|G> <subject> <permissions in rwx format>
#
# Operations supported are 'CHECK_SUPPORT', 'GETFACL' and 'SETFACL' for
# ACL support check on the path, ACL get request and ACL set request respectively.
# TSI_ACL_COMMAND is reguired only for SETFACL
# TSI_ACL_COMMAND_SPEC is only required for SETFACL, with command MODIFY or RM.
#
# the COMMAND and COMMAND_SPEC are required only for SETFACL operation.
# In the COMMAND_SPEC D is used to modify/remove default directory ACL. Subject can be empty
# - then the standard group/owner ACL is modified (as with chmod).
#
#
#   The following output is generated for operations:
#
#    - 'CHECK_SUPPORT': either 'true' or 'false'
#    - 'GETFACL': lines in getfacl format, only containing
#      user or group entries, e.g.:
#
#           user::rwx
#           group::rw-
#           user:nobody:r--
#           group:wheel:rwx
#      Note that this form of output may be a result of translation done
#      by TSI; it is not necessarly the exact output of the command run.
#
#    - 'SETFACL': nothing if operation was succesful or error report if not.
#
#    - any other: 'UNSUPPORTED_OPERATION' string is returned.
#
#     Every line is terminated by \n
#
#
###########################################################################

package ACL;

require Exporter;
@ISA = qw(Exporter);

@EXPORT_OK = qw(process_acl);

use Reporting qw(ok_report failed_report debug_report);
use Cwd;
use strict;

#Checks if a directory is on a FS configured with ACL support. Returns:
# NONE, POSIX or NFS
sub check_support {
    my $path       = shift;
    my $abs_path   = Cwd::abs_path($path);
    my $best_match = 0;
    my $best_res;
    my ( $k, $v );
    if ( !defined $abs_path ) {
        return "NONE";    # in fact the path was invalid, can't be resolved
    }

    while ( ( $k, $v ) = each %main::acl ) {
        if ( $abs_path =~ /^$k/ ) {
            if ( $best_match < $+[0] ) {
                $best_match = $+[0];
                $best_res   = $v;
            }
        }
    }
    if ( $best_match > 0 ) {
        return $best_res;
    }
    else {
        return "NONE";
    }
}

sub getfacl_posix {
    my $file = shift;
    delete $ENV{'POSIXLY_CORRECT'};
    Reporting::command_report("$main::getfacl_cmd \"$file\"");
    my $result = `$main::getfacl_cmd "$file" 2>&1`;

    if ( $? != 0 ) {
        Reporting::failed_report("$? $! $result");
    }
    else {
        Reporting::ok_report($result);
        my @result_a = split( /\n/, $result );
        foreach (@result_a) {
            if ( /^user:/ || /^group:/ ) {
                print main::CMD_SOCK "$_\n";
            }
            if ( /^default:user:/ || /^default:group:/ ) {
                print main::CMD_SOCK "$_\n";
            }
        }
        print main::CMD_SOCK "\n";
    }
}

sub prepare_posix_arg {
    my $val_a  = shift;
    my $remove = shift;
    my @oargs  = split( / /, $val_a );

    my $ret = "";
    if ( $oargs[0] =~ /[D]?U/ ) {
        $ret = "user:$oargs[1]:";
    }
    elsif ( $oargs[0] =~ /[D]?G/ ) {
        $ret = "group:$oargs[1]:";
    }

    if ( $remove == 0 ) {
        $ret = $ret . "$oargs[2]";
    }
    return $ret;
}

sub setfacl_posix {
    my $file  = shift;
    my $op_a  = shift;
    my $val_a = shift;

    my $command;
    my $base_arg = "";
    my $remove   = 0;
    delete $ENV{'POSIXLY_CORRECT'};

    my $recursive = "";
    if ( $op_a =~ /RECURSIVE$/ ) {
        $recursive = "-R ";
    }

    if ( $op_a =~ /^RM_ALL/ ) {
        $command = "$main::setfacl_cmd -b $recursive\"$file\" 2>&1";
    }
    else {
        if ( $val_a =~ /^D/ ) {
            $base_arg = "-d ";
        }
        if ( $op_a =~ /^MODIFY/ ) {
            $base_arg = $base_arg . "-m";
        }
        elsif ( $op_a eq "RM" ) {
            $base_arg = $base_arg . "-x";
            $remove   = 1;
        }
        else {
            Reporting::failed_report("WRONG SETFACL SYNTAX\n");
            return;
        }

        my $arg = prepare_posix_arg( $val_a, $remove );

        $command = "$main::setfacl_cmd $recursive$base_arg $arg \"$file\" 2>&1";
    }

    Reporting::command_report($command);
    my $result = `$command`;
    if ( $? != 0 ) {
        Reporting::failed_report("$? $! $result");
    }
    else {
        Reporting::ok_report($result);
    }
}

sub getfacl_nfs {

}

sub setfacl_nfs {

}

sub process_acl {
    my $from_njs = shift;
    $from_njs =~ /.*#TSI_ACL_OPERATION (.+)\n/;
    my $operation = $1;
    $from_njs =~ /.*#TSI_ACL_PATH (.+)\n/;
    my $path = $1;

    if ( $operation eq "CHECK_SUPPORT" ) {
        if ( check_support($path) eq "NONE" ) {
            Reporting::ok_report("false");
            print main::CMD_SOCK "false\n";
        }
        else {
            Reporting::ok_report("true");
            print main::CMD_SOCK "true\n";
        }
    }
    elsif ( $operation eq "GETFACL" ) {
        my $support = check_support($path);
        if ( $support eq "POSIX" ) {
            getfacl_posix($path);
        }
        elsif ( $support eq "NFS" ) {
            getfacl_nfs($path);
        }
        else {
            Reporting::failed_report(
                "ERROR: Getting ACL on this file system is unsupported.");
        }
    }
    elsif ( $operation eq "SETFACL" ) {
        my $support = check_support($path);
        $from_njs =~ /.*#TSI_ACL_COMMAND (.+)\n/;
        my $command = $1;
        $from_njs =~ /.*#TSI_ACL_COMMAND_SPEC (.+)\n/;
        my $command_spec = $1;
        if ( $support eq "POSIX" ) {
            setfacl_posix( $path, $command, $command_spec );
        }
        elsif ( $support eq "NFS" ) {
            setfacl_nfs( $path, $command, $command_spec );
        }
        else {
            Reporting::failed_report(
                "ERROR: Setting ACL on this file system is unsupported.\n");
        }
    }
    else {
        Reporting::failed_report("UNSUPPORTED_OPERATION\n");
    }
}
