package Submit;

require Exporter;
@ISA = qw(Exporter);

@EXPORT_OK = qw(submit);

use File::Path qw(mkpath);

use Reporting
  qw(debug_report failed_report ok_report start_report command_report report_and_die);
use NoBatchOnly qw(add_zombie reap_zombies);
use CommonUtils qw(addperms);

use strict;

# Submit the script to the BSS
#
# arg 1 = The script (this is called because the script contains the string "#TSI_SUBMIT");
#
# Returns void to TSI
#         on success returns the BSS identifier assigned to the job
#         on fail return a message
#
# SPECIALISATION FOR WITHOUT BATCH SUB-SYSTEM

BEGIN {

    # Set a unique prefix for this invocation for all files
    # that it creates (based on process id)
    $Submit::counter = $$ * 100;

    # Always cd to some neutral place when not doing something
    $Submit::neutral_dir = $ENV{PWD};
    if ( $Submit::neutral_dir =~ m/(.*)/s ) {
        $Submit::neutral_dir = $1;
    }

}

sub submit {

   # Reap children to stop zombies (PP p240)
   # Is this really necessary after Perl 5.004?
   #   local $SIG{CHLD} = sub { wait };
   # Reaping zombies by changing the signal handler locally (the handler is set
   # to 'DEFAULT' in Initialisation.pm) does not work. In general, the parent
   # process has already left the scope of the local change before the child
   # process becomes a zombie and sends a signal. However, making 'IGNORE' the
   # default in Initialisation.pm does not work either, because then no child
   # processes can be started at several places in the TSI.
   # Use reap_zombies() to catch all zombie processes instead. If reap_zombies()
   # is not activated here, it has to be active in MainLoop.pm. The difference
   # is that the reap_zombies() here can only reap previously created processes,
   # and not the new one forked below.
   #   reap_zombies();

    command_report("Submitting a script");

    my $script = shift;

    # Correction for $HOME or $USER in root directories SvdB 11 Feb 2002
    # Substitute for these values as the executed script's
    # environment will not (and if it did the values will
    # be wrong). This is safe as thes script seen here is
    # completely generated by the NJS and so we will not be
    # messing with any user created code.

    $script =~ s/\$USER/$ENV{USER}/g;
    $script =~ s/\$HOME/$ENV{HOME}/g;

    my $outcome_dir;
    my $uspace_dir;
    my $stdout      = "stdout";
    my $stderr      = "stderr";
    my $interactive = "";
    my $job         = "";

    $_ = $script;
    while (/#TSI_(\S+) (.*)\n/g) {
        $outcome_dir = $2 if $1 eq "OUTCOME_DIR";
        $uspace_dir  = $2 if $1 eq "USPACE_DIR";
        $stdout      = $2 if $1 eq "STDOUT";
        $stderr      = $2 if $1 eq "STDERR";
        $interactive = $2 if $1 eq "PREFER_INTERACTIVE";
        $job         = $2 if $1 eq "JOBNAME";
    }

    if ( $interactive eq "true" ) {
        start_report("Interactively executing a job");
    }
    else {
        start_report("Submitting a job to the pseudo-BSS");
    }

    # We are going to run this in a separate process
    # so do it as soon as possible (PP p167)

    my $jobid;
    my $pid;

    # add the following amount to the child process' niceness
    my $nice=100;
    
    # on systems that support it, "ionice" can be useful to
    # reduce the I/O load on the TSI frontend
    # see "man ionice"
    my $ionice="ionice -c 3";

    # First make a unique Job Identifier (as we
    # do not know process ID until after the fork and the
    # child needs the identifier
    $jobid = $Submit::counter;
    $Submit::counter++;

    # chdir here so that failures can be reported to NJS
    if ( chdir($uspace_dir) == 0 ) {
        failed_report("Could not cd to Uspace $uspace_dir because $!");
        return 0;
    }

    # make sure that the outcome dir exists
    mkpath $outcome_dir unless -e $outcome_dir;
    addperms( 0700, $outcome_dir );

    my $cmds_file_name;

    if ($interactive) {
        $cmds_file_name = "UNICORE_" . $jobid . "_Commands";
        open( EMSCRIPT, ">$cmds_file_name" );
        print EMSCRIPT $script;
        close(EMSCRIPT);
`chmod u+rwx $cmds_file_name; $ionice nice -n $nice ./$cmds_file_name > $outcome_dir/$stdout 2> $outcome_dir/$stderr`;
        chdir $Submit::neutral_dir;
        ok_report();
        return;
    }

    $cmds_file_name = "UNICORE_" . $jobid . "_Job";

  FORK: {
        if ( $pid = fork ) {

            # remember forked child process which is a potential zombie
            add_zombie($pid);

            if ( chdir($Submit::neutral_dir) == 0 ) {
                debug_report(
"Could not cd to neutral dir $Submit::neutral_dir because $!"
                );
            }

            # Parent
            # Succeeded, return Job Identifier
            debug_report(
                "Job submitted OK. Identifier: $jobid $cmds_file_name");
            print main::CMD_SOCK "$jobid\n";

        }
        elsif ( defined $pid ) {

            # Child

            # Write the commands to a file
            open( EMSCRIPT, ">$cmds_file_name" );
            print EMSCRIPT $script;
            close(EMSCRIPT);

            # Do the work, cleaning up too
            exec "chmod u+rwx $cmds_file_name; $ionice nice -n $nice ./$cmds_file_name > $outcome_dir/$stdout 2> $outcome_dir/$stderr";

        }
        elsif ( $! =~ /No more process/ ) {

            # sensible to try again
            sleep 5;
            redo FORK;
        }
        else {
            report_and_die("Can't fork for a submit: $!");
        }
    }
}

#
#                   Copyright (c) Fujitsu Ltd 2000 - 2004
#
#                Use and distribution is subject a License.
# A copy was supplied with the distribution (see documentation or the jar file).
#
# This product includes software developed by Fujitsu Limited (http://www.fujitsu.com).
