#!/bin/bash

#
# Install script for the TSI (.rpm/.deb version)
#
# Allows the user to select a TSI and copies specific
# files into predefined directory "/usr/share/unicore/tsi_selected"
#
# Two groups of TSIs are known:
#   TSI without batch sub-system: composed from tsi/SHARED and
#                                 specific files from tsi/NOBATCH
#   TSI with batch sub-system: composed from tsi/SHARED and
#                              specific files from tsi/<specific_system>
#
#
# Called without parameters, the user will be asked which TSI to install.
# 
# This script can also be called non-interactively using
#   
#   Install.sh <tsi_choice> <install_dir>
# 
# where 'tsi_choice' is one of the platform specific TSIs and 
# 'install_dir' is the destination directory. 
# 
# Example:
#   Install.sh tsi/tsi_NOBATCH ./tsi_NOBATCH
# will install the nobatch TSI into 'tsi_NOBATCH'
#

nobatch_tsi=tsi/NOBATCH
shared_tsi=tsi/SHARED

# make sure we're in the correct base directory
cd $(dirname $0)

#
# get the list of all TSIs
#
alltsis=`ls -fd tsi/* tsi_contrib/* `

#
# list all TSIs
#
listTSIs() {
  echo ""
  echo "Available TSI implementations are:"
  echo ""
  i=0
  for tsi in $alltsis; do
    if [ $tsi != $shared_tsi ]; then
      i=`expr $i + 1`
      echo "$i: $tsi"
    fi
  done
}

#
# let the user select one TSI for installation
#
choose() {
  echo ""
  echo "The installation will copy all required files into a new directory."
  echo "Files common to all TSI implementations are copied from tsi/bin,"
  echo "and tsi/SHARED."
  echo "The TSI without a batch sub-system requires also files from tsi/NOBATCH."
  echo "Each TSI with a batch sub-system needs the files from tsi/<specific_system>."
  echo ""
  echo "Select a TSI to install (enter number)"
  read input
  i=0
  for tsi in $alltsis; do
    if [ "$tsi" != "$shared_tsi" ]; then
      i=`expr $i + 1`
      if [ $i = $input ]; then
        FOUND="true"
        break
      fi
    fi
  done
  if [ -z "$FOUND" ]
  then
    echo "Invalid selection"
    exit 1
  fi
}

#
# get the installation directory from the user
# as default, $installdir is offered
#
getdir() {
  installdir=/usr/share/unicore/tsi_selected
  echo ""
  echo "Enter installation directory for $tsi (relative or absolute path) [$installdir]"
  read indir
  if [ "$indir" != "" ]; then
    installdir=$indir
  fi
}

#
# let the user confirm the installation
#
confirm_tsi() {
  echo ""
  echo "Please confirm installation of $1 into directory $2 by typing y/n [y]"
  read yesno
  case $yesno in
    y*|Y*) echo "";;
    "") echo "";;
    n*|N*) echo "Installation stopped because of missing confirmation"; exit 1;;
    *) echo "Installation stopped because of doubtful confirmation"; exit 1;;
  esac
}

#
# copy files from directory $1 to $2
# - if VERBOSE is non-zero, actions will be echoed
# - if CHECKEXIST is non-zero, existing files will be backed up
#
copy_dir() {
  for file in $1/*; do
    DEST="$2/`basename $file`"

    [ "$VERBOSE" != "" ] && echo "Copy $file to $DEST"

    if [ "$CHECKEXIST" != "" ]
    then     
      if [ -e "$DEST" ]
      then
         diff -q $file $DEST
         if [ $? -ne 0 ]
         then
           echo " ** $DEST exists, saving as ${DEST}.bak"
           cp $DEST ${DEST}.bak
         fi
      fi 
    fi
    cp $file $2
  done
}

#
# installation

interactive="true"

tsi=$1
if [ "$tsi" = "" ]; then
  listTSIs
  choose
else
  interactive="false"
fi

installdir=$2
if [ "$installdir" = "" ]; then
  getdir
  confirm_tsi $tsi $installdir
fi


#
# test existence of $installdir
#
if [ -d $installdir ]; then
  if [ "$2" == "" ]; then
    echo "Installation directory $installdir already exists"
    confirm_tsi $tsi $installdir
    echo ""
    echo "Files from $installdir will be overwritten"
    echo ""
    chmod -R u+w $installdir
  fi
elif [ -f $installdir ]; then
  echo "Installation directory $installdir identical to existing file."
  echo "Installation stopped because of doubtful confirmation,"
  echo "Remove file $installdir first."
fi

#
# generate installation directory
#
mkdir -p $installdir
mkdir -p $installdir/bin
mkdir -p $installdir/perl

#
# copy common files
#
 
VERBOSE="true"
CHECKEXIST="true"

echo "Copy common files (bin) first:"
copy_dir bin $installdir/bin

tmpdir=/tmp/tsi_install`date +_%H_%M_%S`/perl
mkdir -p $tmpdir

#
# copy shared files first into tmp dir
#

VERBOSE=""
CHECKEXIST=""

echo "Copy shared files (common to all installations)"
copy_dir $shared_tsi $tmpdir

#
# copy specific files into tmp dir
#
echo "Adding batch-system specific files"
copy_dir $tsi $tmpdir
 
VERBOSE="true"
CHECKEXIST="true"

#
# now copy all Perl modules into install dir
#
echo "Copy all Perl files into final install dir"
copy_dir $tmpdir $installdir/perl

# cleanup tmp dir
rm -rf $tmpdir

#
# restrict file permissions to owner
#
chmod 711 $installdir
chmod 600 $installdir/*
chmod 711 $installdir/perl
chmod 644 $installdir/perl/tsi_ls
chmod 644 $installdir/perl/tsi_df

# ensure helper scripts are executable
chmod 750 $installdir/bin 
chmod 750 $installdir/bin/*

if [ "$interactive" = "true" ]; then
  echo ""
  echo "##########################################################"
  if [ "$tsi" = "$nobatch_tsi" ]; then
    echo "NOTE: for NON-LINUX systems, copy the correct variation of"
    echo "  $installdir/perl/tsi.*"
    echo "to $installdir/tsi"
  fi

  echo "Finish installation by editing $installdir/perl/tsi."
  echo "and /etc/unicore/tsi/tsi.properties."
  echo "Finally start the TSI."
  echo "##########################################################"
  echo ""
  echo "NB: When porting to new platform all specific files have to be adapted!"
  echo ""
fi

