# Package of some commons functions used for UNICORE
# monitoring plugins.
#
# Author: Mariusz Strzelecki <szczeles@mat.umk.pl>

package commons;

require Exporter;
@ISA = qw(Exporter);
@EXPORT_OK = qw(%config &exit_plugin &setup_plugin &message &check_conditions &create_temp_file &check_config &run &check_file_existence &is_debug_enabled);

use Getopt::Long qw(:config no_ignore_case);

use Error qw(:try);

sub exit_plugin;
sub quit;

## common variables
%ERRORS=('OK'=>0,'WARNING'=>1,'CRITICAL'=>2,'UNKNOWN'=>3,'DEPENDENT'=>4);
our $verbosity = 1;  # 0 - only one line on exit
                    # 1 - standard (many lines on exit, normal execution)
                    # 2 - verbose (printing messages + output)
                    # 3 - extra verbose - using debug log4j + leaving temp files

## subroutines

sub quit {
  $code = shift;
  if (defined $config{'testmode'})  {
    print "Exit code: $code\n";
    return;
  }
  clear_temp();
  exit $code;
}

sub setup_plugin {
  $config{'readme_location'} = shift;
  $config{'version'} = shift;
  my $requiredArgs = shift;
  $requiredArgs = "LOGS_DIR" if not defined $requiredArgs;
  $config{'timeout'}=300; # 5 minutes for each script is the default
  
  GetOptions ("file=s"     => \$config{'config_location'},
              "verbose|v=i"  => \$verbosity,
              "timeout|t=i"  => \$config{'timeout'},
              "version|V"  => sub { show_version() },
              "help"       => sub { show_help() },
              "warning=i" => \$config{'warning'},
              "critical=i" => \$config{'critical'},
              );

  message("Timeout set to $config{'timeout'} seconds", 2);

  $verbosity = 3 if $verbosity > 3;
  $config{'originalTimeout'} = $config{'timeout'};
  $config{'timeout'} += time - 5; # 5 seconds for the perl instructions

  exit_plugin("UNKNOWN: No config file given! See $0 --help") if ($config{'config_location'} eq '');
  open( CONF, '<', $config{'config_location'} ) or exit_plugin("UNKNOWN: Unable to open file ".$config{'config_location'});

  while ( my $line = <CONF> ) {
    if ( $line =~ m/^\s*([^#\!;]\s*[a-z_0-9]*)\s*[=:]+\s*(.*?)\s*$/i ) {
      my $prop = $1;
      my $val  = ( defined $2 ) ? $2 : "";
      $config{$prop} = eval($val);
      message("Added property from configuration file: $prop = $config{$prop}", 2);
    }
  }
  close(CONF);
  check_config($requiredArgs);
  chop $config{'LOGS_DIR'} if (defined $config{'LOGS_DIR'} and $config{'LOGS_DIR'} =~ m/\/$/); # removes trailaing slash
  setup_environment();
}

# tested
sub setup_environment {
  my $dir = get_wd();
  mkdir($dir) unless -d $dir;
  chdir ($dir);
  return $dir;
}

# tested
sub is_debug_enabled {
  return $verbosity >= 2;
}

sub locate_log4j {
  my $clientname = shift;
  my $filename = "log4j-$clientname".($verbosity>=2 ? "-debug":"").".properties";
  chomp(my $dir1 = `dirname $config{'config_location'}`);
  $dir1 .= "/$filename";
  return $dir1 if (-e $dir1);

  check_config('UCC_CONFIG');
  chomp(my $dir2 = `dirname $config{'UCC_CONFIG'}`);
  $dir2 .= "/$filename";
  return $dir2 if (-e $dir2);

  $dir3 = $config{'LOGS_DIR'}."/$filename";
  return $dir3 if (-e $dir3);

  exit_plugin("UNKNOWN: Unable to find log4j properties file in any place: $dir1, $dir2, $dir3");
}

sub run {
  my $clientname = shift;
  my $arguments = shift;
  my $outputfile = shift;
  my $joinstderr = shift;
  my $log4jconfig;

  $outputfile = '/dev/null' unless (defined $outputfile);

  # Checking for config variables
  if ($clientname eq "ucc") {
    check_config("UCC_PATH, UCC_CONFIG");
  } elsif ($clientname eq "uvosclc") {
    check_config("UVOS_CLIENT_PATH, UVOS_CLIENT_CONFIG");
  } elsif ($clientname eq "java") {
    if (check_config("JAVA_PATH", 1)==1) {
      $config{'JAVA_PATH'} = 'java';
    }
  } elsif ($clientname ne "probe" and $clientname ne "other") {
    exit_plugin("UKNOWN: Unknown UNICORE client to run!");
  }

  # Activating log4j
  if ($clientname ne "java" and $clientname ne "probe" and $clientname ne "other") {
    $log4jconfig = locate_log4j($clientname);
    message("Found log4j file in $log4jconfig", 2);
  }

  my $dFlag = is_debug_enabled();
  $ENV{'LANG'} = 'C';
  # Gathering command to execute
  if ($clientname eq "ucc") {
    $ENV{'UCC_OPTS'} = "-Djline.terminal=jline.UnsupportedTerminal -Duser.language=en -Duser.country=GB -Dlog4j.configuration=file://$log4jconfig -Ducc.preferences=$config{'UCC_CONFIG'}";
    message("Set environment variable UCC_OPTS to $ENV{'UCC_OPTS'}", 2);
    $command = "$config{'UCC_PATH'} $arguments";
    if (defined $config{'REGISTRY_URL'}) {
      $command .= " -r '$config{'REGISTRY_URL'}'";
    }
    $command .= " -v" if ($verbosity >= 2);
    $ENV{'HOME'} = $config{'LOGS_DIR'};
  } elsif ($clientname eq "uvosclc") {
    $dFlag = 1;
    $ENV{'UVOSCLC_OPTS'} = "-Dlog4j.configuration=file://$log4jconfig";
    message("Set environment variable UVOSCLC_OPTS to $ENV{'UVOSCLC_OPTS'}", 2);
    $ENV{'UVOSCLC_CONFIG'} = $config{'UVOS_CLIENT_CONFIG'};
    message("Set environment variable UVOSCLC_CONFIG to $ENV{'UVOSCLC_CONFIG'}", 2);
    $ENV{'HOME'} = $config{'LOGS_DIR'};
    $command ="$config{'UVOS_CLIENT_PATH'} $arguments";
  } elsif ($clientname eq "java") {
    $command ="$config{'JAVA_PATH'} $arguments";
  } else {
    $command = $arguments;
  }

  # Executing 
  $command .= " > $outputfile";
  if ($dFlag or  defined $joinstderr) {
    $command .= " 2>&1";
  } else {
    $command .= " 2>/dev/null";
  }
  open (MYFILE, ">>$clientname.log");
  print MYFILE "\n".(localtime time)." Running $command\n";
  close (MYFILE);
  message("Running $command", 2);
  backtick($command);

  my $exit_code = $? >> 8;
  `sed -i -e\"s/\r\\(\r\\)\\? \\+//g\" $outputfile` if ($outputfile ne '/dev/null');
  
  # TODO: copying output from $outputfile log to log (always)
  # TODO: check for WARN, ERROR, FATAL in log after execution (always)
  
  # print output if verbosity >= 2
  if ($verbosity >= 2) {
    message("Listing output from command:", 2);
    message("----------------------------", 2);
    print `cat $outputfile`;
    message("----------------------------", 2);
  }

  return $exit_code;
}

# tested
sub get_plugin_name {
  my $plugin = $0;
  $plugin =~ s/^(.*\/)?([^\/\.]*).*$/$2/;
  return $plugin;
}

# tested
sub show_version {
  message("UNICORE Monitoring Infrastructure: ".get_plugin_name().", version $config{'version'}", 0);
  quit $ERRORS{'OK'};
}

# tested
sub check_file_existence {
  my $script_location = shift;
  return exit_plugin("UNKNOWN: Missing required file: $script_location!") unless (-e $script_location);
}

# tested
sub show_help {
  open README, $config{'readme_location'} or return exit_plugin "UNKNOWN: Couldn't load readme file $config{'readme_location'}!";
  my $show_file = 0;
  while (<README>) { 
    $_ =~ m#^.Description# and last;
    print $_ if $show_file;
    $_ eq "\n" and $show_file = 1;
  }
  close README;
  quit $ERRORS{"OK"};
}

# tested
sub message {
  chomp(my $message = shift);
  my $level = shift;
  $level = 1 unless defined $level;
  print "$message\n" if ($level <= $verbosity);
}

# tested
sub exit_plugin {
  if ($_[0] =~ /^([A-Z]*):/ and exists($ERRORS{$1})) {
    message("Listing checker output:", 2);
    message("-----------------------", 2);
    message($_[0], 0);
    if ($_[1]) {
      my $debug_info = eval($_[1]);
      $debug_info =~ s/</&lt;/g;
      $debug_info =~ s/>/&gt;/g;
      if ($debug_info) { message($debug_info, 1); } # show debug output
    }
    message("-----------------------", 2);
    quit $ERRORS{$1};
  } else { 
    message("Wrong plugin output line: $_[0]", 0);
    quit $ERRORS{"UNKNOWN"};
  }
}

# tested
sub clear_temp {
  eval($main::CLEANUP) if defined $main::CLEANUP and $verbosity < 3; 
  die("UNKNOWN: Eval cleanup failed: $@") if defined $main::CLEANUP and $@;
  while (scalar(@tempfiles)) {
    $file = pop(@tempfiles);
    if ($verbosity == 3) {
      message("Temporary file NOT DELETED: $file", 3);
    } else {
      `rm -f $file`;
      message("Temporary file deleted: $file", 2);
    }
  }
}

# tested
sub check_conditions {
  my (@conditions) = @{$_[0]};
  foreach (@conditions) {
    my (%c) = %{$_};
    $c{'show_debug'} = 0 unless defined $c{'show_debug'};
    message("Checking $c{'test'}...", 2);
    if (eval($c{'test'})) { 
      exit_plugin($c{'output'}, $c{'show_debug'});
    }
  }
}

# tested
sub get_wd {
  my $dir = defined $config{'LOGS_DIR'} ? $config{'LOGS_DIR'} : "/tmp";
  $dir .= "/".get_plugin_name()."/";
  `mkdir -p $dir`;
  return $dir;
}

# tested 
sub create_temp_file {
  my $name = get_wd().shift;
  `rm -f $name` if (-e $name);
  $output = `touch $name 2>&1`;
  return exit_plugin("UNKNOWN: Unable to create temporary file: $output") if ($output ne '');
  push (@tempfiles, $name);
  message("Created temp file: $name", 2);
  return $name;
}
                                      
# tested
sub check_config {
  @variables = split(/,\s*/, shift);
  my $notForce = shift;
  my $notFound = 0;
  for my $v (@variables)  {
    unless (defined $config{$v}) {
      return exit_plugin("UNKNOWN: Configuration option $v is missing!") if not defined $notForce;
      $notFound += 1;
    }
  }
  return $notFound;
}

# Code of below functions is copied from http://stackoverflow.com/questions/2562931/how-can-i-terminate-a-system-command-with-alarm-in-perl/3763712#3763712 and adapted to actual needs.
$SIG{ALRM} = sub {
  my $sig_name = shift;
  die "Timeout by signal [$sig_name]\n";
};

sub backtick {
  my $command = shift;
  my @output;

  $timeout = $main::config{'timeout'} - time;
  return if not defined $command;

  close(KID);
  defined( my $pid = open( KID, "-|" ) ) or die "Can't fork: $!\n";
  if ($pid) {
    try {
      alarm( $timeout );
      while (<KID>) {
        chomp;
        push @output, $_;
      }
      wait;
      alarm(0);
    } catch Error with {
      kill -15, $pid;
      sleep 2;
      kill -9, $pid;
      alarm(0);
      exit_plugin("CRITICAL: Plugin timed out ($timeout sec) on command: $command");
    }
    finally {};
  }
  else {
    setpgrp( 0, 0 );
    exec $command;
    exit;
  }
  
}

1;
