import org.w3.x2005.x08.addressing.EndpointReferenceType;
import org.chemomentum.common.ws.IWorkflowFactory;
import org.chemomentum.common.impl.workflow.WorkflowSubmissionClient;
import org.chemomentum.common.impl.workflow.WorkflowManagementClient;
import org.chemomentum.common.api.workflow.WorkflowWrapper;
import de.fzj.unicore.ucc.helpers.TargetSystemFactoryLister;
import de.fzj.unicore.ucc.helpers.Location;
import de.fzj.unicore.uas.client.StorageClient;
import de.fzj.unicore.ucc.util.Mode;
import org.chemomentum.workflow.xmlbeans.WorkflowDocument;
import org.apache.xmlbeans.XmlObject;
import org.chemomentum.workflow.xmlbeans.RulesDocument;
import org.chemomentum.workflow.xmlbeans.StatusType;
import de.fzj.unicore.uas.TargetSystemFactory;
import org.chemomentum.common.ws.ILocationManager;
import org.chemomentum.dataManagement.locationManager.QueryLocationRequestDocument;
import org.chemomentum.workflow.xmlbeans.WorkflowResourcePropertiesDocument;
import org.chemomentum.tracer.xmlbeans.FilterEntriesRequestDocument;
import org.chemomentum.common.ws.ITrace;
import org.chemomentum.tracer.xmlbeans.EntryDocument.Entry;
import org.unigrids.services.atomic.types.ProtocolType;
import eu.unicore.security.xfireutil.client.SecuredXFireClientFactory;
import groovy.xml.QName;
import javax.xml.bind.DatatypeConverter;
import java.util.Calendar;
import eu.unicore.security.etd.DelegationRestrictions;
import eu.unicore.security.etd.TrustDelegation;
import eu.unicore.security.UnicoreSecurityFactory;
import de.fzj.unicore.uas.client.TSFClient;

class TestException extends Exception {
  public TestException(String message) {
    super(message);
  }
}

class WorkflowTester {
  // global UCC params
  private registry;
  private securityProperties;
  private messageWriter;

  // sms client and workflow file name are common for most operations
  private smsClient = null;
  private wsClient = null;
  private wmClient = null;
  private lm = null; //locationMapper
  private workflowID = null;
  public static final String textToWrite = "UNICORE Workflow Test\n";

  // output statistics
  private workflowDefaultTT = null;
  private unusedTSFs = null;
  private performanceData = '';
  private workflowTime = 0;
  private checkFileStatus = ''
  private failedSteps = [:];
  private failedActivities = [];
  private executedSteps = [:];
  private tsfsNeeded = null;

  public WorkflowTester(registry, securityProperties, messageWriter, workflowServiceUrl, storageUrl) {
    this.registry = registry;
    this.securityProperties = securityProperties;
    this.messageWriter = messageWriter;

    def smsEpr = EndpointReferenceType.Factory.newInstance();
    smsEpr.addNewAddress().setStringValue(storageUrl);
    this.smsClient = new StorageClient(storageUrl, smsEpr, securityProperties);

    def wsEpr = EndpointReferenceType.Factory.newInstance();
    wsEpr.addNewAddress().setStringValue(workflowServiceUrl);
    this.wsClient = new WorkflowSubmissionClient(workflowServiceUrl, wsEpr, securityProperties);
  }

  /**
   * Reads workflow file and checks if all targed systems are available.
   * Also creates list of target system that are available but are not used by workflow
   */
  public checkTSFsAvailability(tsfsToCheck) {
    this.tsfsNeeded = tsfsToCheck;
    if (!registry.checkConnection())
      throw new TestException('Can\'t create list of target systems.');

    def eprs;
    try {
      eprs = registry.listServices(TargetSystemFactory.TSF_PORT)
    } catch (Exception e) {
      throw new TestException('Unable to list Target Systems in Registry: ' + e.getMessage());
    }
    unusedTSFs = []

    eprs.each {
      def url = it.address.stringValue;
      if (tsfsNeeded.contains(url)) {
        def tsf = new TSFClient(it.address.stringValue, it, securityProperties);
        def tsss = tsf.accessibleTargetSystems;
        if (tsss == null) {
          messageWriter.verbose 'Unable to list target systems from: '+url;
          return;
        }
        def num = tsss.size();
        if (num >= 1) 
          messageWriter.verbose "Found "+num+" TSS at "+url;
        else {
          messageWriter.verbose "None TSS found, creating one at "+url;
          try {
            tsf.createTSS();
          } catch (Exception e) {
            messageWriter.verbose "Unable to create TSS in "+url+", reason: "+e.message;
            return;
          }
        }
        tsfsNeeded -= url;
      } else {
        messageWriter.verbose "Found unused TSF: "+url;
        unusedTSFs.push(url);
      }
    }
    return tsfsNeeded;
  }

  public attachTrustDelegation() {
    def subject = wsClient.getResourcePropertiesDocument().getWorkflowFactoryProperties().getAccessibleWorkflowEnumeration().getMetadata().getStringValue().replace('x:Enumeration', '');
    messageWriter.verbose 'TD subject: ' + subject;
    DelegationRestrictions restr=new DelegationRestrictions(new Date(),1,10);

   TrustDelegation td=UnicoreSecurityFactory.getETDEngine().generateTD(
        securityProperties.getCredential().getSubjectName(),
        securityProperties.getCredential().getCertificateChain(),
        securityProperties.getCredential().getKey(),
        subject,
        restr);

    List<TrustDelegation>res=new ArrayList<TrustDelegation>();
    res.add(td);
    securityProperties.getETDSettings().setTrustDelegationTokens(res);

    wsClient = new WorkflowSubmissionClient(wsClient.epr.address.stringValue, wsClient.epr, securityProperties);
    messageWriter.verbose "Trust delegation attached"
  }

  public prepareWorkflow(content) {
    // preparing to workflow submission
    def String workflowName = "UMI 2.0 check_workflow test, submitted at "+(new Date().format('yyyy-MM-dd HH:mm:ss'));
    if (wsClient == null || wsClient.checkConnection() != true) 
      throw new TestException("WorkflowFactory service exists in Registry but doesn't respond");

    try {
      wmClient = wsClient.createWorkflow(workflowName, null, smsClient.getEPR());
    } catch (Exception e) {
      throw new TestException("Unable to create workflow: " + e.getMessage());
    }
    this.workflowID = wmClient.getWorkflowID();

    // checking default TT
    def myTT = wmClient.getTerminationTime();
    myTT.add(Calendar.MINUTE, 10); // sumbission overhead
    workflowDefaultTT = ((myTT.getTimeInMillis() - Calendar.getInstance().getTimeInMillis())/86400000).intValue();

    def terminationTime=Calendar.getInstance();
    terminationTime.add(Calendar.HOUR, 3);
    wmClient.setTerminationTime(terminationTime);

    // generating workflow document
    def wd = WorkflowDocument.Factory.newInstance();
    def x = XmlObject.Factory.parse(content.toString().replace('u6://SHARE', 'BFT:'+smsClient.getUrl()+'#').replace('${WORKFLOW_ID}', wmClient.getWorkflowID()).replace('INPUT', getInputFileName()));
    wd.addNewWorkflow().set(x);
    wd.getWorkflow().setDialect(x.newDomNode().getFirstChild().getNamespaceURI());
    def rd = RulesDocument.Factory.newInstance();
    rd.addNewRules();
    return new WorkflowWrapper(wd.getWorkflow(), rd.getRules());
  }

  private getInputFileName() {
    return 'input_for_workflow_' + this.workflowID + '.txt';
  }

  public uploadTestFile(workflowID) {
    def randFileName = this.getInputFileName();
    try {
      def fis = new ByteArrayInputStream(textToWrite.getBytes());
      def ftc = smsClient.getImport(randFileName, [ProtocolType.BFT,ProtocolType.RBYTEIO] as ProtocolType.Enum[]);
      ftc.writeAllData(fis);
      ftc.destroy();
      fis.close();
    } catch (Exception e) {
      throw new TestException("Unable to upload file to " + smsClient.getUrl()+ ": "+e.message);
    }
    messageWriter.verbose "Upload succeded";
    return randFileName;
  }
  
  public submitWorkflow(workflowWrapper) {
    wmClient.submitWorkflow(workflowWrapper);
    messageWriter.verbose "Workflow submitted with id "+wmClient.getWorkflowID() + " ("+ wmClient.epr.address.stringValue+")";
  }

  public waitToFinish(timeoutAt) {
    def finalStatus = null;
    try {
      while (wmClient.status == StatusType.RUNNING) {
        sleep 5000;
        def currentTime = (long)System.currentTimeMillis()/1000;
        if (timeoutAt < currentTime) {
          messageWriter.verbose "Workflow timed out!";
          try {
            wmClient.abort();
          } catch (Exception e) {
          }
          return;
        }
        messageWriter.verbose "Worfklow in RUNNING state, timeout in "+(timeoutAt - currentTime)+" seconds";
      }
      finalStatus = wmClient.status;
    } catch (Exception e) {
      throw new TestException("Workflow Service failed during workflow execution");
    }

    if (finalStatus != StatusType.SUCCESSFUL) {
      throw new TestException("Workflow execution failed (status: " + wmClient.status.toString() + ")");
    }
    messageWriter.verbose "Workflow succeded"
  }


  public checkOutput(numberOfResuls) {
    def available=registry.listServices(ILocationManager.PORT);
    if (available.size() == 0) {
      throw new TestException("Location mapping service is not available!");
    }
    this.lm = new SecuredXFireClientFactory(securityProperties).createPlainWSProxy(ILocationManager.class, available.get(0).getAddress().getStringValue());
    for (i in 1..numberOfResuls) {
      checkFile('result_' + i + '.txt');
    }
  }


  public checkFile(fileName) {
    def req=QueryLocationRequestDocument.Factory.newInstance();
    req.addNewQueryLocationRequest().addNewInput().setLogicalName('c9m:'+workflowID+'/'+fileName);
    def res=lm.queryLocation(req);

    if (res.getQueryLocationResponse().getFaultsOccurred()){
      checkFileStatus += "Could not resolve location, result " + res + "\n";
      return;
    }

    def String[] location=res.getQueryLocationResponse().getResultArray(0).getPhysicalLocationArray()[0].split('#');
    messageWriter.verbose 'Location mapped as '+location[1];

    def ftc = null
    try {
      ftc = smsClient.getExport(location[1], [ProtocolType.BFT,ProtocolType.RBYTEIO] as ProtocolType.Enum[]);
    } catch (Exception e) {
      checkFileStatus += "Unable to download file from " + location[1] + ": " + e.getMessage() + "\n";
      return;
    }
    def fos = new ByteArrayOutputStream();
    ftc.readAllData(fos);
    messageWriter.verbose "Download succeded"

    def result  = fos.toString();
    if (!result.contains(textToWrite)) {
      checkFileStatus += 'Unexpected output: sent "' + textToWrite + '" but received "' + result +"\"\n";
    }
    messageWriter.verbose "File content matches!"
  }

 public trace() {
    if (wmClient == null) {
      return;
    }

    messageWriter.verbose "Using tracer at " + wmClient.tracerAddress.address.stringValue;

    def ferd = FilterEntriesRequestDocument.Factory.newInstance();
    def fer = ferd.addNewFilterEntriesRequest();
    fer.setField(ITrace.MESSAGE_ID);
    fer.setPattern("%"+workflowID+"%");
    def start = null, end = null;
    def realTime = 0.0;
    def starts = [:];
    def results = [:];
    try {
      def tracer=new SecuredXFireClientFactory(securityProperties).createPlainWSProxy(ITrace.class, wmClient.tracerAddress.address.stringValue);
      Entry[] entries=tracer.filterEntries(ferd).getFilterEntriesResponse().getTrace().getEntryArray();
      def lastSite = "";
      def startTime = null;
      entries.each {
        if (it.getOperation().equals("Submit") || it.getOperation().equals("finishedJob")) {
          def jobId = it.getMessageId() =~ /^[^\/]+\/(cat_[^\/]+)/
          if (it.getOperation().equals("Submit")) {
            starts.put(jobId[0][1], it.getTimeStamp());
          } else {
            results.put(jobId[0][1], ((it.getTimeStamp().getTimeInMillis() - starts.get(jobId[0][1]).getTimeInMillis()) / 1000));
          }
        }

        if (it.getOperation().equalsIgnoreCase("submitWorkflow"))
          start = it.getTimeStamp();
        else
          end = it.getTimeStamp();
      }
    } catch (Exception e) {
      throw new TestException("Tracer service is not responding: "+e.getMessage());
    }

    if (end == null || start == null)
      throw new TestException("Tracer service is not responding: unable to get start or finish date of workflow");

    def report = [:]
    for (s in starts) {
      def m = s.key =~ /^cat_([A-Z\-_0-9]+):/;
      def site = m[0][1];

      if (!results.containsKey(s.key)) {
        failedActivities.add(s.key);
        if (failedSteps.containsKey(site)) 
          failedSteps.put(site, failedSteps.get(site) + 1)
        else
          failedSteps.put(site, 1)
      } else {
        def t = report.containsKey(site) ? report.get(site) : 0;
        t += results[s.key] / 2;
        report.put(site, t);
      }
      def ex = executedSteps.containsKey(site) ? executedSteps.get(site) : 0;
      ex += 1;
      executedSteps.put(site, ex);
    }
    
    def extraData = "";
    for (r in report) {
      extraData += " " + r.key + "=" + r.value + "s";
    }

    this.workflowTime = (end.getTimeInMillis() - start.getTimeInMillis()) / 1000;
    this.performanceData = " | WorkflowExecutionTime="+workflowTime+"s"+extraData;
  }

  public finalizeTest(allowedToFail) {
    // wrong TSFs list
    if (tsfsNeeded.size() > allowedToFail) {
      messageWriter.message "Workflow was unable to be executed on too many sites: " + tsfsNeeded;
    } else if (tsfsNeeded.size() > 0) {
      messageWriter.message "Workflow was unable to be executed on some sites: " + tsfsNeeded;
    }

    // failed activities
    if (failedSteps.size() > allowedToFail) {
      messageWriter.message "Workflow failed: jobs failed on too many sites ("+ failedSteps.size() + ")";
    } else if (failedSteps.size() > 0) {
      messageWriter.message "Workflow failed on " + failedSteps.size() + " site" + (failedSteps.size() > 1 ? "s" : "")+ ": " + failedSteps.keySet()+  performanceData;
    } else if (!checkFileStatus.isEmpty()) {
      messageWriter.message "All jobs succeded, but some output files are missing";
    } else {
      messageWriter.message("Wokflow succeeded! Time elapsed: " + workflowTime + " sec." + performanceData);
    }
    messageWriter.message("Default Termination Time: "+workflowDefaultTT+" day"+(workflowDefaultTT > 1 ? "s" : "")+".");
    for (fs in failedSteps) {
      def ex = executedSteps[fs.key].value;
      messageWriter.message "Jobs at " + fs.key + " failed in " + fs.value + "/" + ex +" run" + (ex > 1 ? "s" : "");
    }
    for (st in executedSteps)  {
      if (st.value != 2) 
        messageWriter.message "Only one job was started on " + st.key;
    }
    if (failedActivities.size() > 0)
      messageWriter.message "Failed activities: " + failedActivities;
    if (!checkFileStatus.isEmpty()) 
      messageWriter.message(checkFileStatus);

  }

  public clearData() {
    if (wmClient != null && wmClient.checkConnection() == true) {
      try {
        smsClient.delete(getInputFileName());
      } catch (Exception e) { }

      try {
        smsClient.delete(workflowID);
      } catch (Exception e) { }

      messageWriter.verbose("Files deleted");

      wmClient.destroy();
      messageWriter.verbose("Workflow destroyed");
    }
  }
}


class WorkflowDocumentGenerator {
  def xml;

  private getTSFName(url) {
    return url.find("https?://[^:]+:[0-9]+/([^/]+)/") { it[1] };
  }

  private addSite(url, inputFiles, outputFile, number) {
    def name = getTSFName(url);
    this.xml += 
  "  <s:Activity Id=\"cat_" + name + ":" + number + "\" Name=\"JSDL\">\n" +
  "    <s:Option name=\"MAX_RESUBMITS\">1</s:Option>\n" + 
  "    <s:Option name=\"IGNORE_FAILURE\">true</s:Option>\n" + 
  "    <s:JSDL>\n" +
  "      <jsdl:JobDescription>\n" +
  "        <jsdl:Application>\n" +
  "           <jsdl1:POSIXApplication>\n" +
  "             <jsdl1:Environment name=\"UC_IGNORE_FAILURE\">true</jsdl1:Environment>\n" +
  "             <jsdl1:Executable>cat</jsdl1:Executable>\n" +
  "             <jsdl1:Argument>*.input</jsdl1:Argument>\n" +
  "          </jsdl1:POSIXApplication>\n" +
  "        </jsdl:Application>\n" +
  "        <jsdl:Resources>\n" +
  "          <jsdl:CandidateHosts>\n" +
  "            <jsdl:HostName>"+url+"</jsdl:HostName>\n" +
  "          </jsdl:CandidateHosts>\n" +
  "          <jsdl:IndividualPhysicalMemory>\n" +
  "            <jsdl:Exact>52428800.0</jsdl:Exact>\n" +
  "          </jsdl:IndividualPhysicalMemory>\n" +
  "        </jsdl:Resources>\n" +
  "        <jsdl:DataStaging>\n" +
  "          <jsdl:FileName>stdout</jsdl:FileName>\n" +
  "          <jsdl:CreationFlag>overwrite</jsdl:CreationFlag>\n" +
  "          <jsdl:Target>\n" +
  "            <jsdl:URI>"+outputFile+"</jsdl:URI>\n" +
  "          </jsdl:Target>\n" +
  "        </jsdl:DataStaging>\n";
 
    inputFiles.each {
      def m = it =~ "([^/]+)\$"
      this.xml +=
  "        <jsdl:DataStaging>\n" +
  "          <jsdl:FileName>" + m[0][1] + ".input</jsdl:FileName>\n" +
  "          <jsdl:CreationFlag>overwrite</jsdl:CreationFlag>\n" +
  "          <jsdl:Source>\n" +
  "            <jsdl:URI>"+it+"</jsdl:URI>\n" +
  "          </jsdl:Source>\n" +
  "          <jsdl2:IgnoreFailure>true</jsdl2:IgnoreFailure>\n" + 
  "        </jsdl:DataStaging>\n";
    }
    this.xml +=
  "      </jsdl:JobDescription>\n" +
  "    </s:JSDL>\n" +
  "  </s:Activity>\n";

  }
  
  private addTransition(from, to) {
    from = getTSFName(from);
    to = getTSFName(to);
    this.xml += "<s:Transition Id=\"transition_" + from + "_" + to + "\" From=\"cat_" + from + ":1\" To=\"cat_" + to +":2\"/>\n";
  }

  protected theSame(a, b) {
    for (i in  0..(a.size()-1)) {
      if (a[i].equals(b[i])) {
        return true;
      }
    }
    return false;
  }

  public generate(TSFs) {
    this.xml = "<s:Workflow xmlns:s=\"http://www.chemomentum.org/workflow/simple\"\n" +
  "      xmlns:jsdl=\"http://schemas.ggf.org/jsdl/2005/11/jsdl\"\n" +
  "      xmlns:jsdl1=\"http://schemas.ggf.org/jsdl/2005/11/jsdl-posix\"\n" + 
  "      xmlns:jsdl2=\"http://www.unicore.eu/unicore/jsdl-extensions\">\n" +
  "  <s:Documentation>\n" +
  "    <s:Comment>\n" +
  "      UNICORE Workflow test\n" +
  "    </s:Comment>\n" +
  "  </s:Documentation>\n";
    def secondStep = TSFs.clone();
    if (TSFs.size() > 1) 
      while (theSame(TSFs, secondStep)) {
        Collections.shuffle(secondStep);
      }
    for (i in 0..(TSFs.size()-1)) {
      def output = "c9m:\${WORKFLOW_ID}/output_from_" + getTSFName(TSFs[i]);
      addSite(TSFs[i], ["u6://SHARE/INPUT"], output, 1);
      addSite(secondStep[i], [output], "c9m:\${WORKFLOW_ID}/result_" + (i+1) + ".txt", 2);
      addTransition(TSFs[i], secondStep[i]);
    }

    // initial file
    this.xml += 
  "  <s:FileSet recurse=\"false\">\n" + 
  "    <s:Base>u6://SHARE/</s:Base>\n" + 
  "    <s:Include>INPUT</s:Include>\n" + 
  "  </s:FileSet>\n" + 
  "</s:Workflow>";
    return xml;
  }
}


if (commandLine.args.size() < 4) // minimum one TSF
  throw new TestException("Usage: run-groovy -f test_workflow.groovy WORKFLOW_SERVICE_NAME STORAGE_URL AVAILABLE_TIME <TSF1> <TSF2> ... <TSFN>");

def workflowServiceName = commandLine.args[1];
def smsUrl = commandLine.args[2];
def timeoutAt = (long)System.currentTimeMillis()/1000 + new Long(commandLine.args[3]);
def allowedToFailPercent = 50;
def TSFs = [];
for (i=4; i<commandLine.args.size(); i++) {
  TSFs.push(commandLine.args[i]);
  messageWriter.verbose("Added TSF address from command line: " + commandLine.args[i]);
}
def tester = null;
def allowedToFail = (int)(TSFs.size() * allowedToFailPercent / 100);
messageWriter.verbose "Allowed to fail: " + allowedToFail + " sites";

try {
  tester = new WorkflowTester(registry, securityProperties, messageWriter, workflowServiceName, smsUrl);
  def excludeList = tester.checkTSFsAvailability(TSFs);
  tester.attachTrustDelegation();

  excludeList.each {
    messageWriter.verbose 'Removing ' + it + ' from TSFs list (connection problems)';
    TSFs -= it;
  }

  workflowDescription = new WorkflowDocumentGenerator().generate(TSFs);
  def wrapper = tester.prepareWorkflow(workflowDescription);
  tester.uploadTestFile();
  tester.submitWorkflow(wrapper);
  tester.waitToFinish(timeoutAt);
  tester.checkOutput(TSFs.size());
} catch (TestException e) {
  messageWriter.message e.message;
} catch (Exception e1) {
  messageWriter.message e1.message;
  e1.printStackTrace(System.out);
} finally {
  if (tester != null) {
    try {
      tester.trace();
      tester.finalizeTest(allowedToFail);
      tester.clearData();
    } catch (TestException e) {
      messageWriter.message e.message;
    } catch (Exception e1) {
      messageWriter.message e1.message;
      e1.printStackTrace(System.out);
    }
  }
}
