#!/usr/bin/perl
# Author: Mariusz Strzelecki <szczeles@mat.umk.pl>
# Don't use Nagios embedded perl
# nagios: -epn

use strict;
use warnings;
use LWP::Simple;
use XML::RSS;
use Date::Parse;
use POSIX qw/strftime/;
use Sort::Versions;
our $plugin_version = '2.0';

# Let $Bin store absolute path of script directory
use FindBin '$Bin';

# Look for commons.pm in following directories
use lib "$Bin";
use lib "$Bin/..";

# Load common functions
use commons qw(%config &exit_plugin &setup_plugin &message &check_conditions &create_temp_file &check_config &run &check_file_existence);

# Load configuration and check for commandline arguments
setup_plugin("$Bin/check_freespace.README", $plugin_version);
my $script_location = "$Bin/versions.groovy";
check_file_existence($script_location);

# Execute something
# ###  uvos-clc -c /tmp/unicore-test/logs/check_uvos/uvosclc_config5058 -b getServerInfo
our $TMPFILE = create_temp_file("check_freespace$$");
if (!run("ucc", "run-groovy -f $script_location", $TMPFILE)) {

  my $url = "http://www.unicore.eu/_script/unicore_releases.rss";
  my $data = get $url;
  my $rss = XML::RSS->new;

  exit_plugin("UNKNOWN: Unable to get retrieve UNICORE releases rss feed") if not defined $data;

  $rss->parse($data);
  my @entries = @{$rss->{'items'}};

  my $coreVersion;
  my $coreInfo;

  my $workflowVersion;
  my $workflowInfo;

  foreach my $entry (@entries) {
    my $i = '(released '.strftime('%Y-%m-%d', localtime(str2time($entry->{'pubDate'}))).')';
    if (not defined $coreVersion and $entry->{'title'} =~ m|Unicore6CoreServers ([^ ]*)|) {
      $coreVersion = $1;
      $coreInfo = $i;
    }
    if (not defined $workflowVersion and $entry->{'title'} =~ m|Unicore6WorkflowSystem ([^ ]*)|) {
      $workflowVersion = $1;
      $workflowInfo = $i;
    }
  }

  my $servicesOK = 0;
  my $servicesWARN = 0;
  my $servicesUNKNOWN = 0;
  my $additionalInfo = '';

  # Check for discovered services
  open (FILE, $TMPFILE);
  while (<FILE>) {
    chomp($_);
    my @info = split("\t", $_);
    my $n;
    my $i;
    if ($info[0] eq 'W') {
      $info[2] =~ s|^1|6|;
      $i = $workflowInfo;
      $n = $workflowVersion;
    } elsif ($info[0] eq 'U') {
      $info[2] =~ s|^1|6|;
      $i = $coreInfo;
      $n = $coreVersion;
    } else {
      exit_plugin("UNKNOWN: Wrong line: $_");
    }

    my $c = ($info[2] eq $n) ? 0 : versioncmp($info[2], $n.'-x');
    if ($info[2] eq '0') {
      $servicesUNKNOWN++;
      $additionalInfo.="Service $info[1] does not provide version information!\n";
    } elsif ($c == 0) {
      $servicesOK++;
    } elsif ($c == 1) {
      $servicesUNKNOWN++;
      $additionalInfo.="Service $info[1] is too new ($info[2] is not released yet, last release: $n)!\n";
    } else {
      $servicesWARN++;
      $additionalInfo.="Service $info[1] is outdated, installed version: $info[2], last version: $n $i.\n";
    }

  }
  close (FILE);

  my $status = (($servicesWARN == 0) and ($servicesUNKNOWN == 0)) ? "OK" : "WARNING";
  exit_plugin("$status: Up-to-date: $servicesOK, outdated: $servicesWARN, unknown: $servicesUNKNOWN", "'$additionalInfo'");

} else {
  my $output = `head -1 $TMPFILE`;
  if ($output =~ /^Unable to/) {
    exit_plugin("UNKNOWN: $output", '`tail -n +2 $main::TMPFILE`');
  }
  exit_plugin("UNKNOWN: Unknown error", '`cat $main::TMPFILE`');
}
