#!/usr/bin/perl
# Author: Mariusz Strzelecki <szczeles@mat.umk.pl>
# Don't use Nagios embedded perl
# nagios: -epn

use strict;
use Date::Parse;
our $plugin_version = '2.0';

# Let $Bin store absolute path of script directory
use FindBin '$Bin';

# Look for commons.pm in following directories
use lib "$Bin";
use lib "$Bin/..";

# Load common functions
use commons qw(%config &exit_plugin &setup_plugin &message &check_conditions &create_temp_file &check_config &run &check_file_existence);

# Load configuration and check for commandline arguments
our $work_dir = setup_plugin("$Bin/check_application.README", $plugin_version);
check_config('SITE_NAME');

our $extraInfo = '';

sub wrong_params_number {
	exit_plugin("UNKNOWN: Wrong parameters number for ".(caller(1))[3]."() function!");
}

sub file_exists {
	if (!-e $_[0]) {
		exit_plugin("CRITICAL: File $_[0] doesn't exist!");
	}
}

sub equals {
	if ($main::test_syntax) {
		wrong_params_number() unless @_ == 2;
		if (!(substr($_[0], 0, 1) eq "#")) { file_exists($_[0]); }
		if (!(substr($_[1], 0, 1) eq "#")) { file_exists($_[1]); }
		return 1;
	}
	my $status = `cmp $_[0] $_[1] 2>&1`;
  
	our $globalInfo .= "Files $_[0] and $_[1] are different!\n" unless $? == 0; 
	return $? == 0;
}

sub valid_pdf {
	if ($main::test_syntax) {
		wrong_params_number() unless @_ == 1;
		if (!(substr($_[0], 0, 1) eq "#")) { file_exists($_[0]); }
		return 1;
	}
	`pdfinfo $_[0] 2>&1`;
	our $globalInfo .= "$_[0] is not valid PDF file\n" unless $? == 0; 
	return $? == 0;
}

sub not_fully_equals {
	wrong_params_number() if $main::test_syntax and @_ != 3;
  my $file1 = shift;
  my $file2 = shift;
  my $allowed_different_lines = shift;
	if ($main::test_syntax) {
		if (!(substr($file1, 0, 1) eq "#")) { file_exists($file1); }
		if (!(substr($file2, 0, 1) eq "#")) { file_exists($file2); }
		return 1;
	}
  chomp(my $file1Length = `wc -l $file1 | cut -d' ' -f1`);
  chomp(my $file2Length = `wc -l $file2 | cut -d' ' -f1`);
  our $globalInfo .= "Files has different number of lines ($file1: $file1Length, $file2: $file2Length)!\n" and return 0 if (abs($file1Length - $file2Length) > $allowed_different_lines);
	chomp(my $status = `diff --suppress-common-lines --speed-large-files -y $file1 $file2 | wc -l 2>&1`);
	$globalInfo .= "Files $file1 and $file2 are different!\n" and return 0 if $status > $allowed_different_lines;
  return 1;
}

sub grep_file {
	wrong_params_number() if $main::test_syntax and @_ != 2;
  my $text = shift;
  my $file = shift;
  if ($main::test_syntax) {
    file_exists($file) if not substr($file, 0, 1) eq "#";
    return 1;
  }  

  $text =~ s/\//\/\//g;
  `grep -q '$text' $file`;
  my $status = $? == 0 ? 1 : 0;
  our $globalInfo .= "Grep for $text on $file failed." if $status eq 0;
  return $status;
}

sub contains {
  wrong_params_number() if $main::test_syntax and @_ != 2;
  my $fragment = shift;
  my $file = shift;
  if ($main::test_syntax) {
    file_exists($fragment);
    return 1;
  }  

  `diff $file $fragment -y --suppress-common-lines | grep -q -v '<\$'`;
  my $status = $?;
  our $globalInfo .= "File $file does not contain file $fragment\n" if (!$status);
  return $status;
}

sub number_of_lines_between {
  wrong_params_number() if $main::test_syntax and @_ != 3;
  my $file = shift;
  my $minLines = shift;
  my $maxLines = shift;
  if ($main::test_syntax) {

    $minLines =~ m#^[0-9]+$# or exit_plugin("UNKNOWN: Min lines value is not a number");
    $maxLines =~ m#^[0-9]+$# or exit_plugin("UNKNOWN: Max lines value is not a number");
    return 1;
  }

  chomp(my $nol = `wc -l $file`);
  if ($nol < $minLines) {
    our $globalInfo .= "File $file contains less than $minLines lines: $nol\n";
    return 0;
  } elsif ($nol > $maxLines) {
    our $globalInfo .= "File $file contains more than $maxLines lines: $nol\n";
    return 0;
  }
  return 1;
}

sub not_empty {
	if ($main::test_syntax) {
		wrong_params_number() unless @_ == 1;
		if (!(substr($_[0], 0, 1) eq "#")) { file_exists($_[0]); }
		return 1;
	}
	our $globalInfo .= "$_[0] is empty file\n" if (-z $_[0]); 
	return (!-z $_[0]);
}

sub exit_code {
  my $exitCode = shift;
  if ($main::test_syntax) {
    wrong_params_number() if (@_ != 0); # already shifted
    $exitCode =~ m#^[0-9]+$# or exit_plugin("UNKNOWN: Exit code is not a number");
    return 1;
  }
  my $returnLine = `grep 'SUCCESSFUL exit code: [0-9]\\+\$' $main::TMPFILE`; 
  $returnLine =~ m#: ([0-9]+)#;
  our $globalInfo .= "Exit code does not match $exitCode\n" and return 0 if ($1 ne $exitCode);
  return 1;
}

# Check program agruments
if ($#ARGV != 1) {
  exit_plugin("UNKNOWN: See README file for plugin syntax");
}
my $job_file = $ARGV[0];
my $condition = $ARGV[1];

# check if job file exists
check_file_existence($job_file);

# check if condition syntax is OK
$condition =~ s/,([^ ])/\",\"$1/g;
$condition =~ s/, /\",\"/g;
$condition =~ s/\(([^\)\(]*)\)/\(\"$1\"\)/g;
$condition =~ s/grep\(/grep_file(/g;
$condition =~ s/^"+//;
$condition =~ s/"+$//;
# enable syntax testing
our $test_syntax = 1;
eval $condition;
exit_plugin("UNKNOWN: Wrong condition syntax! See README file.", "print $@") if $@;
# disable syntax testing now
$test_syntax = 0;

# connect
run("ucc", "connect", '/dev/null');

#submit a job
our $TMPFILE = create_temp_file("check_application$$");
run("ucc", "run -v -s $config{'SITE_NAME'} $job_file", $TMPFILE, 1);

if (`grep -c 'SUCCESSFUL exit code: ' $TMPFILE` == 0) {
  our @conditions = (
    {
    'test'       => '`grep -c "Application could not be mapped to an executable." $main::TMPFILE` != 0',
    'output'     => "CRITICAL: Application is not present in IDB",
    'show_debug' => 0
    },
    {
    'test'       => '`grep -c "No matching target system available" $main::TMPFILE` != 0',
    'output'     => "CRITICAL: TSS is not available!",
    'show_debug' => '`echo "See ucc log for details:"; cat $main::TMPFILE`'
    },
    {
    'test'       => '`grep -ic \'Result: Failed\' $main::TMPFILE` != 0',
    'output'     => "CRITICAL: Job failed",
    'show_debug' => '`echo "See ucc log for details:"; cat $main::TMPFILE;`'
    },
    {
    'test'       => '`grep -c "Error waiting for job to finish" $main::TMPFILE` != 0',
    'output'     => "CRITICAL: Error waiting for job to finish!",
    'show_debug' => '`echo "See ucc log for details:"; cat $main::TMPFILE`'
    },
    {
    'test'       => '`grep -c \'Path <.*> not found on storage.\' $main::TMPFILE` != 0',
    'output'     => "CRITICAL: Some output files are missing!",
    'show_debug' => '`echo "See ucc log for details:"; cat $main::TMPFILE;`'
    },
    {
    'test'       => '1', # last check - uknown error
    'output'     => "CRITICAL: Cannot submit job!",
    'show_debug' => '`echo "See ucc log for details:"; cat $main::TMPFILE;`'
    }
    );
  check_conditions(\@main::conditions);
}

# new cleanup function
chomp(our $jms = `grep 'job url' $main::TMPFILE`);
exit_plugin("UNKNOWN: Unable to match job id from output") unless ($jms =~ /=([0-9a-f\-]{36})$/);
our $job_id = $1;
$work_dir .= "/".$job_id;
$condition =~ s/#/$main::work_dir\//g; 
our $CLEANUP = 'run("ucc", "wsrf destroy $main::jms"); `rm -r $main::work_dir`';

# Test condition
my $ok = eval $condition;
if (!$ok) {
  our $globalInfo;
  $CLEANUP = 'run("ucc", "wsrf destroy $main::jms");';
	exit_plugin("WARNING: Unable to pass application condition string.", "'$condition\n$globalInfo\nCheck files in directory $main::work_dir\n'");
}
# Get information and exit
my $props = `grep Total: $work_dir/$job_id.properties`;
$props =~ /Total: ([0-9\.,]*)/;
my $time = $1;
$time =~ s/,/./g;

chomp(my $tt = `grep rl:TerminationTime $work_dir/$job_id.properties`);
$tt =~ m|>([^<]+)<|;
$tt = int((str2time($1) - time() + 60)/(60*60*24));

my $desc = '';
chomp(my $app = `grep ApplicationName $work_dir/$job_id.properties`);
if ($app =~ m|>([^<]+)<|) {
  $desc = $1;
  chomp(my $version = `grep 'module load' $work_dir/$job_id.properties | head -1`);
  $version =~ s|module load [^/]+/?||;
  $desc .= " $version" if ($version);
  $desc .= " test job";
} else {
  $desc = "Test job";
}

my $additionalInfo = "'Termination Time: $tt day".($tt != 1 ? 's' : '')."\n";
if (`grep -c 'SUCCESSFUL exit code: 0' $TMPFILE` == 0) {
  my $returnLine = `grep 'SUCCESSFUL exit code: [0-9]\\+\$' $TMPFILE`; 
  $returnLine =~ m#: ([0-9]+)#;
  if ($1 ne "0") {
    $additionalInfo .= "Exit status: $1 (configured to be accepted)\n";
  }
}
$additionalInfo .= "'";

if ($time) {
  exit_plugin("OK: $desc succeded, time elapsed: $time sec. | ExecutionTime=${time}s", $additionalInfo);
} else {
  exit_plugin("OK: $desc succeded\n(Time profile data are not available!)", $additionalInfo);
}
