#!/usr/bin/perl
# Author: Mariusz Strzelecki <szczeles@mat.umk.pl>
# Don't use Nagios embedded perl
# nagios: -epn

use strict;
our $plugin_version = '2.0';

# Let $Bin store absolute path of script directory
use FindBin '$Bin';
use Date::Parse;

# Look for commons.pm in following directories
use lib "$Bin";
use lib "$Bin/..";

# Load common functions
use commons qw(%config &exit_plugin &setup_plugin &message &check_conditions &create_temp_file &check_config &run &check_file_existence);

# Load configuration and check for commandline arguments
setup_plugin("$Bin/check_activemq.README", $plugin_version);
$config{'ACTIVEMQ_QUEUE'} = '' if (check_config('ACTIVEMQ_QUEUE', 1) == 1);
check_config('ACTIVEMQ_URL, ACTIVEMQ_USERNAME, ACTIVEMQ_PASSWORD, ACTIVEMQ_QUEUE');

my $script_location = "$Bin/CheckAMQStatistics.class";
check_file_existence($script_location);

# Execute something
our $TMPFILE = create_temp_file("check_activemq$$");
run("java", "-cp $Bin:$Bin/activemq-all-5.6.0.jar:$Bin/commons-logging-1.1.1.jar CheckAMQStatistics $config{'UCC_CONFIG'} $config{'ACTIVEMQ_URL'} $config{'ACTIVEMQ_USERNAME'} $config{'ACTIVEMQ_PASSWORD'} $config{'ACTIVEMQ_QUEUE'}", $TMPFILE, 1);

our %stats = ();
my @output = split("\n", `cat $TMPFILE`);
for my $line (@output) {
  next unless ($line =~ m#^([^=]*)=(.*)#);
  my $key = $1;
  my $value = $2;
  $stats{$key} = $value;
}

my $SIZE_WARNING = 10000;
my $SIZE_CRITICAL = 1000000;
my $MEM_USAGE_WARNING = 80;
my $MEM_USAGE_CRITICAL = 95;
my $CERT_DAYS_WARNING = 14;
my $CERT_DAYS_CRITICAL = 3;

my $daysToExpire = int(((str2time($stats{'certificateExpireDate'}) - time()) / 60 / 60 / 24)) if defined $stats{'certificateExpireDate'};

our @conditions = (
  {
    'test'       => 'not defined $main::stats{"ssl"}',
    'output'     => "CRITICAL: Unable to connect broker at $config{'ACTIVEMQ_URL'}",
    'show_debug' => '`echo "See check command output to determine failure:"; cat $main::TMPFILE`'
  },
  {
    'test'       => 'not defined $main::stats{"storePercentUsage"}',
    'output'     => "CRITICAL: Missing store usage info"
  },
  {
    'test'       => 'not defined $main::stats{"memoryPercentUsage"}',
    'output'     => "CRITICAL: Missing memory usage info"
  },
  {
    'test'       => 'not defined $main::stats{"certificateExpireDate"}',
    'output'     => "CRITICAL: Missing certificate expire date"
  },
  {
    'test'       => 'not defined $main::stats{"size"}',
    'output'     => "CRITICAL: Missing queue size"
  },
  {
    'test'       => "$main::stats{'size'} >= $SIZE_CRITICAL",
    'output'     => "CRITICAL: Queue size exceedes critical level ($SIZE_CRITICAL): $main::stats{'size'}"
  },
  {
    'test'       => "$main::stats{'size'} >= $SIZE_WARNING",
    'output'     => "WARNING: Queue size exceedes warning level ($SIZE_WARNING): $main::stats{'size'}"
  },
  {
    'test'       => "$main::stats{'memoryPercentUsage'} >= $MEM_USAGE_CRITICAL",
    'output'     => "CRITICAL: Memory usage exceedes critical level ($MEM_USAGE_CRITICAL%): $main::stats{'memoryPercentUsage'}%"
  },
  {
    'test'       => "$main::stats{'memoryPercentUsage'} >= $MEM_USAGE_WARNING",
    'output'     => "WARNING: Memory usage exceedes warning level ($MEM_USAGE_WARNING%): $main::stats{'memoryPercentUsage'}%"
  },
  {
    'test'       => "$main::stats{'storePercentUsage'} >= $MEM_USAGE_CRITICAL",
    'output'     => "CRITICAL: Store usage exceedes critical level ($MEM_USAGE_CRITICAL%): $main::stats{'storePercentUsage'}%"
  },
  {
    'test'       => "$main::stats{'storePercentUsage'} >= $MEM_USAGE_WARNING",
    'output'     => "WARNING: Store usage exceedes warning level ($MEM_USAGE_WARNING%): $main::stats{'storePercentUsage'}%"
  },
  {
    'test'       => "$daysToExpire <= 0",
    'output'     => "CRITICAL: SSL certificate expired"
  },
  {
    'test'       => "$daysToExpire <= $CERT_DAYS_CRITICAL",
    'output'     => "CRITICAL: SSL certificate expires in $daysToExpire days"
  },
  {
    'test'       => "$daysToExpire <= $CERT_DAYS_WARNING",
    'output'     => "WARNING: SSL certificate expires in $daysToExpire days"
  },
  {
    'test'       => 1,
    'output'     => "OK: ActiveMQ Broker at $config{'ACTIVEMQ_URL'} works | certificateExpiresIn=$daysToExpire memUsage=$main::stats{'storePercentUsage'}% storeUsage=$main::stats{'storePercentUsage'}% queueSize=$main::stats{'size'}"
  },
);
check_conditions(\@main::conditions);
