'''
    Utility methods allowing to reconfigure UNICORE components easily.
    Low level interface, rather not used directly in configurators.

    @organization: ICM UW
    @author: K.Benedyczak @: golbi@icm.edu.pl
    @author: R.Kluszczynski @: klusi@icm.edu.pl
'''
import tempfile, os
import sys
import stat
import shutil
import re

from datetime import datetime
from lxml import etree


def info(options, msg):
    ''' Prints information message to stdout '''
    if (options.quiet):
        return
    print options.indent + msg

def error(msg):
    ''' Prints error message to stderr '''
    print >> sys.stderr, "ERROR " + msg

def fatal(msg):
    ''' Prints fatal message to stderr and exits '''
    print >> sys.stderr, "FATAL " + msg
    sys.exit(1)


def getConfigRoot(options, component):
    """Returns a configuration root directory, depending on installation type"""
    if options.manualConfigurationRoot != None:
        if os.path.isdir(options.manualConfigurationRoot + '/' + component):
            return options.manualConfigurationRoot + '/' + component + '/'
        return options.manualConfigurationRoot

    if options.systemInstall:
        return options.configsRoot + component + '/'
    else:
        return options.configsRoot + component + '/conf/'


def getFile(options, component, cfgFile):
    """Returns a configuration file path, depending on installation type"""
    if re.match('^/|^[a-zA-Z]:', cfgFile):
        return cfgFile
    else:
        return getConfigRoot(options, component) + cfgFile


def copyPreserve(options, src, dest, moveSrc, preserveSrc):
    '''
        Copy a file with respect to permissions. 
    
        If '--dry-run' is used leave it immediately. 
    '''
    if options.dry == True:
        return
    if preserveSrc == True:
        st = os.stat(src)
    else:
        if os.path.isfile(dest):
            st = os.stat(dest)
        else:
            st = None
    if moveSrc == True:
        shutil.move(src, dest)
        if st != None:
            __copyFileMetadata(dest, st)
    else:
        shutil.copy(src, dest)
        __copyFileMetadata(dest, st)

def __copyFileMetadata(destFilename, srcStatData):
    ''' Copy permissions and owner of a file '''
    os.chown(destFilename, srcStatData[stat.ST_UID], srcStatData[stat.ST_GID])
    os.chmod(destFilename, srcStatData[stat.ST_MODE])


def backupFile(options, filename):
    '''
        Backups a file if there is no existing one. 
        
        If '--always-backup' is used stores it with a timestamp. 
        If '--no-backup' is used leave it immediately. 
    '''
    if options.backup == False:
        return
    if not os.path.isfile(filename):
        return
    originFile = filename + '_origin'
    #if not exists, make a copy of the original config file
    if not os.path.isfile(originFile):
        info(options, "Making backup of %s to %s" % (filename, originFile))
        copyPreserve(options, filename, originFile, False, True)
    else:
        if options.backupAlways:
            timestampName = filename + options.backupsuffix
            info(options, "File '%s' exists. Saving backup as %s" % (originFile, timestampName))
            copyPreserve(options, filename, timestampName, False, True)
        else:
            info(options, "File '%s' exists. Skipping backup..." % originFile)


def loadXMLDocumentFromFile(options, filename, stripComments = False):
    '''
        Loads XML document from file and return it.
        
        @return: XML document (etree.XMLElementTree object)
    '''
    info(options, 'File ' + filename)
    doc = etree.parse(filename, __getXMLParser(options, stripComments))
    return doc


def writeXMLDocument(options, xmlDoc, filename = None):
    '''
        Writes XML document to file descriptor (default is standard output).
    '''
    if filename == None:
        fd = sys.stdout
    else:
        fh, tmpFilename = tempfile.mkstemp()
        fd = os.fdopen(fh, 'w')
    xmlDoc.write(fd, encoding = 'utf-8', method = 'xml', pretty_print = True, xml_declaration = True)
    if filename != None:
        fd.close()
        copyPreserve(options, tmpFilename, filename, True, False)
        if options.dry == True:  os.unlink(tmpFilename)


def setXMLElementAttribute(options, xmlDoc, xPath, name, value, nsPrefixMap = {}):
    '''
        Sets one attribute value of a XML document's tag.
        
        @param     options:    global options
        @param         doc:    XML document (etree.XMLElementTree object)  
        @param       xPath:    XPath determining element
        @param        name:    attribute's name
        @param       value:    attribute's value
        @param nsPrefixMap:    prefix map of namespaces used in XPath
    '''
    __setXMLElementAttributes(options, xmlDoc, xPath, { name : value }, nsPrefixMap)

def setXMLElementAttributes(options, xmlDoc, xPath, newAttrsValues = {}, nsPrefixMap = {}):
    __setXMLElementAttributes(options, xmlDoc, xPath, newAttrsValues, nsPrefixMap)


def __setXMLElementAttributes(options, xmlDoc, elementXPath, newAttrsValues, nsPrefixMap = {}):
    '''
        Sets attributes values of a XML document's tag or add them if not exists.
        
        @param        options:    global options
        @param            doc:    XML document (etree.XMLElementTree object)  
        @param          xPath:    XPath determining element
        @param newAttrsValues:    dictionary containing attributes names and values
        @param    nsPrefixMap:    prefix map of namespaces used in XPath
    '''
    result = xmlDoc.xpath(elementXPath, namespaces = nsPrefixMap)
    if len(result) > 0:
        if len(result) > 1:
            info(options, "More then 1 element under '" + elementXPath + "'. Changing only the first one.")
        element = result[0]
        for attribute in newAttrsValues:
            oldVal = element.get(attribute)
            newVal = newAttrsValues[attribute]
            if oldVal != newVal:
                element.set(attribute, newVal)
                if oldVal == None:
                    oldVal = 'NONE'
                info(options, " - setting " + elementXPath + " attribute '" + attribute + "' to '" + newVal + "' (old value: '" + oldVal + "')")
    else:
        error("No elements under '" + elementXPath + "'.")

def checkXPathExistence(options, xmlDoc, elementXPath, nsPrefixMap = {}):
    '''
        Check if XPath refers to at least one element in XML document.
        
        @param      options:    global options
        @param          doc:    XML document (etree.XMLElementTree object)  
        @param elementXPath:    XPath string
        @param  nsPrefixMap:    prefix map of namespaces used in XPath
        
        @return: True if exists at least one element pointed by XPath expression, otherwise False     
    '''
    return len(xmlDoc.xpath(elementXPath, namespaces = nsPrefixMap)) > 0


def addAdditionalXMLs(options, xmlDoc, stringElements2Add, addAfterXPath = None, nsPrefixMap = {}):
    '''
        Adds additional XML elements (put by strings) to the document.  
        
        @param            options:    global options
        @param           filename:    name of file
        @param stringElements2Add:    list of strings containing parts of XML 
        @param      addAfterXPath:    XPath string refers to an element after which new elements should be added 
        @param        nsPrefixMap:    prefix map of namespaces used in XPath
    '''
    if addAfterXPath:
        stringElements2Add = reversed(stringElements2Add)
    for newXMLElement in stringElements2Add:
        __addAdditionalXML(options, xmlDoc, newXMLElement, addAfterXPath, nsPrefixMap)

def addAdditionalXML(options, xmlDoc, stringElement2Add, addAfterXPath = None, nsPrefixMap = {}):
    __addAdditionalXML(options, xmlDoc, stringElement2Add, addAfterXPath, nsPrefixMap)

def __addAdditionalXML(options, doc, stringElement2Add, addAfterXPath = None, nsPrefixMap = {}):
    '''
        Adds additional XML element to the document.  
        
        @param           options:    global options
        @param               doc:    XML document (etree.XMLElementTree object)  
        @param stringElement2Add:    string containing part of XML 
        @param     addAfterXPath:    XPath string refers to an element after which stringElement2Add should be added
        @param       nsPrefixMap:    prefix map of namespaces used in XPath
    '''
    matcher = re.match("<!--(.*)-->", stringElement2Add)
    if matcher == None:
        el = etree.fromstring(stringElement2Add, __getXMLParser(options))
    else:
        strip = matcher.group(1)
        el = etree.Comment(strip)
        
    addAfterElement = None
    if addAfterXPath != None  and  addAfterXPath != '':
        result = doc.xpath(addAfterXPath, namespaces = nsPrefixMap)
        if len(result) > 0:
            addAfterElement = result[0]
    if addAfterElement == None:
        doc.getroot().append(el)
    else:
        addAfterElement.addnext(el)


def removeXPathElements(options, xmlDoc, xPathExpression, nsPrefixMap = {}):
    '''
        Removes elements pointed by XPath expression. 
        
        @param         options:    global options
        @param          xmlDoc:    XML document (etree.XMLElementTree object)  
        @param xPathExpression:    XPath expression 
        @param     nsPrefixMap:    prefix map of namespaces used in XPath  
    '''
    elements = xmlDoc.xpath(xPathExpression, namespaces = nsPrefixMap)
    if len(elements) > 0:
        parent = elements[0].getparent()
        if parent != None:
            for e in elements:
                parent.remove(e)
                info(options, " - removed element at '" + xPathExpression + "'")
        else:
            info(options, " - can not remove XML root element")
    else:
        info(options, " - nothing to remove at '" + xPathExpression + "'")


def __getXMLParser(options, noComments = False):
    '''
        Gets XMLParser object for parsing XML documents. 
        
        @param      options:    global options
        
        @return: XML parser (etree.XMLParser object)     
    '''
    if options.parser == None:
        options.parser = etree.XMLParser(remove_blank_text = True, remove_comments = noComments)
    return options.parser


def setJavaProperties(options, filename, propsDict):
    processJavaProperties(options, filename, propsDict, "value")

def processJavaProperties(options, filename, propsDict, mode):
    '''
        Sets property, if it finds commented one it replaces the first occurrence.
        The rest, if detected, will be commented out. The mode of operation
        can be changed to comment out properties or to change keys.
        
        @param    options:    global options
        @param   filename:    filename to change
        @param  propsDict:    dictionary with java properties to set
        @param    mode:      for 'value' -> updates property value
                             for 'key' -> updates property key
                             for 'comment' -> comments the property   
    
        @todo: handle keys with spaces
        @todo: handle comments after key & value pair
    '''
    info(options, "Updating properties file '%s'" % filename)
    fd, tmpFilename = tempfile.mkstemp()
    tmpFile = os.fdopen(fd, 'w')
    propertiesFile = file(filename, "r")
    previousLines = ''
    for line in propertiesFile:
        strippedLine = previousLines + line.strip()
        if len(strippedLine) > 0:
            #check a multiline
            if strippedLine[-1] == '\\':
                previousLines = strippedLine[:-1]
                continue
            else:
                previousLines = ''
            #strip comment characters at the begging if exists
            comment = re.match('[#!]+', strippedLine)
            if comment != None:
                propDef = strippedLine[comment.end():].strip()
            else:
                propDef = strippedLine
            name, value = __getNameAndValueOfPropertyLine(propDef)
            if name in propsDict:
                newValue = propsDict[name]                
                if mode == "value":
                    #check if it was already used 
                    if newValue != None:
                        if value != newValue:
                            info(options, " - setting '" + name + "' to '" + newValue + "' (old value: '" + value + "')")
                            strippedLine = name + '=' + newValue
                        propsDict[name] = None
                    else:
                        if not re.match('[#!]', strippedLine):
                            strippedLine = '# ' + strippedLine
                elif mode == "key":
                    info(options, " - converting '" + name + "' to '" + newValue + "'")
                    if not re.match('[#!]', strippedLine):
                        strippedLine = newValue + '=' + value
                    else:
                        strippedLine = "#" + newValue + '=' + value
                elif mode == "comment":
                    if not re.match('[#!]', strippedLine):
                        info(options, " - commenting the " + name + "=" + value + " entry " + newValue)
                        strippedLine = "#"+strippedLine
                    if newValue != "":
                        strippedLine = "# !UNICORE CONFIGURATION UPDATE MESSAGE! "+newValue+"\n"+strippedLine
                        info(options, "WARNING " + newValue)

                
        tmpFile.write(strippedLine + '\n')
    #closing original file
    if previousLines != '':
        tmpFile.write("# " + previousLines + '\n')
    propertiesFile.close()

    if mode == "value":
        #adding remaining variables
        firstOne = True
        for name in sorted(propsDict.keys()):
            newValue = propsDict[name]
            if newValue != None:
                if firstOne:
                    tmpFile.write('\n## Added new properties by configurator (' + str(datetime.now()) + ')\n')
                    firstOne = False
                tmpFile.write(name + '=' + newValue + '\n')
                info(options, " - setting new property '" + name + "' to '" + newValue + "'")
    
    #closing temporary file
    tmpFile.close()

    copyPreserve(options, tmpFilename, filename, True, False)
    if options.dry == True:  os.unlink(tmpFilename)


def getJavaProperty(filename, name):
    if not os.path.isfile(filename):
        return None;
    propertiesFile = file(filename, "r")
    previousLines = ''
    for line in propertiesFile:
        strippedLine = previousLines + line.strip()
        if len(strippedLine) > 0:
            #check a multiline
            if strippedLine[-1] == '\\':
                previousLines = strippedLine[:-1]
                continue
            else:
                previousLines = ''
            #strip comment characters at the begging if exists
            comment = re.match('[#!]+', strippedLine)
            if comment != None:
                continue
            else:
                propDef = strippedLine
            key, value = __getNameAndValueOfPropertyLine(propDef)
            if (key == name):
                propertiesFile.close()
                return value
    propertiesFile.close()
    return None
    
def getJavaPropertyKeys(filename):
    R = []
    if not os.path.isfile(filename):
        return R;
    propertiesFile = file(filename, "r")
    previousLines = ''
    for line in propertiesFile:
        strippedLine = previousLines + line.strip()
        if len(strippedLine) > 0:
            #strip comment characters at the begging if exists
            comment = re.match('[#!]+', strippedLine)
            if comment != None:
                continue
            else:
                propDef = strippedLine

            #check a multiline
            if strippedLine[-1] == '\\':
                previousLines = strippedLine[:-1]
                continue
            else:
                previousLines = ''
            key, value = __getNameAndValueOfPropertyLine(propDef)
            R.append(key)
    propertiesFile.close()
    return R
    

def setShellVariables(options, filename, varsDict):
    '''
        Sets config variable, if it finds commented one it replaces the first occurrence.
        
        @param    options:    global options
        @param   filename:    filename to change
        @param  propsDict:    dictionary with shell variables to set    
        
        @todo: handle mutlilines (when line ends wiht backslash)
    '''
    info(options, "Updating shell variables file '%s'" % filename)
    fd, tmpFilename = tempfile.mkstemp()
    tmpFile = os.fdopen(fd, 'w')
    configFile = file(filename, 'r')
    for line in configFile:
        spacesPrefix = ''
        matchPrefixSpaces = re.match('[ \t]*', line)
        if matchPrefixSpaces != None:
            spacesPrefix = line[0:matchPrefixSpaces.end()]
        strippedLine = line.strip()
        if len(strippedLine) > 0:
            ''' strip comment characters at the begging it exists '''
            comment = re.match('#+', strippedLine)
            if comment != None:
                propDef = strippedLine[comment.end():].strip()
            else:
                propDef = strippedLine
            name, value = __getNameAndValueOfPropertyLine(propDef)
            if name in varsDict:
                newValue = varsDict[name]
                ''' check if it was already used '''
                if newValue != None:
                    if value != newValue:
                        info(options, " - setting '" + name + "' to '" + newValue + "' (old value: '" + value + "')")
                        strippedLine = name + '=' + newValue
                    varsDict[name] = None
                else:
                    if strippedLine[0] != '#':
                        strippedLine = '# ' + strippedLine
        tmpFile.write(spacesPrefix + strippedLine + '\n')
    ''' closing original file '''
    configFile.close()

    ''' adding remaining variables '''
    firstOne = True
    for name in varsDict.iterkeys():
        newValue = varsDict[name]
        if newValue != None:
            if firstOne:
                tmpFile.write('\n## Added new variables by configurator (' + str(datetime.now()) + ')\n')
                firstOne = False
            tmpFile.write(name + '=' + newValue + '\n')
            info(options, " - setting new property '" + name + "' to '" + newValue + "'")
    ''' closing temporary file '''
    tmpFile.close()

    copyPreserve(options, tmpFilename, filename, True, False)
    if options.dry == True:  os.unlink(tmpFilename)



def __getNameAndValueOfPropertyLine(strippedLine):
    '''
        Gets name and value from stripped line of properties/shell config file.
        
        Based on: 
            http://www.linuxtopia.org/online_books/programming_books/python_programming/python_ch34s04.html
    '''
    keyDelimeterChars = ':='
#    keyDelimeterChars = ':= '
#    if strippedLine.find(':') >= 0  or  strippedLine.find('=') >= 0:
#        keyDelimeterChars = ':='
    punctuation = [ strippedLine.find(c) for c in keyDelimeterChars ] + [ len(strippedLine) ]
    found = min([ pos for pos in punctuation if pos != -1 ])
    name = strippedLine[:found].rstrip()
    value = strippedLine[found:].lstrip(keyDelimeterChars).rstrip()
    return (name, value)




def appendLinesIfNotAlreadyExist(options, filename, linesList):
    '''
        Appends a line to a file if not exists so far.
    '''
    info(options, "Appending lines to file '%s'" % filename)
    linesSet = set(linesList)
    configFile = file(filename, 'r')
    for line in configFile:
        strippedLine = line.strip()
        if strippedLine in linesSet:
            linesSet.remove(strippedLine)
            info(options, " - line already exists: " + strippedLine)
    configFile.close()

    ''' adding left lines '''
    if len(linesSet) > 0:
        try:
            if options.dry:
                fd = None
            else:
                fd = open(filename, 'a')
            firstOne = True
            for line in linesList:
                if line in linesSet:
                    if fd != None:
                        if firstOne :
                            fd.write('\n## Added new lines by configurator (' + str(datetime.now()) + ')\n')
                            firstOne = False
                        fd.write(line + '\n')
                    info(options, " - line appended: " + line)
            if fd != None:  fd.close()
        except IOError, (errno, strerror):
            error("Processing configuration " + file + ": " + "I/O error(%s): %s" % (errno, strerror))



def removeLinesIfExist(options, filename, linesList):
    '''
        Removes existing lines.
    '''
    info(options, "Removing lines from file file '%s'" % filename)
    linesSet = set(linesList)
    fd, tmpFilename = tempfile.mkstemp()
    tmpFile = os.fdopen(fd, 'w')
    configFile = file(filename, 'r')
    for line in configFile:
        rightStrippedLine = line.rstrip()
        if not rightStrippedLine in linesSet:
            tmpFile.write(rightStrippedLine + '\n')
        else:
            info(options, ' - removing line:' + rightStrippedLine)
    configFile.close()
    tmpFile.close()

    copyPreserve(options, tmpFilename, filename, True, False)
    if options.dry == True:  os.unlink(tmpFilename)



def replaceFileLines(options, filename, linesDict):
    '''
        Replaces a line in a file.
    '''
    info(options, "Replacing lines in file '" + filename + "'")
    fd, tmpFilename = tempfile.mkstemp()
    tmpFile = os.fdopen(fd, 'w')
    configFile = file(filename, 'r')
    for line in configFile:
        rightStrippedLine = line.rstrip()
        if rightStrippedLine in linesDict:
            info(options, " - replacing line '%s' with '%s'" % (rightStrippedLine, linesDict[rightStrippedLine]))
            tmpFile.write(linesDict[rightStrippedLine] + '\n')
        else:
            tmpFile.write(rightStrippedLine + '\n')
    ''' closing original file '''
    configFile.close()
    ''' closing temporary file '''
    tmpFile.close()

    copyPreserve(options, tmpFilename, filename, True, False)
    if options.dry == True:  os.unlink(tmpFilename)

