'''
    Utility methods allowing to reconfigure UNICORE components easily.
    Low level interface, rather not used directly in configurators.

    @organization: ICM UW
'''
import ucfg_utils

from lxml import etree
from ucfg_utils import info


__engineNS = 'http://www.fz-juelich.de/unicore/xnjs/engine'


def setXNJSEngineProperty(options, xmlDoc, name, value):
    '''
        Sets engine property in XNJS configuration file.
        
        @param     options:    global options
        @param    xmlDoc:    XML document (etree.XMLElementTree object)  
        @param        name:    property's name
        @param       value:    property's value
    '''
    propertyXPath = "/eng:EngineConfiguration/eng:Properties/eng:Property[@name='" + name + "']"
    nsPrefixMap = { 'eng' : __engineNS }
    if not ucfg_utils.checkXPathExistence(options, xmlDoc, propertyXPath, nsPrefixMap):
        propEl = etree.Element('{' + __engineNS + '}Property')
        propEl.set('name', name)
        propEl.set('value', value)
        engPropsEl = xmlDoc.xpath('/eng:EngineConfiguration/eng:Properties', namespaces = nsPrefixMap)
        assert len(engPropsEl) == 1
        engPropsEl[0].append(propEl)
    else:
        ucfg_utils.setXMLElementAttributes(options, xmlDoc, propertyXPath, { 'value' : value }, nsPrefixMap)


def appendToJSDLProcessingChain(options, xmlDoc, processorClass):
    '''
        Appends processor to JSDL processing chain in XNJS configuration file.
        
        @param          options:    global options
        @param         xmlDoc:    XML document (etree.XMLElementTree object)  
        @param processorClass:    name of processor class to append 
    '''
    processorXPath = "/eng:EngineConfiguration/eng:ProcessingChain/eng:Processor[text()='" + processorClass + "']"
    nsPrefixMap = { 'eng' : __engineNS }
    if not ucfg_utils.checkXPathExistence(options, xmlDoc, processorXPath, nsPrefixMap):
        element = etree.Element('{' + __engineNS + '}Processor')
        element.text = processorClass
        parents = xmlDoc.xpath('/eng:EngineConfiguration/eng:ProcessingChain[@actionType="JSDL"]', namespaces = nsPrefixMap)
        assert len(parents) == 1
        info(options, " - appending processor '" + processorClass + "' to processing chain")
        parents[0].append(element)
    else:
        info(options, " - processor '" + processorClass + "' already exists")


def setWSRFLiteProperty(options, xmlDoc, name, value, mode):
    '''
        Sets property element in WSRFLite configuration file.
        
        @param     options:    global options
        @param      xmlDoc:    XML document (etree.XMLElementTree object)  
        @param        name:    property's name
        @param       value:    property's value
        @param        mode:    'key' - updates key, 'value' - updates value, 'comment' - comments with extra info.
    '''
    propertyXPath = "/services/property[@name='" + name + "']"
    if not ucfg_utils.checkXPathExistence(options, xmlDoc, propertyXPath):
        if mode != 'value': #when no such element we don't have to neither comment nor change key
            return;
        if value != None:
            propertyV = etree.Element('property')
            propertyV.set('name', name)
            propertyV.set('value', value)
            wsrfProps = xmlDoc.xpath('/services')
            assert len(wsrfProps) == 1
            info(options, " - appending property '" + name + "' with value '" + value + "'")
            wsrfProps[0].append(propertyV)
        else:
            info(options, " - property value is None, skipping '" + propertyXPath + "'...")
    else:
        if mode == 'value':
            ucfg_utils.setXMLElementAttributes(options, xmlDoc, propertyXPath, { 'value' : value })
        elif mode == 'key':
            ucfg_utils.setXMLElementAttributes(options, xmlDoc, propertyXPath, { 'name' : value })
        elif mode == 'comment':
            xpath = "/services/property[@name='" + name + "']"
            info(options, "WARNING " + value)
            ucfg_utils.addAdditionalXML(options, xmlDoc, "<!-- !UNICORE CONFIGURATION UPDATE MESSAGE! " + value + " -->", xpath);
            ucfg_utils.removeXPathElements(options, xmlDoc, xpath)


def toggleWSRFLiteService(options, xmlDoc, serviceName, enabled = True):
    '''
        Sets enabled attribute in WSRFLite configuration file.
        
        @param        options:    global options
        @param         xmlDoc:    XML document (etree.XMLElementTree object)  
        @param  serviceName:    service name 
        @param        enabled:    if service is enabled (default: True)
    '''
    serviceXPath = "/services/service[@name='" + serviceName + "']"
    if ucfg_utils.checkXPathExistence(options, xmlDoc, serviceXPath):
        ucfg_utils.setXMLElementAttributes(options, xmlDoc, serviceXPath, { 'enabled' : enabled })
    else:
        info(options, " - service '" + serviceName + "' not found. Skipping it...")


